/*
 *	TICKR - GTK-based Feed Reader - Copyright (C) Emmanuel Thomas-Maurin 2009-2011
 *	<manutm007@gmail.com>
 *
 * 	This program is free software: you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation, either version 3 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "tickr.h"

#if USE_GUI
#ifdef G_OS_WIN32
extern FILE	*stdout_fp, *stderr_fp;
#endif
static char	*opml_str_global;

static gint esc_key_pressed(GtkWidget *dialog2, GdkEventKey *event, gpointer unused)
{
	unused = unused;
	if (event->keyval == GDK_Escape) {
		gtk_dialog_response(GTK_DIALOG(dialog2), GTK_RESPONSE_CANCEL);
		return TRUE;
	} else
		return FALSE;
}

static void force_quit_dialog(GtkWidget *dialog2)
{
	gtk_dialog_response(GTK_DIALOG(dialog2), GTK_RESPONSE_CANCEL);
}

/*
 * importing
 */

void import_opml_file()
{
	char		*file_name, *listfname;
	FILE		*fp;
	/* same as stuff in news_rsswin.c */
	char		url_array[NURLMAX][FILE_NAME_MAXLEN + 1];
	char		*p_url[NURLMAX + 1];
	int		suspend_rq_bak, i;

	if ((file_name = pick_opml_file())[0] != '\0') {
		listfname = (char *)get_datafile_full_name_from_name(URLLIST);
		if ((fp = g_fopen(listfname, "rb")) != NULL) {
			fclose(fp);
			if (question_win("Imported URLs will be merged with currently saved ones. "
					"Continue ?") == NO)
				return;
		}
		suspend_rq_bak = get_ticker_env()->suspend_rq;
		get_ticker_env()->suspend_rq = TRUE;
		win_with_spinner(WIN_WITH_SPINNER_OPEN,
			"Importing data from OPML file, please wait...\n"
			"(updating may take some time)");
		opml_str_global = l_str_new(NULL);
		if (parse_opml_xml_file(file_name) == OK) {
			if ((fp = g_fopen(listfname, "ab")) != NULL) {
				fprintf(fp, "\n%s\n", opml_str_global);
				fclose(fp);
				for (i = 0; i < NURLMAX; i++)
					p_url[i] = url_array[i];
				p_url[NURLMAX] = NULL;
				if (load_url_list(p_url) == OK) {
					sort_url_list(p_url, 1);
					save_url_list(p_url);
					load_selected_url_list(init_selected_url_list());
					l_str_free(opml_str_global);
					win_with_spinner(WIN_WITH_SPINNER_CLOSE, NULL);
					get_ticker_env()->suspend_rq = suspend_rq_bak;
					info_win(APP_NAME, "\nFeed list has been imported\n",
						INFO, FALSE);
					return;
				}
			}
		}
		l_str_free(opml_str_global);
		win_with_spinner(WIN_WITH_SPINNER_CLOSE, NULL);
		get_ticker_env()->suspend_rq = suspend_rq_bak;
	}
}

char *pick_opml_file()
{
	GtkWidget	*dialog;
	GtkFileFilter	*filter;
	char		*file_name;
	static char	file_name2[FILE_NAME_MAXLEN + 1];

	gtk_window_set_keep_above(GTK_WINDOW(get_ticker_env()->win), FALSE);

	dialog = gtk_file_chooser_dialog_new("Import Feed List (OPML)", GTK_WINDOW(get_ticker_env()->win),
			GTK_FILE_CHOOSER_ACTION_OPEN,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			GTK_STOCK_OK, GTK_RESPONSE_OK,
			NULL);

	set_news_icon_to_dialog(GTK_WINDOW(dialog));
	gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);

	g_signal_connect(G_OBJECT(dialog), "key-press-event", G_CALLBACK(esc_key_pressed), NULL);
	g_signal_connect(G_OBJECT(dialog), "delete_event", G_CALLBACK(force_quit_dialog), NULL);

	gtk_widget_show_all(dialog);

	filter = gtk_file_filter_new();
	gtk_file_filter_add_pattern(filter, "*.opml");
	gtk_file_filter_add_pattern(filter, "*.xml");
	gtk_file_chooser_set_filter(GTK_FILE_CHOOSER(dialog), filter);

	file_name2[0] = '\0';
	if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK) {
		if ((file_name = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog))) != NULL) {
			str_n_cpy(file_name2, file_name, FILE_NAME_MAXLEN);
			g_free(file_name);
		}
	}
	gtk_widget_destroy(dialog);
	check_main_win_always_on_top();
	return file_name2;
}

int parse_opml_xml_file(const char *file_name)
{
	xmlDoc		*doc = NULL;
	xmlNode		*cur_node = NULL, *cur_node_start = NULL;
	int		body_element = NO, outline_element = NO;

	fprintf(STD_OUT, "Parsing OPML file... ");
	if ((doc = xmlParseFile(file_name)) == NULL) {
		warning("Can't parse XML file:", xmlGetLastError()->message,
			"", "", FALSE);
		return XML_UNPARSABLE;
	} else if ((cur_node = xmlDocGetRootElement(doc)) == NULL) {
		warning("Empty XML document:", file_name, "", "", FALSE);
		xmlFreeDoc(doc);
		return XML_EMPTY;
	} else if (xmlStrcmp(cur_node->name, (const xmlChar *)"opml") != 0) {
		warning("Not an OPML document:", file_name, "", "", FALSE);
		xmlFreeDoc(doc);
		return OPML_ERROR;
	}
	cur_node_start = cur_node;
	for (cur_node = cur_node->children; cur_node != NULL; cur_node = cur_node->next) {
		if (xmlStrcmp(cur_node->name, (const xmlChar *)"body") == 0) {
			body_element = YES;
			break;
		}
	}
	if (body_element == YES) {
		for (cur_node = cur_node->children; cur_node != NULL; cur_node = cur_node->next) {
			if (xmlStrcmp(cur_node->name, (const xmlChar *)"outline") == 0) {
				outline_element = YES;
				break;
			}
		}
	}
	if (body_element == YES && outline_element == YES) {
		get_opml_selected_element(cur_node_start, "outline", "type");
		fprintf(STD_OUT, "done\n");
		xmlFreeDoc(doc);
		return OK;
	} else {
		warning("Couldn't find all required OPML elements in:",
			file_name, "", "", FALSE);
		xmlFreeDoc(doc);
		return OPML_ERROR;
	}
}

void get_opml_selected_element(xmlNode *some_node, const char *selected_element, const char *selected_attribute)
{
	xmlNode	*cur_node;
	xmlChar	*str1, *str2;
	gchar	*unesc_str;
	char	feed_title[FEED_TITLE_MAXLEN + 1];

	for (cur_node = some_node; cur_node != NULL; cur_node = cur_node->next) {
		if (xmlStrcmp(cur_node->name, (const xmlChar *)selected_element) == 0) {
			/* need to get attribute name and value
			 * then use only those whose value = "rss" for name = 'type'
			 * then get value for name = 'xmlUrl' -> rss feed url
			 */
			if ((str1 = xmlGetProp(cur_node, (xmlChar *)selected_attribute)) != NULL) {
				if (xmlStrcmp(str1, (xmlChar *)"rss") == 0) {
					if ((str2 = xmlGetProp(cur_node, (xmlChar *)"xmlUrl")) != NULL) {
						/* we add 1st one UNSELECTED_URL_CHAR */
						opml_str_global = l_str_cat(opml_str_global, UNSELECTED_URL_STR);
						unesc_str = g_uri_unescape_string((const char *)str2, NULL);
						opml_str_global = l_str_cat(opml_str_global, (const char*)unesc_str);
						/* get feed title from feed url - ok like that? */
						if (get_feed_info((const char *)unesc_str, feed_title, NULL, NULL) == OK) {
							if (feed_title[0] != '\0') {
								opml_str_global = l_str_cat(opml_str_global, TITLE_TAG_STR);
								opml_str_global = l_str_cat(opml_str_global, feed_title);
							}
						}
						opml_str_global = l_str_cat(opml_str_global, "\n");
						g_free(unesc_str);
						xmlFree(str2);
					}
				}
				xmlFree(str1);
			}
		}
		get_opml_selected_element(cur_node->children, selected_element, selected_attribute);
	}
}

/*
 * exporting
 */
void export_opml_file()
{
	/* same as stuff in news_rsswin.c */
	char	url_array[NURLMAX + 1][FILE_NAME_MAXLEN + 1];
	char	*p_url[NURLMAX + 1];
	char	*str;
	int	i;

	for (i = 0; i < NURLMAX - 1; i++)
		p_url[i] = url_array[i];
	p_url[NURLMAX] = NULL;

	if (load_url_list(p_url) == OK) {
		str = create_opml_str_from_url_list(p_url);
		save_str_as_opml_file(str);
		l_str_free(str);
	}
}

/* must l_str_free() opml_str afterwards */
char *create_opml_str_from_url_list(char **url_list)
{
#define OPML_TITLE	APP_NAME" Feed List"
#define STR_SIZE	(8 * 1024)

	char 	*opml_header =
		"<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
		"<opml version=\"1.0\">\n"
		"  <head>\n"
		"    <title>%s</title>\n"
		"  </head>\n"
		"  <body>\n";
	char 	*opml_item =
		"    <outline title=\"%s\" type=\"rss\" xmlUrl=\"%s\"/>\n";
	char 	*opml_footer =
		"  </body>\n"
		"</opml>\n";
	char	*opml_str, feed_title[FEED_TITLE_MAXLEN + 1], tmp[STR_SIZE];
	gchar	*p, *esc_str;
	int	i;

	snprintf(tmp, STR_SIZE, opml_header, OPML_TITLE);
	opml_str = l_str_new(tmp);

	/* TODO: is escape stuff good? / should title be searched again for each feed?
	*/
	for (i = 0; i < NURLMAX && url_list[i] != NULL; i++) {
		url_list[i] = url_list[i] + 1;	/* we skip 1st char = (UN)SELECTED_URL_CHAR */
		if (!g_utf8_validate(url_list[i], -1, NULL)) {
			warning("Non UTF-8 encoded string:", url_list[i],
				"-", "Skipped", FALSE);
			continue;
		}
		feed_title[0] = '\0';
		for (p = url_list[i]; p < url_list[i] + FILE_NAME_MAXLEN - 1; p = g_utf8_find_next_char(p, NULL)) {
			if (*p == TITLE_TAG_CHAR) {
				esc_str = g_uri_escape_string(g_utf8_find_next_char(p, NULL),
					" AÄääàçéèêÖöôÜüùà'()[]-_|{}/.,;:?!%€£$*+=", TRUE);
				str_n_cpy(feed_title, esc_str, FEED_TITLE_MAXLEN);
				g_free(esc_str);
				*p = '\0';
				break;
			} else if (*p == '\0')
				break;
		}
		esc_str = g_uri_escape_string(url_list[i], G_URI_RESERVED_CHARS_GENERIC_DELIMITERS, TRUE);
		snprintf(tmp, STR_SIZE, opml_item, feed_title, esc_str);
		g_free(esc_str);
		opml_str = l_str_cat(opml_str, tmp);
	}
	return l_str_cat(opml_str, opml_footer);
}

int save_str_as_opml_file(const char *str)
{
	GtkWidget	*dialog;
	char		*file_name = NULL;
	FILE		*fp;
	int		exit_status = CREATE_FILE_ERROR;
	char		tmp[TMPSTR_SIZE + 1];

	gtk_window_set_keep_above(GTK_WINDOW(get_ticker_env()->win), FALSE);

	dialog = gtk_file_chooser_dialog_new("Export Feed List (OPML)", GTK_WINDOW(get_ticker_env()->win),
			GTK_FILE_CHOOSER_ACTION_SAVE,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			GTK_STOCK_OK, GTK_RESPONSE_OK,
			NULL);

	set_news_icon_to_dialog(GTK_WINDOW(dialog));
	gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);

	g_signal_connect(G_OBJECT(dialog), "key-press-event", G_CALLBACK(esc_key_pressed), NULL);
	g_signal_connect(G_OBJECT(dialog), "delete_event", G_CALLBACK(force_quit_dialog), NULL);

	gtk_widget_show_all(dialog);

	/*gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dialog), "~");*/
	gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(dialog), APP_CMD"-feed-list.opml");
	gtk_file_chooser_set_do_overwrite_confirmation(GTK_FILE_CHOOSER(dialog), TRUE);

	if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK) {
		if ((file_name = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog))) != NULL) {
			if ((fp = g_fopen(file_name, "wb")) != NULL) {
				fprintf(fp, "%s",  str);
				fclose(fp);
				exit_status = OK;
			} else
				warning("Can't save OPML file:", file_name, "-",
					strerror(errno), FALSE);
		}
	}
	gtk_widget_destroy(dialog);
	if (exit_status == OK) {
		snprintf(tmp, TMPSTR_SIZE + 1,
			"\nFeed list has been exported to OPML file: %s\n", file_name);
		info_win(APP_NAME, tmp, INFO, FALSE);
	}
	if (file_name != NULL)
			g_free(file_name);
	check_main_win_always_on_top();
	return exit_status;
}
#endif
