/* UndoManager.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.undoredo;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import javax.swing.event.EventListenerList;


/**
 * <p>Manages a list of changes of type {@link UndoableChange}. Allows changes 
 * of this type to be undone and redone. 
 * Listeners can be registered with 
 * this manager and will be notified whenever a change has been undone or redone.
 * <p>Changes come in logical <i>groups</i>.  A single call to {@link #undoLast} or
 * {@link #redoLast} always undoes or redoes one group at the time. Groups are 
 * constructed by merging subsequent changes with the leader of the group.
 * The leader also determines the caption for the undo and redo buttons of the entire
 * group. If this is not the desired behaviour an extra dummy change (with trivial 
 * undo and redo operations and the desired captions) can be used as a leader.
 * <p>Additionally the manager can be used to keep track of whether the 
 * state of the managed data is <i>dirty</i>, i.e., whether important
 * information would be lost if the application would be aborted without saving.
 * This is done by <i>marking</i> the manager state (i.e., the index of the
 * current change in the list) 
 * when a save operation has just completed. If at a later time the current
 * index position is different from the marked index position, the state of the 
 * data should be considered <i>dirty</i>. Registered listeners
 * will also be notified whenever the mark changes.
 * <p><b>Note:</b> For this simple mark strategy to be valid, you must make
 * sure that <i>every</i> action which changes the managed data is registered with
 * the undo manager and that actions which do not change the data in a 
 * significant way, are not.
 */
public class UndoManager {

    /**
     * List of undoable changes.
     */
    private final List<UndoableChange> changes;
    
    //
    private final BitSet leaders;
    
    /**
     * Next position in the list for a new change.
     */
    private int top;

    /**
     * Default constructor.
     */
    public UndoManager () {
        changes = new ArrayList<UndoableChange>();
        leaders = new BitSet ();
        leaders.set (0); //[kc] provides sentinel for undo
        top = 0;
        mark = 0;
    }

    /**
     * Are there currently any changes that can be undone?
     */
    public boolean canUndo () {
        return top > 0;
    }

    /**
     * Are there currently any changes that can be redone?
     */
    public boolean canRedo () {
        return top < changes.size ();
    }
    
    //
    private int mark;
    
    /**
     * Put the <i>mark</i> at the current index position. Typically this
     * is done just after the managed data has been saved.
     */
    public void setMark () {
        if (mark != top) {
            mark = top;
            fireUndoStateChanged ();
        }
    }
    
    /**
     * Check whether the current index position is different from the mark.
     * If true, the managed data should probably be considered in a dirty state.
     */
    public boolean isDirty () {
        return mark != top;
    } 
    
    /**
     * Register an undoable change with the manager.
     * @param leader If true, the call
     * will start a new group of which the
     * given element becomes the leader.
     * If false the given undoable change will be appended to the
     * current group.
     */
    public void add (UndoableChange change, boolean leader) {

        for (int i = changes.size () - 1; i >= top; i--)
            changes.remove (i);
        changes.add (change);
        if (leader)
            leaders.set (top);
        else
            leaders.clear(top);
        top++;
        fireUndoStateChanged ();
    }
    
    /**
     * Clear the list of undoable commands.
     */
    public void clear () {
        if (! changes.isEmpty()) {
            for (int i= changes.size()-1; i >= 0; i--) {
                changes.remove (i);
            }
            top = 0;
            mark = 0;
            fireUndoStateChanged ();
        }
    }

    /**
     * Undo te last group of commands. Commands are undone in reverse order
     * of registration: the leader of the group will be undone last.
     */
    public void undoLast () {
        if (canUndo ()) {
            do {
                top--;
                ((UndoableChange)changes.get (top)).undo ();
            } while (! leaders.get(top));
            fireUndoStateChanged ();
        }
    }

    /**
     * Redo the last group of commands which was previously undone.
     */
    public void redoLast () {
        if (canRedo ()) {
            do {
                ((UndoableChange)changes.get (top)).redo ();
                top++;
            } while (top < changes.size() && ! leaders.get (top));
            fireUndoStateChanged ();
        }
    }
  
    
    //
    private final EventListenerList listenerList = new EventListenerList ();
    
    // [kc] for performance reasons
    private static final Class<UndoListener>
            UNDO_LISTENER_CLASS = UndoListener.class;

    /**
     * Register a listener with this object.
     */
    public void addUndoListener (UndoListener l) {
        listenerList.add (UNDO_LISTENER_CLASS, l);
    }
    
    /**
     * Return the caption for an undo button. This caption is obtained
     * from the leader of the next group to be undone.
     */
    public String getUndoCaption () {
        if (top > 0) {
            int index = top - 1;
            while (!leaders.get(index))
                index --;
            return changes.get (index).getUndoCaption ();
        }
        else
            return null;
    }
    
    /**
     * Return the caption for a redo button. This caption is obtained
     * from the leader of the next group to be redone.
     */
    public String getRedoCaption () {
        if (top < changes.size())
            return changes.get (top).getRedoCaption ();
        else
            return null;
    }

    /**
     * Unregister a listener with this object.
     */
    public void removeUndoListener (UndoListener l) {
        listenerList.remove (UNDO_LISTENER_CLASS, l);
    }
    
    /**
     * Notify  all listeners of a change to the undo state.
     */
    protected  void fireUndoStateChanged () {
        Object[] listeners = listenerList.getListenerList ();
        for (int i = listeners.length-2; i>=0; i-=2)
            if (listeners[i]==UNDO_LISTENER_CLASS)
                ((UndoListener)listeners[i+1]).undoStateChanged ();
    }
    
}
