// $Id: xdr_cxx.h 4272 2011-03-18 17:40:18Z roystgnr $

// The libMesh Finite Element Library.
// Copyright (C) 2002-2008 Benjamin S. Kirk, John W. Peterson, Roy H. Stogner
  
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
  
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
  
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA



#ifndef __xdr_cxx_h__
#define __xdr_cxx_h__

// Local includes
#include "libmesh_common.h"
#include "libmesh.h"
#include "enum_xdr_mode.h"
#include "auto_ptr.h"

// C++ includes
#ifdef LIBMESH_HAVE_XDR
#  include <rpc/rpc.h>
#endif

#include <iosfwd>
#include <vector>
#include <string>
#ifdef LIBMESH_USE_COMPLEX_NUMBERS
# include <complex>
#endif

#define xdr_MAX_STRING_LENGTH 256

#ifndef LIBMESH_DEFAULT_SINGLE_PRECISION
#define xdr_REAL xdr_double
#else
#define xdr_REAL xdr_float
#endif

namespace libMesh
{



//--------------------------------------------------------------
// Xdr class definition

/**
 * This class implements a C++ interface to the XDR 
 * (eXternal Data Representation) format.  XDR is useful for
 * creating platform-independent binary files.  This class was
 * created to handle equation system output as a replacement for
 * XdrIO since that is somewhat limited.
 */

class Xdr
{
  
public:

  /**
   * Constructor.  Takes the filename and the mode.
   * Valid modes are ENCODE, DECODE, READ, and WRITE.
   */
  Xdr (const std::string& name="", const libMeshEnums::XdrMODE m=UNKNOWN);

  /**
   * Destructor.  Closes the file if it is open.
   */
  ~Xdr ();

  /**
   * Opens the file.
   */ 
  void open (const std::string& name);

  /**
   * Closes the file if it is open.
   */ 
  void close();

  /**
   * Returns true if the Xdr file is open, false
   * if it is closed.
   */ 
  bool is_open() const;

  /**
   * Returns true if the file is opened in a reading
   * state, false otherwise.
   */
  bool reading() const { return ((mode == DECODE) || (mode == READ)); }

  /**
   * Returns true if the file is opened in a writing
   * state, false otherwise.
   */
  bool writing() const { return ((mode == ENCODE) || (mode == WRITE)); }

  /**
   * Returns the mode used to access the file.  Valid modes
   * are ENCODE, DECODE, READ, or WRITE.
   */
  XdrMODE access_mode () const { return mode; }

  // Data access methods

  /**
   * Inputs or outputs a single value.
   */
  template <typename T>
  void data(T& a, const char* comment="");

  /**
   * Same, but provides an \p ostream like interface.
   */
  template <typename T>
  Xdr& operator << (T& a) { libmesh_assert (writing()); data(a); return *this; }

  /**
   * Same, but provides an \p istream like interface.
   */
  template <typename T>
  Xdr& operator >> (T& a) { libmesh_assert (reading()); data(a); return *this; }
  
  /**
   * Inputs or outputs a raw data stream.
   */
  template <typename T>
  void data_stream (T *val, const unsigned int len, const unsigned int line_break=libMesh::invalid_uint);

  /**
   * Writes or reads (ignores) a comment line.
   */   
  void comment (std::string &);
  

private:

  /**
   * Helper method for reading different data types
   */
  template <typename T>
  void do_read(T& a);

  template <typename T>
  void do_read(std::complex<T>& a);

  template <typename T>
  void do_read(std::vector<T>& a);

  template <typename T>
  void do_read(std::vector<std::complex<T> >& a);

  /**
   * Helper method for writing different data types
   */
  template <typename T>
  void do_write(T& a);

  template <typename T>
  void do_write(std::complex<T>& a);

  template <typename T>
  void do_write(std::vector<T>& a);

  template <typename T>
  void do_write(std::vector<std::complex<T> >& a);

  /**
   * The mode used for accessing the file.
   */ 
  const XdrMODE mode;

  /**
   * The file name
   */
  std::string file_name;

#ifdef LIBMESH_HAVE_XDR
  
  /**
   * Pointer to the standard @p xdr
   * struct.  See the standard
   * header file rpc/rpc.h
   * for more information.
   */
  XDR* xdrs;

  /**
   * File pointer.
   */
  FILE* fp;
  
#endif

  /**
   * The input file stream.
   */
  AutoPtr<std::istream> in;

  /**
   * The output file stream.
   */
  AutoPtr<std::ostream> out;

  /**
   * A buffer to put comment strings into.
   */
  const int comm_len;
  char comm[xdr_MAX_STRING_LENGTH];  

  /**
   * Are we reading/writing zipped files?
   */
  bool gzipped_file, bzipped_file, xzipped_file;
};


} // namespace libMesh


#endif
