/*
 * Copyright(C) 2010 Neil Jagdish Patel
 * Copyright(C) 2010 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 * Authored by Ken VaDine <ken.vandine@canonical.com>
 */

[DBus (name = "com.Gwibber.Service")]
private interface ServiceInterface : DBus.Object {
        public abstract void Start () throws DBus.Error;
        public abstract void Quit () throws DBus.Error;
        public abstract void Refresh () throws DBus.Error;
        public abstract void SendMessage (string msg) throws DBus.Error;
        public abstract string GetFeatures () throws DBus.Error;
        public abstract string GetServices () throws DBus.Error;
        public abstract string GetVersion () throws DBus.Error;
        public signal void LoadingStarted ();
        public signal void LoadingComplete ();
        public signal void Error (string error_str);
}

namespace Gwibber
{
    public class Service : Object
    {
        private const string service_name  = "com.Gwibber.Service";
        private const string service_path  = "/com/gwibber/Service";
        private const string service_iface = "com.Gwibber.Service";

        private ServiceInterface service;
        private HashTable<string,HashTable> services_table;
        private HashTable<string,Value?> service_table;
        private HashTable<string,Value?> service_attr_table;
        private HashTable<string,HashTable> features_table;
        private HashTable<string,Value?> feature_table;
	private Gwibber.Utils utils;

        /** 
            Service::is_available:
            @arg0: The current state

            Emitted when com.Gwibber.Service availability state changes
        */
        public signal void is_available(bool is_up);

        /**
            Service::loading_started:

            Emitted when an operations starts
        */

        public signal void loading_started();
        /**
            Service::loading_started:

            Emitted when an operations is complete
        */
        public signal void loading_complete();

        /**
            Service::error:
            @arg0: String containing the account id
            @arg1: String containing the error type
            @arg2: String containing the error message

            Emitted when an error is received from the service
        */
        public signal void error(string id, string type, string message);

	public Service ()
	{
	}

        construct
        {
            try
            {
                var dbus_conn = DBus.Bus.get(DBus.BusType.SESSION);

                this.service = (ServiceInterface) dbus_conn.get_object(this.service_name,
                                                           this.service_path,
                                                           this.service_iface);
		this.utils = new Gwibber.Utils();
		this.utils.setup(service_name);
		this.utils.available.connect(this.service_available);
		this.service.LoadingStarted.connect(this.on_loading_started);
		this.service.LoadingComplete.connect(this.on_loading_complete);
                this.service.Error.connect((source) => {
                        this.on_error (source);});


            }
            catch (GLib.Error e)
            {
                warning ("Unable to get Gwibber service");
            }
        }

	public void service_available(bool is_up)
	{
		this.is_available(is_up);
	}

	public void on_loading_started()
	{
		this.loading_started();
	}

	public void on_loading_complete()
	{
		this.loading_complete();
	}

        public void on_error(string error_str)
        {
            
            try {
                var parser = new Json.Parser();
                parser.load_from_data(error_str, -1);
                Json.Object root_object = parser.get_root().get_object();
                var error_object = root_object.get_member("error").get_object();
                var account_object = error_object.get_member("account").get_object();
                string id = account_object.get_member("id").get_string();
                string type = error_object.get_member("type").get_string();
                string message = error_object.get_member("message").get_string();
		/*
                debug ("ID is %s", id);
                debug ("type is %s", type);
                debug ("message is %s", message);
		*/
                this.error(id, type, message);
            } catch(GLib.Error e) {
                warning (e.message);
            }
        }

        /**
            start:

            Tells the gwibber service to perform a refresh.
        */
        public void start()
        {
            try {
                this.service.Start();
            } catch (DBus.Error e) {
                warning (e.message);
            }
        }

        /**
            quit:

            Tells the gwibber service to shutdown.
        */
        public void quit()
        {
            try {
                this.service.Quit();
            } catch (DBus.Error e) {
                warning (e.message);
            }
        }

        /**
            refresh:

            Tells the gwibber service to perform a refresh.
        */
        public void refresh()
        {
            try {
                this.service.Refresh();
            } catch (DBus.Error e) {
                warning (e.message);
            }
        }

        /**
            send_message:
            @message: The message to post to Gwibber

            Posts a message to the enabled services.
        */
        public void send_message(string message)
        {
            try {
                service.SendMessage(message);
            } catch (DBus.Error e) {
                warning (e.message);
            }
        }

        public GLib.HashTable? services()
        {
            services_table = new HashTable<string,HashTable>.full(str_hash, str_equal, g_free, GLib.HashTable.remove_all);
            try {
                string services = service.GetServices();
                var parser = new Json.Parser();
                try {
                    parser.load_from_data(services);
                } catch (GLib.Error e) {
                    warning (e.message);
                    return null;
                }
                var root_object = parser.get_root().get_object();
                foreach(var service in root_object.get_members()) {
                    service_table = new HashTable<string,Value?>.full(str_hash, str_equal, g_free, g_free);
                    var services_object = root_object.get_member(service).get_object();
                    foreach(var member in services_object.get_members()) {
                        GLib.Value? value;
                        if (services_object.get_member(member).type_name() == "JsonArray") {
                            Json.Array service_array = parser.get_root().get_array();
                            for(int i = 0; i < service_array.get_length(); i++) {
                                service_attr_table = new HashTable<string,Value?>.full(str_hash, str_equal, g_free, g_free);
                                foreach(var service_attr in service_array.get_element(i).get_object().get_members()) {
                                    if(service_array.get_element(i).get_object().get_member(service_attr).is_null()) {
                                        continue;
                                    } else if(service_array.get_element(i).get_object().get_member(service_attr).type_name() == "gboolean") {
                                        value = service_array.get_element(i).get_object().get_member(service_attr).get_boolean();
                                    } else {
                                        service_array.get_element(i).get_object().get_member(service_attr).get_value(out value);
                                    }
                                    service_attr_table.insert(service_attr, value);
                                }
                            }
                            service_table.insert(member, service_attr_table);
                            continue;
                        } else if(services_object.get_member(member).is_null()) {
                            continue;
                        } else if(services_object.get_member(member).type_name() == "gboolean") {
                            value = services_object.get_member(member).get_boolean();
                        } else {
                            services_object.get_member(member).get_value(out value);
                        }
                        service_table.insert(member, value);
                    }
                    services_table.insert(service, service_table);
                }
                return services_table;
            } catch(GLib.Error e) {
                warning(e.message);
                return null;
            }
        }

        public GLib.HashTable? features()
        {
            features_table = new HashTable<string,HashTable>.full(str_hash, str_equal, g_free, GLib.HashTable.remove_all);
            try {
                string features = service.GetFeatures();
                var parser = new Json.Parser();
                try {
                    parser.load_from_data(features, -1);
                } catch (GLib.Error e) {
                    warning (e.message);
                    return null;
                }
                var root_object = parser.get_root().get_object();
                foreach(var feature in root_object.get_members()) {
                    feature_table = new HashTable<string,Value?>.full(str_hash, str_equal, g_free, g_free);
                    var features_object = root_object.get_member(feature).get_object();
                    foreach(var member in features_object.get_members()) {
                        GLib.Value? value = null;
                        if(features_object.get_member(member).is_null()) {
                            continue;
                        } else if(features_object.get_member(member).type_name() == "gboolean") {
                            value = features_object.get_member(member).get_boolean();
                        } else { 
                            features_object.get_member(member).get_value(out value);
                        }
                        feature_table.insert(member, value);
                    }
                    features_table.insert(feature, feature_table);
                }
                return features_table;
            } catch(DBus.Error e) {
                warning(e.message);
                return null;
            }
        }
    

        /**
            get_version:

            Return value: the version of the Gwibber service that is running.
        */
        public string version()
        {
            try {
                return service.GetVersion();
            } catch(DBus.Error e) {
                warning(e.message);
            }
            return "";
        }
    }
}
