/* treemap.vala
 *
 * Copyright (C) 2009  Maciej Piechotka
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Maciej Piechotka <uzytkownik2@gmail.com>
 */

#include <glib.h>
#include <glib-object.h>


#define GEE_TYPE_MAP (gee_map_get_type ())
#define GEE_MAP(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_MAP, GeeMap))
#define GEE_IS_MAP(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_MAP))
#define GEE_MAP_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_MAP, GeeMapIface))

typedef struct _GeeMap GeeMap;
typedef struct _GeeMapIface GeeMapIface;

#define GEE_TYPE_ITERABLE (gee_iterable_get_type ())
#define GEE_ITERABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_ITERABLE, GeeIterable))
#define GEE_IS_ITERABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_ITERABLE))
#define GEE_ITERABLE_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_ITERABLE, GeeIterableIface))

typedef struct _GeeIterable GeeIterable;
typedef struct _GeeIterableIface GeeIterableIface;

#define GEE_TYPE_ITERATOR (gee_iterator_get_type ())
#define GEE_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_ITERATOR, GeeIterator))
#define GEE_IS_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_ITERATOR))
#define GEE_ITERATOR_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_ITERATOR, GeeIteratorIface))

typedef struct _GeeIterator GeeIterator;
typedef struct _GeeIteratorIface GeeIteratorIface;

#define GEE_TYPE_COLLECTION (gee_collection_get_type ())
#define GEE_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_COLLECTION, GeeCollection))
#define GEE_IS_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_COLLECTION))
#define GEE_COLLECTION_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_COLLECTION, GeeCollectionIface))

typedef struct _GeeCollection GeeCollection;
typedef struct _GeeCollectionIface GeeCollectionIface;

#define GEE_TYPE_SET (gee_set_get_type ())
#define GEE_SET(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_SET, GeeSet))
#define GEE_IS_SET(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_SET))
#define GEE_SET_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_SET, GeeSetIface))

typedef struct _GeeSet GeeSet;
typedef struct _GeeSetIface GeeSetIface;

#define GEE_TYPE_ABSTRACT_MAP (gee_abstract_map_get_type ())
#define GEE_ABSTRACT_MAP(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_ABSTRACT_MAP, GeeAbstractMap))
#define GEE_ABSTRACT_MAP_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_TYPE_ABSTRACT_MAP, GeeAbstractMapClass))
#define GEE_IS_ABSTRACT_MAP(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_ABSTRACT_MAP))
#define GEE_IS_ABSTRACT_MAP_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_TYPE_ABSTRACT_MAP))
#define GEE_ABSTRACT_MAP_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_TYPE_ABSTRACT_MAP, GeeAbstractMapClass))

typedef struct _GeeAbstractMap GeeAbstractMap;
typedef struct _GeeAbstractMapClass GeeAbstractMapClass;
typedef struct _GeeAbstractMapPrivate GeeAbstractMapPrivate;

#define GEE_TYPE_TREE_MAP (gee_tree_map_get_type ())
#define GEE_TREE_MAP(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_TREE_MAP, GeeTreeMap))
#define GEE_TREE_MAP_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_TYPE_TREE_MAP, GeeTreeMapClass))
#define GEE_IS_TREE_MAP(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_TREE_MAP))
#define GEE_IS_TREE_MAP_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_TYPE_TREE_MAP))
#define GEE_TREE_MAP_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_TYPE_TREE_MAP, GeeTreeMapClass))

typedef struct _GeeTreeMap GeeTreeMap;
typedef struct _GeeTreeMapClass GeeTreeMapClass;
typedef struct _GeeTreeMapPrivate GeeTreeMapPrivate;
typedef struct _GeeTreeMapNode GeeTreeMapNode;

#define GEE_TYPE_ABSTRACT_COLLECTION (gee_abstract_collection_get_type ())
#define GEE_ABSTRACT_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_ABSTRACT_COLLECTION, GeeAbstractCollection))
#define GEE_ABSTRACT_COLLECTION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_TYPE_ABSTRACT_COLLECTION, GeeAbstractCollectionClass))
#define GEE_IS_ABSTRACT_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_ABSTRACT_COLLECTION))
#define GEE_IS_ABSTRACT_COLLECTION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_TYPE_ABSTRACT_COLLECTION))
#define GEE_ABSTRACT_COLLECTION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_TYPE_ABSTRACT_COLLECTION, GeeAbstractCollectionClass))

typedef struct _GeeAbstractCollection GeeAbstractCollection;
typedef struct _GeeAbstractCollectionClass GeeAbstractCollectionClass;

#define GEE_TREE_MAP_TYPE_KEY_SET (gee_tree_map_key_set_get_type ())
#define GEE_TREE_MAP_KEY_SET(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TREE_MAP_TYPE_KEY_SET, GeeTreeMapKeySet))
#define GEE_TREE_MAP_KEY_SET_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_TREE_MAP_TYPE_KEY_SET, GeeTreeMapKeySetClass))
#define GEE_TREE_MAP_IS_KEY_SET(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TREE_MAP_TYPE_KEY_SET))
#define GEE_TREE_MAP_IS_KEY_SET_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_TREE_MAP_TYPE_KEY_SET))
#define GEE_TREE_MAP_KEY_SET_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_TREE_MAP_TYPE_KEY_SET, GeeTreeMapKeySetClass))

typedef struct _GeeTreeMapKeySet GeeTreeMapKeySet;
typedef struct _GeeTreeMapKeySetClass GeeTreeMapKeySetClass;

#define GEE_TREE_MAP_TYPE_VALUE_COLLECTION (gee_tree_map_value_collection_get_type ())
#define GEE_TREE_MAP_VALUE_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TREE_MAP_TYPE_VALUE_COLLECTION, GeeTreeMapValueCollection))
#define GEE_TREE_MAP_VALUE_COLLECTION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_TREE_MAP_TYPE_VALUE_COLLECTION, GeeTreeMapValueCollectionClass))
#define GEE_TREE_MAP_IS_VALUE_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TREE_MAP_TYPE_VALUE_COLLECTION))
#define GEE_TREE_MAP_IS_VALUE_COLLECTION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_TREE_MAP_TYPE_VALUE_COLLECTION))
#define GEE_TREE_MAP_VALUE_COLLECTION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_TREE_MAP_TYPE_VALUE_COLLECTION, GeeTreeMapValueCollectionClass))

typedef struct _GeeTreeMapValueCollection GeeTreeMapValueCollection;
typedef struct _GeeTreeMapValueCollectionClass GeeTreeMapValueCollectionClass;

#define GEE_TREE_MAP_NODE_TYPE_COLOR (gee_tree_map_node_color_get_type ())
typedef struct _GeeAbstractCollectionPrivate GeeAbstractCollectionPrivate;
typedef struct _GeeTreeMapKeySetPrivate GeeTreeMapKeySetPrivate;

#define GEE_TREE_MAP_TYPE_KEY_ITERATOR (gee_tree_map_key_iterator_get_type ())
#define GEE_TREE_MAP_KEY_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TREE_MAP_TYPE_KEY_ITERATOR, GeeTreeMapKeyIterator))
#define GEE_TREE_MAP_KEY_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_TREE_MAP_TYPE_KEY_ITERATOR, GeeTreeMapKeyIteratorClass))
#define GEE_TREE_MAP_IS_KEY_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TREE_MAP_TYPE_KEY_ITERATOR))
#define GEE_TREE_MAP_IS_KEY_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_TREE_MAP_TYPE_KEY_ITERATOR))
#define GEE_TREE_MAP_KEY_ITERATOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_TREE_MAP_TYPE_KEY_ITERATOR, GeeTreeMapKeyIteratorClass))

typedef struct _GeeTreeMapKeyIterator GeeTreeMapKeyIterator;
typedef struct _GeeTreeMapKeyIteratorClass GeeTreeMapKeyIteratorClass;
typedef struct _GeeTreeMapValueCollectionPrivate GeeTreeMapValueCollectionPrivate;

#define GEE_TREE_MAP_TYPE_VALUE_ITERATOR (gee_tree_map_value_iterator_get_type ())
#define GEE_TREE_MAP_VALUE_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TREE_MAP_TYPE_VALUE_ITERATOR, GeeTreeMapValueIterator))
#define GEE_TREE_MAP_VALUE_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_TREE_MAP_TYPE_VALUE_ITERATOR, GeeTreeMapValueIteratorClass))
#define GEE_TREE_MAP_IS_VALUE_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TREE_MAP_TYPE_VALUE_ITERATOR))
#define GEE_TREE_MAP_IS_VALUE_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_TREE_MAP_TYPE_VALUE_ITERATOR))
#define GEE_TREE_MAP_VALUE_ITERATOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_TREE_MAP_TYPE_VALUE_ITERATOR, GeeTreeMapValueIteratorClass))

typedef struct _GeeTreeMapValueIterator GeeTreeMapValueIterator;
typedef struct _GeeTreeMapValueIteratorClass GeeTreeMapValueIteratorClass;
typedef struct _GeeTreeMapKeyIteratorPrivate GeeTreeMapKeyIteratorPrivate;
typedef struct _GeeTreeMapValueIteratorPrivate GeeTreeMapValueIteratorPrivate;

/**
 * Implemented by classes that support a simple iteration over instances of the
 * collection.
 */
struct _GeeIteratorIface {
	GTypeInterface parent_iface;
	gboolean (*next) (GeeIterator* self);
	gpointer (*get) (GeeIterator* self);
};

/**
 * Implemented by classes that support a simple iteration over instances of the
 * collection.
 */
struct _GeeIterableIface {
	GTypeInterface parent_iface;
	GeeIterator* (*iterator) (GeeIterable* self);
	GType (*get_element_type) (GeeIterable* self);
};

/**
 * Serves as the base interface for implementing collection classes. Defines
 * size, iteration, and modification methods.
 */
struct _GeeCollectionIface {
	GTypeInterface parent_iface;
	gboolean (*contains) (GeeCollection* self, gconstpointer item);
	gboolean (*add) (GeeCollection* self, gconstpointer item);
	gboolean (*remove) (GeeCollection* self, gconstpointer item);
	void (*clear) (GeeCollection* self);
	gboolean (*add_all) (GeeCollection* self, GeeCollection* collection);
	gboolean (*contains_all) (GeeCollection* self, GeeCollection* collection);
	gboolean (*remove_all) (GeeCollection* self, GeeCollection* collection);
	gboolean (*retain_all) (GeeCollection* self, GeeCollection* collection);
	gpointer* (*to_array) (GeeCollection* self, int* result_length1);
	gint (*get_size) (GeeCollection* self);
	gboolean (*get_is_empty) (GeeCollection* self);
};

/**
 * A set is a collection without duplicates.
 */
struct _GeeSetIface {
	GTypeInterface parent_iface;
};

/**
 * A map is a generic collection of key/value pairs.
 */
struct _GeeMapIface {
	GTypeInterface parent_iface;
	GeeSet* (*get_keys) (GeeMap* self);
	GeeCollection* (*get_values) (GeeMap* self);
	gboolean (*contains) (GeeMap* self, gconstpointer key);
	gpointer (*get) (GeeMap* self, gconstpointer key);
	void (*set) (GeeMap* self, gconstpointer key, gconstpointer value);
	gboolean (*remove) (GeeMap* self, gconstpointer key, gpointer* value);
	void (*clear) (GeeMap* self);
	void (*set_all) (GeeMap* self, GeeMap* map);
	gboolean (*remove_all) (GeeMap* self, GeeMap* map);
	gboolean (*contains_all) (GeeMap* self, GeeMap* map);
	gint (*get_size) (GeeMap* self);
	gboolean (*get_is_empty) (GeeMap* self);
};

/**
 * Serves as the base class for implementing map classes.
 */
struct _GeeAbstractMap {
	GObject parent_instance;
	GeeAbstractMapPrivate * priv;
};

struct _GeeAbstractMapClass {
	GObjectClass parent_class;
	GeeSet* (*get_keys) (GeeAbstractMap* self);
	GeeCollection* (*get_values) (GeeAbstractMap* self);
	gboolean (*contains) (GeeAbstractMap* self, gconstpointer key);
	gpointer (*get) (GeeAbstractMap* self, gconstpointer key);
	void (*set) (GeeAbstractMap* self, gconstpointer key, gconstpointer value);
	gboolean (*remove) (GeeAbstractMap* self, gconstpointer key, gpointer* value);
	void (*clear) (GeeAbstractMap* self);
	void (*set_all) (GeeAbstractMap* self, GeeMap* map);
	gboolean (*remove_all) (GeeAbstractMap* self, GeeMap* map);
	gboolean (*contains_all) (GeeAbstractMap* self, GeeMap* map);
	gint (*get_size) (GeeAbstractMap* self);
	gboolean (*get_is_empty) (GeeAbstractMap* self);
};

/**
 * Left-leaning red-black tree implementation of the Map interface.
 */
struct _GeeTreeMap {
	GeeAbstractMap parent_instance;
	GeeTreeMapPrivate * priv;
};

struct _GeeTreeMapClass {
	GeeAbstractMapClass parent_class;
};

struct _GeeTreeMapPrivate {
	GType k_type;
	GBoxedCopyFunc k_dup_func;
	GDestroyNotify k_destroy_func;
	GType v_type;
	GBoxedCopyFunc v_dup_func;
	GDestroyNotify v_destroy_func;
	GCompareFunc _key_compare_func;
	GEqualFunc _value_equal_func;
	gint _size;
	GeeTreeMapNode* root;
	GeeTreeMapNode* first;
	gint stamp;
};

typedef enum  {
	GEE_TREE_MAP_NODE_COLOR_RED,
	GEE_TREE_MAP_NODE_COLOR_BLACK
} GeeTreeMapNodeColor;

struct _GeeTreeMapNode {
	gpointer key;
	gpointer value;
	GeeTreeMapNodeColor color;
	GeeTreeMapNode* left;
	GeeTreeMapNode* right;
	GeeTreeMapNode* prev;
	GeeTreeMapNode* next;
};

/**
 * Serves as the base class for implementing collection classes.
 */
struct _GeeAbstractCollection {
	GObject parent_instance;
	GeeAbstractCollectionPrivate * priv;
};

struct _GeeAbstractCollectionClass {
	GObjectClass parent_class;
	gboolean (*contains) (GeeAbstractCollection* self, gconstpointer item);
	gboolean (*add) (GeeAbstractCollection* self, gconstpointer item);
	gboolean (*remove) (GeeAbstractCollection* self, gconstpointer item);
	void (*clear) (GeeAbstractCollection* self);
	gpointer* (*to_array) (GeeAbstractCollection* self, int* result_length1);
	gboolean (*add_all) (GeeAbstractCollection* self, GeeCollection* collection);
	gboolean (*contains_all) (GeeAbstractCollection* self, GeeCollection* collection);
	gboolean (*remove_all) (GeeAbstractCollection* self, GeeCollection* collection);
	gboolean (*retain_all) (GeeAbstractCollection* self, GeeCollection* collection);
	GeeIterator* (*iterator) (GeeAbstractCollection* self);
	gint (*get_size) (GeeAbstractCollection* self);
	gboolean (*get_is_empty) (GeeAbstractCollection* self);
};

struct _GeeTreeMapKeySet {
	GeeAbstractCollection parent_instance;
	GeeTreeMapKeySetPrivate * priv;
};

struct _GeeTreeMapKeySetClass {
	GeeAbstractCollectionClass parent_class;
};

struct _GeeTreeMapKeySetPrivate {
	GType k_type;
	GBoxedCopyFunc k_dup_func;
	GDestroyNotify k_destroy_func;
	GType v_type;
	GBoxedCopyFunc v_dup_func;
	GDestroyNotify v_destroy_func;
	GeeTreeMap* _map;
};

struct _GeeTreeMapValueCollection {
	GeeAbstractCollection parent_instance;
	GeeTreeMapValueCollectionPrivate * priv;
};

struct _GeeTreeMapValueCollectionClass {
	GeeAbstractCollectionClass parent_class;
};

struct _GeeTreeMapValueCollectionPrivate {
	GType k_type;
	GBoxedCopyFunc k_dup_func;
	GDestroyNotify k_destroy_func;
	GType v_type;
	GBoxedCopyFunc v_dup_func;
	GDestroyNotify v_destroy_func;
	GeeTreeMap* _map;
};

struct _GeeTreeMapKeyIterator {
	GObject parent_instance;
	GeeTreeMapKeyIteratorPrivate * priv;
};

struct _GeeTreeMapKeyIteratorClass {
	GObjectClass parent_class;
};

struct _GeeTreeMapKeyIteratorPrivate {
	GType k_type;
	GBoxedCopyFunc k_dup_func;
	GDestroyNotify k_destroy_func;
	GType v_type;
	GBoxedCopyFunc v_dup_func;
	GDestroyNotify v_destroy_func;
	GeeTreeMap* _map;
	gint stamp;
	GeeTreeMapNode* current;
	gboolean run;
};

struct _GeeTreeMapValueIterator {
	GObject parent_instance;
	GeeTreeMapValueIteratorPrivate * priv;
};

struct _GeeTreeMapValueIteratorClass {
	GObjectClass parent_class;
};

struct _GeeTreeMapValueIteratorPrivate {
	GType k_type;
	GBoxedCopyFunc k_dup_func;
	GDestroyNotify k_destroy_func;
	GType v_type;
	GBoxedCopyFunc v_dup_func;
	GDestroyNotify v_destroy_func;
	GeeTreeMap* _map;
	gint stamp;
	GeeTreeMapNode* current;
	gboolean run;
};


static gpointer gee_tree_map_key_set_parent_class = NULL;
static GeeSetIface* gee_tree_map_key_set_gee_set_parent_iface = NULL;
static gpointer gee_tree_map_value_collection_parent_class = NULL;
static gpointer gee_tree_map_key_iterator_parent_class = NULL;
static GeeIteratorIface* gee_tree_map_key_iterator_gee_iterator_parent_iface = NULL;
static gpointer gee_tree_map_value_iterator_parent_class = NULL;
static GeeIteratorIface* gee_tree_map_value_iterator_gee_iterator_parent_iface = NULL;
static gpointer gee_tree_map_parent_class = NULL;

GType gee_iterator_get_type (void);
GType gee_iterable_get_type (void);
GType gee_collection_get_type (void);
GType gee_set_get_type (void);
GType gee_map_get_type (void);
GType gee_abstract_map_get_type (void);
GType gee_tree_map_get_type (void);
#define GEE_TREE_MAP_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_TYPE_TREE_MAP, GeeTreeMapPrivate))
enum  {
	GEE_TREE_MAP_DUMMY_PROPERTY,
	GEE_TREE_MAP_SIZE,
	GEE_TREE_MAP_KEY_COMPARE_FUNC,
	GEE_TREE_MAP_VALUE_EQUAL_FUNC,
	GEE_TREE_MAP_K_TYPE,
	GEE_TREE_MAP_K_DUP_FUNC,
	GEE_TREE_MAP_K_DESTROY_FUNC,
	GEE_TREE_MAP_V_TYPE,
	GEE_TREE_MAP_V_DUP_FUNC,
	GEE_TREE_MAP_V_DESTROY_FUNC
};
gint gee_direct_compare (void* _val1, void* _val2);
static void gee_tree_map_set_key_compare_func (GeeTreeMap* self, GCompareFunc value);
static void gee_tree_map_set_value_equal_func (GeeTreeMap* self, GEqualFunc value);
GeeTreeMap* gee_tree_map_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GCompareFunc key_compare_func, GEqualFunc value_equal_func);
GeeTreeMap* gee_tree_map_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GCompareFunc key_compare_func, GEqualFunc value_equal_func);
static GeeTreeMapKeySet* gee_tree_map_key_set_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeTreeMap* map);
static GeeTreeMapKeySet* gee_tree_map_key_set_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeTreeMap* map);
GType gee_abstract_collection_get_type (void);
static GType gee_tree_map_key_set_get_type (void);
static GeeSet* gee_tree_map_real_get_keys (GeeAbstractMap* base);
static GeeTreeMapValueCollection* gee_tree_map_value_collection_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeTreeMap* map);
static GeeTreeMapValueCollection* gee_tree_map_value_collection_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeTreeMap* map);
static GType gee_tree_map_value_collection_get_type (void);
static GeeCollection* gee_tree_map_real_get_values (GeeAbstractMap* base);
GType gee_tree_map_node_color_get_type (void);
static void gee_tree_map_rotate_right (GeeTreeMap* self, GeeTreeMapNode** root);
static void gee_tree_map_rotate_left (GeeTreeMap* self, GeeTreeMapNode** root);
static gboolean gee_tree_map_is_red (GeeTreeMap* self, GeeTreeMapNode* n);
static gboolean gee_tree_map_is_black (GeeTreeMap* self, GeeTreeMapNode* n);
GCompareFunc gee_tree_map_get_key_compare_func (GeeTreeMap* self);
static gboolean gee_tree_map_real_contains (GeeAbstractMap* base, gconstpointer key);
static gpointer gee_tree_map_real_get (GeeAbstractMap* base, gconstpointer key);
static GeeTreeMapNode* gee_tree_map_node_new (gpointer key, gpointer value, GeeTreeMapNode* prev, GeeTreeMapNode* next);
static GeeTreeMapNode* gee_tree_map_node_new (gpointer key, gpointer value, GeeTreeMapNode* prev, GeeTreeMapNode* next);
static void gee_tree_map_node_flip (GeeTreeMapNode* self);
static void gee_tree_map_set_to_node (GeeTreeMap* self, GeeTreeMapNode** node, gconstpointer key, gconstpointer value, GeeTreeMapNode* prev, GeeTreeMapNode* next);
static void gee_tree_map_fix_up (GeeTreeMap* self, GeeTreeMapNode** node);
static void gee_tree_map_real_set (GeeAbstractMap* base, gconstpointer key, gconstpointer value);
static void gee_tree_map_move_red_left (GeeTreeMap* self, GeeTreeMapNode** root);
static void gee_tree_map_move_red_right (GeeTreeMap* self, GeeTreeMapNode** root);
static void gee_tree_map_remove_minimal (GeeTreeMap* self, GeeTreeMapNode** node, gpointer* key, gpointer* value);
static gboolean gee_tree_map_remove_from_node (GeeTreeMap* self, GeeTreeMapNode** node, gconstpointer key, gpointer* value);
static gboolean gee_tree_map_real_remove (GeeAbstractMap* base, gconstpointer key, gpointer* value);
static void gee_tree_map_real_clear (GeeAbstractMap* base);
GEqualFunc gee_tree_map_get_value_equal_func (GeeTreeMap* self);
static void gee_tree_map_node_free (GeeTreeMapNode* self);
static GeeTreeMapNodeColor gee_tree_map_node_color_flip (GeeTreeMapNodeColor self);
static void gee_tree_map_node_instance_init (GeeTreeMapNode * self);
#define GEE_TREE_MAP_KEY_SET_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_TREE_MAP_TYPE_KEY_SET, GeeTreeMapKeySetPrivate))
enum  {
	GEE_TREE_MAP_KEY_SET_DUMMY_PROPERTY,
	GEE_TREE_MAP_KEY_SET_MAP,
	GEE_TREE_MAP_KEY_SET_SIZE,
	GEE_TREE_MAP_KEY_SET_K_TYPE,
	GEE_TREE_MAP_KEY_SET_K_DUP_FUNC,
	GEE_TREE_MAP_KEY_SET_K_DESTROY_FUNC,
	GEE_TREE_MAP_KEY_SET_V_TYPE,
	GEE_TREE_MAP_KEY_SET_V_DUP_FUNC,
	GEE_TREE_MAP_KEY_SET_V_DESTROY_FUNC
};
static void gee_tree_map_key_set_set_map (GeeTreeMapKeySet* self, GeeTreeMap* value);
static GeeTreeMap* gee_tree_map_key_set_get_map (GeeTreeMapKeySet* self);
static GeeTreeMapKeyIterator* gee_tree_map_key_iterator_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeTreeMap* map);
static GeeTreeMapKeyIterator* gee_tree_map_key_iterator_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeTreeMap* map);
static GType gee_tree_map_key_iterator_get_type (void);
static GeeIterator* gee_tree_map_key_set_real_iterator (GeeAbstractCollection* base);
static gboolean gee_tree_map_key_set_real_add (GeeAbstractCollection* base, gconstpointer key);
static void gee_tree_map_key_set_real_clear (GeeAbstractCollection* base);
static gboolean gee_tree_map_key_set_real_remove (GeeAbstractCollection* base, gconstpointer key);
gboolean gee_abstract_map_contains (GeeAbstractMap* self, gconstpointer key);
static gboolean gee_tree_map_key_set_real_contains (GeeAbstractCollection* base, gconstpointer key);
static gboolean gee_tree_map_key_set_real_add_all (GeeAbstractCollection* base, GeeCollection* collection);
static gboolean gee_tree_map_key_set_real_remove_all (GeeAbstractCollection* base, GeeCollection* collection);
static gboolean gee_tree_map_key_set_real_retain_all (GeeAbstractCollection* base, GeeCollection* collection);
gint gee_abstract_map_get_size (GeeAbstractMap* self);
static void gee_tree_map_key_set_finalize (GObject* obj);
static void gee_tree_map_key_set_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void gee_tree_map_key_set_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
#define GEE_TREE_MAP_VALUE_COLLECTION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_TREE_MAP_TYPE_VALUE_COLLECTION, GeeTreeMapValueCollectionPrivate))
enum  {
	GEE_TREE_MAP_VALUE_COLLECTION_DUMMY_PROPERTY,
	GEE_TREE_MAP_VALUE_COLLECTION_MAP,
	GEE_TREE_MAP_VALUE_COLLECTION_SIZE,
	GEE_TREE_MAP_VALUE_COLLECTION_K_TYPE,
	GEE_TREE_MAP_VALUE_COLLECTION_K_DUP_FUNC,
	GEE_TREE_MAP_VALUE_COLLECTION_K_DESTROY_FUNC,
	GEE_TREE_MAP_VALUE_COLLECTION_V_TYPE,
	GEE_TREE_MAP_VALUE_COLLECTION_V_DUP_FUNC,
	GEE_TREE_MAP_VALUE_COLLECTION_V_DESTROY_FUNC
};
static void gee_tree_map_value_collection_set_map (GeeTreeMapValueCollection* self, GeeTreeMap* value);
static GeeTreeMap* gee_tree_map_value_collection_get_map (GeeTreeMapValueCollection* self);
static GeeTreeMapValueIterator* gee_tree_map_value_iterator_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeTreeMap* map);
static GeeTreeMapValueIterator* gee_tree_map_value_iterator_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeTreeMap* map);
static GType gee_tree_map_value_iterator_get_type (void);
static GeeIterator* gee_tree_map_value_collection_real_iterator (GeeAbstractCollection* base);
static gboolean gee_tree_map_value_collection_real_add (GeeAbstractCollection* base, gconstpointer key);
static void gee_tree_map_value_collection_real_clear (GeeAbstractCollection* base);
static gboolean gee_tree_map_value_collection_real_remove (GeeAbstractCollection* base, gconstpointer key);
GeeIterator* gee_abstract_collection_iterator (GeeAbstractCollection* self);
gboolean gee_iterator_next (GeeIterator* self);
gpointer gee_iterator_get (GeeIterator* self);
static gboolean gee_tree_map_value_collection_real_contains (GeeAbstractCollection* base, gconstpointer key);
static gboolean gee_tree_map_value_collection_real_add_all (GeeAbstractCollection* base, GeeCollection* collection);
static gboolean gee_tree_map_value_collection_real_remove_all (GeeAbstractCollection* base, GeeCollection* collection);
static gboolean gee_tree_map_value_collection_real_retain_all (GeeAbstractCollection* base, GeeCollection* collection);
static void gee_tree_map_value_collection_finalize (GObject* obj);
static void gee_tree_map_value_collection_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void gee_tree_map_value_collection_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
#define GEE_TREE_MAP_KEY_ITERATOR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_TREE_MAP_TYPE_KEY_ITERATOR, GeeTreeMapKeyIteratorPrivate))
enum  {
	GEE_TREE_MAP_KEY_ITERATOR_DUMMY_PROPERTY,
	GEE_TREE_MAP_KEY_ITERATOR_MAP,
	GEE_TREE_MAP_KEY_ITERATOR_K_TYPE,
	GEE_TREE_MAP_KEY_ITERATOR_K_DUP_FUNC,
	GEE_TREE_MAP_KEY_ITERATOR_K_DESTROY_FUNC,
	GEE_TREE_MAP_KEY_ITERATOR_V_TYPE,
	GEE_TREE_MAP_KEY_ITERATOR_V_DUP_FUNC,
	GEE_TREE_MAP_KEY_ITERATOR_V_DESTROY_FUNC
};
static void gee_tree_map_key_iterator_set_map (GeeTreeMapKeyIterator* self, GeeTreeMap* value);
static GeeTreeMap* gee_tree_map_key_iterator_get_map (GeeTreeMapKeyIterator* self);
static gboolean gee_tree_map_key_iterator_real_next (GeeIterator* base);
static gpointer gee_tree_map_key_iterator_real_get (GeeIterator* base);
static GObject * gee_tree_map_key_iterator_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static void gee_tree_map_key_iterator_finalize (GObject* obj);
static void gee_tree_map_key_iterator_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void gee_tree_map_key_iterator_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
#define GEE_TREE_MAP_VALUE_ITERATOR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_TREE_MAP_TYPE_VALUE_ITERATOR, GeeTreeMapValueIteratorPrivate))
enum  {
	GEE_TREE_MAP_VALUE_ITERATOR_DUMMY_PROPERTY,
	GEE_TREE_MAP_VALUE_ITERATOR_MAP,
	GEE_TREE_MAP_VALUE_ITERATOR_K_TYPE,
	GEE_TREE_MAP_VALUE_ITERATOR_K_DUP_FUNC,
	GEE_TREE_MAP_VALUE_ITERATOR_K_DESTROY_FUNC,
	GEE_TREE_MAP_VALUE_ITERATOR_V_TYPE,
	GEE_TREE_MAP_VALUE_ITERATOR_V_DUP_FUNC,
	GEE_TREE_MAP_VALUE_ITERATOR_V_DESTROY_FUNC
};
static void gee_tree_map_value_iterator_set_map (GeeTreeMapValueIterator* self, GeeTreeMap* value);
static GeeTreeMap* gee_tree_map_value_iterator_get_map (GeeTreeMapValueIterator* self);
static gboolean gee_tree_map_value_iterator_real_next (GeeIterator* base);
static gpointer gee_tree_map_value_iterator_real_get (GeeIterator* base);
static GObject * gee_tree_map_value_iterator_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static void gee_tree_map_value_iterator_finalize (GObject* obj);
static void gee_tree_map_value_iterator_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void gee_tree_map_value_iterator_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
static void gee_tree_map_finalize (GObject* obj);
static void gee_tree_map_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void gee_tree_map_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);



GeeTreeMap* gee_tree_map_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GCompareFunc key_compare_func, GEqualFunc value_equal_func) {
	GParameter * __params;
	GParameter * __params_it;
	GeeTreeMap * self;
	__params = g_new0 (GParameter, 8);
	__params_it = __params;
	__params_it->name = "k-type";
	g_value_init (&__params_it->value, G_TYPE_GTYPE);
	g_value_set_gtype (&__params_it->value, k_type);
	__params_it++;
	__params_it->name = "k-dup-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, k_dup_func);
	__params_it++;
	__params_it->name = "k-destroy-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, k_destroy_func);
	__params_it++;
	__params_it->name = "v-type";
	g_value_init (&__params_it->value, G_TYPE_GTYPE);
	g_value_set_gtype (&__params_it->value, v_type);
	__params_it++;
	__params_it->name = "v-dup-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, v_dup_func);
	__params_it++;
	__params_it->name = "v-destroy-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, v_destroy_func);
	__params_it++;
	__params_it->name = "key-compare-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, key_compare_func);
	__params_it++;
	__params_it->name = "value-equal-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, value_equal_func);
	__params_it++;
	self = g_object_newv (object_type, __params_it - __params, __params);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


GeeTreeMap* gee_tree_map_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GCompareFunc key_compare_func, GEqualFunc value_equal_func) {
	return gee_tree_map_construct (GEE_TYPE_TREE_MAP, k_type, k_dup_func, k_destroy_func, v_type, v_dup_func, v_destroy_func, key_compare_func, value_equal_func);
}


static GeeSet* gee_tree_map_real_get_keys (GeeAbstractMap* base) {
	GeeTreeMap * self;
	GeeSet* result;
	self = (GeeTreeMap*) base;
	result = (GeeSet*) gee_tree_map_key_set_new (self->priv->k_type, (GBoxedCopyFunc) self->priv->k_dup_func, self->priv->k_destroy_func, self->priv->v_type, (GBoxedCopyFunc) self->priv->v_dup_func, self->priv->v_destroy_func, self);
	return result;
}


static GeeCollection* gee_tree_map_real_get_values (GeeAbstractMap* base) {
	GeeTreeMap * self;
	GeeCollection* result;
	self = (GeeTreeMap*) base;
	result = (GeeCollection*) gee_tree_map_value_collection_new (self->priv->k_type, (GBoxedCopyFunc) self->priv->k_dup_func, self->priv->k_destroy_func, self->priv->v_type, (GBoxedCopyFunc) self->priv->v_dup_func, self->priv->v_destroy_func, self);
	return result;
}


static void gee_tree_map_rotate_right (GeeTreeMap* self, GeeTreeMapNode** root) {
	GeeTreeMapNode* _tmp0_;
	GeeTreeMapNode* pivot;
	GeeTreeMapNode* _tmp2_;
	GeeTreeMapNode* _tmp1_;
	GeeTreeMapNode* _tmp4_;
	GeeTreeMapNode* _tmp3_;
	GeeTreeMapNode* _tmp6_;
	GeeTreeMapNode* _tmp5_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (root != NULL);
	_tmp0_ = NULL;
	pivot = (_tmp0_ = (*root)->left, (*root)->left = NULL, _tmp0_);
	pivot->color = (*root)->color;
	(*root)->color = GEE_TREE_MAP_NODE_COLOR_RED;
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	(*root)->left = (_tmp2_ = (_tmp1_ = pivot->right, pivot->right = NULL, _tmp1_), ((*root)->left == NULL) ? NULL : ((*root)->left = (gee_tree_map_node_free ((*root)->left), NULL)), _tmp2_);
	_tmp4_ = NULL;
	_tmp3_ = NULL;
	pivot->right = (_tmp4_ = (_tmp3_ = (*root), (*root) = NULL, _tmp3_), (pivot->right == NULL) ? NULL : (pivot->right = (gee_tree_map_node_free (pivot->right), NULL)), _tmp4_);
	_tmp6_ = NULL;
	_tmp5_ = NULL;
	(*root) = (_tmp6_ = (_tmp5_ = pivot, pivot = NULL, _tmp5_), ((*root) == NULL) ? NULL : ((*root) = (gee_tree_map_node_free ((*root)), NULL)), _tmp6_);
	(pivot == NULL) ? NULL : (pivot = (gee_tree_map_node_free (pivot), NULL));
}


static void gee_tree_map_rotate_left (GeeTreeMap* self, GeeTreeMapNode** root) {
	GeeTreeMapNode* _tmp0_;
	GeeTreeMapNode* pivot;
	GeeTreeMapNode* _tmp2_;
	GeeTreeMapNode* _tmp1_;
	GeeTreeMapNode* _tmp4_;
	GeeTreeMapNode* _tmp3_;
	GeeTreeMapNode* _tmp6_;
	GeeTreeMapNode* _tmp5_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (root != NULL);
	_tmp0_ = NULL;
	pivot = (_tmp0_ = (*root)->right, (*root)->right = NULL, _tmp0_);
	pivot->color = (*root)->color;
	(*root)->color = GEE_TREE_MAP_NODE_COLOR_RED;
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	(*root)->right = (_tmp2_ = (_tmp1_ = pivot->left, pivot->left = NULL, _tmp1_), ((*root)->right == NULL) ? NULL : ((*root)->right = (gee_tree_map_node_free ((*root)->right), NULL)), _tmp2_);
	_tmp4_ = NULL;
	_tmp3_ = NULL;
	pivot->left = (_tmp4_ = (_tmp3_ = (*root), (*root) = NULL, _tmp3_), (pivot->left == NULL) ? NULL : (pivot->left = (gee_tree_map_node_free (pivot->left), NULL)), _tmp4_);
	_tmp6_ = NULL;
	_tmp5_ = NULL;
	(*root) = (_tmp6_ = (_tmp5_ = pivot, pivot = NULL, _tmp5_), ((*root) == NULL) ? NULL : ((*root) = (gee_tree_map_node_free ((*root)), NULL)), _tmp6_);
	(pivot == NULL) ? NULL : (pivot = (gee_tree_map_node_free (pivot), NULL));
}


static gboolean gee_tree_map_is_red (GeeTreeMap* self, GeeTreeMapNode* n) {
	gboolean result;
	gboolean _tmp0_;
	g_return_val_if_fail (self != NULL, FALSE);
	_tmp0_ = FALSE;
	if (n != NULL) {
		_tmp0_ = n->color == GEE_TREE_MAP_NODE_COLOR_RED;
	} else {
		_tmp0_ = FALSE;
	}
	result = _tmp0_;
	return result;
}


static gboolean gee_tree_map_is_black (GeeTreeMap* self, GeeTreeMapNode* n) {
	gboolean result;
	gboolean _tmp0_;
	g_return_val_if_fail (self != NULL, FALSE);
	_tmp0_ = FALSE;
	if (n == NULL) {
		_tmp0_ = TRUE;
	} else {
		_tmp0_ = n->color == GEE_TREE_MAP_NODE_COLOR_BLACK;
	}
	result = _tmp0_;
	return result;
}


static gboolean gee_tree_map_real_contains (GeeAbstractMap* base, gconstpointer key) {
	GeeTreeMap * self;
	gboolean result;
	GeeTreeMapNode* cur;
	self = (GeeTreeMap*) base;
	cur = self->priv->root;
	while (TRUE) {
		gint res;
		if (!(cur != NULL)) {
			break;
		}
		res = gee_tree_map_get_key_compare_func (self) (key, cur->key);
		if (res == 0) {
			result = TRUE;
			return result;
		} else {
			if (res < 0) {
				cur = cur->left;
			} else {
				cur = cur->right;
			}
		}
	}
	result = FALSE;
	return result;
}


static gpointer gee_tree_map_real_get (GeeAbstractMap* base, gconstpointer key) {
	GeeTreeMap * self;
	gpointer result;
	GeeTreeMapNode* cur;
	self = (GeeTreeMap*) base;
	cur = self->priv->root;
	while (TRUE) {
		gint res;
		if (!(cur != NULL)) {
			break;
		}
		res = gee_tree_map_get_key_compare_func (self) (key, cur->key);
		if (res == 0) {
			gconstpointer _tmp0_;
			_tmp0_ = NULL;
			result = (_tmp0_ = cur->value, ((_tmp0_ == NULL) || (self->priv->v_dup_func == NULL)) ? ((gpointer) _tmp0_) : self->priv->v_dup_func ((gpointer) _tmp0_));
			return result;
		} else {
			if (res < 0) {
				cur = cur->left;
			} else {
				cur = cur->right;
			}
		}
	}
	result = NULL;
	return result;
}


static void gee_tree_map_set_to_node (GeeTreeMap* self, GeeTreeMapNode** node, gconstpointer key, gconstpointer value, GeeTreeMapNode* prev, GeeTreeMapNode* next) {
	gboolean _tmp3_;
	gint cmp;
	g_return_if_fail (self != NULL);
	if ((*node) == NULL) {
		GeeTreeMapNode* _tmp2_;
		gconstpointer _tmp1_;
		gconstpointer _tmp0_;
		_tmp2_ = NULL;
		_tmp1_ = NULL;
		_tmp0_ = NULL;
		(*node) = (_tmp2_ = gee_tree_map_node_new ((_tmp0_ = key, ((_tmp0_ == NULL) || (self->priv->k_dup_func == NULL)) ? ((gpointer) _tmp0_) : self->priv->k_dup_func ((gpointer) _tmp0_)), (_tmp1_ = value, ((_tmp1_ == NULL) || (self->priv->v_dup_func == NULL)) ? ((gpointer) _tmp1_) : self->priv->v_dup_func ((gpointer) _tmp1_)), prev, next), ((*node) == NULL) ? NULL : ((*node) = (gee_tree_map_node_free ((*node)), NULL)), _tmp2_);
		if (prev == NULL) {
			self->priv->first = (*node);
		}
		self->priv->_size++;
	}
	_tmp3_ = FALSE;
	if (gee_tree_map_is_red (self, (*node)->left)) {
		_tmp3_ = gee_tree_map_is_red (self, (*node)->right);
	} else {
		_tmp3_ = FALSE;
	}
	if (_tmp3_) {
		gee_tree_map_node_flip ((*node));
	}
	cmp = gee_tree_map_get_key_compare_func (self) (key, (*node)->key);
	if (cmp == 0) {
		gpointer _tmp5_;
		gconstpointer _tmp4_;
		_tmp5_ = NULL;
		_tmp4_ = NULL;
		(*node)->value = (_tmp5_ = (_tmp4_ = value, ((_tmp4_ == NULL) || (self->priv->v_dup_func == NULL)) ? ((gpointer) _tmp4_) : self->priv->v_dup_func ((gpointer) _tmp4_)), (((*node)->value == NULL) || (self->priv->v_destroy_func == NULL)) ? NULL : ((*node)->value = (self->priv->v_destroy_func ((*node)->value), NULL)), _tmp5_);
	} else {
		if (cmp < 0) {
			gee_tree_map_set_to_node (self, &(*node)->left, key, value, (*node)->prev, (*node));
		} else {
			gee_tree_map_set_to_node (self, &(*node)->right, key, value, (*node), (*node)->next);
		}
	}
	gee_tree_map_fix_up (self, &(*node));
}


static void gee_tree_map_real_set (GeeAbstractMap* base, gconstpointer key, gconstpointer value) {
	GeeTreeMap * self;
	self = (GeeTreeMap*) base;
	gee_tree_map_set_to_node (self, &self->priv->root, key, value, NULL, NULL);
	self->priv->root->color = GEE_TREE_MAP_NODE_COLOR_BLACK;
}


static void gee_tree_map_move_red_left (GeeTreeMap* self, GeeTreeMapNode** root) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (root != NULL);
	gee_tree_map_node_flip ((*root));
	if (gee_tree_map_is_red (self, (*root)->right->left)) {
		gee_tree_map_rotate_right (self, &(*root)->right);
		gee_tree_map_rotate_left (self, &(*root));
		gee_tree_map_node_flip ((*root));
	}
}


static void gee_tree_map_move_red_right (GeeTreeMap* self, GeeTreeMapNode** root) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (root != NULL);
	gee_tree_map_node_flip ((*root));
	if (gee_tree_map_is_red (self, (*root)->left->left)) {
		gee_tree_map_rotate_right (self, &(*root)->right);
		gee_tree_map_node_flip ((*root));
	}
}


static void gee_tree_map_remove_minimal (GeeTreeMap* self, GeeTreeMapNode** node, gpointer* key, gpointer* value) {
	gboolean _tmp6_;
	gpointer _tmp10_;
	gpointer _tmp9_;
	gpointer _tmp8_;
	gpointer _tmp7_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (node != NULL);
	if ((*node)->left == NULL) {
		GeeTreeMapNode* _tmp0_;
		GeeTreeMapNode* n;
		gpointer _tmp2_;
		gpointer _tmp1_;
		gpointer _tmp4_;
		gpointer _tmp3_;
		GeeTreeMapNode* _tmp5_;
		_tmp0_ = NULL;
		n = (_tmp0_ = (*node), (*node) = NULL, _tmp0_);
		_tmp2_ = NULL;
		_tmp1_ = NULL;
		(*key) = (_tmp2_ = (_tmp1_ = n->key, n->key = NULL, _tmp1_), (((*key) == NULL) || (self->priv->k_destroy_func == NULL)) ? NULL : ((*key) = (self->priv->k_destroy_func ((*key)), NULL)), _tmp2_);
		_tmp4_ = NULL;
		_tmp3_ = NULL;
		(*value) = (_tmp4_ = (_tmp3_ = n->value, n->value = NULL, _tmp3_), (((*value) == NULL) || (self->priv->v_destroy_func == NULL)) ? NULL : ((*value) = (self->priv->v_destroy_func ((*value)), NULL)), _tmp4_);
		_tmp5_ = NULL;
		(*node) = (_tmp5_ = NULL, ((*node) == NULL) ? NULL : ((*node) = (gee_tree_map_node_free ((*node)), NULL)), _tmp5_);
		(n == NULL) ? NULL : (n = (gee_tree_map_node_free (n), NULL));
		return;
	}
	_tmp6_ = FALSE;
	if (gee_tree_map_is_black (self, (*node)->left)) {
		_tmp6_ = gee_tree_map_is_black (self, (*node)->left->left);
	} else {
		_tmp6_ = FALSE;
	}
	if (_tmp6_) {
		gee_tree_map_move_red_left (self, &(*node));
	}
	_tmp10_ = NULL;
	_tmp9_ = NULL;
	_tmp8_ = NULL;
	_tmp7_ = NULL;
	(gee_tree_map_remove_minimal (self, &(*node)->left, &_tmp7_, &_tmp9_), (*key) = (_tmp8_ = _tmp7_, (((*key) == NULL) || (self->priv->k_destroy_func == NULL)) ? NULL : ((*key) = (self->priv->k_destroy_func ((*key)), NULL)), _tmp8_));
	(*value) = (_tmp10_ = _tmp9_, (((*value) == NULL) || (self->priv->v_destroy_func == NULL)) ? NULL : ((*value) = (self->priv->v_destroy_func ((*value)), NULL)), _tmp10_);
	gee_tree_map_fix_up (self, &(*node));
}


static gboolean gee_tree_map_remove_from_node (GeeTreeMap* self, GeeTreeMapNode** node, gconstpointer key, gpointer* value) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	if ((*node) == NULL) {
		result = FALSE;
		return result;
	} else {
		if (gee_tree_map_get_key_compare_func (self) (key, (*node)->key) < 0) {
			GeeTreeMapNode* left;
			gboolean _tmp0_;
			gpointer _tmp3_;
			gboolean _tmp2_;
			gpointer _tmp1_;
			gboolean r;
			left = (*node)->left;
			if (left == NULL) {
				result = FALSE;
				return result;
			}
			_tmp0_ = FALSE;
			if (gee_tree_map_is_black (self, left)) {
				_tmp0_ = gee_tree_map_is_black (self, left->left);
			} else {
				_tmp0_ = FALSE;
			}
			if (_tmp0_) {
				gee_tree_map_move_red_left (self, &(*node));
			}
			_tmp3_ = NULL;
			_tmp1_ = NULL;
			r = (_tmp2_ = gee_tree_map_remove_from_node (self, &(*node)->left, key, &_tmp1_), (*value) = (_tmp3_ = _tmp1_, (((*value) == NULL) || (self->priv->v_destroy_func == NULL)) ? NULL : ((*value) = (self->priv->v_destroy_func ((*value)), NULL)), _tmp3_), _tmp2_);
			gee_tree_map_fix_up (self, &(*node));
			result = r;
			return result;
		} else {
			GeeTreeMapNode* r;
			gboolean _tmp4_;
			gboolean _tmp8_;
			if (gee_tree_map_is_red (self, (*node)->left)) {
				gee_tree_map_rotate_right (self, &(*node));
			}
			r = (*node)->right;
			_tmp4_ = FALSE;
			if (gee_tree_map_get_key_compare_func (self) (key, (*node)->key) == 0) {
				_tmp4_ = r == NULL;
			} else {
				_tmp4_ = FALSE;
			}
			if (_tmp4_) {
				gpointer _tmp6_;
				gpointer _tmp5_;
				GeeTreeMapNode* _tmp7_;
				_tmp6_ = NULL;
				_tmp5_ = NULL;
				(*value) = (_tmp6_ = (_tmp5_ = (*node)->value, (*node)->value = NULL, _tmp5_), (((*value) == NULL) || (self->priv->v_destroy_func == NULL)) ? NULL : ((*value) = (self->priv->v_destroy_func ((*value)), NULL)), _tmp6_);
				_tmp7_ = NULL;
				(*node) = (_tmp7_ = NULL, ((*node) == NULL) ? NULL : ((*node) = (gee_tree_map_node_free ((*node)), NULL)), _tmp7_);
				self->priv->_size--;
				result = TRUE;
				return result;
			}
			_tmp8_ = FALSE;
			if (gee_tree_map_is_black (self, r)) {
				_tmp8_ = gee_tree_map_is_black (self, r->left);
			} else {
				_tmp8_ = FALSE;
			}
			if (_tmp8_) {
				gee_tree_map_move_red_right (self, &(*node));
			}
			if (gee_tree_map_get_key_compare_func (self) (key, (*node)->key) == 0) {
				gpointer _tmp10_;
				gpointer _tmp9_;
				gpointer _tmp14_;
				gpointer _tmp13_;
				gpointer _tmp12_;
				gpointer _tmp11_;
				_tmp10_ = NULL;
				_tmp9_ = NULL;
				(*value) = (_tmp10_ = (_tmp9_ = (*node)->value, (*node)->value = NULL, _tmp9_), (((*value) == NULL) || (self->priv->v_destroy_func == NULL)) ? NULL : ((*value) = (self->priv->v_destroy_func ((*value)), NULL)), _tmp10_);
				_tmp14_ = NULL;
				_tmp13_ = NULL;
				_tmp12_ = NULL;
				_tmp11_ = NULL;
				(gee_tree_map_remove_minimal (self, &(*node)->right, &_tmp11_, &_tmp13_), (*node)->key = (_tmp12_ = _tmp11_, (((*node)->key == NULL) || (self->priv->k_destroy_func == NULL)) ? NULL : ((*node)->key = (self->priv->k_destroy_func ((*node)->key), NULL)), _tmp12_));
				(*node)->value = (_tmp14_ = _tmp13_, (((*node)->value == NULL) || (self->priv->v_destroy_func == NULL)) ? NULL : ((*node)->value = (self->priv->v_destroy_func ((*node)->value), NULL)), _tmp14_);
				gee_tree_map_fix_up (self, &(*node));
				self->priv->_size--;
				result = TRUE;
				return result;
			} else {
				gpointer _tmp17_;
				gboolean _tmp16_;
				gpointer _tmp15_;
				gboolean re;
				_tmp17_ = NULL;
				_tmp15_ = NULL;
				re = (_tmp16_ = gee_tree_map_remove_from_node (self, &(*node)->right, key, &_tmp15_), (*value) = (_tmp17_ = _tmp15_, (((*value) == NULL) || (self->priv->v_destroy_func == NULL)) ? NULL : ((*value) = (self->priv->v_destroy_func ((*value)), NULL)), _tmp17_), _tmp16_);
				gee_tree_map_fix_up (self, &(*node));
				result = re;
				return result;
			}
		}
	}
}


static void gee_tree_map_fix_up (GeeTreeMap* self, GeeTreeMapNode** node) {
	gboolean _tmp0_;
	gboolean _tmp1_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (node != NULL);
	_tmp0_ = FALSE;
	if (gee_tree_map_is_black (self, (*node)->left)) {
		_tmp0_ = gee_tree_map_is_red (self, (*node)->right);
	} else {
		_tmp0_ = FALSE;
	}
	if (_tmp0_) {
		gee_tree_map_rotate_left (self, &(*node));
	}
	_tmp1_ = FALSE;
	if (gee_tree_map_is_red (self, (*node)->left)) {
		_tmp1_ = gee_tree_map_is_black (self, (*node)->right);
	} else {
		_tmp1_ = FALSE;
	}
	if (_tmp1_) {
		gee_tree_map_rotate_right (self, &(*node));
	}
}


static gboolean gee_tree_map_real_remove (GeeAbstractMap* base, gconstpointer key, gpointer* value) {
	GeeTreeMap * self;
	gboolean result;
	gpointer node_value;
	gpointer _tmp2_;
	gboolean _tmp1_;
	gpointer _tmp0_;
	gboolean b;
	self = (GeeTreeMap*) base;
	node_value = NULL;
	_tmp2_ = NULL;
	_tmp0_ = NULL;
	b = (_tmp1_ = gee_tree_map_remove_from_node (self, &self->priv->root, key, &_tmp0_), node_value = (_tmp2_ = _tmp0_, ((node_value == NULL) || (self->priv->v_destroy_func == NULL)) ? NULL : (node_value = (self->priv->v_destroy_func (node_value), NULL)), _tmp2_), _tmp1_);
	if ((&(*value)) != NULL) {
		gpointer _tmp4_;
		gpointer _tmp3_;
		_tmp4_ = NULL;
		_tmp3_ = NULL;
		(*value) = (_tmp4_ = (_tmp3_ = node_value, node_value = NULL, _tmp3_), (((*value) == NULL) || (self->priv->v_destroy_func == NULL)) ? NULL : ((*value) = (self->priv->v_destroy_func ((*value)), NULL)), _tmp4_);
	}
	if (self->priv->root != NULL) {
		self->priv->root->color = GEE_TREE_MAP_NODE_COLOR_BLACK;
	}
	self->priv->stamp++;
	result = b;
	((node_value == NULL) || (self->priv->v_destroy_func == NULL)) ? NULL : (node_value = (self->priv->v_destroy_func (node_value), NULL));
	return result;
}


static void gee_tree_map_real_clear (GeeAbstractMap* base) {
	GeeTreeMap * self;
	GeeTreeMapNode* _tmp0_;
	self = (GeeTreeMap*) base;
	_tmp0_ = NULL;
	self->priv->root = (_tmp0_ = NULL, (self->priv->root == NULL) ? NULL : (self->priv->root = (gee_tree_map_node_free (self->priv->root), NULL)), _tmp0_);
	self->priv->_size = 0;
	self->priv->stamp++;
}


static gint gee_tree_map_real_get_size (GeeAbstractMap* base) {
	gint result;
	GeeTreeMap* self;
	self = (GeeTreeMap*) base;
	result = self->priv->_size;
	return result;
}


GCompareFunc gee_tree_map_get_key_compare_func (GeeTreeMap* self) {
	GCompareFunc result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_key_compare_func;
	return result;
}


static void gee_tree_map_set_key_compare_func (GeeTreeMap* self, GCompareFunc value) {
	g_return_if_fail (self != NULL);
	self->priv->_key_compare_func = value;
	g_object_notify ((GObject *) self, "key-compare-func");
}


GEqualFunc gee_tree_map_get_value_equal_func (GeeTreeMap* self) {
	GEqualFunc result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_value_equal_func;
	return result;
}


static void gee_tree_map_set_value_equal_func (GeeTreeMap* self, GEqualFunc value) {
	g_return_if_fail (self != NULL);
	self->priv->_value_equal_func = value;
	g_object_notify ((GObject *) self, "value-equal-func");
}


static GeeTreeMapNodeColor gee_tree_map_node_color_flip (GeeTreeMapNodeColor self) {
	GeeTreeMapNodeColor result;
	if (self == GEE_TREE_MAP_NODE_COLOR_RED) {
		result = GEE_TREE_MAP_NODE_COLOR_BLACK;
		return result;
	} else {
		result = GEE_TREE_MAP_NODE_COLOR_RED;
		return result;
	}
}



GType gee_tree_map_node_color_get_type (void) {
	static GType gee_tree_map_node_color_type_id = 0;
	if (G_UNLIKELY (gee_tree_map_node_color_type_id == 0)) {
		static const GEnumValue values[] = {{GEE_TREE_MAP_NODE_COLOR_RED, "GEE_TREE_MAP_NODE_COLOR_RED", "red"}, {GEE_TREE_MAP_NODE_COLOR_BLACK, "GEE_TREE_MAP_NODE_COLOR_BLACK", "black"}, {0, NULL, NULL}};
		gee_tree_map_node_color_type_id = g_enum_register_static ("GeeTreeMapNodeColor", values);
	}
	return gee_tree_map_node_color_type_id;
}


static GeeTreeMapNode* gee_tree_map_node_new (gpointer key, gpointer value, GeeTreeMapNode* prev, GeeTreeMapNode* next) {
	GeeTreeMapNode* self;
	gpointer _tmp0_;
	gpointer _tmp1_;
	self = g_slice_new0 (GeeTreeMapNode);
	gee_tree_map_node_instance_init (self);
	_tmp0_ = NULL;
	self->key = (_tmp0_ = key, key = NULL, _tmp0_);
	_tmp1_ = NULL;
	self->value = (_tmp1_ = value, value = NULL, _tmp1_);
	self->color = GEE_TREE_MAP_NODE_COLOR_RED;
	self->prev = prev;
	self->next = next;
	if (prev != NULL) {
		prev->next = self;
	}
	if (next != NULL) {
		next->prev = self;
	}
	return self;
}


static void gee_tree_map_node_flip (GeeTreeMapNode* self) {
	g_return_if_fail (self != NULL);
	gee_tree_map_node_color_flip (self->color);
	if (self->left != NULL) {
		self->left->color = gee_tree_map_node_color_flip (self->left->color);
	}
	if (self->right != NULL) {
		self->right->color = gee_tree_map_node_color_flip (self->right->color);
	}
}


static void gee_tree_map_node_instance_init (GeeTreeMapNode * self) {
}


static void gee_tree_map_node_free (GeeTreeMapNode* self) {
	(self->left == NULL) ? NULL : (self->left = (gee_tree_map_node_free (self->left), NULL));
	(self->right == NULL) ? NULL : (self->right = (gee_tree_map_node_free (self->right), NULL));
	{
		if (self->prev != NULL) {
			self->prev->next = self->next;
		}
		if (self->next != NULL) {
			self->next->prev = self->prev;
		}
	}
	g_slice_free (GeeTreeMapNode, self);
}


static GeeTreeMapKeySet* gee_tree_map_key_set_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeTreeMap* map) {
	GParameter * __params;
	GParameter * __params_it;
	GeeTreeMapKeySet * self;
	g_return_val_if_fail (map != NULL, NULL);
	__params = g_new0 (GParameter, 7);
	__params_it = __params;
	__params_it->name = "k-type";
	g_value_init (&__params_it->value, G_TYPE_GTYPE);
	g_value_set_gtype (&__params_it->value, k_type);
	__params_it++;
	__params_it->name = "k-dup-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, k_dup_func);
	__params_it++;
	__params_it->name = "k-destroy-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, k_destroy_func);
	__params_it++;
	__params_it->name = "v-type";
	g_value_init (&__params_it->value, G_TYPE_GTYPE);
	g_value_set_gtype (&__params_it->value, v_type);
	__params_it++;
	__params_it->name = "v-dup-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, v_dup_func);
	__params_it++;
	__params_it->name = "v-destroy-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, v_destroy_func);
	__params_it++;
	__params_it->name = "map";
	g_value_init (&__params_it->value, GEE_TYPE_TREE_MAP);
	g_value_set_object (&__params_it->value, map);
	__params_it++;
	self = g_object_newv (object_type, __params_it - __params, __params);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


static GeeTreeMapKeySet* gee_tree_map_key_set_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeTreeMap* map) {
	return gee_tree_map_key_set_construct (GEE_TREE_MAP_TYPE_KEY_SET, k_type, k_dup_func, k_destroy_func, v_type, v_dup_func, v_destroy_func, map);
}


static GeeIterator* gee_tree_map_key_set_real_iterator (GeeAbstractCollection* base) {
	GeeTreeMapKeySet * self;
	GeeIterator* result;
	self = (GeeTreeMapKeySet*) base;
	result = (GeeIterator*) gee_tree_map_key_iterator_new (self->priv->k_type, (GBoxedCopyFunc) self->priv->k_dup_func, self->priv->k_destroy_func, self->priv->v_type, (GBoxedCopyFunc) self->priv->v_dup_func, self->priv->v_destroy_func, self->priv->_map);
	return result;
}


static gboolean gee_tree_map_key_set_real_add (GeeAbstractCollection* base, gconstpointer key) {
	GeeTreeMapKeySet * self;
	gboolean result;
	self = (GeeTreeMapKeySet*) base;
	g_assert_not_reached ();
}


static void gee_tree_map_key_set_real_clear (GeeAbstractCollection* base) {
	GeeTreeMapKeySet * self;
	self = (GeeTreeMapKeySet*) base;
	g_assert_not_reached ();
}


static gboolean gee_tree_map_key_set_real_remove (GeeAbstractCollection* base, gconstpointer key) {
	GeeTreeMapKeySet * self;
	gboolean result;
	self = (GeeTreeMapKeySet*) base;
	g_assert_not_reached ();
}


static gboolean gee_tree_map_key_set_real_contains (GeeAbstractCollection* base, gconstpointer key) {
	GeeTreeMapKeySet * self;
	gboolean result;
	self = (GeeTreeMapKeySet*) base;
	result = gee_abstract_map_contains ((GeeAbstractMap*) self->priv->_map, key);
	return result;
}


static gboolean gee_tree_map_key_set_real_add_all (GeeAbstractCollection* base, GeeCollection* collection) {
	GeeTreeMapKeySet * self;
	gboolean result;
	self = (GeeTreeMapKeySet*) base;
	g_return_val_if_fail (collection != NULL, FALSE);
	g_assert_not_reached ();
}


static gboolean gee_tree_map_key_set_real_remove_all (GeeAbstractCollection* base, GeeCollection* collection) {
	GeeTreeMapKeySet * self;
	gboolean result;
	self = (GeeTreeMapKeySet*) base;
	g_return_val_if_fail (collection != NULL, FALSE);
	g_assert_not_reached ();
}


static gboolean gee_tree_map_key_set_real_retain_all (GeeAbstractCollection* base, GeeCollection* collection) {
	GeeTreeMapKeySet * self;
	gboolean result;
	self = (GeeTreeMapKeySet*) base;
	g_return_val_if_fail (collection != NULL, FALSE);
	g_assert_not_reached ();
}


static GeeTreeMap* gee_tree_map_key_set_get_map (GeeTreeMapKeySet* self) {
	GeeTreeMap* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_map;
	return result;
}


static void gee_tree_map_key_set_set_map (GeeTreeMapKeySet* self, GeeTreeMap* value) {
	GeeTreeMap* _tmp1_;
	GeeTreeMap* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_map = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)), (self->priv->_map == NULL) ? NULL : (self->priv->_map = (g_object_unref (self->priv->_map), NULL)), _tmp1_);
	g_object_notify ((GObject *) self, "map");
}


static gint gee_tree_map_key_set_real_get_size (GeeAbstractCollection* base) {
	gint result;
	GeeTreeMapKeySet* self;
	self = (GeeTreeMapKeySet*) base;
	result = gee_map_get_size ((GeeMap*) self->priv->_map);
	return result;
}


static void gee_tree_map_key_set_class_init (GeeTreeMapKeySetClass * klass) {
	gee_tree_map_key_set_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GeeTreeMapKeySetPrivate));
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->iterator = gee_tree_map_key_set_real_iterator;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->add = gee_tree_map_key_set_real_add;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->clear = gee_tree_map_key_set_real_clear;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->remove = gee_tree_map_key_set_real_remove;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->contains = gee_tree_map_key_set_real_contains;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->add_all = gee_tree_map_key_set_real_add_all;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->remove_all = gee_tree_map_key_set_real_remove_all;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->retain_all = gee_tree_map_key_set_real_retain_all;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->get_size = gee_tree_map_key_set_real_get_size;
	G_OBJECT_CLASS (klass)->get_property = gee_tree_map_key_set_get_property;
	G_OBJECT_CLASS (klass)->set_property = gee_tree_map_key_set_set_property;
	G_OBJECT_CLASS (klass)->finalize = gee_tree_map_key_set_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_KEY_SET_K_TYPE, g_param_spec_gtype ("k-type", "type", "type", G_TYPE_NONE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_KEY_SET_K_DUP_FUNC, g_param_spec_pointer ("k-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_KEY_SET_K_DESTROY_FUNC, g_param_spec_pointer ("k-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_KEY_SET_V_TYPE, g_param_spec_gtype ("v-type", "type", "type", G_TYPE_NONE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_KEY_SET_V_DUP_FUNC, g_param_spec_pointer ("v-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_KEY_SET_V_DESTROY_FUNC, g_param_spec_pointer ("v-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_KEY_SET_MAP, g_param_spec_object ("map", "map", "map", GEE_TYPE_TREE_MAP, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_override_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_KEY_SET_SIZE, "size");
}


static void gee_tree_map_key_set_gee_set_interface_init (GeeSetIface * iface) {
	gee_tree_map_key_set_gee_set_parent_iface = g_type_interface_peek_parent (iface);
}


static void gee_tree_map_key_set_instance_init (GeeTreeMapKeySet * self) {
	self->priv = GEE_TREE_MAP_KEY_SET_GET_PRIVATE (self);
}


static void gee_tree_map_key_set_finalize (GObject* obj) {
	GeeTreeMapKeySet * self;
	self = GEE_TREE_MAP_KEY_SET (obj);
	(self->priv->_map == NULL) ? NULL : (self->priv->_map = (g_object_unref (self->priv->_map), NULL));
	G_OBJECT_CLASS (gee_tree_map_key_set_parent_class)->finalize (obj);
}


static GType gee_tree_map_key_set_get_type (void) {
	static GType gee_tree_map_key_set_type_id = 0;
	if (gee_tree_map_key_set_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeTreeMapKeySetClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_tree_map_key_set_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeTreeMapKeySet), 0, (GInstanceInitFunc) gee_tree_map_key_set_instance_init, NULL };
		static const GInterfaceInfo gee_set_info = { (GInterfaceInitFunc) gee_tree_map_key_set_gee_set_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_tree_map_key_set_type_id = g_type_register_static (GEE_TYPE_ABSTRACT_COLLECTION, "GeeTreeMapKeySet", &g_define_type_info, 0);
		g_type_add_interface_static (gee_tree_map_key_set_type_id, GEE_TYPE_SET, &gee_set_info);
	}
	return gee_tree_map_key_set_type_id;
}


static void gee_tree_map_key_set_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	GeeTreeMapKeySet * self;
	gpointer boxed;
	self = GEE_TREE_MAP_KEY_SET (object);
	switch (property_id) {
		case GEE_TREE_MAP_KEY_SET_MAP:
		g_value_set_object (value, gee_tree_map_key_set_get_map (self));
		break;
		case GEE_TREE_MAP_KEY_SET_SIZE:
		g_value_set_int (value, gee_abstract_collection_get_size ((GeeAbstractCollection*) self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gee_tree_map_key_set_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	GeeTreeMapKeySet * self;
	self = GEE_TREE_MAP_KEY_SET (object);
	switch (property_id) {
		case GEE_TREE_MAP_KEY_SET_MAP:
		gee_tree_map_key_set_set_map (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
		case GEE_TREE_MAP_KEY_SET_K_TYPE:
		self->priv->k_type = g_value_get_gtype (value);
		break;
		case GEE_TREE_MAP_KEY_SET_K_DUP_FUNC:
		self->priv->k_dup_func = g_value_get_pointer (value);
		break;
		case GEE_TREE_MAP_KEY_SET_K_DESTROY_FUNC:
		self->priv->k_destroy_func = g_value_get_pointer (value);
		break;
		case GEE_TREE_MAP_KEY_SET_V_TYPE:
		self->priv->v_type = g_value_get_gtype (value);
		break;
		case GEE_TREE_MAP_KEY_SET_V_DUP_FUNC:
		self->priv->v_dup_func = g_value_get_pointer (value);
		break;
		case GEE_TREE_MAP_KEY_SET_V_DESTROY_FUNC:
		self->priv->v_destroy_func = g_value_get_pointer (value);
		break;
	}
}


static GeeTreeMapValueCollection* gee_tree_map_value_collection_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeTreeMap* map) {
	GParameter * __params;
	GParameter * __params_it;
	GeeTreeMapValueCollection * self;
	g_return_val_if_fail (map != NULL, NULL);
	__params = g_new0 (GParameter, 7);
	__params_it = __params;
	__params_it->name = "k-type";
	g_value_init (&__params_it->value, G_TYPE_GTYPE);
	g_value_set_gtype (&__params_it->value, k_type);
	__params_it++;
	__params_it->name = "k-dup-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, k_dup_func);
	__params_it++;
	__params_it->name = "k-destroy-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, k_destroy_func);
	__params_it++;
	__params_it->name = "v-type";
	g_value_init (&__params_it->value, G_TYPE_GTYPE);
	g_value_set_gtype (&__params_it->value, v_type);
	__params_it++;
	__params_it->name = "v-dup-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, v_dup_func);
	__params_it++;
	__params_it->name = "v-destroy-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, v_destroy_func);
	__params_it++;
	__params_it->name = "map";
	g_value_init (&__params_it->value, GEE_TYPE_TREE_MAP);
	g_value_set_object (&__params_it->value, map);
	__params_it++;
	self = g_object_newv (object_type, __params_it - __params, __params);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


static GeeTreeMapValueCollection* gee_tree_map_value_collection_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeTreeMap* map) {
	return gee_tree_map_value_collection_construct (GEE_TREE_MAP_TYPE_VALUE_COLLECTION, k_type, k_dup_func, k_destroy_func, v_type, v_dup_func, v_destroy_func, map);
}


static GeeIterator* gee_tree_map_value_collection_real_iterator (GeeAbstractCollection* base) {
	GeeTreeMapValueCollection * self;
	GeeIterator* result;
	self = (GeeTreeMapValueCollection*) base;
	result = (GeeIterator*) gee_tree_map_value_iterator_new (self->priv->k_type, (GBoxedCopyFunc) self->priv->k_dup_func, self->priv->k_destroy_func, self->priv->v_type, (GBoxedCopyFunc) self->priv->v_dup_func, self->priv->v_destroy_func, self->priv->_map);
	return result;
}


static gboolean gee_tree_map_value_collection_real_add (GeeAbstractCollection* base, gconstpointer key) {
	GeeTreeMapValueCollection * self;
	gboolean result;
	self = (GeeTreeMapValueCollection*) base;
	g_assert_not_reached ();
}


static void gee_tree_map_value_collection_real_clear (GeeAbstractCollection* base) {
	GeeTreeMapValueCollection * self;
	self = (GeeTreeMapValueCollection*) base;
	g_assert_not_reached ();
}


static gboolean gee_tree_map_value_collection_real_remove (GeeAbstractCollection* base, gconstpointer key) {
	GeeTreeMapValueCollection * self;
	gboolean result;
	self = (GeeTreeMapValueCollection*) base;
	g_assert_not_reached ();
}


static gboolean gee_tree_map_value_collection_real_contains (GeeAbstractCollection* base, gconstpointer key) {
	GeeTreeMapValueCollection * self;
	gboolean result;
	GeeIterator* it;
	self = (GeeTreeMapValueCollection*) base;
	it = gee_abstract_collection_iterator ((GeeAbstractCollection*) self);
	while (TRUE) {
		gpointer _tmp0_;
		gboolean _tmp1_;
		if (!gee_iterator_next (it)) {
			break;
		}
		_tmp0_ = NULL;
		if ((_tmp1_ = gee_tree_map_get_value_equal_func (self->priv->_map) (key, _tmp0_ = gee_iterator_get (it)), ((_tmp0_ == NULL) || (self->priv->v_destroy_func == NULL)) ? NULL : (_tmp0_ = (self->priv->v_destroy_func (_tmp0_), NULL)), _tmp1_)) {
			result = TRUE;
			(it == NULL) ? NULL : (it = (g_object_unref (it), NULL));
			return result;
		}
	}
	result = FALSE;
	(it == NULL) ? NULL : (it = (g_object_unref (it), NULL));
	return result;
}


static gboolean gee_tree_map_value_collection_real_add_all (GeeAbstractCollection* base, GeeCollection* collection) {
	GeeTreeMapValueCollection * self;
	gboolean result;
	self = (GeeTreeMapValueCollection*) base;
	g_return_val_if_fail (collection != NULL, FALSE);
	g_assert_not_reached ();
}


static gboolean gee_tree_map_value_collection_real_remove_all (GeeAbstractCollection* base, GeeCollection* collection) {
	GeeTreeMapValueCollection * self;
	gboolean result;
	self = (GeeTreeMapValueCollection*) base;
	g_return_val_if_fail (collection != NULL, FALSE);
	g_assert_not_reached ();
}


static gboolean gee_tree_map_value_collection_real_retain_all (GeeAbstractCollection* base, GeeCollection* collection) {
	GeeTreeMapValueCollection * self;
	gboolean result;
	self = (GeeTreeMapValueCollection*) base;
	g_return_val_if_fail (collection != NULL, FALSE);
	g_assert_not_reached ();
}


static GeeTreeMap* gee_tree_map_value_collection_get_map (GeeTreeMapValueCollection* self) {
	GeeTreeMap* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_map;
	return result;
}


static void gee_tree_map_value_collection_set_map (GeeTreeMapValueCollection* self, GeeTreeMap* value) {
	GeeTreeMap* _tmp1_;
	GeeTreeMap* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_map = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)), (self->priv->_map == NULL) ? NULL : (self->priv->_map = (g_object_unref (self->priv->_map), NULL)), _tmp1_);
	g_object_notify ((GObject *) self, "map");
}


static gint gee_tree_map_value_collection_real_get_size (GeeAbstractCollection* base) {
	gint result;
	GeeTreeMapValueCollection* self;
	self = (GeeTreeMapValueCollection*) base;
	result = gee_map_get_size ((GeeMap*) self->priv->_map);
	return result;
}


static void gee_tree_map_value_collection_class_init (GeeTreeMapValueCollectionClass * klass) {
	gee_tree_map_value_collection_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GeeTreeMapValueCollectionPrivate));
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->iterator = gee_tree_map_value_collection_real_iterator;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->add = gee_tree_map_value_collection_real_add;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->clear = gee_tree_map_value_collection_real_clear;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->remove = gee_tree_map_value_collection_real_remove;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->contains = gee_tree_map_value_collection_real_contains;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->add_all = gee_tree_map_value_collection_real_add_all;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->remove_all = gee_tree_map_value_collection_real_remove_all;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->retain_all = gee_tree_map_value_collection_real_retain_all;
	GEE_ABSTRACT_COLLECTION_CLASS (klass)->get_size = gee_tree_map_value_collection_real_get_size;
	G_OBJECT_CLASS (klass)->get_property = gee_tree_map_value_collection_get_property;
	G_OBJECT_CLASS (klass)->set_property = gee_tree_map_value_collection_set_property;
	G_OBJECT_CLASS (klass)->finalize = gee_tree_map_value_collection_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_VALUE_COLLECTION_K_TYPE, g_param_spec_gtype ("k-type", "type", "type", G_TYPE_NONE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_VALUE_COLLECTION_K_DUP_FUNC, g_param_spec_pointer ("k-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_VALUE_COLLECTION_K_DESTROY_FUNC, g_param_spec_pointer ("k-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_VALUE_COLLECTION_V_TYPE, g_param_spec_gtype ("v-type", "type", "type", G_TYPE_NONE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_VALUE_COLLECTION_V_DUP_FUNC, g_param_spec_pointer ("v-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_VALUE_COLLECTION_V_DESTROY_FUNC, g_param_spec_pointer ("v-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_VALUE_COLLECTION_MAP, g_param_spec_object ("map", "map", "map", GEE_TYPE_TREE_MAP, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_override_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_VALUE_COLLECTION_SIZE, "size");
}


static void gee_tree_map_value_collection_instance_init (GeeTreeMapValueCollection * self) {
	self->priv = GEE_TREE_MAP_VALUE_COLLECTION_GET_PRIVATE (self);
}


static void gee_tree_map_value_collection_finalize (GObject* obj) {
	GeeTreeMapValueCollection * self;
	self = GEE_TREE_MAP_VALUE_COLLECTION (obj);
	(self->priv->_map == NULL) ? NULL : (self->priv->_map = (g_object_unref (self->priv->_map), NULL));
	G_OBJECT_CLASS (gee_tree_map_value_collection_parent_class)->finalize (obj);
}


static GType gee_tree_map_value_collection_get_type (void) {
	static GType gee_tree_map_value_collection_type_id = 0;
	if (gee_tree_map_value_collection_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeTreeMapValueCollectionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_tree_map_value_collection_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeTreeMapValueCollection), 0, (GInstanceInitFunc) gee_tree_map_value_collection_instance_init, NULL };
		gee_tree_map_value_collection_type_id = g_type_register_static (GEE_TYPE_ABSTRACT_COLLECTION, "GeeTreeMapValueCollection", &g_define_type_info, 0);
	}
	return gee_tree_map_value_collection_type_id;
}


static void gee_tree_map_value_collection_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	GeeTreeMapValueCollection * self;
	gpointer boxed;
	self = GEE_TREE_MAP_VALUE_COLLECTION (object);
	switch (property_id) {
		case GEE_TREE_MAP_VALUE_COLLECTION_MAP:
		g_value_set_object (value, gee_tree_map_value_collection_get_map (self));
		break;
		case GEE_TREE_MAP_VALUE_COLLECTION_SIZE:
		g_value_set_int (value, gee_abstract_collection_get_size ((GeeAbstractCollection*) self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gee_tree_map_value_collection_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	GeeTreeMapValueCollection * self;
	self = GEE_TREE_MAP_VALUE_COLLECTION (object);
	switch (property_id) {
		case GEE_TREE_MAP_VALUE_COLLECTION_MAP:
		gee_tree_map_value_collection_set_map (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
		case GEE_TREE_MAP_VALUE_COLLECTION_K_TYPE:
		self->priv->k_type = g_value_get_gtype (value);
		break;
		case GEE_TREE_MAP_VALUE_COLLECTION_K_DUP_FUNC:
		self->priv->k_dup_func = g_value_get_pointer (value);
		break;
		case GEE_TREE_MAP_VALUE_COLLECTION_K_DESTROY_FUNC:
		self->priv->k_destroy_func = g_value_get_pointer (value);
		break;
		case GEE_TREE_MAP_VALUE_COLLECTION_V_TYPE:
		self->priv->v_type = g_value_get_gtype (value);
		break;
		case GEE_TREE_MAP_VALUE_COLLECTION_V_DUP_FUNC:
		self->priv->v_dup_func = g_value_get_pointer (value);
		break;
		case GEE_TREE_MAP_VALUE_COLLECTION_V_DESTROY_FUNC:
		self->priv->v_destroy_func = g_value_get_pointer (value);
		break;
	}
}


static GeeTreeMapKeyIterator* gee_tree_map_key_iterator_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeTreeMap* map) {
	GParameter * __params;
	GParameter * __params_it;
	GeeTreeMapKeyIterator * self;
	g_return_val_if_fail (map != NULL, NULL);
	__params = g_new0 (GParameter, 7);
	__params_it = __params;
	__params_it->name = "k-type";
	g_value_init (&__params_it->value, G_TYPE_GTYPE);
	g_value_set_gtype (&__params_it->value, k_type);
	__params_it++;
	__params_it->name = "k-dup-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, k_dup_func);
	__params_it++;
	__params_it->name = "k-destroy-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, k_destroy_func);
	__params_it++;
	__params_it->name = "v-type";
	g_value_init (&__params_it->value, G_TYPE_GTYPE);
	g_value_set_gtype (&__params_it->value, v_type);
	__params_it++;
	__params_it->name = "v-dup-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, v_dup_func);
	__params_it++;
	__params_it->name = "v-destroy-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, v_destroy_func);
	__params_it++;
	__params_it->name = "map";
	g_value_init (&__params_it->value, GEE_TYPE_TREE_MAP);
	g_value_set_object (&__params_it->value, map);
	__params_it++;
	self = g_object_newv (object_type, __params_it - __params, __params);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


static GeeTreeMapKeyIterator* gee_tree_map_key_iterator_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeTreeMap* map) {
	return gee_tree_map_key_iterator_construct (GEE_TREE_MAP_TYPE_KEY_ITERATOR, k_type, k_dup_func, k_destroy_func, v_type, v_dup_func, v_destroy_func, map);
}


static gboolean gee_tree_map_key_iterator_real_next (GeeIterator* base) {
	GeeTreeMapKeyIterator * self;
	gboolean result;
	self = (GeeTreeMapKeyIterator*) base;
	if (self->priv->current != NULL) {
		self->priv->current = self->priv->current->next;
		result = self->priv->current != NULL;
		return result;
	} else {
		if (!self->priv->run) {
			self->priv->run = TRUE;
			self->priv->current = self->priv->_map->priv->first;
			result = self->priv->current != NULL;
			return result;
		} else {
			result = FALSE;
			return result;
		}
	}
}


static gpointer gee_tree_map_key_iterator_real_get (GeeIterator* base) {
	GeeTreeMapKeyIterator * self;
	gpointer result;
	gconstpointer _tmp0_;
	self = (GeeTreeMapKeyIterator*) base;
	g_assert (self->priv->stamp == self->priv->_map->priv->stamp);
	g_assert (self->priv->current != NULL);
	_tmp0_ = NULL;
	result = (_tmp0_ = self->priv->current->key, ((_tmp0_ == NULL) || (self->priv->k_dup_func == NULL)) ? ((gpointer) _tmp0_) : self->priv->k_dup_func ((gpointer) _tmp0_));
	return result;
}


static GeeTreeMap* gee_tree_map_key_iterator_get_map (GeeTreeMapKeyIterator* self) {
	GeeTreeMap* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_map;
	return result;
}


static void gee_tree_map_key_iterator_set_map (GeeTreeMapKeyIterator* self, GeeTreeMap* value) {
	GeeTreeMap* _tmp1_;
	GeeTreeMap* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_map = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)), (self->priv->_map == NULL) ? NULL : (self->priv->_map = (g_object_unref (self->priv->_map), NULL)), _tmp1_);
	g_object_notify ((GObject *) self, "map");
}


static GObject * gee_tree_map_key_iterator_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	GeeTreeMapKeyIteratorClass * klass;
	GObjectClass * parent_class;
	GeeTreeMapKeyIterator * self;
	klass = GEE_TREE_MAP_KEY_ITERATOR_CLASS (g_type_class_peek (GEE_TREE_MAP_TYPE_KEY_ITERATOR));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = GEE_TREE_MAP_KEY_ITERATOR (obj);
	{
		self->priv->stamp = self->priv->_map->priv->stamp;
	}
	return obj;
}


static void gee_tree_map_key_iterator_class_init (GeeTreeMapKeyIteratorClass * klass) {
	gee_tree_map_key_iterator_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GeeTreeMapKeyIteratorPrivate));
	G_OBJECT_CLASS (klass)->get_property = gee_tree_map_key_iterator_get_property;
	G_OBJECT_CLASS (klass)->set_property = gee_tree_map_key_iterator_set_property;
	G_OBJECT_CLASS (klass)->constructor = gee_tree_map_key_iterator_constructor;
	G_OBJECT_CLASS (klass)->finalize = gee_tree_map_key_iterator_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_KEY_ITERATOR_K_TYPE, g_param_spec_gtype ("k-type", "type", "type", G_TYPE_NONE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_KEY_ITERATOR_K_DUP_FUNC, g_param_spec_pointer ("k-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_KEY_ITERATOR_K_DESTROY_FUNC, g_param_spec_pointer ("k-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_KEY_ITERATOR_V_TYPE, g_param_spec_gtype ("v-type", "type", "type", G_TYPE_NONE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_KEY_ITERATOR_V_DUP_FUNC, g_param_spec_pointer ("v-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_KEY_ITERATOR_V_DESTROY_FUNC, g_param_spec_pointer ("v-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_KEY_ITERATOR_MAP, g_param_spec_object ("map", "map", "map", GEE_TYPE_TREE_MAP, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
}


static void gee_tree_map_key_iterator_gee_iterator_interface_init (GeeIteratorIface * iface) {
	gee_tree_map_key_iterator_gee_iterator_parent_iface = g_type_interface_peek_parent (iface);
	iface->next = gee_tree_map_key_iterator_real_next;
	iface->get = gee_tree_map_key_iterator_real_get;
}


static void gee_tree_map_key_iterator_instance_init (GeeTreeMapKeyIterator * self) {
	self->priv = GEE_TREE_MAP_KEY_ITERATOR_GET_PRIVATE (self);
	self->priv->run = FALSE;
}


static void gee_tree_map_key_iterator_finalize (GObject* obj) {
	GeeTreeMapKeyIterator * self;
	self = GEE_TREE_MAP_KEY_ITERATOR (obj);
	(self->priv->_map == NULL) ? NULL : (self->priv->_map = (g_object_unref (self->priv->_map), NULL));
	G_OBJECT_CLASS (gee_tree_map_key_iterator_parent_class)->finalize (obj);
}


static GType gee_tree_map_key_iterator_get_type (void) {
	static GType gee_tree_map_key_iterator_type_id = 0;
	if (gee_tree_map_key_iterator_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeTreeMapKeyIteratorClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_tree_map_key_iterator_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeTreeMapKeyIterator), 0, (GInstanceInitFunc) gee_tree_map_key_iterator_instance_init, NULL };
		static const GInterfaceInfo gee_iterator_info = { (GInterfaceInitFunc) gee_tree_map_key_iterator_gee_iterator_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_tree_map_key_iterator_type_id = g_type_register_static (G_TYPE_OBJECT, "GeeTreeMapKeyIterator", &g_define_type_info, 0);
		g_type_add_interface_static (gee_tree_map_key_iterator_type_id, GEE_TYPE_ITERATOR, &gee_iterator_info);
	}
	return gee_tree_map_key_iterator_type_id;
}


static void gee_tree_map_key_iterator_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	GeeTreeMapKeyIterator * self;
	gpointer boxed;
	self = GEE_TREE_MAP_KEY_ITERATOR (object);
	switch (property_id) {
		case GEE_TREE_MAP_KEY_ITERATOR_MAP:
		g_value_set_object (value, gee_tree_map_key_iterator_get_map (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gee_tree_map_key_iterator_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	GeeTreeMapKeyIterator * self;
	self = GEE_TREE_MAP_KEY_ITERATOR (object);
	switch (property_id) {
		case GEE_TREE_MAP_KEY_ITERATOR_MAP:
		gee_tree_map_key_iterator_set_map (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
		case GEE_TREE_MAP_KEY_ITERATOR_K_TYPE:
		self->priv->k_type = g_value_get_gtype (value);
		break;
		case GEE_TREE_MAP_KEY_ITERATOR_K_DUP_FUNC:
		self->priv->k_dup_func = g_value_get_pointer (value);
		break;
		case GEE_TREE_MAP_KEY_ITERATOR_K_DESTROY_FUNC:
		self->priv->k_destroy_func = g_value_get_pointer (value);
		break;
		case GEE_TREE_MAP_KEY_ITERATOR_V_TYPE:
		self->priv->v_type = g_value_get_gtype (value);
		break;
		case GEE_TREE_MAP_KEY_ITERATOR_V_DUP_FUNC:
		self->priv->v_dup_func = g_value_get_pointer (value);
		break;
		case GEE_TREE_MAP_KEY_ITERATOR_V_DESTROY_FUNC:
		self->priv->v_destroy_func = g_value_get_pointer (value);
		break;
	}
}


static GeeTreeMapValueIterator* gee_tree_map_value_iterator_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeTreeMap* map) {
	GParameter * __params;
	GParameter * __params_it;
	GeeTreeMapValueIterator * self;
	g_return_val_if_fail (map != NULL, NULL);
	__params = g_new0 (GParameter, 7);
	__params_it = __params;
	__params_it->name = "k-type";
	g_value_init (&__params_it->value, G_TYPE_GTYPE);
	g_value_set_gtype (&__params_it->value, k_type);
	__params_it++;
	__params_it->name = "k-dup-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, k_dup_func);
	__params_it++;
	__params_it->name = "k-destroy-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, k_destroy_func);
	__params_it++;
	__params_it->name = "v-type";
	g_value_init (&__params_it->value, G_TYPE_GTYPE);
	g_value_set_gtype (&__params_it->value, v_type);
	__params_it++;
	__params_it->name = "v-dup-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, v_dup_func);
	__params_it++;
	__params_it->name = "v-destroy-func";
	g_value_init (&__params_it->value, G_TYPE_POINTER);
	g_value_set_pointer (&__params_it->value, v_destroy_func);
	__params_it++;
	__params_it->name = "map";
	g_value_init (&__params_it->value, GEE_TYPE_TREE_MAP);
	g_value_set_object (&__params_it->value, map);
	__params_it++;
	self = g_object_newv (object_type, __params_it - __params, __params);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


static GeeTreeMapValueIterator* gee_tree_map_value_iterator_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeTreeMap* map) {
	return gee_tree_map_value_iterator_construct (GEE_TREE_MAP_TYPE_VALUE_ITERATOR, k_type, k_dup_func, k_destroy_func, v_type, v_dup_func, v_destroy_func, map);
}


static gboolean gee_tree_map_value_iterator_real_next (GeeIterator* base) {
	GeeTreeMapValueIterator * self;
	gboolean result;
	self = (GeeTreeMapValueIterator*) base;
	if (self->priv->current != NULL) {
		self->priv->current = self->priv->current->next;
		result = self->priv->current != NULL;
		return result;
	} else {
		if (!self->priv->run) {
			self->priv->run = TRUE;
			self->priv->current = self->priv->_map->priv->first;
			result = self->priv->current != NULL;
			return result;
		} else {
			result = FALSE;
			return result;
		}
	}
}


static gpointer gee_tree_map_value_iterator_real_get (GeeIterator* base) {
	GeeTreeMapValueIterator * self;
	gpointer result;
	gconstpointer _tmp0_;
	self = (GeeTreeMapValueIterator*) base;
	g_assert (self->priv->stamp == self->priv->_map->priv->stamp);
	g_assert (self->priv->current != NULL);
	_tmp0_ = NULL;
	result = (_tmp0_ = self->priv->current->value, ((_tmp0_ == NULL) || (self->priv->v_dup_func == NULL)) ? ((gpointer) _tmp0_) : self->priv->v_dup_func ((gpointer) _tmp0_));
	return result;
}


static GeeTreeMap* gee_tree_map_value_iterator_get_map (GeeTreeMapValueIterator* self) {
	GeeTreeMap* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_map;
	return result;
}


static void gee_tree_map_value_iterator_set_map (GeeTreeMapValueIterator* self, GeeTreeMap* value) {
	GeeTreeMap* _tmp1_;
	GeeTreeMap* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_map = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)), (self->priv->_map == NULL) ? NULL : (self->priv->_map = (g_object_unref (self->priv->_map), NULL)), _tmp1_);
	g_object_notify ((GObject *) self, "map");
}


static GObject * gee_tree_map_value_iterator_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	GeeTreeMapValueIteratorClass * klass;
	GObjectClass * parent_class;
	GeeTreeMapValueIterator * self;
	klass = GEE_TREE_MAP_VALUE_ITERATOR_CLASS (g_type_class_peek (GEE_TREE_MAP_TYPE_VALUE_ITERATOR));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = GEE_TREE_MAP_VALUE_ITERATOR (obj);
	{
		self->priv->stamp = self->priv->_map->priv->stamp;
	}
	return obj;
}


static void gee_tree_map_value_iterator_class_init (GeeTreeMapValueIteratorClass * klass) {
	gee_tree_map_value_iterator_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GeeTreeMapValueIteratorPrivate));
	G_OBJECT_CLASS (klass)->get_property = gee_tree_map_value_iterator_get_property;
	G_OBJECT_CLASS (klass)->set_property = gee_tree_map_value_iterator_set_property;
	G_OBJECT_CLASS (klass)->constructor = gee_tree_map_value_iterator_constructor;
	G_OBJECT_CLASS (klass)->finalize = gee_tree_map_value_iterator_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_VALUE_ITERATOR_K_TYPE, g_param_spec_gtype ("k-type", "type", "type", G_TYPE_NONE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_VALUE_ITERATOR_K_DUP_FUNC, g_param_spec_pointer ("k-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_VALUE_ITERATOR_K_DESTROY_FUNC, g_param_spec_pointer ("k-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_VALUE_ITERATOR_V_TYPE, g_param_spec_gtype ("v-type", "type", "type", G_TYPE_NONE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_VALUE_ITERATOR_V_DUP_FUNC, g_param_spec_pointer ("v-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_VALUE_ITERATOR_V_DESTROY_FUNC, g_param_spec_pointer ("v-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_VALUE_ITERATOR_MAP, g_param_spec_object ("map", "map", "map", GEE_TYPE_TREE_MAP, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
}


static void gee_tree_map_value_iterator_gee_iterator_interface_init (GeeIteratorIface * iface) {
	gee_tree_map_value_iterator_gee_iterator_parent_iface = g_type_interface_peek_parent (iface);
	iface->next = gee_tree_map_value_iterator_real_next;
	iface->get = gee_tree_map_value_iterator_real_get;
}


static void gee_tree_map_value_iterator_instance_init (GeeTreeMapValueIterator * self) {
	self->priv = GEE_TREE_MAP_VALUE_ITERATOR_GET_PRIVATE (self);
	self->priv->run = FALSE;
}


static void gee_tree_map_value_iterator_finalize (GObject* obj) {
	GeeTreeMapValueIterator * self;
	self = GEE_TREE_MAP_VALUE_ITERATOR (obj);
	(self->priv->_map == NULL) ? NULL : (self->priv->_map = (g_object_unref (self->priv->_map), NULL));
	G_OBJECT_CLASS (gee_tree_map_value_iterator_parent_class)->finalize (obj);
}


static GType gee_tree_map_value_iterator_get_type (void) {
	static GType gee_tree_map_value_iterator_type_id = 0;
	if (gee_tree_map_value_iterator_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeTreeMapValueIteratorClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_tree_map_value_iterator_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeTreeMapValueIterator), 0, (GInstanceInitFunc) gee_tree_map_value_iterator_instance_init, NULL };
		static const GInterfaceInfo gee_iterator_info = { (GInterfaceInitFunc) gee_tree_map_value_iterator_gee_iterator_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_tree_map_value_iterator_type_id = g_type_register_static (G_TYPE_OBJECT, "GeeTreeMapValueIterator", &g_define_type_info, 0);
		g_type_add_interface_static (gee_tree_map_value_iterator_type_id, GEE_TYPE_ITERATOR, &gee_iterator_info);
	}
	return gee_tree_map_value_iterator_type_id;
}


static void gee_tree_map_value_iterator_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	GeeTreeMapValueIterator * self;
	gpointer boxed;
	self = GEE_TREE_MAP_VALUE_ITERATOR (object);
	switch (property_id) {
		case GEE_TREE_MAP_VALUE_ITERATOR_MAP:
		g_value_set_object (value, gee_tree_map_value_iterator_get_map (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gee_tree_map_value_iterator_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	GeeTreeMapValueIterator * self;
	self = GEE_TREE_MAP_VALUE_ITERATOR (object);
	switch (property_id) {
		case GEE_TREE_MAP_VALUE_ITERATOR_MAP:
		gee_tree_map_value_iterator_set_map (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
		case GEE_TREE_MAP_VALUE_ITERATOR_K_TYPE:
		self->priv->k_type = g_value_get_gtype (value);
		break;
		case GEE_TREE_MAP_VALUE_ITERATOR_K_DUP_FUNC:
		self->priv->k_dup_func = g_value_get_pointer (value);
		break;
		case GEE_TREE_MAP_VALUE_ITERATOR_K_DESTROY_FUNC:
		self->priv->k_destroy_func = g_value_get_pointer (value);
		break;
		case GEE_TREE_MAP_VALUE_ITERATOR_V_TYPE:
		self->priv->v_type = g_value_get_gtype (value);
		break;
		case GEE_TREE_MAP_VALUE_ITERATOR_V_DUP_FUNC:
		self->priv->v_dup_func = g_value_get_pointer (value);
		break;
		case GEE_TREE_MAP_VALUE_ITERATOR_V_DESTROY_FUNC:
		self->priv->v_destroy_func = g_value_get_pointer (value);
		break;
	}
}


static void gee_tree_map_class_init (GeeTreeMapClass * klass) {
	gee_tree_map_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GeeTreeMapPrivate));
	GEE_ABSTRACT_MAP_CLASS (klass)->get_keys = gee_tree_map_real_get_keys;
	GEE_ABSTRACT_MAP_CLASS (klass)->get_values = gee_tree_map_real_get_values;
	GEE_ABSTRACT_MAP_CLASS (klass)->contains = gee_tree_map_real_contains;
	GEE_ABSTRACT_MAP_CLASS (klass)->get = gee_tree_map_real_get;
	GEE_ABSTRACT_MAP_CLASS (klass)->set = gee_tree_map_real_set;
	GEE_ABSTRACT_MAP_CLASS (klass)->remove = gee_tree_map_real_remove;
	GEE_ABSTRACT_MAP_CLASS (klass)->clear = gee_tree_map_real_clear;
	GEE_ABSTRACT_MAP_CLASS (klass)->get_size = gee_tree_map_real_get_size;
	G_OBJECT_CLASS (klass)->get_property = gee_tree_map_get_property;
	G_OBJECT_CLASS (klass)->set_property = gee_tree_map_set_property;
	G_OBJECT_CLASS (klass)->finalize = gee_tree_map_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_K_TYPE, g_param_spec_gtype ("k-type", "type", "type", G_TYPE_NONE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_K_DUP_FUNC, g_param_spec_pointer ("k-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_K_DESTROY_FUNC, g_param_spec_pointer ("k-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_V_TYPE, g_param_spec_gtype ("v-type", "type", "type", G_TYPE_NONE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_V_DUP_FUNC, g_param_spec_pointer ("v-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_V_DESTROY_FUNC, g_param_spec_pointer ("v-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_override_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_SIZE, "size");
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_KEY_COMPARE_FUNC, g_param_spec_pointer ("key-compare-func", "key-compare-func", "key-compare-func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_TREE_MAP_VALUE_EQUAL_FUNC, g_param_spec_pointer ("value-equal-func", "value-equal-func", "value-equal-func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
}


static void gee_tree_map_instance_init (GeeTreeMap * self) {
	self->priv = GEE_TREE_MAP_GET_PRIVATE (self);
	self->priv->_size = 0;
	self->priv->stamp = 0;
}


static void gee_tree_map_finalize (GObject* obj) {
	GeeTreeMap * self;
	self = GEE_TREE_MAP (obj);
	(self->priv->root == NULL) ? NULL : (self->priv->root = (gee_tree_map_node_free (self->priv->root), NULL));
	G_OBJECT_CLASS (gee_tree_map_parent_class)->finalize (obj);
}


GType gee_tree_map_get_type (void) {
	static GType gee_tree_map_type_id = 0;
	if (gee_tree_map_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeTreeMapClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_tree_map_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeTreeMap), 0, (GInstanceInitFunc) gee_tree_map_instance_init, NULL };
		gee_tree_map_type_id = g_type_register_static (GEE_TYPE_ABSTRACT_MAP, "GeeTreeMap", &g_define_type_info, 0);
	}
	return gee_tree_map_type_id;
}


static void gee_tree_map_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	GeeTreeMap * self;
	gpointer boxed;
	self = GEE_TREE_MAP (object);
	switch (property_id) {
		case GEE_TREE_MAP_SIZE:
		g_value_set_int (value, gee_abstract_map_get_size ((GeeAbstractMap*) self));
		break;
		case GEE_TREE_MAP_KEY_COMPARE_FUNC:
		g_value_set_pointer (value, gee_tree_map_get_key_compare_func (self));
		break;
		case GEE_TREE_MAP_VALUE_EQUAL_FUNC:
		g_value_set_pointer (value, gee_tree_map_get_value_equal_func (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gee_tree_map_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	GeeTreeMap * self;
	self = GEE_TREE_MAP (object);
	switch (property_id) {
		case GEE_TREE_MAP_KEY_COMPARE_FUNC:
		gee_tree_map_set_key_compare_func (self, g_value_get_pointer (value));
		break;
		case GEE_TREE_MAP_VALUE_EQUAL_FUNC:
		gee_tree_map_set_value_equal_func (self, g_value_get_pointer (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
		case GEE_TREE_MAP_K_TYPE:
		self->priv->k_type = g_value_get_gtype (value);
		break;
		case GEE_TREE_MAP_K_DUP_FUNC:
		self->priv->k_dup_func = g_value_get_pointer (value);
		break;
		case GEE_TREE_MAP_K_DESTROY_FUNC:
		self->priv->k_destroy_func = g_value_get_pointer (value);
		break;
		case GEE_TREE_MAP_V_TYPE:
		self->priv->v_type = g_value_get_gtype (value);
		break;
		case GEE_TREE_MAP_V_DUP_FUNC:
		self->priv->v_dup_func = g_value_get_pointer (value);
		break;
		case GEE_TREE_MAP_V_DESTROY_FUNC:
		self->priv->v_destroy_func = g_value_get_pointer (value);
		break;
	}
}




