/*--------------------------------------------------------------------------+
$Id: RegionSetTest.java 28332 2010-06-16 10:10:00Z hummelb $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.region;

import edu.tum.cs.commons.test.CCSMTestCaseBase;

/**
 * Test case for {@link RegionSet}
 * 
 * @author Elmar Juergens
 * @author $Author: hummelb $
 * 
 * @version $Revision: 28332 $
 * @levd.rating GREEN Hash: CD49F311E9E6C87A84EEF526EDE2F6DE
 */
public class RegionSetTest extends CCSMTestCaseBase {

	/** Utility method: counts number of regions */
	private int countRegions(RegionSet regions) {
		int count = 0;
		for (Region region : regions) {
			assertNotNull(region);
			count++;
		}
		return count;
	}

	/**
	 * Tests correct containment checks for a region set that contains a single
	 * region
	 */
	public void testContainsOnSetWithSingleRegion() {
		RegionSet regions = set(r(15, 20));

		assertTrue(regions.contains(15));
		assertTrue(regions.contains(16));
		assertTrue(regions.contains(20));

		assertFalse(regions.contains(14));
		assertFalse(regions.contains(21));
	}

	/**
	 * Tests correct containment checks for a region set that contains multiple,
	 * overlapping regions
	 */
	public void testContainsOnSetWithMultipleRegions() {
		RegionSet regions = set(r(15, 20), r(14, 17), r(25, 30));

		assertTrue(regions.contains(14));
		assertTrue(regions.contains(15));
		assertTrue(regions.contains(16));
		assertTrue(regions.contains(20));

		assertFalse(regions.contains(13));
		assertFalse(regions.contains(21));
		assertFalse(regions.contains(24));
		assertFalse(regions.contains(31));
	}

	/** Tests correct containment of region */
	public void testContainsAny() {
		RegionSet regions = set(r(15, 20), r(14, 17), r(25, 30));

		assertFalse(regions.containsAny(r(0, 13)));
		assertFalse(regions.containsAny(r(21, 24)));
		assertFalse(regions.containsAny(r(31, 34)));

		assertTrue(regions.containsAny(r(14, 15)));
		assertTrue(regions.containsAny(r(15, 15)));
		assertTrue(regions.containsAny(r(20, 20)));
		assertTrue(regions.containsAny(r(18, 20)));
		assertTrue(regions.containsAny(r(16, 17)));
		assertTrue(regions.containsAny(r(20, 28)));
		assertTrue(regions.containsAny(r(24, 32)));
	}

	/** Tests correct containment of region */
	public void testContains() {
		RegionSet regions = set(r(15, 20), r(14, 17), r(25, 30));

		assertTrue(regions.contains(r(15, 20)));
		assertTrue(regions.contains(r(16, 19)));
		assertTrue(regions.contains(r(15, 16)));

		assertFalse(regions.contains(r(0, 13)));
		assertFalse(regions.contains(r(21, 24)));
		assertFalse(regions.contains(r(31, 34)));

		assertFalse(regions.contains(r(15, 21)));
		assertFalse(regions.contains(r(13, 17)));
		assertFalse(regions.contains(r(0, 5)));
		assertFalse(regions.contains(r(31, 35)));

		regions.add(r(1, 69));
		assertTrue(regions.contains(r(1, 68)));
	}

	/**
	 * Smoke test for the implementation of {@link Iterable} in
	 * {@link RegionSet}
	 */
	public void testIterator() {
		RegionSet regions = set(r(25, 30), r(15, 20), r(5, 10));

		int count = countRegions(regions);
		assertEquals(3, count);
	}

	/** Tests addition of a region that has length of 1 */
	public void testSinglePosition() {
		RegionSet regions = set(r(10, 10));
		assertTrue(regions.contains(10));
	}

	/** Tests if overlapping regions are merged during add */
	public void testAddOverlappingRegions() {
		RegionSet regions = set(r(10, 12), r(9, 11));

		// assert that all positions are resolved correctly
		assertFalse(regions.contains(8));
		assertTrue(regions.contains(9));
		assertTrue(regions.contains(10));
		assertTrue(regions.contains(11));
		assertTrue(regions.contains(12));
		assertFalse(regions.contains(13));
	}

	/** Tests the correct computation of the PositionCount */
	public void testPositionCount() {
		RegionSet regions = set(r(10, 12));
		assertEquals(3, regions.getPositionCount());

		regions.add(r(9, 11));
		assertEquals(4, regions.getPositionCount());
	}

	/** Tests inversion of a {@link RegionSet} */
	public void testInvert() {
		RegionSet regions = set(r(2, 3), r(5, 6), r(10, 10));

		RegionSet invertedRegions = regions.createInverted("inverted", 10);
		assertNotNull(invertedRegions);

		assertEquals(3, invertedRegions.size());
		assertTrue(invertedRegions.contains(0));
		assertTrue(invertedRegions.contains(1));
		assertFalse(invertedRegions.contains(2));
		assertFalse(invertedRegions.contains(3));
		assertTrue(invertedRegions.contains(4));
		assertFalse(invertedRegions.contains(5));
		assertFalse(invertedRegions.contains(6));
		assertTrue(invertedRegions.contains(7));
		assertTrue(invertedRegions.contains(8));
		assertTrue(invertedRegions.contains(9));
		assertFalse(invertedRegions.contains(10));
		assertFalse(invertedRegions.contains(11));
	}

	/** Test inversion of a region set that only contains a single region */
	public void testInvertSingleRegion() {
		RegionSet set = set(r(1, 2));

		RegionSet inverted = set.createInverted("inverted", 2);
		RegionSet original = inverted.createInverted("original", 2);

		assertEquals(1, original.size());
		assertFalse(original.contains(0));
		assertTrue(original.contains(1));
		assertTrue(original.contains(2));
		assertFalse(original.contains(3));
	}

	/** Test {@link RegionSet#getLastPosition()} */
	public void testGetLastPosition() {
		RegionSet regionSet = set(r(1, 2));
		assertEquals(2, regionSet.getLastPosition());

		regionSet.add(r(1, 2));
		assertEquals(2, regionSet.getLastPosition());

		regionSet.add(r(1, 1));
		assertEquals(2, regionSet.getLastPosition());

		regionSet.add(r(1, 3));
		assertEquals(3, regionSet.getLastPosition());

		regionSet.add(r(9, 9));
		assertEquals(9, regionSet.getLastPosition());

		regionSet.add(r(0, 1000));
		assertEquals(1000, regionSet.getLastPosition());
	}

	/** Test {@link RegionSet#getFirstPosition()} */
	public void testGetFirstPosition() {
		RegionSet regionSet = set(r(10, 20));
		assertEquals(10, regionSet.getFirstPosition());

		regionSet.add(r(10, 20));
		assertEquals(10, regionSet.getFirstPosition());

		regionSet.add(r(21, 25));
		assertEquals(10, regionSet.getFirstPosition());

		regionSet.add(r(5, 8));
		assertEquals(5, regionSet.getFirstPosition());

		regionSet.add(r(5, 50));
		assertEquals(5, regionSet.getFirstPosition());

		regionSet.add(r(1, 2));
		assertEquals(1, regionSet.getFirstPosition());

		regionSet.add(r(0, 2));
		assertEquals(0, regionSet.getFirstPosition());
	}

	/** Creates {@link Region} */
	public static Region r(int start, int end) {
		return new Region(start, end);
	}

	/** Create {@link RegionSet} */
	public static RegionSet set(Region... regions) {
		RegionSet regionSet = new RegionSet();
		for (Region region : regions) {
			regionSet.add(region);
		}
		return regionSet;
	}

	/** Test case for {@link RegionSet#positionsEqual(RegionSet)} */
	public void testPositionsDontEqualForNull() {
		assertFalse(set(r(1, 2)).positionsEqual(null));
	}

	/** Test case for {@link RegionSet#positionsEqual(RegionSet)} */
	public void testPositionsEqualForEmptySets() {
		assertTrue(set().positionsEqual(set()));
	}

	/** Test case for {@link RegionSet#positionsEqual(RegionSet)} */
	public void testPositionsDontEqualDifferentSets() {
		assertFalse(set(r(1, 2)).positionsEqual(set()));
	}

	/** Test case for {@link RegionSet#positionsEqual(RegionSet)} */
	public void testPositionsEqual() {
		RegionSet set1 = set(r(1, 3));
		RegionSet set2 = set(r(1, 1), r(2, 3));
		assertTrue(set1.positionsEqual(set2));
	}
}