# Copyright (C) 2008-2010 LottaNZB Development Team
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 3.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.

import re

import logging
log = logging.getLogger("lottanzb.plugins.inhibit_suspending")

try:
    import dbus
except ImportError:
    dbus = None

from subprocess import Popen, PIPE
from distutils.version import StrictVersion

from lottanzb.plugins import PluginBase, PluginEnablingError
from lottanzb.util import _
from lottanzb.modes import standalone

class Plugin(PluginBase):
    title = _("Inhibit suspending")
    description = _("Prevents the computer from being suspended while "
        "downloading.")
    author = _("LottaNZB Development Team")
    requires_modes = [standalone.Mode]
    
    def __init__(self, app, config):
        # Holds an `InhibitionProvider` if it's possible to instantiate one.
        self._provider = None
        
        PluginBase.__init__(self, app, config)
        
        self.connect_when_enabled(self.app.backend, "updated",
            self.on_backend_updated)
    
    def import_dependencies(self):
        """
        Make sure that the D-Bus module is available on the user's machine.
        
        Try to set up an `InhibitionProvider` that provides the methods to
        inhibit or uninhibit suspending.
        """
        
        global dbus
        
        if not dbus:
            import dbus
        
        # A provider should only be set up once, but this method may be called
        # several times.
        if self._provider is None:
            self._setup_provider()
    
    def _setup_provider(self):
        inhibition_providers_classes = (
            GnomeSessionInhibitionProvider,
            HALInhibitionProvider
        )
        
        for inhibition_providers_class in inhibition_providers_classes:
            try:
                provider = inhibition_providers_class("lottanzb")
            except InhibitionProviderError, error:
                # Try the next provider class.
                pass
            else:
                self._provider = provider
        
        if self._provider is None:
            # Bad luck.
            raise PluginEnablingError(self, _("Your desktop environment "
                "isn't supported by the plug-in."))
    
    def refresh(self):
        """
        Inhibit suspending if the plug-in is enabled and there are active
        downloads or an active post-processing operation. Allows suspending
        otherwise.
        """
        
        if self._provider is not None:
            if self.enabled:
                downloads = self.app.backend.downloads
                inhibit = bool(downloads.get_processing() \
                    or (downloads.get_active_download() \
                    and not self.app.backend.paused))
                
                if inhibit:
                    self.inhibit()
                else:
                    self.uninhibit()
            else:
                self.uninhibit()
    
    def on_backend_updated(self, backend):
        """
        Check if suspending needs to be inhibited/uninhibited whenever the
        download list is updated.
        """
        
        self.refresh()
    
    def inhibit(self):
        """Try to prevent the computer from being suspended."""
        
        if not self._provider.is_inhibited():
            reason = _("Download activity")
            
            try:
                self._provider.inhibit(reason)
            except InhibitionProviderError, error:
                log.warning(_("Could not disallow suspending: %s") % str(error))
            else:
                log.info(_("Disallowing suspending."))
    
    def uninhibit(self):
        """Allow the computer to be suspended."""
        
        if self._provider.is_inhibited():
            try:
                self._provider.uninhibit()
            except InhibitionProviderError, error:
                log.warning(_("Could not reallow suspending: %s") % \
                    str(error))
            else:
                log.info(_("Allowing suspending."))


class InhibitionProvider:
    def __init__(self, application):
        self.application = application
    
    def inhibit(self, reason=""):
        raise NotImplementedError
    
    def uninhibit(self):
        raise NotImplementedError
    
    def is_inhibited(self):
        raise NotImplementedError
    
    def tear_down(self):
        pass


class HALInhibitionProvider(InhibitionProvider):
    def __init__(self, application):
        InhibitionProvider.__init__(self, application)
        
        try:
            session_bus = dbus.SessionBus()
            interface_name = "org.freedesktop.PowerManagement.Inhibit"
            proxy = session_bus.get_object(
                "org.freedesktop.PowerManagement",
                "/org/freedesktop/PowerManagement/Inhibit"
            )
            
            self._interface = dbus.Interface(proxy, interface_name)
            self._cookie = None
        except dbus.exceptions.DBusException, error:
            raise InhibitionProviderError(error.message)
    
    def inhibit(self, reason=""):
        """Try to prevent the computer from being suspended."""
        
        try:
            self._cookie = self._interface.Inhibit(self.application, reason)
        except dbus.exceptions.DBusException, error:
            raise InhibitionProviderError(error.message)
    
    def uninhibit(self):
        """Allow the computer to be suspended."""
        
        try:
            self._interface.UnInhibit(self._cookie)
        except dbus.exceptions.DBusException, error:
            raise InhibitionProviderError(error.message)
        finally:
            self._cookie = None
    
    def is_inhibited(self):
        return bool(self._cookie is not None)


class GnomeSessionInhibitionProvider(InhibitionProvider):
    VERSION_PATTERN = re.compile("\d[\d\.]*")
    
    # The GNOME Power Manager didn't actually use the SessionManager D-Bus API
    # to check if there are any suspending inhibitors until version 2.29.2.
    # Thus, this `InhibitionProvider` will not work on systems like Ubuntu 9.10
    # that use the GNOME 2.28 or older.
    # See https://bugzilla.gnome.org/show_bug.cgi?id=607748
    MIN_GPM_VERSION = StrictVersion("2.29.2")
    
    def __init__(self, application):
        InhibitionProvider.__init__(self, application)
        
        try:
            process = Popen(["gnome-power-manager", "--version"], stdout=PIPE)
            process.wait()
        except OSError, error:
            raise InhibitionProviderError(str(error))
        else:
            match = self.VERSION_PATTERN.search(process.stdout.read())
            
            # Don't raise an `InhibitionProviderError` if the version output
            # doesn't have the expected format, so that potential future changes
            # to the GNOME Power Manager don't erroneously fail the check.
            if match and StrictVersion(match.group()) < self.MIN_GPM_VERSION:
                raise InhibitionProviderError(
                    "gnome-power-manager %s is not supported" % match.group())
        
        try:
            session_bus = dbus.SessionBus()
            interface_name = "org.gnome.SessionManager"
            proxy = session_bus.get_object(
                "org.gnome.SessionManager",
                "/org/gnome/SessionManager"
            )
            
            self._interface = dbus.Interface(proxy, interface_name)
            self._client_id = self._interface.RegisterClient(self.application,
                self.application)
            self._cookie = None
        except dbus.exceptions.DBusException, error:
            raise InhibitionProviderError(error.message)
    
    def inhibit(self, reason=""):
        """Try to prevent the computer from being suspended."""
        
        try:
            self._cookie = self._interface.Inhibit(self.application, 0, reason,
                0x4)
        except dbus.exceptions.DBusException, error:
            raise InhibitionProviderError(error.message)
    
    def uninhibit(self):
        """Allow the computer to be suspended."""
        
        try:
            self._interface.Uninhibit(self._cookie)
        except dbus.exceptions.DBusException, error:
            raise InhibitionProviderError(error.message)
        finally:
            self._cookie = None
    
    def tear_down(self):
        try:
            self._interface.UnregisterClient(self._client_id)
        except dbus.exceptions.DBusException, error:
            raise InhibitionProviderError(error.message)
    
    def is_inhibited(self):
        return bool(self._cookie is not None)


class InhibitionProviderError(Exception):
    pass
