#!/usr/bin/env python

# Copyright (C) 2008 LottaNZB Development Team
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 3.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.

import os

from distutils.core import Command, setup as distutils_setup
from distutils.command import clean, install, build, bdist_rpm
from distutils.errors import DistutilsFileError, DistutilsArgError
from distutils.dir_util import remove_tree
from distutils import log

from imp import find_module, load_module
from subprocess import call, Popen, PIPE
from glob import glob
from os.path import isfile, isdir, join, basename, exists, abspath
from stat import ST_MODE

import kiwi

from kiwi.dist import listpackages, setup as kiwi_setup
from kiwi.i18n.i18n import update_po, compile_po_files, list_languages

from lottanzb import __version__ as version

NAME             = "lottanzb"
VERSION          = version
DESCRIPTION      = "LottaNZB - Automated Usenet Client"
LONG_DESCRIPTION = ("LottaNZB is a Usenet client that automates the download "
                   "of Usenet files with the help of NZB files. It uses "
                   "HellaNZB as its backend and PyGTK for its user interface.")
AUTHOR           = "LottaNZB Development Team"
AUTHOR_EMAIL     = "avirulence@lottanzb.org"
URL              = "http://www.lottanzb.org/"
LICENSE          = "GPL"
KEYWORDS         = ["usenet", "nzb", "newzbin", "download", "hellanzb", 
                    "frontend"]
CLASSIFIERS      = [
                        "Development Status :: 4 - Beta",
                        "Environment :: X11 Applications :: Gnome",
                        "Environment :: X11 Applications :: GTK",
                        "Intended Audience :: End Users/Desktop",
                        "License :: OSI Approved :: "
                        "GNU General Public License (GPL)",
                        "Operating System :: POSIX :: Linux",
                        "Natural Language :: Dutch",
                        "Natural Language :: English",
                        "Natural Language :: French",
                        "Natural Language :: German",
                        "Natural Language :: Spanish",
                        "Programming Language :: Python",
                        "Topic :: Communications :: Usenet News"
                   ]
DATA_FILES       = [
                        ("$datadir/glade", glob("data/glade/*.*")),
                   ]

def remove(path, dry_run=False):
    """
    Remove a certain file or directory.
    
    Similar to the remove_tree function provided by the distutils module,
    it has an optional dry_run parameter. If set to true,
    no files are removed at all.
    """
    
    if not exists(path):
        log.debug("%s does not exist.", path)
        return
    
    if isfile(path):
        try:
            log.info("removing %s", path)
            
            if not dry_run:
                os.remove(path)
        except OSError:
            raise DistutilsFileError("Unable to remove %s.", path)
    elif isdir(path):
        remove_tree(path, dry_run=dry_run)

class CleanCommand(clean.clean):
    description = "remove files and folders created during the build process"
    
    def run(self):
        clean.clean.run(self)
        
        # Remove pyc files.
        for root, dirs, files in os.walk("."):
            for a_file in files:
                if a_file.endswith(".pyc"):
                    self.remove(join(root, a_file))
        
        self.remove("MANIFEST")
        self.remove(".xml2po.mo")
        
        if self.all:
            self.remove(self.build_base)
    
    def remove(self, path):
        """
        Wraps the remove function, so that the dry_run parameter doesn't need
        to be set whenever it's called.
        """
        
        remove(path, self.dry_run)

class InstallWrapper(install.install):
    """
    Doesn't contain much more than the complete_installation method, used to
    complete the GNOME integration or to reverse it, respectively.
    """
    
    def complete_installation(self):
        """
        Try to register LottaNZB as an application used to handle NZB files,
        make sure NZB files and the LottaNZB menu entry get a shiny icon
        and register the help content.
        """
        
        omf_dir = join(self.prefix, "share", "omf")
        mime_dir = join(self.prefix, "share", "mime")
        
        def try_to_call(cmd):
            try:
                if not self.dry_run:
                    call(cmd)
            except OSError:
                log.warn("Could not call %s.", cmd)
        
        log.info("updating MIME types")
        try_to_call(["update-mime-database", mime_dir])
        
        log.info("updating desktop database")
        try_to_call(["update-desktop-database"])
        
        for loc in ["gnome", "hicolor"]:
            icon_dir = join(self.prefix, "share", "icons", loc)
            
            log.info("updating icon cache %s", icon_dir)
            try_to_call(["gtk-update-icon-cache", "-q", "-f", "-t", icon_dir])
        
        log.info("updating scrollkeeper database")
        try_to_call(["scrollkeeper-update", "-q", "-o", omf_dir])
    
    def remove(self, path):
        """
        Wraps the remove function, so that the dry_run parameter doesn't need
        to be set whenever it's called.
        """
        
        remove(path, self.dry_run)

INSTALLED_FILES = "INSTALLED_FILES"

class InstallCommand(InstallWrapper):
    user_options = InstallWrapper.user_options + [
        ("packaging-mode", "p", "don't perform post-installation operations"),
        ("upgrade", "u", "upgrade existing installations without confirmation")
    ]
    
    boolean_options = InstallWrapper.boolean_options + [
        "packaging-mode", "upgrade"
    ]
    
    def initialize_options(self):
        self.packaging_mode = False
        self.upgrade = False
        self.record = None
        
        InstallWrapper.initialize_options(self)
    
    def finalize_options(self):
        InstallWrapper.finalize_options(self)
        
        if self.upgrade:
            self.force = True
        
        if not self.record and not self.packaging_mode:
            self.record = INSTALLED_FILES
    
    def run(self):
        if not self.packaging_mode:
            self.upgrade_existing_installation()
        
        install.install.run(self)
        
        if os.name == "posix":
            # Make postprocessor.py executable.
            script = join(self.install_purelib, "lottanzb", "postprocessor.py")
            self.make_executable(script)
        
        if not self.packaging_mode:
            self.complete_installation()
        
        if self.record and isfile(self.record):
            log.info("\nLottaNZB can be uninstalled using 'python setup.py "
                "uninstall'.")
            log.info("This command depends on the newly created file %s.", \
                self.record)
    
    def make_executable(self, a_file):
        if not self.dry_run:
            os.chmod(a_file, ((os.stat(a_file)[ST_MODE]) | 0555) & 07777)
        
        log.info("making %s executable", a_file)
    
    def upgrade_existing_installation(self):
        """
        Looks for exiting installations of LottaNZB in the target installation
        directory, which is usually /usr/lib/python2.x/site-packages.
        
        If the user didn't specify the force or upgrade flag explicitly,
        request a confirmation for the removal of all found installation dirs.
        
        Data files aren't removed, but overwritten by setting the force flag.
        """
        
        lotta_dirs = glob(join(self.install_purelib, "%s*" % NAME))
        installed_version = ""
        
        try:
            module_info = find_module(NAME, [self.install_purelib])
            module = load_module(NAME, *module_info)
        except ImportError:
            pass
        else:
            if hasattr(module, "__version__"):
                installed_version = module.__version__
            else:
                installed_version = "<= 0.3"
        
        if installed_version:
            log.info("An existing installation of LottaNZB %s has been "
                "detected on your system.", installed_version)
            log.info("To avoid conflicts with the new version, the following "
                "folders will be removed:")
            
            for lotta_dir in lotta_dirs:
                log.info(" * %s", lotta_dir)
            
            if not self.upgrade and not self.force:
                selection = raw_input("\nWould you like to upgrade to LottaNZB "
                    "%s? [Y/n]: " % version)
                
                if selection.lower() != "y":
                    raise DistutilsArgError("Aborting installation...")
            
            log.info("Upgrading to LottaNZB %s...", version)
            
            for lotta_dir in lotta_dirs:
                self.remove(lotta_dir)
            
            self.force = True

class UpgradeCommand(InstallCommand):
    description = "upgrade existing installation"
    
    def finalize_options(self):
        InstallCommand.finalize_options(self)
        
        self.upgrade = True

class UninstallCommand(InstallWrapper):
    def run(self):
        try:
            files = file(INSTALLED_FILES, "r").readlines()
        except:
            raise DistutilsFileError("Could not find list of installed files "
                "(%s)." % INSTALLED_FILES)
        
        for a_file in [a_file.strip() for a_file in files]:
            try:
                self.remove(a_file)
            except OSError, error:
                if error.errno == 13:
                    raise DistutilsFileError("Root privileges are required to "
                        "uninstall LottaNZB.")
        
        if not self.dry_run:
            self.remove(INSTALLED_FILES)
            self.complete_installation()
        
        log.info("\nLottaNZB has been uninstalled successfully.")

class BuildExtra(build.build):
    """Adds the extra commands to the build target."""
    
    def finalize_options(self):
        build.build.finalize_options(self)
        
        self.sub_commands.append(("build_i18n",  None))
        self.sub_commands.append(("build_icons", None))
        self.sub_commands.append(("build_help",  None))

class BuildIcons(Command):
    description = "select all icons for installation"
    user_options = [("icon-dir=", "i", "icon directory of the source tree")]
    
    def initialize_options(self):
        self.icon_dir = None
    
    def finalize_options(self):
        if self.icon_dir is None:
            self.icon_dir = "data/icons"
    
    def run(self):
        def add(theme, size, category, pattern):
            dest = join("share", "icons", theme, size, category)
            files = glob(join(self.icon_dir, size, category, pattern))
            
            if files:
                # Don't create empty directories
                self.distribution.data_files.append((dest, files))
        
        for size_path in glob(join(self.icon_dir, "*")):
            size = basename(size_path)
            
            for category_path in glob(join(size_path, "*")):
                category = basename(category_path)
                pattern = "*.png"
                
                if category == "apps":
                    add("hicolor", size, category, pattern)
                elif category == "mimetypes":
                    if size == "scalable":
                        pattern = "*.*"
                    
                    add("gnome", size, category, pattern)

class BuildI18N(Command):
    description = "integrate the gettext framework"
    boolean_options = ["update", "compile", "merge"]
    user_options = [
        ("desktop-files=",   None, ".desktop.in files that should be merged"),
        ("xml-files=",       None, ".xml.in files that should be merged"),
        ("key-files=",       None, ".key.in files that should be merged"),
        ("update",           "u",  "update .pot file and all .po files"),
        ("compile",          "c",  "compile all .po files into .mo"),
        ("merge",            "m",  "generate .desktop, .xml and key files")
    ]
    
    def initialize_options(self):
        self.domain = self.distribution.metadata.name
        
        self.desktop_files = []
        self.xml_files = []
        self.key_files = []
        
        self.update = False
        self.compile = False
        self.merge = False
    
    def finalize_options(self):
        pass
    
    def run(self):
        data_files = self.distribution.data_files
        root = os.getcwd()
        intltool_map = (
            (self.xml_files, "-x"),
            (self.desktop_files, "-d"),
            (self.key_files, "-k"),
        )
        
        if self.update:
            update_po(root, self.domain)
        
        if self.compile:
            compile_po_files(root, self.domain)
        
        for lang in list_languages(root):
            mo_dir = join("locale", lang, "LC_MESSAGES")
            mo_file = join(mo_dir, "%s.mo" % self.domain)
            
            if isfile(mo_file):
                data_files.append((join("share", mo_dir), [mo_file]))
        
        for (option, switch) in intltool_map:
            try:
                file_set = eval(option)
            except:
                continue
            for (target, files) in file_set:
                for file in files:
                    file_merged = file[:-3]
                    
                    if self.merge:
                        call(["intltool-merge", switch, "po", file, file_merged])
                    
                    if isfile(file_merged):
                        data_files.append((target, [file_merged]))

class BuildHelp(Command):
    description = "install a docbook based documentation"
    
    user_options = [
        ("help-dir=", None, "help directory of the source tree"),
        ("update",   "u",   "update po, pot and xml files")
    ]
    
    boolean_options = ["update"]
    
    def initialize_options(self):
        self.help_dir = None
        self.update = False
    
    def finalize_options(self):
        if self.help_dir is None:
            self.help_dir = "help"
    
    def run(self):
        def help_dir(*args):
            return join(self.help_dir, *args)
        
        langs = [basename(path) for path in glob(help_dir("*")) if isdir(path) \
                 and not basename(path) == "po"]
        
        original_omf = help_dir("C", "lottanzb-C.omf")
        original_xml = help_dir("C", "lottanzb.xml")
        pot_file = help_dir("po", "documentation.pot")
        
        if self.update:
            if self.has_xml2po():
                call(["xml2po", "-k", "-o", pot_file, original_xml, 
                      original_omf])
                
                for lang in langs:
                    if not lang == "C":
                        dest_omf = abspath(help_dir(lang, "lottanzb-%s.omf" % lang))
                        dest_xml = help_dir(lang, "lottanzb.xml")
                        dest_po = help_dir("po", "%s.po" % lang)
                        
                        call(["msgmerge", "-U", dest_po, pot_file])
                        call(["xml2po", "-k", "-p", dest_po, "-o", dest_xml, original_xml])
                        call(["xml2po", "-k", "-p", dest_po, "-o", dest_omf, original_omf])
                        
                        cmd = ("sed 's/\"C\"/\"%s\"/' %s | " + \
                            "sed 's/\/C\//\/%s\//'") % (lang, dest_omf, lang)
                        
                        process = Popen(cmd, shell=True, stdout=PIPE)
                        
                        if not process.wait():
                            file = open(dest_omf, "w")
                            file.write(process.stdout.read())
            else:
                self.error("Please install the gnome-doc-utils package first.")
        
        data_files = self.distribution.data_files
        app_name = self.distribution.metadata.name
        
        for lang in langs:
            path_xml = join("share", "gnome", "help", app_name, lang)
            path_figures = join("share", "gnome", "help", app_name, lang, "figures")
            
            data_files.append((path_xml, (glob(help_dir(lang, "*.xml")))))
            data_files.append((path_figures, (glob(help_dir(lang, "figures", "*.png")))))
        
        data_files.append((join("share", "omf", app_name), glob(help_dir("*", "*.omf"))))
    
    @staticmethod
    def has_xml2po():
        try:
            return call(["which", "xml2po"]) == 0
        except:
            return False

class BuildRPM(bdist_rpm.bdist_rpm):
    def finalize_options(self):
        bdist_rpm.bdist_rpm.finalize_options(self)
        
        if isfile("/etc/mandriva-release"):
            dist_release = open("/etc/mandriva-release").read().split(" ")[3]
            
            self.distribution_name = "Mandriva Linux %s" % dist_release
            self.release = "%s.mdv%s" % (self.release, dist_release)
            self.requires = "python >= 2.4 hellanzb >= 0.13 pygtk2.0 " + \
                "pygtk2.0-libglade python-kiwi >= 1.9.9"
        
        elif isfile("/etc/fedora-release"):
            dist_release = open("/etc/fedora-release").read().split(" ")[2]
            
            self.distribution_name = "Fedora %s" % dist_release
            self.release = "%s.fc%s" % (self.release, dist_release)
            self.requires = "python >= 2.4 hellanzb >= 0.13 pygtk2 " + \
                "pygtk2-libglade python-kiwi >= 1.9.9"

def setup(**kwargs):
    if "cmdclass" in kwargs:
        def fixed_setup(**inner_kwargs):
            inner_kwargs["cmdclass"].update(kwargs["cmdclass"])
            distutils_setup(**inner_kwargs)
        
        kiwi.dist.DS_setup = fixed_setup
    
    kiwi_setup(**kwargs)

options = {
    "name"             : NAME,
    "version"          : VERSION,
    "description"      : DESCRIPTION,
    "long_description" : LONG_DESCRIPTION,
    "author"           : AUTHOR,
    "author_email"     : AUTHOR_EMAIL,
    "url"              : URL,
    "packages"         : listpackages("lottanzb"),
    "license"          : LICENSE,
    "scripts"          : ["data/lottanzb"],
    "keywords"         : KEYWORDS,
    "data_files"       : DATA_FILES,
    "classifiers"      : CLASSIFIERS,
    "global_resources" : {
                            "glade": "$datadir/glade",
                            "locale": "$prefix/share/locale"
                         },
    "resources"        : {
                            "help": "$prefix/share/gnome/help/lottanzb"
                         },
    "cmdclass"         : {
                            "build"         : BuildExtra,
                            "build_i18n"    : BuildI18N,
                            "build_help"    : BuildHelp,
                            "build_icons"   : BuildIcons,
                            "install"       : InstallCommand,
                            "upgrade"       : UpgradeCommand,
                            "uninstall"     : UninstallCommand,
                            "clean"         : CleanCommand,
                            "bdist_rpm"     : BuildRPM
                         }
}

if __name__ == "__main__":
    setup(**options)
