/** @file wlan_cmdresp.c
  * @brief This file contains the handling of command
  * responses as well as events generated by firmware.
  *
  * (c) Copyright  2003-2007, Marvell International Ltd. 
  *  
  * This software file (the "File") is distributed by Marvell International 
  * Ltd. under the terms of the GNU General Public License Version 2, June 1991 
  * (the "License").  You may use, redistribute and/or modify this File in 
  * accordance with the terms and conditions of the License, a copy of which 
  * is available along with the File in the gpl.txt file or by writing to 
  * the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 
  * 02111-1307 or on the worldwide web at http://www.gnu.org/licenses/gpl.txt.
  *
  * THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE 
  * IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE 
  * ARE EXPRESSLY DISCLAIMED.  The License provides additional details about 
  * this warranty disclaimer.
  *
  */
/********************************************************
Change log:
	10/10/05: Add Doxygen format comments
	11/11/05: Add support for WMM Status change event
	12/13/05: Add Proprietary periodic sleep support
	12/23/05: Fix bug in adhoc start where the current index was
	          not properly being assigned before it was used.
	01/05/06: Add kernel 2.6.x support	
	01/11/06: Conditionalize new scan/join structures.
	          Update assoc response handling; entire IEEE response returned
	04/06/06: Add TSPEC, queue metrics, and MSDU expiry support
	04/10/06: Add hostcmd generic API
	04/18/06: Remove old Subscrive Event and add new Subscribe Event
	          implementation through generic hostcmd API
	05/04/06: Add IBSS coalescing related new hostcmd response handling
	05/08/06: Remove PermanentAddr from Adapter
	06/08/06: Remove function HandleMICFailureEvent()
	08/29/06: Add ledgpio private command
********************************************************/

#include	"wlan_headers.h"

/********************************************************
		Local Variables
********************************************************/

/********************************************************
		Global Variables
********************************************************/

/********************************************************
		Local Functions
********************************************************/

/** 
 *  @brief This function handles disconnect event. it
 *  reports disconnect to upper layer, clean tx/rx packets,
 *  reset link state etc.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   n/a
 */
void
wlan_reset_connect_state(wlan_private * priv)
{
    wlan_adapter *Adapter = priv->adapter;
    union iwreq_data wrqu;

    ENTER();

    if (Adapter->MediaConnectStatus != WlanMediaStateConnected)
        return;

    PRINTM(INFO, "Handles disconnect event.\n");

    /* Free Tx and Rx packets, report disconnect to upper layer */
    wlan_clean_txrx(priv);

    memset(wrqu.ap_addr.sa_data, 0x00, ETH_ALEN);
    wrqu.ap_addr.sa_family = ARPHRD_ETHER;

    wireless_send_event(priv->wlan_dev.netdev, SIOCGIWAP, &wrqu, NULL);

    /* reset SNR/NF/RSSI values */
    memset(Adapter->SNR, 0x00, sizeof(Adapter->SNR));
    memset(Adapter->NF, 0x00, sizeof(Adapter->NF));
    memset(Adapter->RSSI, 0x00, sizeof(Adapter->RSSI));
    memset(Adapter->rawSNR, 0x00, sizeof(Adapter->rawSNR));
    memset(Adapter->rawNF, 0x00, sizeof(Adapter->rawNF));
    Adapter->nextSNRNF = 0;
    Adapter->numSNRNF = 0;
    Adapter->RxPDRate = 0;
    PRINTM(INFO, "Current SSID=%s, Ssid Length=%u\n",
           Adapter->CurBssParams.BSSDescriptor.Ssid.Ssid,
           Adapter->CurBssParams.BSSDescriptor.Ssid.SsidLength);
    PRINTM(INFO, "Previous SSID=%s, Ssid Length=%u\n",
           Adapter->PreviousSSID.Ssid, Adapter->PreviousSSID.SsidLength);

    Adapter->SecInfo.WPAEnabled = FALSE;
    Adapter->SecInfo.WPA2Enabled = FALSE;
    Adapter->Wpa_ie_len = 0;
    Adapter->SecInfo.EncryptionMode = CIPHER_NONE;

    Adapter->MediaConnectStatus = WlanMediaStateDisconnected;
    Adapter->AdhocState = ADHOC_IDLE;
    Adapter->AdhocLinkSensed = FALSE;

    /* 
     * memorize the previous SSID and BSSID
     * it could be used for re-assoc
     */
    memcpy(&Adapter->PreviousSSID,
           &Adapter->CurBssParams.BSSDescriptor.Ssid,
           sizeof(WLAN_802_11_SSID));
    memcpy(Adapter->PreviousBSSID,
           Adapter->CurBssParams.BSSDescriptor.MacAddress, ETH_ALEN);

    /* need to erase the current SSID and BSSID info */
    memset(&Adapter->CurBssParams, 0x00, sizeof(Adapter->CurBssParams));

    if (Adapter->PSState != PS_STATE_FULL_POWER) {
        /* make firmware to exit PS mode */
        PRINTM(INFO, "Disconnected, so exit PS mode.\n");
        wlan_exit_ps(priv, 0);
    }

    LEAVE();
}

/** 
 *  @brief This function handles link lost, deauth and
 *  disassoc events.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   n/a
 */
static void
wlan_handle_disconnect_event(wlan_private * priv)
{
    wlan_adapter *Adapter = priv->adapter;

    ENTER();

    if (Adapter->MediaConnectStatus == WlanMediaStateConnected) {
        wlan_reset_connect_state(priv);
#ifdef REASSOCIATION
        if (Adapter->Reassoc_on == TRUE) {
            PRINTM(INFO, "RE-ASSOC: trigger the timer\n");
            Adapter->ReassocTimerIsSet = TRUE;
            wlan_mod_timer(&Adapter->MrvDrvTimer, 0);
        }
#endif /* REASSOCIATION */
    }
}

/** 
 *  @brief This function handles the command response of reg_access
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param type	   the type of reg access (MAC, BBP or RF)
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int
wlan_ret_reg_access(wlan_private * priv, u16 type, HostCmd_DS_COMMAND * resp)
{
    wlan_adapter *Adapter = priv->adapter;
    wlan_offset_value *pOffsetValue =
        (wlan_offset_value *) Adapter->CurCmd->pdata_buf;

    ENTER();

    switch (type) {
    case HostCmd_CMD_MAC_REG_ACCESS:
        {
            HostCmd_DS_MAC_REG_ACCESS *reg;

            reg = (HostCmd_DS_MAC_REG_ACCESS *) & resp->params.macreg;

            pOffsetValue->offset = wlan_le16_to_cpu(reg->Offset);
            pOffsetValue->value = wlan_le32_to_cpu(reg->Value);
            break;
        }

    case HostCmd_CMD_BBP_REG_ACCESS:
        {
            HostCmd_DS_BBP_REG_ACCESS *reg;
            reg = (HostCmd_DS_BBP_REG_ACCESS *) & resp->params.bbpreg;

            pOffsetValue->offset = wlan_le16_to_cpu(reg->Offset);
            pOffsetValue->value = (u8) reg->Value;
            break;
        }

    case HostCmd_CMD_RF_REG_ACCESS:
        {
            HostCmd_DS_RF_REG_ACCESS *reg;
            reg = (HostCmd_DS_RF_REG_ACCESS *) & resp->params.rfreg;

            pOffsetValue->offset = wlan_le16_to_cpu(reg->Offset);
            pOffsetValue->value = (u8) reg->Value;
            break;
        }

    default:
        LEAVE();
        return WLAN_STATUS_FAILURE;
    }

    LEAVE();
    return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of get_hw_spec
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int
wlan_ret_get_hw_spec(wlan_private * priv, HostCmd_DS_COMMAND * resp)
{
    u32 i;
    HostCmd_DS_GET_HW_SPEC *hwspec = &resp->params.hwspec;
    wlan_adapter *Adapter = priv->adapter;
    int ret = WLAN_STATUS_SUCCESS;

    ENTER();

    Adapter->fwCapInfo = wlan_le32_to_cpu(hwspec->fwCapInfo);

    if (IS_SUPPORT_MULTI_BANDS(Adapter)) {
        Adapter->fw_bands = GET_FW_DEFAULT_BANDS(Adapter);
        Adapter->is_multiband = 1;
    } else {
        Adapter->adhoc_start_band = BAND_B;
        Adapter->fw_bands = BAND_B;
        Adapter->is_multiband = 0;
    }

    Adapter->config_bands = Adapter->fw_bands;

    if (Adapter->fw_bands & BAND_A) {
        Adapter->adhoc_start_band = BAND_A;
        Adapter->AdhocChannel = DEFAULT_AD_HOC_CHANNEL_A;
    } else if (Adapter->fw_bands & BAND_B) {
        Adapter->adhoc_start_band = BAND_B;
        Adapter->AdhocChannel = DEFAULT_AD_HOC_CHANNEL;
    } else if (Adapter->fw_bands & BAND_G) {
        Adapter->adhoc_start_band = BAND_G;
        Adapter->AdhocChannel = DEFAULT_AD_HOC_CHANNEL;
    }

    Adapter->FWReleaseNumber = wlan_le32_to_cpu(hwspec->FWReleaseNumber);

    PRINTM(INFO, "GET_HW_SPEC: FWReleaseVersion- 0x%X\n",
           Adapter->FWReleaseNumber);
    PRINTM(INFO, "GET_HW_SPEC: Permanent addr- %2x:%2x:%2x:%2x:%2x:%2x\n",
           hwspec->PermanentAddr[0], hwspec->PermanentAddr[1],
           hwspec->PermanentAddr[2], hwspec->PermanentAddr[3],
           hwspec->PermanentAddr[4], hwspec->PermanentAddr[5]);
    PRINTM(INFO, "GET_HW_SPEC: HWIfVersion=0x%X  Version=0x%X\n",
           wlan_le16_to_cpu(hwspec->HWIfVersion),
           wlan_le16_to_cpu(hwspec->Version));

    Adapter->RegionCode = wlan_le16_to_cpu(hwspec->RegionCode);

    for (i = 0; i < MRVDRV_MAX_REGION_CODE; i++) {
        /* use the region code to search for the index */
        if (Adapter->RegionCode == RegionCodeToIndex[i]) {
            break;
        }
    }

    /* if it's unidentified region code, use the default (USA) */
    if (i >= MRVDRV_MAX_REGION_CODE) {
        Adapter->RegionCode = 0x10;
        PRINTM(WARN, "unidentified region code, use the default (USA)\n");
    }

    if (Adapter->CurrentAddr[0] == 0xff) {
        memmove(Adapter->CurrentAddr, hwspec->PermanentAddr, ETH_ALEN);
    }
    memcpy(priv->wlan_dev.netdev->dev_addr, Adapter->CurrentAddr, ETH_ALEN);

    if (wlan_set_regiontable(priv, Adapter->RegionCode, Adapter->fw_bands)) {
        ret = WLAN_STATUS_FAILURE;
        goto done;
    }

    if (wlan_set_universaltable(priv, Adapter->fw_bands)) {
        ret = WLAN_STATUS_FAILURE;
        goto done;
    }

  done:
    LEAVE();
    return ret;
}

/** 
 *  @brief This function handles the command response of host_sleep_cfg
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int
wlan_ret_host_sleep_cfg(wlan_private * priv, HostCmd_DS_COMMAND * resp)
{
    wlan_adapter *Adapter = priv->adapter;
    HostCmd_DS_802_11_HOST_SLEEP_CFG *hscfg = &resp->params.hostsleepcfg;
    int ret = WLAN_STATUS_SUCCESS;

    ENTER();

    if (hscfg->conditions != HOST_SLEEP_CFG_CANCEL) {
        Adapter->bHostSleepConfigured = TRUE;
    } else {
        Adapter->bHostSleepConfigured = FALSE;
        if (Adapter->PSState == PS_STATE_FULL_POWER && Adapter->HS_Activated) {
            wlan_host_sleep_deactivated_event(priv);
        }
        os_start_queue(priv);
        os_carrier_on(priv);
        wmm_start_queue(priv);
    }

    LEAVE();
    return ret;
}

/** 
 *  @brief This function handles the command response of fw_wakeup_method
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int
wlan_ret_fw_wakeup_method(wlan_private * priv, HostCmd_DS_COMMAND * resp)
{
    wlan_adapter *Adapter = priv->adapter;
    HostCmd_DS_802_11_FW_WAKEUP_METHOD *fwwm = &resp->params.fwwakeupmethod;
    u16 action;

    ENTER();

    action = wlan_le16_to_cpu(fwwm->Action);

    switch (action) {
    case HostCmd_ACT_GEN_GET:
    case HostCmd_ACT_GEN_SET:
        Adapter->fwWakeupMethod = wlan_le16_to_cpu(fwwm->Method);
        break;
    default:
        break;
    }

    LEAVE();
    return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of sleep_params
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int
wlan_ret_802_11_sleep_params(wlan_private * priv, HostCmd_DS_COMMAND * resp)
{
    HostCmd_DS_802_11_SLEEP_PARAMS *sp = &resp->params.sleep_params;
    wlan_adapter *Adapter = priv->adapter;

    ENTER();

    PRINTM(INFO, "error=%x offset=%x stabletime=%x calcontrol=%x\n"
           " extsleepclk=%x\n", sp->Error, sp->Offset,
           sp->StableTime, sp->CalControl, sp->ExternalSleepClk);
    Adapter->sp.sp_error = wlan_le16_to_cpu(sp->Error);
    Adapter->sp.sp_offset = wlan_le16_to_cpu(sp->Offset);
    Adapter->sp.sp_stabletime = wlan_le16_to_cpu(sp->StableTime);
    Adapter->sp.sp_calcontrol = wlan_le16_to_cpu(sp->CalControl);
    Adapter->sp.sp_extsleepclk = wlan_le16_to_cpu(sp->ExternalSleepClk);
    Adapter->sp.sp_reserved = wlan_le16_to_cpu(sp->Reserved);

    LEAVE();
    return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of sleep_params
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int
wlan_ret_802_11_sleep_period(wlan_private * priv, HostCmd_DS_COMMAND * resp)
{
    HostCmd_DS_802_11_SLEEP_PERIOD *sp_period = &resp->params.ps_sleeppd;
    wlan_adapter *Adapter = priv->adapter;

    ENTER();

    Adapter->sleep_period.period = wlan_le16_to_cpu(sp_period->Period);

    LEAVE();
    return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of bca_timeshare
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int
wlan_ret_802_11_bca_timeshare(wlan_private * priv, HostCmd_DS_COMMAND * resp)
{
    HostCmd_DS_802_11_BCA_TIMESHARE *bca_ts = &resp->params.bca_timeshare;
    wlan_adapter *Adapter = priv->adapter;

    ENTER();

    PRINTM(MSG, "TrafficType=%x TimeShareInterva=%x BTTime=%x\n",
           bca_ts->TrafficType, bca_ts->TimeShareInterval, bca_ts->BTTime);

    Adapter->bca_ts.TrafficType = wlan_le16_to_cpu(bca_ts->TrafficType);
    Adapter->bca_ts.TimeShareInterval =
        wlan_le32_to_cpu(bca_ts->TimeShareInterval);
    Adapter->bca_ts.BTTime = wlan_le32_to_cpu(bca_ts->BTTime);

    LEAVE();
    return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of mac_control
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int
wlan_ret_mac_control(wlan_private * priv, HostCmd_DS_COMMAND * resp)
{
    return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of set_wep
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int
wlan_ret_802_11_set_wep(wlan_private * priv, HostCmd_DS_COMMAND * resp)
{
    return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of reset
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int
wlan_ret_802_11_reset(wlan_private * priv, HostCmd_DS_COMMAND * resp)
{
    ENTER();
    PRINTM(INFO, "HWAC - Reset command successful\n");

    return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of snmp_mib
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int
wlan_ret_802_11_snmp_mib(wlan_private * priv, HostCmd_DS_COMMAND * resp)
{
    HostCmd_DS_802_11_SNMP_MIB *smib = &resp->params.smib;
    u16 OID = wlan_le16_to_cpu(smib->OID);
    u16 QueryType = wlan_le16_to_cpu(smib->QueryType);

    ENTER();

    PRINTM(INFO, "SNMP_RESP: value of the OID = %x, QueryType=%x\n", OID,
           QueryType);
    PRINTM(INFO, "SNMP_RESP: Buf size  = %x\n",
           wlan_le16_to_cpu(smib->BufSize));

    if (QueryType == HostCmd_ACT_GEN_GET) {
        switch (OID) {
        case FragThresh_i:
            priv->adapter->FragThsd =
                wlan_le16_to_cpu(*((PUSHORT) (smib->Value)));
            PRINTM(INFO, "SNMP_RESP: FragThsd =%u\n",
                   priv->adapter->FragThsd);
            break;
        case RtsThresh_i:
            priv->adapter->RTSThsd =
                wlan_le16_to_cpu(*((PUSHORT) (smib->Value)));
            PRINTM(INFO, "SNMP_RESP: RTSThsd =%u\n", priv->adapter->RTSThsd);
            break;
        case ShortRetryLim_i:
            priv->adapter->TxRetryCount =
                wlan_le16_to_cpu(*((PUSHORT) (smib->Value)));
            PRINTM(INFO, "SNMP_RESP: TxRetryCount =%u\n",
                   priv->adapter->RTSThsd);
            break;
        default:
            break;
        }
    }

    LEAVE();
    return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of radio_control
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int
wlan_ret_802_11_radio_control(wlan_private * priv, HostCmd_DS_COMMAND * resp)
{
    ENTER();

    LEAVE();
    return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of key_material
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int
wlan_ret_802_11_key_material(wlan_private * priv, HostCmd_DS_COMMAND * resp)
{
    HostCmd_DS_802_11_KEY_MATERIAL *pKey = &resp->params.keymaterial;
    wlan_adapter *Adapter = priv->adapter;

    ENTER();

    if (wlan_le16_to_cpu(pKey->Action) == HostCmd_ACT_GEN_SET) {
        if ((wlan_le16_to_cpu(pKey->KeyParamSet.KeyInfo) &
             KEY_INFO_TKIP_MCAST)
            || (wlan_le16_to_cpu(pKey->KeyParamSet.KeyInfo) &
                KEY_INFO_AES_MCAST)) {
            PRINTM(INFO, "Key: GTK is set\n");
            Adapter->IsGTK_SET = TRUE;
        }
    }

    memcpy(Adapter->aeskey.KeyParamSet.Key, pKey->KeyParamSet.Key,
           sizeof(pKey->KeyParamSet.Key));

    LEAVE();
    return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of mac_address
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int
wlan_ret_802_11_mac_address(wlan_private * priv, HostCmd_DS_COMMAND * resp)
{
    HostCmd_DS_802_11_MAC_ADDRESS *MacAdd = &resp->params.macadd;
    wlan_adapter *Adapter = priv->adapter;

    ENTER();

    memcpy(Adapter->CurrentAddr, MacAdd->MacAdd, ETH_ALEN);

    LEAVE();
    return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of rf_tx_power
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int
wlan_ret_802_11_rf_tx_power(wlan_private * priv, HostCmd_DS_COMMAND * resp)
{
    HostCmd_DS_802_11_RF_TX_POWER *rtp = &resp->params.txp;
    wlan_adapter *Adapter = priv->adapter;
    u16 Action = wlan_le16_to_cpu(rtp->Action);

    ENTER();

    Adapter->TxPowerLevel = wlan_le16_to_cpu(rtp->CurrentLevel);

    if (Action == HostCmd_ACT_GEN_GET) {
        Adapter->MaxTxPowerLevel = rtp->MaxPower;
        Adapter->MinTxPowerLevel = rtp->MinPower;
    }

    PRINTM(INFO, "Current TxPower Level = %d,Max Power=%d, Min Power=%d\n",
           Adapter->TxPowerLevel, Adapter->MaxTxPowerLevel,
           Adapter->MinTxPowerLevel);

    LEAVE();
    return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of rf_antenna
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int
wlan_ret_802_11_rf_antenna(wlan_private * priv, HostCmd_DS_COMMAND * resp)
{
    HostCmd_DS_802_11_RF_ANTENNA *pAntenna = &resp->params.rant;
    wlan_adapter *Adapter = priv->adapter;
    u16 Action = wlan_le16_to_cpu(pAntenna->Action);

    if (Action == HostCmd_ACT_GET_RX)
        Adapter->RxAntennaMode = wlan_le16_to_cpu(pAntenna->AntennaMode);

    if (Action == HostCmd_ACT_GET_TX)
        Adapter->TxAntennaMode = wlan_le16_to_cpu(pAntenna->AntennaMode);

    PRINTM(INFO, "RF_ANT_RESP: Action = 0x%x, Mode = 0x%04x\n",
           Action, wlan_le16_to_cpu(pAntenna->AntennaMode));

    return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of multicast_address
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int
wlan_ret_mac_multicast_adr(wlan_private * priv, HostCmd_DS_COMMAND * resp)
{
    return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of rate_adapt_rateset
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int
wlan_ret_802_11_rate_adapt_rateset(wlan_private * priv,
                                   HostCmd_DS_COMMAND * resp)
{
    HostCmd_DS_802_11_RATE_ADAPT_RATESET *rates = &resp->params.rateset;
    wlan_adapter *Adapter = priv->adapter;

    ENTER();

    if (wlan_le16_to_cpu(rates->Action) == HostCmd_ACT_GEN_GET) {
        Adapter->HWRateDropMode = wlan_le16_to_cpu(rates->HWRateDropMode);
        Adapter->Threshold = wlan_le16_to_cpu(rates->Threshold);
        Adapter->FinalRate = wlan_le16_to_cpu(rates->FinalRate);
        Adapter->RateBitmap = wlan_le16_to_cpu(rates->Bitmap);
    }

    LEAVE();

    return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of rf_channel
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int
wlan_ret_802_11_rf_channel(wlan_private * priv, HostCmd_DS_COMMAND * resp)
{
    HostCmd_DS_802_11_RF_CHANNEL *rfchannel = &resp->params.rfchannel;
    wlan_adapter *Adapter = priv->adapter;
    u16 Action = wlan_le16_to_cpu(rfchannel->Action);
    u16 newChannel = wlan_le16_to_cpu(rfchannel->CurrentChannel);

    ENTER();

    if (Action == HostCmd_OPT_802_11_RF_CHANNEL_GET
        && Adapter->CurBssParams.BSSDescriptor.Channel != newChannel) {
        PRINTM(INFO, "Channel Switch: %d to %d\n",
               Adapter->CurBssParams.BSSDescriptor.Channel, newChannel);

        /* Update the channel again */
        Adapter->CurBssParams.BSSDescriptor.Channel = newChannel;
    }

    LEAVE();
    return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of rssi
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int
wlan_ret_802_11_rssi(wlan_private * priv, HostCmd_DS_COMMAND * resp)
{
    HostCmd_DS_802_11_RSSI_RSP *rssirsp = &resp->params.rssirsp;
    wlan_adapter *Adapter = priv->adapter;

    /* store the non average value */
    Adapter->SNR[TYPE_BEACON][TYPE_NOAVG] = wlan_le16_to_cpu(rssirsp->SNR);
    Adapter->NF[TYPE_BEACON][TYPE_NOAVG] =
        wlan_le16_to_cpu(rssirsp->NoiseFloor);

    Adapter->SNR[TYPE_BEACON][TYPE_AVG] = wlan_le16_to_cpu(rssirsp->AvgSNR);
    Adapter->NF[TYPE_BEACON][TYPE_AVG] =
        wlan_le16_to_cpu(rssirsp->AvgNoiseFloor);

    Adapter->RSSI[TYPE_BEACON][TYPE_NOAVG] =
        CAL_RSSI(Adapter->SNR[TYPE_BEACON][TYPE_NOAVG],
                 Adapter->NF[TYPE_BEACON][TYPE_NOAVG]);

    Adapter->RSSI[TYPE_BEACON][TYPE_AVG] =
        CAL_RSSI(Adapter->SNR[TYPE_BEACON][TYPE_AVG] / AVG_SCALE,
                 Adapter->NF[TYPE_BEACON][TYPE_AVG] / AVG_SCALE);

    PRINTM(INFO, "Beacon RSSI value = 0x%x\n",
           Adapter->RSSI[TYPE_BEACON][TYPE_AVG]);

    return WLAN_STATUS_SUCCESS;
}

#ifdef MFG_CMD_SUPPORT
/** 
 *  @brief This function handles the command response of mfg_cmd
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int
wlan_ret_mfg_cmd(wlan_private * priv, HostCmd_DS_COMMAND * resp)
{
    wlan_adapter *Adapter = priv->adapter;

    ENTER();

    PRINTM(INFO, "MFG command response size = %d\n", resp->Size);

    resp->Size = MIN(resp->Size, MRVDRV_SIZE_OF_CMD_BUFFER);
    memcpy(Adapter->CurCmd->pdata_buf, (void *) resp, resp->Size);

    LEAVE();
    return WLAN_STATUS_SUCCESS;
}
#endif /* MFG_CMD_SUPPORT */

/** 
 *  @brief This function handles the command response of cal_data_ext.
 *  
 *  @param priv		A pointer to wlan_private structure
 *  @param resp		A pointer to HostCmd_DS_COMMAND
 *  @return    		WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int
wlan_ret_802_11_cal_data_ext(wlan_private * priv, HostCmd_DS_COMMAND * resp)
{
    wlan_adapter *Adapter = priv->adapter;
    HostCmd_DS_802_11_CAL_DATA_EXT *pCalDataExt = &resp->params.caldataext;

    ENTER();

    if (wlan_le16_to_cpu(pCalDataExt->Action) == HostCmd_ACT_GEN_GET) {
        pCalDataExt->Action = wlan_le16_to_cpu(pCalDataExt->Action);
        pCalDataExt->Revision = wlan_le16_to_cpu(pCalDataExt->Revision);
        pCalDataExt->CalDataLen = wlan_le16_to_cpu(pCalDataExt->CalDataLen);

        memmove(Adapter->CurCmd->pdata_buf,
                pCalDataExt, pCalDataExt->CalDataLen + CAL_DATA_HEADER_LEN);
    }

    LEAVE();
    return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of eeprom_access
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int
wlan_ret_802_11_eeprom_access(wlan_private * priv, HostCmd_DS_COMMAND * resp)
{
    wlan_adapter *Adapter = priv->adapter;
    wlan_ioctl_regrdwr *pBuf =
        (wlan_ioctl_regrdwr *) Adapter->CurCmd->pdata_buf;

    PRINTM(INFO, "eeprom read len=%x\n",
           wlan_le16_to_cpu(resp->params.rdeeprom.ByteCount));
    if (pBuf->NOB < wlan_le16_to_cpu(resp->params.rdeeprom.ByteCount)) {
        pBuf->NOB = 0;
        PRINTM(INFO, "eeprom read return length is too big\n");
        return WLAN_STATUS_FAILURE;
    }
    pBuf->NOB = wlan_le16_to_cpu(resp->params.rdeeprom.ByteCount);
    if (pBuf->NOB > 0) {
        memcpy(&pBuf->Value, (u8 *) & resp->params.rdeeprom.Value, pBuf->NOB);
        HEXDUMP("EEPROM", (char *) &pBuf->Value, pBuf->NOB);
    }
    return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of get_log
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int
wlan_ret_get_log(wlan_private * priv, HostCmd_DS_COMMAND * resp)
{
    HostCmd_DS_802_11_GET_LOG *LogMessage =
        (HostCmd_DS_802_11_GET_LOG *) & resp->params.glog;
    wlan_adapter *Adapter = priv->adapter;

    ENTER();

    memcpy(&Adapter->LogMsg, LogMessage, sizeof(HostCmd_DS_802_11_GET_LOG));
    endian_convert_GET_LOG(Adapter->LogMsg);

    LEAVE();
    return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief This function handles the command response of crypto
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param resp	   A pointer to HostCmd_DS_COMMAND
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int
wlan_ret_cmd_802_11_crypto(wlan_private * priv, HostCmd_DS_COMMAND * resp)
{
    wlan_adapter *Adapter = priv->adapter;
    HostCmd_DS_802_11_CRYPTO *crypto = &resp->params.crypto;
    MrvlIEtypes_Data_t *data =
        (MrvlIEtypes_Data_t *) ((u8 *) crypto +
                                sizeof(HostCmd_DS_802_11_CRYPTO));

    data->Header.Type = wlan_le16_to_cpu(data->Header.Type);
    data->Header.Len = wlan_le16_to_cpu(data->Header.Len);

    crypto->EncDec = wlan_le16_to_cpu(crypto->EncDec);
    crypto->Algorithm = wlan_le16_to_cpu(crypto->Algorithm);
    crypto->KeyIVLength = wlan_le16_to_cpu(crypto->KeyIVLength);
    crypto->KeyLength = wlan_le16_to_cpu(crypto->KeyLength);

    memmove(Adapter->CurCmd->pdata_buf, crypto,
            sizeof(HostCmd_DS_802_11_CRYPTO) + data->Header.Len +
            sizeof(MrvlIEtypesHeader_t));

    return WLAN_STATUS_SUCCESS;

}

static void
wlan_ret_ibss_coalescing_status(wlan_private * priv,
                                HostCmd_DS_COMMAND * resp)
{
    HostCmd_DS_802_11_IBSS_Status *IBSSStatusRsp;
    wlan_adapter *Adapter;
    union iwreq_data wrqu;
    u8 nullMac[ETH_ALEN] = { 0, 0, 0, 0, 0, 0 };

    Adapter = priv->adapter;
    IBSSStatusRsp = &(resp->params.ibssCoalescing);

    if (Adapter->CurCmd->pdata_buf)
        *(int *) Adapter->CurCmd->pdata_buf = IBSSStatusRsp->Enable;

    if (wlan_le16_to_cpu(IBSSStatusRsp->Action) == HostCmd_ACT_GEN_SET) {
        return;
    }

    PRINTM(INFO, "New BSSID %x:%x:%x:%x:%x:%x\n",
           IBSSStatusRsp->BSSID[0],
           IBSSStatusRsp->BSSID[1],
           IBSSStatusRsp->BSSID[2],
           IBSSStatusRsp->BSSID[3],
           IBSSStatusRsp->BSSID[4], IBSSStatusRsp->BSSID[5]);

    /* if rsp has NULL BSSID, Just return.. No Action */
    if (!memcmp(IBSSStatusRsp->BSSID, nullMac, ETH_ALEN)) {
        PRINTM(MSG, "New BSSID is NULL\n");
        return;
    }

    /* if BSSID is diff, Send evnet to Linux */
    if (memcmp(Adapter->CurBssParams.BSSDescriptor.MacAddress,
               IBSSStatusRsp->BSSID, ETH_ALEN)) {
        memcpy((void *) Adapter->CurBssParams.BSSDescriptor.MacAddress,
               (void *) IBSSStatusRsp->BSSID, ETH_ALEN);

        /* Beacon Interval and ATIM window */
        Adapter->CurBssParams.BSSDescriptor.BeaconPeriod
            = IBSSStatusRsp->BeaconInterval;
        Adapter->CurBssParams.BSSDescriptor.ATIMWindow
            = IBSSStatusRsp->ATIMWindow;
        //ERP Information
        Adapter->CurBssParams.BSSDescriptor.ERPFlags =
            (u8) IBSSStatusRsp->UseGRateProtection;

        memset(&wrqu, 0, sizeof(wrqu));
        memcpy(wrqu.ap_addr.sa_data,
               Adapter->CurBssParams.BSSDescriptor.MacAddress, ETH_ALEN);
        wrqu.ap_addr.sa_family = ARPHRD_ETHER;

        Adapter->AdhocState = ADHOC_COALESCED;
        wireless_send_event(priv->wlan_dev.netdev, SIOCGIWAP, &wrqu, NULL);
    }
}

/********************************************************
		Global Functions
********************************************************/

/** 
 *  @brief This function stop tx/rx queue and free skb
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
void
wlan_clean_txrx(wlan_private * priv)
{
    ENTER();
    os_stop_queue(priv);
    os_carrier_off(priv);

    wmm_stop_queue(priv);
    wmm_cleanup_queues(priv);

    /* Free Tx and Rx packets */
    os_free_tx_packet(priv);
    LEAVE();
}

/** 
 *  @brief This function handles the command response
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int
wlan_process_cmdresp(wlan_private * priv)
{
    u16 RespCmdOrig;
    u16 RespCmd;
    HostCmd_DS_COMMAND *resp;
    wlan_adapter *Adapter = priv->adapter;
    int ret = WLAN_STATUS_SUCCESS;
    ulong flags;
    u16 Result;

    ENTER();

    /* Now we got response from FW, cancel the command timer */
    if (Adapter->CommandTimerIsSet) {
        wlan_cancel_timer(&Adapter->MrvDrvCommandTimer);
        Adapter->CommandTimerIsSet = FALSE;
    }

    if (!Adapter->CurCmd) {
        resp = (HostCmd_DS_COMMAND *) priv->wlan_dev.upld_buf;
        resp->Command = wlan_le16_to_cpu(resp->Command);
        PRINTM(ERROR, "CMD_RESP: NULL CurCmd, 0x%x\n", resp->Command);
        ret = WLAN_STATUS_FAILURE;
        goto done;
    }
    Adapter->num_cmd_timeout = 0;

    DBG_HEXDUMP(CMD_D, "CMD_RESP", Adapter->CurCmd->BufVirtualAddr,
                priv->wlan_dev.upld_len);

    resp = (HostCmd_DS_COMMAND *) (Adapter->CurCmd->BufVirtualAddr);

    resp->Command = wlan_le16_to_cpu(resp->Command);
    resp->Size = wlan_le16_to_cpu(resp->Size);
    resp->SeqNum = wlan_le16_to_cpu(resp->SeqNum);
    resp->Result = wlan_le16_to_cpu(resp->Result);

    RespCmdOrig = resp->Command;
    Result = resp->Result;

    /* Save the last command response to debug log */
    Adapter->dbg.LastCmdRespIndex =
        (Adapter->dbg.LastCmdRespIndex + 1) % DBG_CMD_NUM;
    Adapter->dbg.LastCmdRespId[Adapter->dbg.LastCmdRespIndex] = RespCmdOrig;

    PRINTM(CMND, "CMD_RESP: 0x%x, result %d, len %d, seqno %d @ %lu\n",
           RespCmdOrig, Result, resp->Size, resp->SeqNum, os_time_get());

    if (!(RespCmdOrig & HostCmd_RET_BIT)) {
        PRINTM(ERROR, "CMD_RESP: Invalid response to command!");
        Adapter->CurCmdRetCode = WLAN_STATUS_FAILURE;
        wlan_insert_cmd_to_free_q(priv, Adapter->CurCmd);
        spin_lock_irqsave(&Adapter->QueueSpinLock, flags);
        Adapter->CurCmd = NULL;
        spin_unlock_irqrestore(&Adapter->QueueSpinLock, flags);

        ret = WLAN_STATUS_FAILURE;
        goto done;
    }

    /* Clear the HostCmd_RET_BIT */
    resp->Command &= ~HostCmd_RET_BIT;
    RespCmd = resp->Command;

    /* Store the response code to CurCmdRetCode. */
    Adapter->CurCmdRetCode = resp->Result;

    if (RespCmd == HostCmd_CMD_802_11_PS_MODE) {
        HostCmd_DS_802_11_PS_MODE *psmode;

        psmode = &resp->params.psmode;
        PRINTM(INFO, "CMD_RESP: PS_MODE cmd reply result=%#x action=0x%X\n",
               resp->Result, psmode->Action);
        psmode->Action = wlan_cpu_to_le16(psmode->Action);

        if (Result) {
            PRINTM(ERROR, "CMD_RESP: PS command failed- %#x \n",
                   resp->Result);
            if (Adapter->InfrastructureMode == Wlan802_11IBSS) {
                /* 
                 * We should not re-try enter-ps command in 
                 * ad-hoc mode. It takes place in 
                 * wlan_exec_next_cmd().
                 */
                if (psmode->Action == HostCmd_SubCmd_Enter_PS)
                    Adapter->PSMode = Wlan802_11PowerModeCAM;
            }
        } else if (psmode->Action == HostCmd_SubCmd_Enter_PS) {
            Adapter->NeedToWakeup = FALSE;
            Adapter->PSState = PS_STATE_AWAKE;
            if (Adapter->MediaConnectStatus != WlanMediaStateConnected) {
                /*
                 * When Deauth Event received before Enter_PS command
                 * response, We need to wake up the firmware.
                 */
                PRINTM(INFO,
                       "CMD_RESP: Disconnected, Going to invoke wlan_exit_ps\n");
                wlan_exit_ps(priv, 0);
            }
        } else if (psmode->Action == HostCmd_SubCmd_Exit_PS) {
            Adapter->NeedToWakeup = FALSE;
            Adapter->PSState = PS_STATE_FULL_POWER;
        } else {
            PRINTM(INFO, "CMD_RESP: PS- Action=0x%X\n", psmode->Action);
        }

        wlan_insert_cmd_to_free_q(priv, Adapter->CurCmd);
        spin_lock_irqsave(&Adapter->QueueSpinLock, flags);
        Adapter->CurCmd = NULL;
        spin_unlock_irqrestore(&Adapter->QueueSpinLock, flags);

        ret = WLAN_STATUS_SUCCESS;
        goto done;
    }

    if (Adapter->CurCmd->CmdFlags & CMD_F_HOSTCMD) {
        /* Copy the response back to response buffer */
        memcpy(Adapter->CurCmd->pdata_buf, resp, resp->Size);

        Adapter->CurCmd->CmdFlags &= ~CMD_F_HOSTCMD;

        if ((Result == HostCmd_RESULT_OK)
            && (RespCmd == HostCmd_CMD_802_11_HOST_SLEEP_CFG)) {
            ret = wlan_ret_host_sleep_cfg(priv, resp);
        }
    } else {
        /* If the command is not successful, cleanup and return failure */
        if (Result != HostCmd_RESULT_OK) {
            PRINTM(ERROR, "CMD_RESP: cmd %#x error, result=%#x\n",
                   resp->Command, resp->Result);
            /*
             * Handling errors here
             */
            switch (RespCmd) {
            case HostCmd_CMD_GET_HW_SPEC:
                PRINTM(INFO, "CMD_RESP: HW spec command Failed\n");
                break;

            }

            wlan_insert_cmd_to_free_q(priv, Adapter->CurCmd);
            spin_lock_irqsave(&Adapter->QueueSpinLock, flags);
            Adapter->CurCmd = NULL;
            spin_unlock_irqrestore(&Adapter->QueueSpinLock, flags);

            return WLAN_STATUS_FAILURE;
        }

        switch (RespCmd) {
        case HostCmd_CMD_MAC_REG_ACCESS:
        case HostCmd_CMD_BBP_REG_ACCESS:
        case HostCmd_CMD_RF_REG_ACCESS:
            ret = wlan_ret_reg_access(priv, RespCmd, resp);
            break;

        case HostCmd_CMD_GET_HW_SPEC:
            ret = wlan_ret_get_hw_spec(priv, resp);
            break;

        case HostCmd_CMD_802_11_BG_SCAN_QUERY:
            {
                union iwreq_data wrqu;

                ret = wlan_ret_802_11_scan(priv, resp);
                memset(&wrqu, 0, sizeof(union iwreq_data));
                wireless_send_event(priv->wlan_dev.netdev, SIOCGIWSCAN, &wrqu,
                                    NULL);
                PRINTM(INFO, "CMD_RESP: BG_SCAN result is ready!\n");
                break;
            }
        case HostCmd_CMD_802_11_SCAN:
            ret = wlan_ret_802_11_scan(priv, resp);
            break;

        case HostCmd_CMD_MAC_CONTROL:
            ret = wlan_ret_mac_control(priv, resp);
            break;

        case HostCmd_CMD_802_11_GET_LOG:
            ret = wlan_ret_get_log(priv, resp);
            break;

        case HostCmd_CMD_802_11_ASSOCIATE:
            ret = wlan_ret_802_11_associate(priv, resp);
            break;

        case HostCmd_CMD_802_11_DEAUTHENTICATE:
            ret = wlan_ret_802_11_deauthenticate(priv, resp);
            break;

        case HostCmd_CMD_802_11_SET_WEP:
            ret = wlan_ret_802_11_set_wep(priv, resp);
            break;

        case HostCmd_CMD_802_11_AD_HOC_START:
        case HostCmd_CMD_802_11_AD_HOC_JOIN:
            ret = wlan_ret_802_11_ad_hoc(priv, resp);
            break;

        case HostCmd_CMD_802_11_RESET:
            ret = wlan_ret_802_11_reset(priv, resp);
            break;

        case HostCmd_CMD_802_11_SNMP_MIB:
            ret = wlan_ret_802_11_snmp_mib(priv, resp);
            break;

        case HostCmd_CMD_802_11_RF_TX_POWER:
            ret = wlan_ret_802_11_rf_tx_power(priv, resp);
            break;

        case HostCmd_CMD_802_11_RADIO_CONTROL:
            ret = wlan_ret_802_11_radio_control(priv, resp);
            break;

        case HostCmd_CMD_802_11_HOST_SLEEP_CFG:
            ret = wlan_ret_host_sleep_cfg(priv, resp);
            break;
        case HostCmd_CMD_802_11_WAKEUP_CONFIRM:
            wlan_host_sleep_deactivated_event(priv);
            break;
        case HostCmd_CMD_802_11_HOST_SLEEP_ACTIVATE:
            if (Adapter->bHostSleepConfigured
                && Adapter->HSCfg.gap == HOST_SLEEP_CFG_GAP_FF)
                Adapter->bWakeupDevRequired = TRUE;
            wlan_host_sleep_activated_event(priv);
            break;

        case HostCmd_CMD_802_11_RF_ANTENNA:
            ret = wlan_ret_802_11_rf_antenna(priv, resp);
            break;

        case HostCmd_CMD_MAC_MULTICAST_ADR:
            ret = wlan_ret_mac_multicast_adr(priv, resp);
            break;

        case HostCmd_CMD_802_11_RATE_ADAPT_RATESET:
            ret = wlan_ret_802_11_rate_adapt_rateset(priv, resp);
            break;
        case HostCmd_CMD_802_11_RF_CHANNEL:
            ret = wlan_ret_802_11_rf_channel(priv, resp);
            break;

        case HostCmd_CMD_802_11_RSSI:
            ret = wlan_ret_802_11_rssi(priv, resp);
            break;

        case HostCmd_CMD_802_11_MAC_ADDRESS:
            ret = wlan_ret_802_11_mac_address(priv, resp);
            break;

#ifdef MFG_CMD_SUPPORT
        case HostCmd_CMD_MFG_COMMAND:
            ret = wlan_ret_mfg_cmd(priv, resp);
            break;
#endif
        case HostCmd_CMD_802_11_AD_HOC_STOP:
            ret = wlan_ret_802_11_ad_hoc_stop(priv, resp);
            break;

        case HostCmd_CMD_802_11_CAL_DATA_EXT:
            ret = wlan_ret_802_11_cal_data_ext(priv, resp);
            break;

        case HostCmd_CMD_802_11_KEY_MATERIAL:
            ret = wlan_ret_802_11_key_material(priv, resp);
            break;
        case HostCmd_CMD_802_11_EEPROM_ACCESS:
            ret = wlan_ret_802_11_eeprom_access(priv, resp);
            break;

        case HostCmd_CMD_802_11D_DOMAIN_INFO:
            ret = wlan_ret_802_11d_domain_info(priv, resp);
            break;

        case HostCmd_CMD_802_11_TPC_ADAPT_REQ:
        case HostCmd_CMD_802_11_TPC_INFO:
        case HostCmd_CMD_802_11_CHAN_SW_ANN:
            ret = wlan_11h_cmdresp_process(priv, resp);
            break;
        case HostCmd_CMD_MEASUREMENT_REQUEST:
        case HostCmd_CMD_MEASUREMENT_REPORT:
            ret = wlan_meas_cmdresp_process(priv, resp);
            break;
        case HostCmd_CMD_802_11_SLEEP_PARAMS:
            ret = wlan_ret_802_11_sleep_params(priv, resp);
            break;
        case HostCmd_CMD_802_11_BCA_CONFIG_TIMESHARE:
            ret = wlan_ret_802_11_bca_timeshare(priv, resp);
            break;
        case HostCmd_CMD_802_11_INACTIVITY_TIMEOUT:
            *((u16 *) Adapter->CurCmd->pdata_buf) =
                wlan_le16_to_cpu(resp->params.inactivity_timeout.Timeout);
            break;
        case HostCmd_CMD_802_11_BG_SCAN_CONFIG:
            break;

        case HostCmd_CMD_802_11_FW_WAKE_METHOD:
            ret = wlan_ret_fw_wakeup_method(priv, resp);
            break;

        case HostCmd_CMD_802_11_SLEEP_PERIOD:
            ret = wlan_ret_802_11_sleep_period(priv, resp);
            break;
        case HostCmd_CMD_WMM_GET_STATUS:
            ret = wlan_cmdresp_wmm_get_status(priv, resp);
            break;
        case HostCmd_CMD_WMM_ADDTS_REQ:
            ret = wlan_cmdresp_wmm_addts_req(priv, resp);
            break;
        case HostCmd_CMD_WMM_DELTS_REQ:
            ret = wlan_cmdresp_wmm_delts_req(priv, resp);
            break;
        case HostCmd_CMD_WMM_QUEUE_CONFIG:
            ret = wlan_cmdresp_wmm_queue_config(priv, resp);
            break;
        case HostCmd_CMD_WMM_QUEUE_STATS:
            ret = wlan_cmdresp_wmm_queue_stats(priv, resp);
            break;
        case HostCmd_CMD_WMM_TS_STATUS:
            ret = wlan_cmdresp_wmm_ts_status(priv, resp);
            break;
        case HostCmd_CMD_TX_PKT_STATS:
            memcpy(Adapter->CurCmd->pdata_buf,
                   &resp->params.txPktStats, sizeof(HostCmd_DS_TX_PKT_STATS));
            ret = WLAN_STATUS_SUCCESS;
            break;
        case HostCmd_CMD_802_11_TPC_CFG:
            memmove(Adapter->CurCmd->pdata_buf,
                    &resp->params.tpccfg, sizeof(HostCmd_DS_802_11_TPC_CFG));
            break;
        case HostCmd_CMD_802_11_LED_CONTROL:
            {
                HostCmd_DS_802_11_LED_CTRL *pLedCtrl = &resp->params.ledgpio;
                MrvlIEtypes_LedGpio_t *pGpio = &pLedCtrl->LedGpio;
                MrvlIEtypes_LedBehavior_t *pBehavior = pLedCtrl->LedBehavior;

                pLedCtrl->Action = wlan_le16_to_cpu(pLedCtrl->Action);
                pLedCtrl->LedNums = wlan_le16_to_cpu(pLedCtrl->LedNums);
                pGpio->Header.Type = wlan_le16_to_cpu(pGpio->Header.Type);
                pGpio->Header.Len = wlan_le16_to_cpu(pGpio->Header.Len);
                pBehavior->Header.Type =
                    wlan_le16_to_cpu(pBehavior->Header.Type);
                pBehavior->Header.Len =
                    wlan_le16_to_cpu(pBehavior->Header.Len);
                memmove(Adapter->CurCmd->pdata_buf, &resp->params.ledgpio,
                        sizeof(HostCmd_DS_802_11_LED_CTRL));
                break;
            }
        case HostCmd_CMD_802_11_CRYPTO:
            ret = wlan_ret_cmd_802_11_crypto(priv, resp);
            break;

        case HostCmd_CMD_GET_TSF:
            resp->params.gettsf.TsfValue =
                wlan_le64_to_cpu(resp->params.gettsf.TsfValue);
            memcpy(priv->adapter->CurCmd->pdata_buf,
                   &resp->params.gettsf.TsfValue, sizeof(u64));
            break;
        case HostCmd_CMD_802_11_TX_RATE_QUERY:
            priv->adapter->TxRate =
                wlan_le16_to_cpu(resp->params.txrate.TxRate);
            break;
        case HostCmd_CMD_802_11_IBSS_COALESCING_STATUS:
            wlan_ret_ibss_coalescing_status(priv, resp);
            break;

        case HostCmd_CMD_SDIO_PULL_CTRL:
            memmove(Adapter->CurCmd->pdata_buf,
                    &resp->params.sdiopullctl,
                    sizeof(HostCmd_DS_SDIO_PULL_CTRL));
            break;
        case HostCmd_CMD_ECL_SYSTEM_CLOCK_CONFIG:
            {
                int i;
                HostCmd_DS_ECL_SYSTEM_CLOCK_CONFIG *pSysclockcfg =
                    &resp->params.sysclockcfg;

                pSysclockcfg->Action = wlan_le16_to_cpu(pSysclockcfg->Action);
                pSysclockcfg->SystemClock =
                    wlan_le16_to_cpu(pSysclockcfg->SystemClock);
                pSysclockcfg->SupportedSysClockLen =
                    wlan_le16_to_cpu(pSysclockcfg->SupportedSysClockLen);
                for (i = 0;
                     i <
                     (sizeof(pSysclockcfg->SupportedSysClock) / sizeof(u16));
                     i++)
                    pSysclockcfg->SupportedSysClock[i] =
                        wlan_le16_to_cpu(pSysclockcfg->SupportedSysClock[i]);
                memmove(Adapter->CurCmd->pdata_buf, &resp->params.sysclockcfg,
                        sizeof(HostCmd_DS_ECL_SYSTEM_CLOCK_CONFIG));
                break;
            }

        case HostCmd_CMD_MODULE_TYPE_CONFIG:
            resp->params.moduletypecfg.Action =
                wlan_le16_to_cpu(resp->params.moduletypecfg.Action);
            resp->params.moduletypecfg.Module =
                wlan_le16_to_cpu(resp->params.moduletypecfg.Module);
            memmove(Adapter->CurCmd->pdata_buf, &resp->params.moduletypecfg,
                    sizeof(HostCmd_DS_MODULE_TYPE_CONFIG));
            break;

        case HostCmd_CMD_VERSION_EXT:
            memcpy(Adapter->CurCmd->pdata_buf,
                   &resp->params.verext, sizeof(HostCmd_DS_VERSION_EXT));
            break;
        case HostCmd_CMD_MEF_CFG:
            break;
        case HostCmd_CMD_DBGS_CFG:
            break;
        case HostCmd_CMD_GET_MEM:
            {
                FW_MEM_DATA *pFwData =
                    (FW_MEM_DATA *) Adapter->CurCmd->pdata_buf;
                pFwData->size = resp->Size - S_DS_GEN;
                memcpy((u8 *) & pFwData->data, &resp->params.getmem,
                       pFwData->size);
            }
            break;
        default:
            PRINTM(INFO, "CMD_RESP: Unknown command response %#x\n",
                   resp->Command);
            break;
        }
    }

    if (Adapter->CurCmd) {
        /* Clean up and Put current command back to CmdFreeQ */
        wlan_insert_cmd_to_free_q(priv, Adapter->CurCmd);
        spin_lock_irqsave(&Adapter->QueueSpinLock, flags);
        Adapter->CurCmd = NULL;
        spin_unlock_irqrestore(&Adapter->QueueSpinLock, flags);
    }

  done:
    LEAVE();
    return ret;
}

#if WIRELESS_EXT >= 18
/** 
 *  @brief This function sends mic error event to application.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @para event    MIC ERROR EVENT. 
 *  @return 	   n/a
 */
void
send_mic_error_event(wlan_private * priv, u32 event)
{
    union iwreq_data iwrq;
    struct iw_michaelmicfailure mic;

    ENTER();

    memset(&iwrq, 0, sizeof(iwrq));
    memset(&mic, 0, sizeof(mic));
    if (event == EVENT_MIC_ERR_UNICAST) {
        mic.flags = IW_MICFAILURE_PAIRWISE;
    } else {
        mic.flags = IW_MICFAILURE_GROUP;
    }

    iwrq.data.pointer = &mic;
    iwrq.data.length = sizeof(mic);

    wireless_send_event(priv->wlan_dev.netdev, IWEVMICHAELMICFAILURE, &iwrq,
                        (u8 *) & mic);

    LEAVE();
    return;
}
#endif

/** 
 *  @brief This function handles events generated by firmware
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int
wlan_process_event(wlan_private * priv)
{
    int ret = WLAN_STATUS_SUCCESS;
    wlan_adapter *Adapter = priv->adapter;
    u32 eventcause = Adapter->EventCause;

    ENTER();

    /* Save the last event to debug log */
    Adapter->dbg.LastEventIndex =
        (Adapter->dbg.LastEventIndex + 1) % DBG_CMD_NUM;
    Adapter->dbg.LastEvent[Adapter->dbg.LastEventIndex] = eventcause;

    if (eventcause != EVENT_PS_SLEEP && eventcause != EVENT_PS_AWAKE)
        PRINTM(EVENT, "EVENT: 0x%x @ %lu\n", eventcause, os_time_get());

    switch (eventcause) {
    case EVENT_DUMMY_HOST_WAKEUP_SIGNAL:
        PRINTM(INFO, "EVENT: DUMMY_HOST_WAKEUP_SIGNAL\n");
        if (!priv->adapter->HS_Activated) {
            PRINTM(WARN, "DUMMY_HOST_WAKEUP_SIGNAL (HS_Deactivated)\n");
        } else {
            wlan_host_sleep_wakeup_event(priv);
        }
        break;
    case EVENT_LINK_SENSED:
        PRINTM(INFO, "EVENT: LINK_SENSED\n");
        Adapter->AdhocLinkSensed = TRUE;
        os_carrier_on(priv);
        os_start_queue(priv);
        wmm_start_queue(priv);
        send_iwevcustom_event(priv, CUS_EVT_ADHOC_LINK_SENSED);
        break;

    case EVENT_DEAUTHENTICATED:
        PRINTM(INFO, "EVENT: Deauthenticated\n");
        Adapter->dbg.num_event_deauth++;
        wlan_handle_disconnect_event(priv);
        break;

    case EVENT_DISASSOCIATED:
        PRINTM(INFO, "EVENT: Disassociated\n");
        Adapter->dbg.num_event_disassoc++;
        wlan_handle_disconnect_event(priv);
        break;

    case EVENT_LINK_LOST:
        PRINTM(INFO, "EVENT: Link lost\n");
        Adapter->dbg.num_event_link_lost++;
        wlan_handle_disconnect_event(priv);
        break;

    case EVENT_PS_SLEEP:
        PRINTM(INFO, "EVENT: SLEEP\n");
        PRINTM(EVENT, "_");

        /* handle unexpected PS SLEEP event */
        if (Adapter->PSState == PS_STATE_FULL_POWER) {
            PRINTM(INFO, "EVENT: In FULL POWER mode - ignore PS SLEEP\n");
            break;
        }
        Adapter->PSState = PS_STATE_PRE_SLEEP;

        wlan_ps_cond_check(priv, (u16) Adapter->PSMode);
        break;

    case EVENT_PS_AWAKE:
        PRINTM(INFO, "EVENT: AWAKE \n");
        PRINTM(EVENT, "|");
        Adapter->bWakeupDevRequired = FALSE;
        Adapter->WakeupTries = 0;
        sbi_reset_deepsleep_wakeup(priv);

        /* handle unexpected PS AWAKE event */
        if (Adapter->PSState == PS_STATE_FULL_POWER) {
            PRINTM(INFO, "EVENT: In FULL POWER mode - ignore PS AWAKE\n");
            break;
        }

        Adapter->TxLockFlag = FALSE;
        if (TRUE == wlan_check_last_packet_indication(priv)) {
            if (!priv->wlan_dev.dnld_sent && Adapter->gen_null_pkg) {
                wlan_send_null_packet(priv,
                                      MRVDRV_TxPD_POWER_MGMT_NULL_PACKET |
                                      MRVDRV_TxPD_POWER_MGMT_LAST_PACKET);
                Adapter->TxLockFlag = TRUE;
            }
        }

        Adapter->PSState = PS_STATE_AWAKE;

        if (Adapter->NeedToWakeup == TRUE) {
            /*
             * wait for the command processing to finish
             * before resuming sending 
             * Adapter->NeedToWakeup will be set to FALSE 
             * in PSWakup()
             */
            PRINTM(INFO, "Waking up...\n");
            wlan_exit_ps(priv, 0);
        }
        break;

    case EVENT_DEEP_SLEEP_AWAKE:
        sbi_reset_deepsleep_wakeup(priv);
        PRINTM(INFO, "EVENT: DS_AWAKE\n");
        if (priv->adapter->IsDeepSleep == TRUE) {
            Adapter->IsDeepSleep = FALSE;
            Adapter->bWakeupDevRequired = FALSE;
            Adapter->WakeupTries = 0;
            /* 
             * For auto DS + BGS case, some delay is needed to 
             * avoid going back to DS before getting BGS result
             */
            if (Adapter->IsAutoDeepSleepEnabled &&
                Adapter->bgScanConfig->Enable)
                os_sched_timeout(10);
            if (!priv->enhance_flag) {
                priv->wlan_dev.dnld_sent = DNLD_RES_RECEIVED;
                priv->adapter->HisRegCpy |= HIS_TxDnLdRdy;
            }
        }
        send_iwevcustom_event(priv, CUS_EVT_DEEP_SLEEP_AWAKE);
        wake_up_interruptible(&Adapter->ds_awake_q);
        break;

    case EVENT_HOST_SLEEP_AWAKE:
        PRINTM(INFO, "EVENT: HS_AWAKE\n");
        Adapter->bWakeupDevRequired = FALSE;
        Adapter->WakeupTries = 0;
        sbi_reset_deepsleep_wakeup(priv);
        /*
         * in BG SCAN mode w/ deep sleep, WAKE UP event
         * will be sent first, Deep Sleep Awake will
         * be sent later. 
         */
        if (priv->adapter->IsDeepSleep == TRUE) {
            priv->adapter->IsDeepSleep = FALSE;
            if (!priv->enhance_flag) {
                priv->wlan_dev.dnld_sent = DNLD_RES_RECEIVED;
                priv->adapter->HisRegCpy |= HIS_TxDnLdRdy;
            }
        }
        ret = wlan_prepare_cmd(priv,
                               HostCmd_CMD_802_11_WAKEUP_CONFIRM,
                               0, 0, 0, NULL);
        break;

    case EVENT_MIC_ERR_UNICAST:
        PRINTM(INFO, "EVENT: UNICAST MIC ERROR\n");
#if WIRELESS_EXT >= 18
        send_mic_error_event(priv, EVENT_MIC_ERR_UNICAST);
#else
        send_iwevcustom_event(priv, CUS_EVT_MLME_MIC_ERR_UNI);
#endif
        break;

    case EVENT_MIC_ERR_MULTICAST:
        PRINTM(INFO, "EVENT: MULTICAST MIC ERROR\n");
#if WIRELESS_EXT >= 18
        send_mic_error_event(priv, EVENT_MIC_ERR_MULTICAST);
#else
        send_iwevcustom_event(priv, CUS_EVT_MLME_MIC_ERR_MUL);
#endif
        break;
    case EVENT_MIB_CHANGED:
    case EVENT_INIT_DONE:
        break;

    case EVENT_ADHOC_BCN_LOST:
        PRINTM(INFO, "EVENT: ADHOC_BCN_LOST\n");
        Adapter->AdhocLinkSensed = FALSE;
        wlan_clean_txrx(priv);
        send_iwevcustom_event(priv, CUS_EVT_ADHOC_BCN_LOST);
        break;

    case EVENT_BG_SCAN_REPORT:
        PRINTM(INFO, "EVENT: BGS_REPORT\n");
        Adapter->bgScanConfig->Enable = FALSE;
        ret = wlan_cmd_bgs_query(priv);
        break;
    case EVENT_STOP_TX:
        PRINTM(INFO, "EVENT: Stop Tx (%#x)\n", eventcause);
        wlan_11h_tx_disable(priv);
        break;
    case EVENT_START_TX:
        PRINTM(INFO, "EVENT: Start Tx (%#x)\n", eventcause);
        wlan_11h_tx_enable(priv);
        break;
    case EVENT_CHANNEL_SWITCH:
        PRINTM(INFO, "EVENT: Channel Switch (%#x)\n", eventcause);

        wlan_prepare_cmd(priv, HostCmd_CMD_802_11_RF_CHANNEL,
                         HostCmd_OPT_802_11_RF_CHANNEL_GET, 0, 0, NULL);
        break;
    case EVENT_MEAS_REPORT_RDY:
        PRINTM(INFO, "EVENT: Measurement Report Ready (%#x)\n", eventcause);
        wlan_prepare_cmd(priv, HostCmd_CMD_MEASUREMENT_REPORT,
                         HostCmd_ACT_GEN_SET, 0, 0, NULL);
        break;
    case EVENT_WMM_STATUS_CHANGE:
        PRINTM(INFO, "EVENT: WMM status changed\n");
        ret = wlan_cmd_wmm_status_change(priv);
        break;

    case EVENT_RSSI_LOW:
        PRINTM(INFO, "EVENT: Beacon RSSI_LOW\n");
        send_iwevcustom_event(priv, CUS_EVT_BEACON_RSSI_LOW);
        break;
    case EVENT_SNR_LOW:
        PRINTM(INFO, "EVENT: Beacon SNR_LOW\n");
        send_iwevcustom_event(priv, CUS_EVT_BEACON_SNR_LOW);
        break;
    case EVENT_MAX_FAIL:
        PRINTM(INFO, "EVENT: MAX_FAIL\n");
        send_iwevcustom_event(priv, CUS_EVT_MAX_FAIL);
        break;
    case EVENT_RSSI_HIGH:
        PRINTM(INFO, "EVENT: Beacon RSSI_HIGH\n");
        send_iwevcustom_event(priv, CUS_EVT_BEACON_RSSI_HIGH);
        break;
    case EVENT_SNR_HIGH:
        PRINTM(INFO, "EVENT: Beacon SNR_HIGH\n");
        send_iwevcustom_event(priv, CUS_EVT_BEACON_SNR_HIGH);
        break;
    case EVENT_DATA_RSSI_LOW:
        PRINTM(INFO, "EVENT: Data RSSI_LOW\n");
        send_iwevcustom_event(priv, CUS_EVT_DATA_RSSI_LOW);
        break;
    case EVENT_DATA_SNR_LOW:
        PRINTM(INFO, "EVENT: Data SNR_LOW\n");
        send_iwevcustom_event(priv, CUS_EVT_DATA_SNR_LOW);
        break;
    case EVENT_DATA_RSSI_HIGH:
        PRINTM(INFO, "EVENT: Data RSSI_HIGH\n");
        send_iwevcustom_event(priv, CUS_EVT_DATA_RSSI_HIGH);
        break;
    case EVENT_DATA_SNR_HIGH:
        PRINTM(INFO, "EVENT: Data SNR_HIGH\n");
        send_iwevcustom_event(priv, CUS_EVT_DATA_SNR_HIGH);
        break;
    case EVENT_IBSS_COALESCED:
        PRINTM(INFO, "EVENT: IBSS_COALESCED\n");
        ret = wlan_prepare_cmd(priv,
                               HostCmd_CMD_802_11_IBSS_COALESCING_STATUS,
                               HostCmd_ACT_GEN_GET, 0, 0, NULL);

        break;
    default:
        PRINTM(INFO, "EVENT: unknown event id: %#x\n", eventcause);
        break;
    }

    Adapter->EventCause = 0;
    LEAVE();
    return ret;
}
