/*
 * @file ui_indicator.c  libindicate support
 *
 * Copyright (C) 2010-2011 Maia Kozheva <sikon@ubuntu.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

#ifdef HAVE_CONFIG_H
#   include <config.h>
#endif

#include "ui_indicator.h"

#ifdef HAVE_MESSAGING_MENU

#include <gtk/gtk.h>
#include <messaging-menu.h>
#include "feedlist.h"
#include "ui/feed_list_view.h"
#include "ui/liferea_shell.h"
#include "ui/ui_tray.h"
#include "vfolder.h"

/* The maximum number of feeds to display in the indicator menu. */
#define MAX_INDICATORS      6

static struct indicator_priv {
	MessagingMenuApp *server;
	GPtrArray *indicators;
} *indicator_priv = NULL;

/*
 The desktop file to initialize the indicator menu with. Resolves to
 a string like "/usr/share/applications/liferea.desktop".
*/ 
static const char *DESKTOP_FILE = "liferea.desktop";

static void
remove_all_indicators () {
	g_ptr_array_set_size (indicator_priv->indicators, 0);
}

/*
 Called when the main "Liferea" entry in the indicator menu is clicked.
*/ 
static void
on_indicator_server_clicked (MessagingMenuApp *server, gchar *source_id, gpointer user_data)
{
	liferea_shell_present ();
	remove_all_indicators ();
}

/*
 Called when the indicator menu entry for a specific feed
 is clicked, meaning Liferea should switch to that feed.
*/ 
static void
on_indicator_clicked (MessagingMenuApp *indicator, gchar *source_id, gpointer user_data)
{
	feed_list_view_select ((nodePtr) user_data);
	liferea_shell_present ();
	remove_all_indicators ();
}

static void
destroy_indicator (gpointer indicator)
{
	if (indicator_priv->server == NULL || indicator == NULL) 
		return;
	
	messaging_menu_app_remove_source (indicator_priv->server, (char *)(indicator));
}

void
ui_indicator_init ()
{
	if (indicator_priv != NULL)
		return;
	
	indicator_priv = g_new0 (struct indicator_priv, 1);
	indicator_priv->indicators = g_ptr_array_new_with_free_func (destroy_indicator);
	
	indicator_priv->server = messaging_menu_app_new (DESKTOP_FILE);
	messaging_menu_app_register (indicator_priv->server);

	g_signal_connect (G_OBJECT (indicator_priv->server), "activate-source", G_CALLBACK (on_indicator_server_clicked), NULL);
	
	ui_indicator_update ();
}

void
ui_indicator_destroy ()
{
	if (indicator_priv == NULL)
		return;
	
	remove_all_indicators ();
	g_object_unref (indicator_priv->server);
	indicator_priv->server = NULL;
	g_ptr_array_free (indicator_priv->indicators, TRUE);
	g_free (indicator_priv);
	indicator_priv = NULL;
}

static void
add_node_indicator (nodePtr node)
{
	GFile *file;
	GIcon *icon;
	gchar count[10];
	gchar *signal;
	
	if (indicator_priv->indicators->len >= MAX_INDICATORS)
		return;

	if (IS_VFOLDER(node) || g_slist_length (node->children) > 0) {
		/* Not a feed - walk children and do nothing more */
		node_foreach_child (node, add_node_indicator);
		return;
	}
	
	/* Skip feeds with no unread items */
	if (node->unreadCount == 0)
		return;
	
	file = g_file_new_for_path (node->iconFile);
	icon = g_file_icon_new (file);

	messaging_menu_app_append_source_with_count(
		indicator_priv->server,
		node->id,
		icon,
		node->title,
		node->unreadCount);

	signal = g_strdup_printf("activate-source::%s", node->id);
	g_signal_connect (indicator_priv->server, signal, G_CALLBACK (on_indicator_clicked), node);
	g_ptr_array_add (indicator_priv->indicators, node->id);

	g_object_unref(icon);
	g_object_unref(file);
	g_free(signal);
}

void
ui_indicator_update ()
{
	/* Do not update indicators if the user is interacting with the main window */
	if (!indicator_priv || gtk_window_is_active (GTK_WINDOW (liferea_shell_get_window ())))
		return;

	/* Remove all previous indicators from the menu */
	remove_all_indicators ();
	/* ...then walk the tree and add an indicator for each unread feed */
	feedlist_foreach (add_node_indicator);
}

gboolean
ui_indicator_is_visible ()
{
	return TRUE;
}

#endif  /* HAVE_MESSAGING_MENU */
