package latexDraw.ui.dialog;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.FileOutputStream;
import java.net.URL;
import java.net.URLConnection;
import java.util.Vector;

import javax.imageio.ImageIO;
import javax.print.DocFlavor;
import javax.swing.*;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import latexDraw.filters.*;
import latexDraw.lang.LaTeXDrawLang;
import latexDraw.ui.LaTeXDrawFrame;
import latexDraw.util.LaTeXDrawResources;
import net.sourceforge.jiu.codecs.BMPCodec;
import net.sourceforge.jiu.codecs.CodecMode;
import net.sourceforge.jiu.codecs.PNMCodec;
import net.sourceforge.jiu.gui.awt.BufferedRGB24Image;
import net.sourceforge.jiu.gui.awt.ImageCreator;
import net.sourceforge.jiu.gui.awt.RGBA;
import net.sourceforge.jiu.gui.awt.ToolkitLoader;

import org.sourceforge.jlibeps.epsgraphics.EpsGraphics2D;


/**
 * This frame allows the user to convert several pictures in an another format.<br>
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 02/13/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class BatchConvertFrame extends JFrame implements ActionListener, ListSelectionListener, Runnable
{
	private static final long serialVersionUID = 1L;

	/** The label of the frame */
	public static final String TITLE_FRAME = LaTeXDrawLang.getString1_6("LaTeXDrawFrame.2"); //$NON-NLS-1$
	
	/** The model of the list (Useful to add element to the list) */
	private BatchConvertListModel listModel;
	
	/** The list of pictures of folders to convert */
	private JList list;
	
	/** Allows to know if the user want to overwrite the existing files */
	private JCheckBox overwriteCB;
	
	private JCheckBox sameDirCB;
	
	/** Allows to choose the format of the new pictures */
	private JComboBox formatList;
	
	/** Allows to add severals pictures */
	private JButton addFilesB;
	
	/** Allows to remove a line of the list */
	private JButton removeB;
	
	/** Allows to remove of the lines of the list */
	private JButton removeAllB;
	
	/** Allows to set the path of the output directory */
	private JTextField pathField;
	
	/** Allows to select the path of the output directory */
	private JButton pathSelectB;
	
	/** The file chooser of paths selections */
	private JFileChooser dirChooser;
	
	/** The file chooser of pictures selections */
	private JFileChooser picturesChooser;
	
	/** Allows to start the conversion */
	private JButton beginB;
	
	/** Allows to cancel the conversion */
	private JButton stopB;
	
	/** Allows to close the frame */
	private JButton closeB;
	
	/** Allows to know if the scan of folders must be recursive */
	private JCheckBox recursiveCB;
	
	/** The thread of the conversion */
	private transient Thread thread;
	
	/** Allows to stop the thread which convert the pictures */
	private boolean stopThread;

	protected String pathSelectFigure;
	
	/** The label of the JCheckBox recursiveCB */
	public static final String LABEL_RECURSIVE = LaTeXDrawLang.getString1_6("BatchConvertFrame.1"); //$NON-NLS-1$
	
	/** The label of the button addFilesB */
	public static final String LABEL_ADD_FILES = LaTeXDrawLang.getString1_6("BatchConvertFrame.2"); //$NON-NLS-1$
	
	/** The label of the button  removeB */
	public static final String LABEL_REMOVE = LaTeXDrawLang.getString1_7("ParametersTextFrame.13"); //$NON-NLS-1$
	
	/** The label of the button removeAllB */
	public static final String LABEL_REMOVE_ALL = LaTeXDrawLang.getString1_6("BatchConvertFrame.4"); //$NON-NLS-1$
	
	/** The label of the checkbox overwrite */
	public static final String LABEL_OVERWRITE = LaTeXDrawLang.getString1_6("BatchConvertFrame.5"); //$NON-NLS-1$
	
	public static final String LABEL_SAME_DIR = LaTeXDrawLang.getString1_7("BatchConvertFrame.0"); //$NON-NLS-1$
	
	/** The label of the button pathSelectB */
	public static final String LABEL_SEL_PATH = LaTeXDrawLang.getString1_6("BatchConvertFrame.6"); //$NON-NLS-1$
	
	/** The label of the button formatList */
	public static final String LABEL_FORMAT_LIST = LaTeXDrawLang.getString1_6("BatchConvertFrame.7"); //$NON-NLS-1$
	
	/** The label of the button beginB */
	public static final String LABEL_BEGIN = LaTeXDrawLang.getString1_6("BatchConvertFrame.8"); //$NON-NLS-1$
	
	/** The second label of the button beginB */
	public static final String LABEL_STOP = LaTeXDrawLang.getString1_6("BatchConvertFrame.9"); //$NON-NLS-1$
	
	/** The label of the button closeB */
	public static final String LABEL_CLOSE = LaTeXDrawLang.getString1_8("LaTeXDrawFrame.23"); //$NON-NLS-1$
	
	/** The name of the list */
	public static final String NAME_LIST = LaTeXDrawLang.getString1_6("BatchConvertFrame.11"); //$NON-NLS-1$
	
	
	
	
	/**
	 * The constructor by default
	 */
	public BatchConvertFrame()
	{
		try
		{
	  		setIconImage(LaTeXDrawResources.copyIcon.getImage());
	  		
	  		stopThread = true;
	  		Dimension dim = Toolkit.getDefaultToolkit().getScreenSize();
			setTitle(TITLE_FRAME);

	  		dirChooser = null;
	  		picturesChooser = null;
	  		pathSelectFigure = null;

	  		JPanel buttonListPanel = new JPanel();
			JPanel panelTmp = new JPanel();
			panelTmp.setLayout(new BoxLayout(panelTmp, BoxLayout.Y_AXIS));
            buttonListPanel.setLayout(new BoxLayout(buttonListPanel, BoxLayout.X_AXIS));
	  		addFilesB  = new JButton(LABEL_ADD_FILES, LaTeXDrawResources.plusIcon);
	  		removeB	   = new JButton(LABEL_REMOVE, LaTeXDrawResources.minusIcon);
	  		removeAllB = new JButton(LABEL_REMOVE_ALL, LaTeXDrawResources.minusIcon);
	  		addFilesB.setActionCommand(LABEL_ADD_FILES);
	  		removeB.setActionCommand(LABEL_REMOVE);
	  		removeAllB.setActionCommand(LABEL_REMOVE_ALL);
	  		addFilesB.addActionListener(this);
	  		removeB.addActionListener(this);
	  		removeAllB.addActionListener(this);
	  		removeB.setEnabled(false);
	  		removeAllB.setEnabled(false);
	  		panelTmp.add(addFilesB);
	  		panelTmp.add(removeB);
	  		panelTmp.add(removeAllB);
	  		buttonListPanel.add(panelTmp);
	  		
	  		JPanel listPanel = new JPanel(); 
			list = new JList(); 
			list.setCellRenderer(new BatchConvertListCellRenderer());
			list.addListSelectionListener(this);
			list.setName(NAME_LIST);
			listModel = new BatchConvertListModel();
			list.setModel(listModel);
	  		list.setVisibleRowCount(10); 
	  		JScrollPane scrollPane = new JScrollPane(list); 
	  	 	listPanel.add(scrollPane); 
	  	 	
	  	 	JPanel topPanel = new JPanel(new BorderLayout());
	  	 	topPanel.add(listPanel, BorderLayout.CENTER);
	  	 	topPanel.add(buttonListPanel, BorderLayout.EAST);
			
	  	 	JPanel formatPanel = new JPanel(new GridBagLayout());
	  	 	GridBagConstraints constraint = new GridBagConstraints();
	    	constraint.gridx = 0;
	     	constraint.gridy = 0;
	     	constraint.gridwidth = 1;
	     	constraint.gridheight = 1;
	     	constraint.weightx = 0.1;
	     	constraint.weighty = 0.1;
	     	constraint.fill = GridBagConstraints.HORIZONTAL;
	     	constraint.anchor = GridBagConstraints.EAST;
	     	formatPanel.add(new JLabel(LaTeXDrawLang.getString1_6("BatchConvertFrame.14")),constraint); //$NON-NLS-1$
	     	
	  	 	formatList = new JComboBox();
	  	 	formatList.setActionCommand(LABEL_FORMAT_LIST);
	  	 	formatList.addItem(LaTeXDrawResources.EPS_EXTENSION);
	  	 	formatList.addItem(LaTeXDrawResources.JPG_EXTENSION);
	  	 	formatList.addItem(LaTeXDrawResources.BMP_EXTENSION);
	  	 	formatList.addItem(LaTeXDrawResources.PNG_EXTENSION);
	  	 	formatList.addItem(LaTeXDrawResources.PPM_EXTENSION);
	  	 	formatList.setSelectedIndex(0);
	  	 	formatList.setPreferredSize(new Dimension(300, 30));
	  	 	formatList.setSize(new Dimension(300, 30));
	  	 	formatList.setMaximumSize(new Dimension(300, 30));
	  	 	formatList.setMinimumSize(new Dimension(300, 30));
	     	constraint.gridx = 1;
	     	constraint.gridwidth = 6;
	     	formatPanel.add(formatList,constraint);
	  	 	overwriteCB = new JCheckBox(LABEL_OVERWRITE);
	  	 	overwriteCB.setActionCommand(LABEL_OVERWRITE);
	  	 	pathSelectB = new JButton(LaTeXDrawResources.openIcon);
	  	 	pathSelectB.setActionCommand(LABEL_SEL_PATH);
	  	 	pathSelectB.addActionListener(this);
	  	 	pathSelectB.setMaximumSize(new Dimension(40,40));
	  	 	pathSelectB.setPreferredSize(new Dimension(40,40));
	  	 	pathField = new JTextField();
	  	 	pathField.setEditable(false);
	  	 	pathField.setText(LaTeXDrawFrame.getPathExport());
	  	 	pathField.setMinimumSize(new Dimension(300,40));
	  	 	pathField.setPreferredSize(new Dimension(300,40));
	  	 	sameDirCB = new JCheckBox(LABEL_SAME_DIR);
	  	 	sameDirCB.setActionCommand(LABEL_SAME_DIR);
	  	 	sameDirCB.addActionListener(this);
	  	 	constraint.fill = GridBagConstraints.NONE;
	  	 	JPanel pathPanel = new JPanel(new GridBagLayout());
	    	constraint.gridx = 0;
	     	constraint.gridwidth = 6;
	     	pathPanel.add(pathField, constraint);
	     	
	     	constraint.gridx = 6;
	     	constraint.gridwidth = 1;
	     	pathPanel.add(pathSelectB, constraint);
	     	
	  	 	JPanel optPanel = new JPanel(new GridLayout(4,1));
	  	 	optPanel.add(formatPanel);
	  	 	optPanel.add(pathPanel);
	  	 	optPanel.add(overwriteCB);
	  	 	optPanel.add(sameDirCB);
	  	 	
	  	 	JPanel ctrlPanel = new JPanel(new GridLayout(1, 3));
	  	 	closeB = new JButton(LABEL_CLOSE);
	  	 	beginB = new JButton(LABEL_BEGIN);
	  	 	stopB = new JButton(LABEL_STOP);
	  	 	beginB.setEnabled(false);
	  	 	stopB.setEnabled(false);
	  	 	closeB.addActionListener(this);
	  	 	beginB.addActionListener(this);
	  	 	stopB.addActionListener(this);
	  	 	closeB.setActionCommand(LABEL_CLOSE);
	  	 	beginB.setActionCommand(LABEL_BEGIN);
	  	 	stopB.setActionCommand(LABEL_STOP);
	  	 	ctrlPanel.add(beginB);
	  	 	ctrlPanel.add(stopB);
	  	 	ctrlPanel.add(closeB);
	  	 	ctrlPanel.setPreferredSize(new Dimension(280, 40));
	  	 	ctrlPanel.setMaximumSize(new Dimension(280, 40));
	  	 	
	  	 	BoxLayout bl = new BoxLayout(getContentPane(), BoxLayout.Y_AXIS);
	     	getContentPane().setLayout(bl);
			getContentPane().add(topPanel);
			getContentPane().add(optPanel);
			getContentPane().add(ctrlPanel);
	  		pack();
	  		setLocation(dim.width/2-getWidth()/2, dim.height/2-getHeight()/2);
			setVisible(false);
		}catch(Exception e)
		{
			e.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(e);
		}
	}




	public void actionPerformed(ActionEvent e)
	{
		Object o = e.getSource();
		
		if(o instanceof AbstractButton)
		{
			String cmd = ((AbstractButton)o).getActionCommand();
			
			if(cmd==null) return;
			
			if(cmd.equals(LABEL_SAME_DIR))
			{
				if(sameDirCB.isSelected())
				{
					pathField.setEnabled(false);
					pathSelectB.setEnabled(false);
				}
				else
				{
					pathField.setEnabled(true);
					pathSelectB.setEnabled(true);
				}
				return ;
			}
			
			if(cmd.equals(LABEL_CLOSE))
			{
				setVisible(false);
				return ;
			}
			
			if(cmd.equals(LABEL_BEGIN))
			{
				if(thread==null)
				{
					int i, size = listModel.getSize();
					for(i=0; i<size; i++)
						listModel.changeIconLabel(-1, i);
					list.updateUI();
					
					setStopThread(false);
					beginB.setEnabled(false);
					stopB.setEnabled(true);
					closeB.setEnabled(false);
					addFilesB.setEnabled(false);
					formatList.setEnabled(false);
					overwriteCB.setEnabled(false);
					pathSelectB.setEnabled(false);
					removeAllB.setEnabled(false);
					removeB.setEnabled(false);
					sameDirCB.setEnabled(false);
					thread = new Thread(this);
					thread.start();
				}
				return;
			}
			
			if(cmd.equals(LABEL_STOP))
			{
				stop();
				stopB.setEnabled(false);
				beginB.setEnabled(true);
				closeB.setEnabled(true);
				sameDirCB.setEnabled(true);
				addFilesB.setEnabled(true);
				formatList.setEnabled(true);
				overwriteCB.setEnabled(true);
				removeAllB.setEnabled(true);
				if(!list.isSelectionEmpty())
					removeB.setEnabled(true);
				if(!sameDirCB.isSelected())
					pathSelectB.setEnabled(true);
				thread=null;
				return;
			}
			
			if(cmd.equals(LABEL_ADD_FILES))
			{
				
				if(picturesChooser==null)
				{
			  		picturesChooser = new JFileChooser();
			  		picturesChooser.addChoosableFileFilter(new BMPFilter());
			  		picturesChooser.addChoosableFileFilter(new PNGFilter());
			  		picturesChooser.addChoosableFileFilter(new PPMFilter());
			  		picturesChooser.addChoosableFileFilter(new GIFFilter());
			  		picturesChooser.addChoosableFileFilter(new JPGFilter());			  		
			 		picturesChooser.setAcceptAllFileFilterUsed(false);
			  		recursiveCB = new JCheckBox(LABEL_RECURSIVE);
			  		recursiveCB.setSelected(false);
			  		JPanel custom = new JPanel(); 
			 		custom.setLayout(new BoxLayout(custom, BoxLayout.Y_AXIS)); 
			 		custom.add(recursiveCB); 
			 		picturesChooser.add(custom, BorderLayout.SOUTH);
			  		
					picturesChooser.setCurrentDirectory(new File(LaTeXDrawFrame.getPathOpen()));
			  		picturesChooser.setApproveButtonText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.171")); //$NON-NLS-1$
			  		picturesChooser.setDialogTitle(LaTeXDrawLang.getString1_6("BatchConvertFrame.13"));     //$NON-NLS-1$
			  		picturesChooser.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
			  		picturesChooser.setMultiSelectionEnabled(true);
				}
				
				if(pathSelectFigure!=null)
					picturesChooser.setCurrentDirectory(new File(pathSelectFigure));
				
				picturesChooser.showSaveDialog(this);
				File[] files = picturesChooser.getSelectedFiles();
				
				if(files!=null)
					for(File f : files)
						if(f.canRead() && f.exists())
						{
							if(f.isDirectory())
							{
								listModel.addElement(f.getAbsolutePath(), -1);
								
								if(recursiveCB.isSelected())
									addSubFolders(f);
							}
							else
								listModel.addElement(f.getAbsolutePath(), -1);
						}
				
				list.updateUI();
				if(listModel.getSize()>0)
				{
					removeAllB.setEnabled(true);
					beginB.setEnabled(true);
				}
				return;
			}
			
			if(cmd.equals(LABEL_REMOVE))
			{
				listModel.removeElementAt(list.getSelectedIndex());
				list.updateUI();
				if(listModel.getSize()==0) 
				{
					removeAllB.setEnabled(false);
					beginB.setEnabled(false);
				}
				removeB.setEnabled(false);
			}
			
			if(cmd.equals(LABEL_REMOVE_ALL))
			{
				listModel.removeAllElements();
				list.updateUI();
				if(listModel.getSize()==0) 
					removeAllB.setEnabled(false);
				removeB.setEnabled(false);
				beginB.setEnabled(false);
			}
			
			if(cmd.equals(LABEL_SEL_PATH))
			{
				if(dirChooser==null)
				{
			  		dirChooser = new JFileChooser();
			  		dirChooser.setApproveButtonText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.171")); //$NON-NLS-1$
			  		dirChooser.setDialogTitle(LaTeXDrawLang.getString1_6("BatchConvertFrame.12"));     //$NON-NLS-1$
			  		dirChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
			  		dirChooser.setMultiSelectionEnabled(false);
				}
				dirChooser.setCurrentDirectory(new File(pathField.getText()));
				dirChooser.showSaveDialog(this);
				File file = dirChooser.getSelectedFile();
				
				if(file!=null)
					if(file.isDirectory())
						pathField.setText(file.getPath());
				return;
			}
			
			return ;
		}// if(o instanceof JButton)
	} 
	
	
	
	
	/**
	 * Converts the pictures.
	 * @throws InterruptedException If the user cancel the conversion.
	 */
	public void convert() throws InterruptedException
	{
		try
		{
			int i, size = listModel.getSize();
			
			for(i=0; i<size; i++)
			{
				testEndThread();
				Object o = listModel.getElementAt(i);
				String path=null;
				if(o instanceof String) 
					path = (String)o;
				else if(o instanceof JLabel)
					path = ((JLabel)o).getText();
				else listModel.changeIconLabel(0, i);
				
				if(path!=null)
				{
					File f = new File(path);
					if(f.exists() && f.canRead())
					{
						if(f.isFile())
						{
							boolean state = convertFile(f);
							if(state)
								 listModel.changeIconLabel(1, i);
							else listModel.changeIconLabel(0, i);
						}
						else
						{
							File[] dir = f.listFiles();
							for(File f2 : dir)
							{
								convertFile(f2);
								testEndThread();
							}
							listModel.changeIconLabel(1, i);
						}
					}
					else listModel.changeIconLabel(0, i);
				} //if(path!=null)
				list.updateUI();
			} // for(i=0; i<size; i++)
		}catch(InterruptedException e){throw new InterruptedException(); }
	    catch (Exception e)
	    {
			e.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(e);
	    }
	}


	
	
	/**
	 * Allows to convert a file.
	 * @param f The file to convert.
	 * @return true: the conversion is successful, else false.
	 */
	public boolean convertFile(File f)
	{
		try
		{
			// Creation of the path
			String format = formatList.getSelectedItem().toString();
			String path = null;
			if(sameDirCB.isSelected())
			{
				path = f.getAbsolutePath();
				int end = path.lastIndexOf(".");//$NON-NLS-1$
				path = path.substring(0, end);
				path += format;
			}
			else
			{
				String name = f.getName();
				int end = name.lastIndexOf(".");//$NON-NLS-1$
				name = name.substring(0, end);
				name += format;
				path = pathField.getText()+System.getProperty("file.separator")+name;//$NON-NLS-1$
			}
			
			File file = new File(path);
			
			if(file.exists() && !overwriteCB.isSelected()) return false;
			
			Image image = getImage(f);
			if(image==null) return false;
			
			boolean ok = true;
			// Creation of the buffered image
			BufferedImage bufferedImage = new BufferedImage(image.getWidth(null), 
												image.getHeight(null), BufferedImage.TYPE_INT_RGB );
			Graphics g = bufferedImage.createGraphics();
			g.drawImage(image,0,0,null);
			g.dispose();
			
			// Exportation of the picture following the selected format
			if(format.equals(LaTeXDrawResources.EPS_EXTENSION))
			{
				FileOutputStream finalImage = new FileOutputStream(file);
				EpsGraphics2D graph = new EpsGraphics2D("TeXDrawPicture",  //$NON-NLS-1$
						finalImage, 0, 0, bufferedImage.getWidth(), bufferedImage.getHeight());	
				graph.drawImage(image, 0, 0, null);
				graph.flush();
				graph.close();
				finalImage.close();
				
			}
			else if(format.equals(LaTeXDrawResources.JPG_EXTENSION))
				ImageIO.write(bufferedImage, "jpg", file);//$NON-NLS-1$
			else if(format.equals(LaTeXDrawResources.BMP_EXTENSION))
			{
				BMPCodec codec = new BMPCodec();
				codec.setFile(path, CodecMode.SAVE);
				BufferedRGB24Image img = new BufferedRGB24Image(bufferedImage);
				codec.setImage(img); 
				codec.process();
				codec.close();
			}
			else if(format.equals(LaTeXDrawResources.PNG_EXTENSION))
				ImageIO.write(bufferedImage, "png", file);//$NON-NLS-1$
			else if(format.equals(LaTeXDrawResources.PPM_EXTENSION))
			{
				PNMCodec codec = new PNMCodec();
				codec.setFile(file.getAbsolutePath(), CodecMode.SAVE);
				BufferedRGB24Image img = new BufferedRGB24Image(bufferedImage);
				codec.setImage(img); 
				codec.process();
				codec.close();
			}
			else ok = false;
			
			bufferedImage.flush();
			
			return ok;
		}
		catch(Exception e) { return false; }
	}
	
	
	
	
	/**
	 * Allows to get the image of a file.
	 * @param f The file of the picture.
	 * @return The AWT image.
	 */
	public Image getImage(File f)
	{
		try
		{
			if(f==null || !f.exists() || !f.canRead() || !f.isFile())
				return null;
			
			Image image=null;
			URL url = f.toURL();
			URLConnection connection = url.openConnection();
		    String type = connection.getContentType();
		
		    if(type.equals(DocFlavor.INPUT_STREAM.JPEG.getMimeType()) || 
		       type.equals(DocFlavor.INPUT_STREAM.PNG.getMimeType()) ||
	    	   type.equals(DocFlavor.INPUT_STREAM.GIF.getMimeType()))
		    	image = Toolkit.getDefaultToolkit().getImage(f.getPath());
		    else
		    	if(f.getName().toLowerCase().endsWith(LaTeXDrawResources.BMP_EXTENSION) ||
		    		f.getName().toLowerCase().endsWith(LaTeXDrawResources.PPM_EXTENSION))
			     image = ImageCreator.convertToAwtImage(
			    		 ToolkitLoader.loadViaToolkitOrCodecs(f.getAbsolutePath(), true, null), RGBA.DEFAULT_ALPHA);
		    else
		    	return null;
		    
		    //	We must wait the end of the loading of the picture
		    MediaTracker tracker = new MediaTracker(this);
		    tracker.addImage(image, 0);
		    tracker.waitForID(0, 0);

		    if(tracker.isErrorID(0))
		    	return null;
		    
		    return image;

		 }catch(Exception e) { return null;}
	}
	
	


	/**
	 * Allows to add sub-folders in the list.
	 * @param f The root folder.
	 */
	private void addSubFolders(File f)
	{
		if(f==null || !f.canRead() || !f.isDirectory()) return ;
		
		File[] files = f.listFiles();
		
		for(File f1 : files)
			if(f1!=null && f1.canRead() && f1.isDirectory())
			{
				addSubFolders(f1);
				listModel.addElement(f1.getAbsolutePath(), -1);
			}
	}




	public void valueChanged(ListSelectionEvent e)
	{
		Object o = e.getSource();
		
		if(o instanceof JList)
		{
			JList l = (JList)o;
			
			if(l.getName().equals(NAME_LIST))
			{
				int[] id = l.getSelectedIndices();
				removeB.setEnabled(id!=null && id.length!=0);
				
				return ;
			}
			return ;
		}
	}



	public synchronized void testEndThread() throws InterruptedException 
	{
        if(stopThread) 
               throw new InterruptedException();
	} 

	
	
	public synchronized void stop() 
	{
        stopThread = true;
	}
	
	
	
	
	public synchronized void setStopThread(boolean state)
	{
		stopThread = state;
	}
	
	
	
	
	public void run()
	{
		try 
		{
			convert();
			stopB.setEnabled(false);
			beginB.setEnabled(true);
			closeB.setEnabled(true);
			sameDirCB.setEnabled(true);
			addFilesB.setEnabled(true);
			formatList.setEnabled(true);
			overwriteCB.setEnabled(true);
			
			if(!sameDirCB.isSelected())
				pathSelectB.setEnabled(true);
			
			removeAllB.setEnabled(true);
			
			if(!list.isSelectionEmpty())
				removeB.setEnabled(true);
			
			thread=null;
	    }
		catch(InterruptedException e) { /* nothing to do */ }
	}
	
	
	
	/**
	 * Sets the path used to select pictures.
	 * @param pathOpen The path.
	 * @since 1.9
	 */
	public void setPathSelect(String pathOpen)
	{
		File f = new File(pathOpen);
		
		if(f.exists() && f.canRead())
			pathSelectFigure = pathOpen;
	}




	/**
	 * Sets the path defining the location of the output pictures.
	 * @param pathExport The path.
	 * @since 1.9
	 */
	public void setPathOutput(String pathExport)
	{
		File f = new File(pathExport);
		
		if(f.exists() && f.canRead())
			pathField.setText(pathExport);
	}
}




/**
 * This is the model of the list of the batch conversion frame.<br>
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 02/14/06<br>
 * @author Arnaud BLOUIN<br>
 * 1.9.2<br>
 */
class BatchConvertListModel extends AbstractListModel
{
	private static final long serialVersionUID = 1L;

	/** The element of the list */
	Vector<JLabel> elements;
	
	
	/**
	 * The constructor by default
	 */
	public BatchConvertListModel()
	{
		elements = new Vector<JLabel>();
	}
	
	
	/**
	 * @return the elements
	 */
	public Vector<JLabel> getElements()
	{
		return elements;
	}


	/**
	 * Allows to remove all the elements of the list
	 */
	public void removeAllElements()
	{
		elements.removeAllElements();
	}


	
	/**
	 * Allows to remove an element at the position id in the list
	 * @param id The position of the element to remove from the list
	 */
	public void removeElementAt(int id)
	{
		if(id<0 || id>elements.size()) return ;
		elements.remove(id);
	}
	


	/**
	 * Allows to change the icon of an element of the list
	 * @param ok : 1, the 'ok' icon<br>0 : the 'err' icon<br>else: the 'empty' icon
	 * @param id The position of the element in the list
	 */
	public void changeIconLabel(int ok, int id)
	{
		if(id>=0 && id<elements.size())
		{
			if(ok==1) elements.elementAt(id).setIcon(LaTeXDrawResources.okIcon);
			else  if(ok==0) elements.elementAt(id).setIcon(LaTeXDrawResources.errIcon);
			else elements.elementAt(id).setIcon(LaTeXDrawResources.emptyIcon);
		}
			
	}
	
	
	
	
	/**
	 * Allows to add an element to the list
	 * @param txt The text to add
	 * @param ok : 1, the 'ok' icon<br>0: the 'err' icon<br>else : the 'empty' icon
	 */
	public void addElement(String txt, int ok)
	{
		JLabel label = new JLabel(txt);
		
		if(ok==1) label.setIcon(LaTeXDrawResources.okIcon);
		else  if(ok==0) label.setIcon(LaTeXDrawResources.errIcon);
		else label.setIcon(LaTeXDrawResources.emptyIcon);
		
		elements.add(label);
	}
	
	
	
	
	/**
	 * Allows to get the size of the list
	 */
	public int getSize()
	{
		return elements.size();
	}

	
	
	/**
	 * Allows to get an element of the list
	 * @param id The position of the element in the list
	 */
	public Object getElementAt(int id)
	{
		if(id<0 || id>elements.size()) return null;
		
		return elements.elementAt(id);
	}
	
}




/**
 * This is the renderer of the cells of the list of the batch conversion frame.<br>
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 02/14/06<br>
 * @author Arnaud BLOUIN<br>
 * 1.9.2<br>
 */
class BatchConvertListCellRenderer extends DefaultListCellRenderer 
{ 
	private static final long serialVersionUID = 1L;

	
	@Override
	public Component getListCellRendererComponent(JList list, Object value, 
            				int index, boolean isSelected, boolean cellHasFocus) 
    { 
		Component retValue;
		
		if(value instanceof JLabel)
		{
			retValue = super.getListCellRendererComponent( 
			 							list, ((JLabel)value).getText(), index, isSelected, cellHasFocus); 
			setIcon(((JLabel)value).getIcon());
		}
		else
		{
			retValue = super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus); 
			setIcon(LaTeXDrawResources.emptyIcon);
		}

		return retValue; 
    }
}
