package latexDraw.generators.svg;

import java.awt.geom.Point2D;
import java.text.ParseException;
import java.util.Vector;

import latexDraw.figures.Figure;
import latexDraw.figures.LaTeXDrawRectangle;
import latexDraw.figures.Triangle;
import latexDraw.parsers.latexdraw.PointsParser;
import latexDraw.parsers.svg.SVGAttributes;
import latexDraw.parsers.svg.SVGDocument;
import latexDraw.parsers.svg.elements.SVGElement;
import latexDraw.parsers.svg.elements.SVGGElement;
import latexDraw.parsers.svg.elements.SVGPolygonElement;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.util.LaTeXDrawNamespace;
import latexDraw.util.LaTeXDrawPoint2D;

/**
 * Defines an SVG generator for a triangle.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 11/11/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.1<br>
 */
public class LTriangleSVGGenerator extends LPolygonSVGGenerator
{

	public LTriangleSVGGenerator(Triangle f)
	{
		super(f);
	}

	
	
	
	public LTriangleSVGGenerator(SVGGElement elt)
	{
		this(elt, true);
	}
	
	
	
	/**
	 * Creates a triangle from a latexdraw-SVG element.
	 * @param elt The source element.
	 * @since 2.0.0
	 */
	public LTriangleSVGGenerator(SVGGElement elt, boolean withTransformation)
	{
		this(new Triangle(false));

		setNumber(elt);
		SVGElement elt2 = getLaTeXDrawElement(elt, null);
		Triangle t = (Triangle)getShape();
		
		if(elt==null || elt2==null || !(elt2 instanceof SVGPolygonElement))
			throw new IllegalArgumentException();
		
		SVGPolygonElement main = (SVGPolygonElement)elt2;
		setSVGLatexdrawParameters(elt);
		setSVGParameters(main);
		
		try
		{
			Vector<Point2D> ptsPol = PointsParser.getPoints(elt.getAttribute(
									 elt.getUsablePrefix(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE_URI)+LaTeXDrawNamespace.XML_POINTS));
			
			if(ptsPol==null || ptsPol.size()!=LaTeXDrawRectangle.NB_POINTS_FRAME)
				throw new IllegalArgumentException();
			
			t.getPoint(0).setLocation(ptsPol.elementAt(0));
			t.getPoint(1).setLocation(ptsPol.elementAt(1));
			t.getPoint(2).setLocation(ptsPol.elementAt(2));
			t.getPoint(3).setLocation(ptsPol.elementAt(3));
		}
		catch(ParseException e) { throw new IllegalArgumentException(e.getMessage()); }
		
		t.updateShape();
		t.getBorders().updateNSEWDelimitors();
		
		setSVGShadowParameters(getLaTeXDrawElement(elt, LaTeXDrawNamespace.XML_TYPE_SHADOW));
		setSVGDbleBordersParameters(getLaTeXDrawElement(elt, LaTeXDrawNamespace.XML_TYPE_DBLE_BORDERS));
		
		if(withTransformation)
			applyTransformations(elt);
	}
	
	
	
	
	@Override
	public SVGElement toSVG(SVGDocument doc)
	{
		if(doc==null || doc.getFirstChild().getDefs()==null)
			return null;
		
		Triangle t = (Triangle)getShape();
		SVGElement root = new SVGGElement(doc), elt;
		root.setAttribute(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_TYPE, LaTeXDrawNamespace.XML_TYPE_TRIANGLE);
		root.setAttribute(SVGAttributes.SVG_ID, getSVGID());
	    double gap = getPositionGap()/2.;
	    LaTeXDrawPoint2D pt1 = t.getPoint(0), pt2 = t.getPoint(-1);
		LaTeXDrawPoint2D p1 = new LaTeXDrawPoint2D((pt1.x+pt2.x)/2., pt1.y);
		LaTeXDrawPoint2D p2 = new LaTeXDrawPoint2D(pt2.x, pt2.y);
		LaTeXDrawPoint2D p3 = new LaTeXDrawPoint2D(pt1.x, pt2.y);
	    double cornerGap1 = Figure.getCornerGap(new LaTeXDrawPoint2D(p1.x, p2.y), p1, p2, gap);
	    double cornerGap2 = Figure.getCornerGap(t.getGravityCenter(), p2, p3, gap);
	   
	    if(p2.x>p3.x)
	    	cornerGap2*=-1;
	    
	    if(p1.y>p2.y)
	    	cornerGap1*=-1;
	    
		String points = (float)p1.x + "," + (float)(p1.y - cornerGap1) + " " + //$NON-NLS-1$//$NON-NLS-2$
						(float)(p2.x - cornerGap2) + "," + (float)(p2.y + (p1.y<p2.y ? gap : -gap)) + " " + //$NON-NLS-1$//$NON-NLS-2$
						(float)(p3.x + cornerGap2) + "," + (float)(p2.y + (p1.y<p2.y ? gap : -gap));//$NON-NLS-1$
	    String ltdPoints = pt1.x + " " + pt1.y + " " + t.getPoint(1).x + " " + t.getPoint(1).y +//$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
	    					" " + t.getPoint(2).x + " " + t.getPoint(2).y + " " + pt2.x + " " + pt2.y;//$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$//$NON-NLS-4$
		
		if(t.hasShadow())
		{
			SVGElement shad = new SVGPolygonElement(doc);
	    	
	   		shad.setAttribute(SVGAttributes.SVG_POINTS, points);
	   		setSVGShadowAttributes(shad, true);
	   		root.appendChild(shad);
		}
		
        if(shape.hasShadow() && !shape.getLineStyle().equals(PSTricksConstants.LINE_NONE_STYLE))
        {// The background of the borders must be filled is there is a shadow.
    		elt = new SVGPolygonElement(doc);
    		elt.setAttribute(SVGAttributes.SVG_POINTS, points);
    		setSVGBorderBackground(elt, root);
        }
        
		elt = new SVGPolygonElement(doc);
		elt.setAttribute(SVGAttributes.SVG_POINTS, points);
		root.appendChild(elt);
		root.setAttribute(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_POINTS, ltdPoints);
		
		if(t.hasDoubleBoundary())
		{
			SVGElement dblBord = new SVGPolygonElement(doc);
			
			dblBord.setAttribute(SVGAttributes.SVG_POINTS, points);
			setSVGDoubleBordersAttributes(dblBord);
			root.appendChild(dblBord);
		}
		
		setSVGAttributes(doc, elt, true);
		setSVGRotationAttribute(root);
		
		return root;
	}
}
