package latexDraw.generators.svg;

import static latexDraw.parsers.svg.SVGAttributes.*;

import java.awt.Color;
import java.awt.Polygon;
import java.awt.geom.Point2D;
import java.text.ParseException;
import java.util.Vector;

import latexDraw.figures.Dot;
import latexDraw.figures.Figure;
import latexDraw.parsers.latexdraw.PointsParser;
import latexDraw.parsers.svg.CSSColors;
import latexDraw.parsers.svg.SVGAttributes;
import latexDraw.parsers.svg.SVGDocument;
import latexDraw.parsers.svg.elements.*;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.util.LaTeXDrawNamespace;
import latexDraw.util.LaTeXDrawPoint2D;

/**
 * Defines a SVG generator for a dot.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 11/11/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.1<br>
 */
public class LDotSVGGenerator extends LShapeSVGGenerator
{
	public LDotSVGGenerator(Dot f)
	{
		super(f);
	}


	public LDotSVGGenerator(SVGGElement elt)
	{
		this(elt, true);
	}
	
	
	
	/**
	 * Creates a dot from a latexdraw-SVG element.
	 * @param elt The source element.
	 * @since 2.0.0
	 */
	public LDotSVGGenerator(SVGGElement elt, boolean withTransformation)
	{
		this(new Dot(false));

		if(elt==null)
			throw new IllegalArgumentException();
		
		Dot d = (Dot)getShape();
		String v = elt.getAttribute(elt.getUsablePrefix(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE_URI)+LaTeXDrawNamespace.XML_SIZE);
		SVGElement main = getLaTeXDrawElement(elt, null);
		
		try { d.setCurrentStyle(elt.getAttribute(elt.getUsablePrefix(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE_URI)+LaTeXDrawNamespace.XML_DOT_SHAPE)); }
		catch(IllegalArgumentException e) { /* */ }
		
		if(v!=null)
			try { d.setWidth(Double.valueOf(v).floatValue()); }
			catch(NumberFormatException e) { /* */ }
			
		v = elt.getAttribute(elt.getUsablePrefix(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE_URI)+LaTeXDrawNamespace.XML_POSITION);
		
		try
		{ 
			Vector<Point2D> pos = PointsParser.getPoints(v); 
			
			if(pos!=null && !pos.isEmpty())
				d.setCenter(new LaTeXDrawPoint2D(pos.firstElement()));
		}
		catch(ParseException e) { /* */ }
		
		setSVGLatexdrawParameters(elt);
		setSVGParameters(main);
		setNumber(elt);
		
		if(withTransformation)
			applyTransformations(elt);
		
		if(!d.canBeFilled())
			d.setLinesColor(CSSColors.getRGBColour(main.getFill()));
	}


	@Override
	public SVGElement toSVG(SVGDocument doc)
	{
		if(doc==null)
			return null;
		
		Dot d = (Dot)shape;
		float width = d.getWidth();
		Point2D.Double center = d.getCenter();
		String currentStyle = d.getCurrentStyle();
        SVGElement root = new SVGGElement(doc), dot;
		LaTeXDrawPoint2D NW = new LaTeXDrawPoint2D(center.x-width/2., center.y-width/2.);
		LaTeXDrawPoint2D SE = new LaTeXDrawPoint2D(center.x+width/2., center.y+width/2.);
		float dec = width/Dot.THICKNESS_o_STYLE_FACTOR;
		
        root.setAttribute(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_TYPE, LaTeXDrawNamespace.XML_TYPE_DOT);
        root.setAttribute(SVGAttributes.SVG_ID, getSVGID());

		if(currentStyle.equals(PSTricksConstants.DOT_STYLE))
		{	
			dot = new SVGCircleElement(doc);
			
			dot.setAttribute(SVGAttributes.SVG_FILL, CSSColors.getColorName(d.getLinesColor(), true));
			dot.setAttribute(SVGAttributes.SVG_CX, String.valueOf(NW.x+width/2.));
			dot.setAttribute(SVGAttributes.SVG_CY, String.valueOf(NW.y+width/2.));
			dot.setAttribute(SVGAttributes.SVG_R, String.valueOf(width/2.));
		}
		else if(currentStyle.equals(PSTricksConstants.O_STYLE) || currentStyle.equals(PSTricksConstants.OPLUS_STYLE) ||
				currentStyle.equals(PSTricksConstants.OTIMES_STYLE))
		{	
			float t;
			dot = new SVGGElement(doc);
			SVGElement o = new SVGCircleElement(doc);
			
			if(currentStyle.equals(PSTricksConstants.O_STYLE))
				dec = (float)(width*(0.1/3.6))*2;
			else
				dec = (float)(width*(0.1/2.6))*2;
			
			if(currentStyle.equals(PSTricksConstants.OPLUS_STYLE) || currentStyle.equals(PSTricksConstants.OTIMES_STYLE))
				dot.setAttribute(SVGAttributes.SVG_FILL, SVGAttributes.SVG_VALUE_NONE);
			else
				dot.setAttribute(SVGAttributes.SVG_FILL, 
						CSSColors.getColorName(d.isFilled()? d.getInteriorColor() : Color.WHITE, true));
			
			dot.setAttribute(SVGAttributes.SVG_STROKE, CSSColors.getColorName(d.getLinesColor(), true));
			dot.setAttribute(SVGAttributes.SVG_STROKE_WIDTH, String.valueOf(dec));
			o.setAttribute(SVGAttributes.SVG_CX, String.valueOf(NW.x+width/2.));
			o.setAttribute(SVGAttributes.SVG_CY, String.valueOf(NW.y+width/2.));
			o.setAttribute(SVGAttributes.SVG_R, String.valueOf(width/2.-dec/2.));
			dot.appendChild(o);
			
			if(currentStyle.equals(PSTricksConstants.OPLUS_STYLE))
			{
				SVGElement p1 = new SVGLineElement(doc);
				SVGElement p2 = new SVGLineElement(doc);
				
				t = dec*2;
				
				p1.setAttribute(SVGAttributes.SVG_X1, String.valueOf((NW.x+SE.x)/2.));
				p1.setAttribute(SVGAttributes.SVG_Y1, String.valueOf(NW.y+t));
				p1.setAttribute(SVGAttributes.SVG_X2, String.valueOf((NW.x+SE.x)/2.));
				p1.setAttribute(SVGAttributes.SVG_Y2, String.valueOf(SE.y-t));
				p2.setAttribute(SVGAttributes.SVG_X1, String.valueOf(NW.x+t));
				p2.setAttribute(SVGAttributes.SVG_Y1, String.valueOf((NW.y+SE.y)/2.));
				p2.setAttribute(SVGAttributes.SVG_X2, String.valueOf(SE.x-t));
				p2.setAttribute(SVGAttributes.SVG_Y2, String.valueOf((NW.y+SE.y)/2.));
				dot.appendChild(p1);
				dot.appendChild(p2);
			}
			else if(currentStyle.equals(PSTricksConstants.OTIMES_STYLE))
			{
				SVGElement p1 = new SVGLineElement(doc);
				SVGElement p2 = new SVGLineElement(doc);
				
				t = dec*2+Figure.DEFAULT_THICKNESS*2;
				
				p1.setAttribute(SVGAttributes.SVG_X1, String.valueOf(NW.x+t));
				p1.setAttribute(SVGAttributes.SVG_Y1, String.valueOf(NW.y+t));
				p1.setAttribute(SVGAttributes.SVG_X2, String.valueOf(SE.x-t));
				p1.setAttribute(SVGAttributes.SVG_Y2, String.valueOf(SE.y-t));
				p2.setAttribute(SVGAttributes.SVG_X1, String.valueOf(SE.x-t));
				p2.setAttribute(SVGAttributes.SVG_Y1, String.valueOf(NW.y+t));
				p2.setAttribute(SVGAttributes.SVG_X2, String.valueOf(NW.x+t));
				p2.setAttribute(SVGAttributes.SVG_Y2, String.valueOf(SE.y-t));
				dot.appendChild(p1);
				dot.appendChild(p2);
			}			
		}
		else if(currentStyle.equals(PSTricksConstants.BAR_STYLE))
		{
			dot = new SVGLineElement(doc);
			dot.setAttribute(SVGAttributes.SVG_STROKE_LINECAP, SVGAttributes.SVG_LINECAP_VALUE_SQUARE);
			dot.setAttribute(SVGAttributes.SVG_STROKE, CSSColors.getColorName(d.getLinesColor(), true));
			dot.setAttribute(SVGAttributes.SVG_STROKE_WIDTH, String.valueOf(d.getBarThickness()));
			dot.setAttribute(SVGAttributes.SVG_X1, String.valueOf((NW.x+SE.x)/2.));
			dot.setAttribute(SVGAttributes.SVG_Y1, String.valueOf(NW.y+d.getBarThickness()/2.));
			dot.setAttribute(SVGAttributes.SVG_X2, String.valueOf((NW.x+SE.x)/2.));
			dot.setAttribute(SVGAttributes.SVG_Y2, String.valueOf(SE.y+d.getBarGap()));
		}
		else if(currentStyle.equals(PSTricksConstants.PLUS_STYLE))
		{
			dot = new SVGGElement(doc);
			SVGElement p1 = new SVGLineElement(doc);
			SVGElement p2 = new SVGLineElement(doc);
			double plusGap = d.getPlusGap(); 
				
			dot.setAttribute(SVGAttributes.SVG_STROKE_LINECAP, SVGAttributes.SVG_LINECAP_VALUE_SQUARE);
			dot.setAttribute(SVGAttributes.SVG_STROKE, CSSColors.getColorName(d.getLinesColor(), true));
			dot.setAttribute(SVGAttributes.SVG_STROKE_WIDTH, String.valueOf(width/Dot.PLUS_COEFF_WIDTH));
			
			p1.setAttribute(SVGAttributes.SVG_X1, String.valueOf((NW.x+SE.x)/2.));
			p1.setAttribute(SVGAttributes.SVG_Y1, String.valueOf(NW.y-plusGap));
			p1.setAttribute(SVGAttributes.SVG_X2, String.valueOf((NW.x+SE.x)/2.));
			p1.setAttribute(SVGAttributes.SVG_Y2, String.valueOf(SE.y+plusGap));
			p2.setAttribute(SVGAttributes.SVG_X1, String.valueOf(NW.x-plusGap));
			p2.setAttribute(SVGAttributes.SVG_Y1, String.valueOf((NW.y+SE.y)/2.));
			p2.setAttribute(SVGAttributes.SVG_X2, String.valueOf(SE.x+plusGap));
			p2.setAttribute(SVGAttributes.SVG_Y2, String.valueOf((NW.y+SE.y)/2.));
			dot.appendChild(p1);
			dot.appendChild(p2);
		}
		else if(currentStyle.equals(PSTricksConstants.SQUARE_STYLE) || currentStyle.equals(PSTricksConstants.FSQUARE_STYLE))
		{
			Color c = d.getLinesColor();
			
			if(currentStyle.equals(PSTricksConstants.SQUARE_STYLE))
				if(d.isFilled())
					 c = d.getInteriorColor();
				else
				     c = Color.WHITE;
			
			dec = (float)(width*(0.1/4))*2;
			dot = new SVGRectElement(doc);
			dot.setAttribute(SVG_X, String.valueOf(NW.x+dec*1.5));
			dot.setAttribute(SVG_Y, String.valueOf(NW.y+dec*1.5));
			dot.setAttribute(SVG_WIDTH, String.valueOf(Math.abs(NW.x-SE.x)-3*dec));
			dot.setAttribute(SVG_HEIGHT, String.valueOf(Math.abs(NW.y - SE.y)-3*dec));
			dot.setAttribute(SVGAttributes.SVG_STROKE, CSSColors.getColorName(d.getLinesColor(), true));
			dot.setAttribute(SVGAttributes.SVG_FILL, CSSColors.getColorName(c, true));
			dot.setAttribute(SVGAttributes.SVG_STROKE_WIDTH, String.valueOf(dec));
		}
		else if(currentStyle.equals(PSTricksConstants.TRIANGLE_STYLE) || currentStyle.equals(PSTricksConstants.FTRIANGLE_STYLE) ||
				currentStyle.equals(PSTricksConstants.DIAMOND_STYLE) || currentStyle.equals(PSTricksConstants.FDIAMOND_STYLE) ||
				currentStyle.equals(PSTricksConstants.PENTAGON_STYLE) || currentStyle.equals(PSTricksConstants.FPENTAGON_STYLE))
		{
			Polygon p;
			Color c = d.getLinesColor();
			String path;
			dot = new SVGPathElement(doc);
			
			if(currentStyle.equals(PSTricksConstants.TRIANGLE_STYLE) || currentStyle.equals(PSTricksConstants.FTRIANGLE_STYLE))
				p = d.createTriangle();
			else
				if(currentStyle.equals(PSTricksConstants.PENTAGON_STYLE) || currentStyle.equals(PSTricksConstants.FPENTAGON_STYLE))
					p = d.createPentagon();
				else
					p = d.createDiamond();
			
			if(currentStyle.equals(PSTricksConstants.TRIANGLE_STYLE) || currentStyle.equals(PSTricksConstants.DIAMOND_STYLE) ||
				currentStyle.equals(PSTricksConstants.PENTAGON_STYLE))
				if(d.isFilled())
					c = d.getInteriorColor();
				else
					c = Color.WHITE;
			
			if(p.npoints==0)
				return null;
			
			int[] xp = p.xpoints;
			int[] yp = p.ypoints;
			//TODO use path
			path = "M " + xp[0] + " " + yp[0]; //$NON-NLS-1$ //$NON-NLS-2$
			
			for(int i = 1; i<p.npoints; i++)
				path += " L " + xp[i] + " " + yp[i];  //$NON-NLS-1$ //$NON-NLS-2$
			
			path += " Z"; //$NON-NLS-1$
			
			dot.setAttribute(SVGAttributes.SVG_D, path);
			dot.setAttribute(SVGAttributes.SVG_STROKE, CSSColors.getColorName(d.getLinesColor(), true));
			dot.setAttribute(SVGAttributes.SVG_FILL, CSSColors.getColorName(c, true));
			dot.setAttribute(SVGAttributes.SVG_STROKE_WIDTH, String.valueOf(dec));
		}
		else if(currentStyle.equals(PSTricksConstants.X_STYLE))
		{
			double crossGap = d.getCrossGap();
			dot = new SVGGElement(doc);
			SVGElement p1 = new SVGLineElement(doc);
			SVGElement p2 = new SVGLineElement(doc);
			
			dot.setAttribute(SVGAttributes.SVG_STROKE_LINECAP, SVGAttributes.SVG_LINECAP_VALUE_SQUARE);
			dot.setAttribute(SVGAttributes.SVG_STROKE, CSSColors.getColorName(d.getLinesColor(), true));
			dot.setAttribute(SVGAttributes.SVG_STROKE_WIDTH, String.valueOf(crossGap));
			
			p1.setAttribute(SVGAttributes.SVG_X1, String.valueOf(NW.x+crossGap));
			p1.setAttribute(SVGAttributes.SVG_Y1, String.valueOf(NW.y+crossGap));
			p1.setAttribute(SVGAttributes.SVG_X2, String.valueOf(SE.x-crossGap));
			p1.setAttribute(SVGAttributes.SVG_Y2, String.valueOf(SE.y-crossGap));
			p2.setAttribute(SVGAttributes.SVG_X1, String.valueOf(SE.x-crossGap));
			p2.setAttribute(SVGAttributes.SVG_Y1, String.valueOf(NW.y+crossGap));
			p2.setAttribute(SVGAttributes.SVG_X2, String.valueOf(NW.x+crossGap));
			p2.setAttribute(SVGAttributes.SVG_Y2, String.valueOf(SE.y-crossGap));
			dot.appendChild(p1);
			dot.appendChild(p2);
		}
		else
		{
			dot = new SVGGElement(doc);
			SVGElement l1 = new SVGLineElement(doc);
			SVGElement l2 = new SVGLineElement(doc);
			SVGElement l3 = new SVGLineElement(doc);
			double xCenter = (NW.x+SE.x)/2., yCenter = (NW.y+SE.y)/2.;
			double radius = Math.abs((NW.y+width/10.)-(SE.y-width/10.))/2.+dec;
			
			dot.setAttribute(SVGAttributes.SVG_STROKE_LINECAP, SVGAttributes.SVG_LINECAP_VALUE_SQUARE);
			dot.setAttribute(SVGAttributes.SVG_STROKE, CSSColors.getColorName(d.getLinesColor(), true));
			dot.setAttribute(SVGAttributes.SVG_STROKE_WIDTH, String.valueOf(dec));
			
			l1.setAttribute(SVGAttributes.SVG_X1, String.valueOf((NW.x+SE.x)/2.));
			l1.setAttribute(SVGAttributes.SVG_Y1, String.valueOf(NW.y+width/10.-dec));
			l1.setAttribute(SVGAttributes.SVG_X2, String.valueOf((NW.x+SE.x)/2.));
			l1.setAttribute(SVGAttributes.SVG_Y2, String.valueOf(SE.y-width/10.+dec));
			l2.setAttribute(SVGAttributes.SVG_X1, String.valueOf(Math.cos(Math.PI/6.)*radius+xCenter));
			l2.setAttribute(SVGAttributes.SVG_Y1, String.valueOf(radius/2. + yCenter));
			l2.setAttribute(SVGAttributes.SVG_X2, String.valueOf(Math.cos(7*Math.PI/6.)*radius+xCenter));
			l2.setAttribute(SVGAttributes.SVG_Y2, String.valueOf(Math.sin(7*Math.PI/6.)*radius+yCenter));
			l3.setAttribute(SVGAttributes.SVG_X1, String.valueOf(Math.cos(5*Math.PI/6.)*radius+xCenter));
			l3.setAttribute(SVGAttributes.SVG_Y1, String.valueOf(Math.sin(5*Math.PI/6.)*radius+yCenter));
			l3.setAttribute(SVGAttributes.SVG_X2, String.valueOf(Math.cos(11*Math.PI/6.)*radius+xCenter));
			l3.setAttribute(SVGAttributes.SVG_Y2, String.valueOf(Math.sin(11*Math.PI/6.)*radius+yCenter));
			dot.appendChild(l1);
			dot.appendChild(l2);
			dot.appendChild(l3);
		}
		
		root.setAttribute(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_SIZE, String.valueOf(d.getWidth()));
		root.setAttribute(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_DOT_SHAPE, currentStyle);
		root.setAttribute(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_POSITION, d.getCenter().x + " " + d.getCenter().y); //$NON-NLS-1$
		root.appendChild(dot);
		
		setSVGRotationAttribute(root);
		
		return root;
	}
}
