#--
# Copyright (C) 2008-2009 Harald Sitter <apachelogger@ubuntu.com>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 2 of
# the License or (at your option) version 3 or any later version
# accepted by the membership of KDE e.V. (or its successor approved
# by the membership of KDE e.V.), which shall act as a proxy
# defined in Section 14 of version 3 of the license.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
#++

SCRIPT = File.basename($0)
KEY    = ENV['DEBFULLNAME'] # -k'name' to use for signing

require 'fileutils'

include FileUtils

module Bat
    # Output script synopsis.
    # mandatory contains a comma seperated list of mandatory arguments, while
    # optional contains a comma seperated list of optional arguments.
    #
    #     synopsis("PACKAGENAME,PATH")       #=> ./batget.rb PACKAGENAME|PATH
    #     synopsis("PATH","download,upload") #=> ./batget.rb PATH [download|upload]
    #     synopsis(,"download,upload")       #=> ./batget.rb [download|upload]
    #
    # Bat::Optparser provides a more powerful implementation of option parsing,
    # in a lot of cases you would only need a simple approach however. For
    # example when you only parse one argument anyway.
    def synopsis(mandatory='',optional='')
        string = ""
        for mandatory in mandatory.split(",")
            string += " #{mandatory}"
        end
        for optional in optional.split(",")
            string += " [#{optional}"
        end
        string += "]" unless optional.empty?
        puts "#{SCRIPT}#{string}"
        exit 1
    end

    # Evaluates error code and puts a string as warning (using wrn) if
    # keeprunning is true and code != 0. If keeprunning is false it calls err.
    def exitchecker(exitcode,string="The most recent system call didn't exit properly.",keeprunning=false)
        if exitcode != 0 and keeprunning
            wrn(string)
        elsif exitcode != 0 and not keeprunning
            err(string)
        end
        return exitcode
    end

    # Wrapper around IO#puts. It will add custom formating to ensure batmessages
    # are more visible than regular terminal output. It expects @batpart to be
    # set. Thus you can influence it at runtime, eventually this will change to
    # harddetection of the part using the script name.
    #
    #     pt("Reading your emails")               #=> .!!!~~~~>Bat batget: Reading your emails
    #     pt("Reading your emails", " seriously") #=> .!!!~~~~>Bat batget seriously: Reading your emails
    def pt(string, add='')
        bat = "Bat " if SCRIPT.downcase.start_with?("bat")
        @batpart = SCRIPT.sub(/.*bat/,"") unless @batpart
        add = " #{add}" if not add.start_with?(" ") and not add.empty?
        puts(".!!!~~~~>#{bat}#{@batpart}#{add}: #{string}")
    end

    # Wrapper around Bat::pt. It will add the word WARNING to the output.
    def wrn(string)
        pt(string," WARNING")
    end

    # Wrapper around Bat::pt. It will add the word ERROR to the output.
    # And exit with code 1.
    def err(string)
        pt(string," ERROR")
        exit 1
    end

    # Constructs very basic option parser object. You still will have to spend a
    # fair amount of time on creating the actual options. If I knew how to pass the
    # desired variable name as exactly that, a variable name, to be used for
    # enhancing the OpenStruct one could reduce adding an item to one method call
    # with 4 arguments. So imagine the following being actually 3 lines if I was a
    # better haxx0r (or ultimately I had intarwebs access right now ;-)
    #
    # I imagine something like Optparser::add_bool() to add new options in a
    # comfortable way...
    #
    #     Optparser.new()
    #     $options.build = true
    #     $opts.on("-b", "--no-build", "Don't testbuild") do |nobuild|
    #         $options.build = nobuild
    #     end
    #     $opts.parse!(ARGV)
    #
    # For more information please consult the OptionParser documentatin. The basic
    # method structure is short option | long option | description | do something.
    # You can also describe mandatory arguments by writing them in upper case, or
    # optional arguments using [ARG].
    #      $opts.on("-s", "--build-slave SLAVE", "SLAVE makes it a mandatory argument for --build-slave")
    class Optparser
        require 'optparse'
        require 'ostruct'
        def initialize()
            $options = OpenStruct.new

            $opts = OptionParser.new{ |$opts|
                $opts.banner = "Usage: #{SCRIPT} [options]"

                $opts.on_tail("-h", "--help", "Show this message"){
                    puts $opts
                    exit
                }
            }
        end
    end

end

# import namespace
include Bat
