/*
 * Grdc - GTK+/Gnome Remote Desktop Client
 * Copyright (C) 2009 - Vic Lee 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA.
 */
 
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <stdlib.h>
#include "grdcpublic.h"
#include "grdcstringlist.h"
#include "grdcpref.h"
#include "grdcprefdialog.h"

static const gpointer default_action_list[] =
{
    "0", N_("Open Connection"),
    "1", N_("Edit Settings"),
    NULL
};

static const gpointer scale_quality_list[] =
{
    "0", N_("Nearest"),
    "1", N_("Tiles"),
    "2", N_("Bilinear"),
    "3", N_("Hyper"),
    NULL
};

static const gpointer default_mode_list[] =
{
    "0", N_("Automatic"),
    "1", N_("Scrolled Window"),
    "3", N_("Scrolled Fullscreen"),
    "4", N_("Viewport Fullscreen"),
    NULL
};

static gboolean
grdc_pref_resolution_validation_func (const gchar *new_str, gchar **error)
{
    gint i;
    gint width, height;
    gboolean splitted;
    gboolean result;

    width = 0;
    height = 0;
    splitted = FALSE;
    result = TRUE;
    for (i = 0; new_str[i] != '\0'; i++)
    {
        if (new_str[i] == 'x')
        {
            if (splitted)
            {
                result = FALSE;
                break;
            }
            splitted = TRUE;
            continue;
        }
        if (new_str[i] < '0' || new_str[i] > '9')
        {
                result = FALSE;
                break;
        }
        if (splitted)
        {
            height = 1;
        }
        else
        {
            width = 1;
        }
    }

    if (width == 0 || height == 0) result = FALSE;

    if (!result) *error = g_strdup (_("Please enter format 'widthxheight'."));
    return result;
}

static void
grdc_pref_dialog_clear_recent (GtkWidget *widget, gpointer data)
{
    GtkWidget *dialog;

    grdc_pref_clear_recent ();
    dialog = gtk_message_dialog_new (GTK_WINDOW (data),
        GTK_DIALOG_MODAL, GTK_MESSAGE_INFO, GTK_BUTTONS_OK,
        _("Recent lists cleared."));
    gtk_dialog_run (GTK_DIALOG (dialog));
    gtk_widget_destroy (dialog);
}

gboolean
grdc_pref_dialog_run (GtkWindow *parent, gint default_tab)
{
    GtkWidget *dialog;
    GtkWidget *notebook;
    GtkWidget *tablabel;
    GtkWidget *vbox;
    GtkWidget *hbox;
    GtkWidget *table;
    GtkWidget *widget;
    GtkWidget *separated_process_check;
    GtkWidget *save_view_mode_check;
    GtkWidget *invisible_toolbar_check;
    GtkWidget *default_action_combo;
    GtkWidget *default_mode_combo;
    GtkWidget *scale_quality_combo;
    GtkWidget *sshtunnel_port_entry;
    GtkWidget *recent_maximum_entry;
    GtkWidget *groups_list;
    GtkWidget *resolutions_list;
    GtkWidget *applet_quick_ontop_check;
    GtkWidget *applet_hide_count_check;
    gint response;
    gboolean ret;
    gchar *s, buf[100];

    /* Create the dialog */
    dialog = gtk_dialog_new_with_buttons (_("Preferences"),
        parent, GTK_DIALOG_MODAL,
        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
        GTK_STOCK_OK, GTK_RESPONSE_OK,
        NULL);
    gtk_dialog_set_alternative_button_order (GTK_DIALOG (dialog),
        GTK_RESPONSE_OK, GTK_RESPONSE_CANCEL, -1);
    gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);
    gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);

    /* Create the notebook */
    notebook = gtk_notebook_new ();
    gtk_widget_show (notebook);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), notebook, TRUE, TRUE, 4);
    gtk_container_set_border_width (GTK_CONTAINER (notebook), 4);

    /* Options tab */
    tablabel = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (tablabel);

    widget = gtk_image_new_from_stock (GTK_STOCK_PREFERENCES, GTK_ICON_SIZE_MENU);
    gtk_box_pack_start (GTK_BOX (tablabel), widget, FALSE, FALSE, 0);
    gtk_widget_show (widget);

    widget = gtk_label_new (_("Options"));
    gtk_box_pack_start (GTK_BOX (tablabel), widget, FALSE, FALSE, 0);
    gtk_widget_show (widget);

    /* Options body */
    vbox = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox);
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox, tablabel);

    table = gtk_table_new (8, 2, FALSE);
    gtk_widget_show (table);
    gtk_table_set_row_spacings (GTK_TABLE (table), 4);
    gtk_table_set_col_spacings (GTK_TABLE (table), 4);
    gtk_container_set_border_width (GTK_CONTAINER (table), 8);
    gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 0);

    separated_process_check = gtk_check_button_new_with_label (_("Open Connections in Separated Processes"));
    gtk_widget_show (separated_process_check);
    gtk_table_attach_defaults (GTK_TABLE (table), separated_process_check, 0, 2, 0, 1);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (separated_process_check), grdc_pref.separated_process);

    save_view_mode_check = gtk_check_button_new_with_label (_("Remember Last View Mode for Each Connection"));
    gtk_widget_show (save_view_mode_check);
    gtk_table_attach_defaults (GTK_TABLE (table), save_view_mode_check, 0, 2, 1, 2);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (save_view_mode_check), grdc_pref.save_view_mode);

    invisible_toolbar_check = gtk_check_button_new_with_label (_("Invisible Floating Toolbar"));
    gtk_widget_show (invisible_toolbar_check);
    gtk_table_attach_defaults (GTK_TABLE (table), invisible_toolbar_check, 0, 2, 2, 3);
    if (gtk_widget_is_composited (dialog))
    {
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (invisible_toolbar_check), grdc_pref.invisible_toolbar);
    }
    else
    {
        gtk_widget_set_sensitive (invisible_toolbar_check, FALSE);
    }

    widget = gtk_label_new (_("Double-click Action"));
    gtk_widget_show (widget);
    gtk_misc_set_alignment (GTK_MISC (widget), 0.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 3, 4);

    default_action_combo = grdc_public_create_combo_mapint (default_action_list, grdc_pref.default_action, FALSE);
    gtk_widget_show (default_action_combo);
    gtk_table_attach_defaults (GTK_TABLE (table), default_action_combo, 1, 2, 3, 4);

    widget = gtk_label_new (_("Default View Mode"));
    gtk_widget_show (widget);
    gtk_misc_set_alignment (GTK_MISC (widget), 0.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 4, 5);

    default_mode_combo = grdc_public_create_combo_mapint (default_mode_list, grdc_pref.default_mode, FALSE);
    gtk_widget_show (default_mode_combo);
    gtk_table_attach_defaults (GTK_TABLE (table), default_mode_combo, 1, 2, 4, 5);

    widget = gtk_label_new (_("Scale Quality"));
    gtk_widget_show (widget);
    gtk_misc_set_alignment (GTK_MISC (widget), 0.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 5, 6);

    scale_quality_combo = grdc_public_create_combo_mapint (scale_quality_list, grdc_pref.scale_quality, FALSE);
    gtk_widget_show (scale_quality_combo);
    gtk_table_attach_defaults (GTK_TABLE (table), scale_quality_combo, 1, 2, 5, 6);

    widget = gtk_label_new (_("SSH Tunnel Local Port"));
    gtk_widget_show (widget);
    gtk_misc_set_alignment (GTK_MISC (widget), 0.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 6, 7);

    sshtunnel_port_entry = gtk_entry_new_with_max_length (5);
    gtk_widget_show (sshtunnel_port_entry);
    gtk_table_attach_defaults (GTK_TABLE (table), sshtunnel_port_entry, 1, 2, 6, 7);
    g_snprintf (buf, sizeof (buf), "%i", grdc_pref.sshtunnel_port);
    gtk_entry_set_text (GTK_ENTRY (sshtunnel_port_entry), buf);

    widget = gtk_label_new (_("Maximum Recent Items"));
    gtk_widget_show (widget);
    gtk_misc_set_alignment (GTK_MISC (widget), 0.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 7, 8);

    hbox = gtk_hbox_new (FALSE, 2);
    gtk_widget_show (hbox);
    gtk_table_attach_defaults (GTK_TABLE (table), hbox, 1, 2, 7, 8);

    recent_maximum_entry = gtk_entry_new_with_max_length (2);
    gtk_widget_show (recent_maximum_entry);
    gtk_box_pack_start (GTK_BOX (hbox), recent_maximum_entry, TRUE, TRUE, 0);
    g_snprintf (buf, sizeof (buf), "%i", grdc_pref.recent_maximum);
    gtk_entry_set_text (GTK_ENTRY (recent_maximum_entry), buf);

    widget = gtk_button_new_with_label (_("Clear"));
    gtk_widget_show (widget);
    gtk_box_pack_start (GTK_BOX (hbox), widget, FALSE, FALSE, 0);
    g_signal_connect (G_OBJECT (widget), "clicked", G_CALLBACK (grdc_pref_dialog_clear_recent), dialog);

    /* Groups tab */
    tablabel = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (tablabel);

    widget = gtk_image_new_from_stock (GTK_STOCK_INDEX, GTK_ICON_SIZE_MENU);
    gtk_box_pack_start (GTK_BOX (tablabel), widget, FALSE, FALSE, 0);
    gtk_widget_show (widget);

    widget = gtk_label_new (_("Groups"));
    gtk_box_pack_start (GTK_BOX (tablabel), widget, FALSE, FALSE, 0);
    gtk_widget_show (widget);

    /* Groups body */
    vbox = gtk_vbox_new (FALSE, 2);
    gtk_widget_show (vbox);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), 8);
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox, tablabel);

    groups_list = grdc_string_list_new ();
    gtk_widget_show (groups_list);
    gtk_container_set_border_width (GTK_CONTAINER (groups_list), 8);
    gtk_widget_set_size_request (groups_list, 350, -1);
    grdc_string_list_set_new_entry_tooltip_text (GRDC_STRING_LIST (groups_list),
        _("Press Enter key to add new group"));
    gtk_box_pack_start (GTK_BOX (vbox), groups_list, TRUE, TRUE, 0);

    grdc_string_list_set_text (GRDC_STRING_LIST (groups_list), grdc_pref.groups);

    /* Resolutions tab */
    tablabel = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (tablabel);

    widget = gtk_image_new_from_stock (GTK_STOCK_FULLSCREEN, GTK_ICON_SIZE_MENU);
    gtk_box_pack_start (GTK_BOX (tablabel), widget, FALSE, FALSE, 0);
    gtk_widget_show (widget);

    widget = gtk_label_new (_("Resolutions"));
    gtk_box_pack_start (GTK_BOX (tablabel), widget, FALSE, FALSE, 0);
    gtk_widget_show (widget);

    /* Resolutions body */
    vbox = gtk_vbox_new (FALSE, 2);
    gtk_widget_show (vbox);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), 8);
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox, tablabel);

    resolutions_list = grdc_string_list_new ();
    gtk_widget_show (resolutions_list);
    gtk_container_set_border_width (GTK_CONTAINER (resolutions_list), 8);
    gtk_widget_set_size_request (resolutions_list, 350, -1);
    grdc_string_list_set_new_entry_tooltip_text (GRDC_STRING_LIST (resolutions_list),
        _("Press Enter key to add new resolution"));
    grdc_string_list_set_validation_func (GRDC_STRING_LIST (resolutions_list),
        grdc_pref_resolution_validation_func);
    gtk_box_pack_start (GTK_BOX (vbox), resolutions_list, TRUE, TRUE, 0);

    grdc_string_list_set_text (GRDC_STRING_LIST (resolutions_list), grdc_pref.resolutions);

    /* Applet tab */
    tablabel = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (tablabel);

    widget = gtk_image_new_from_stock (GTK_STOCK_EXECUTE, GTK_ICON_SIZE_MENU);
    gtk_box_pack_start (GTK_BOX (tablabel), widget, FALSE, FALSE, 0);
    gtk_widget_show (widget);

    widget = gtk_label_new (_("Applet"));
    gtk_box_pack_start (GTK_BOX (tablabel), widget, FALSE, FALSE, 0);
    gtk_widget_show (widget);

    /* Applet body */
    vbox = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox);
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox, tablabel);

    table = gtk_table_new (2, 2, FALSE);
    gtk_widget_show (table);
    gtk_table_set_row_spacings (GTK_TABLE (table), 4);
    gtk_table_set_col_spacings (GTK_TABLE (table), 4);
    gtk_container_set_border_width (GTK_CONTAINER (table), 8);
    gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 0);

    applet_quick_ontop_check = gtk_check_button_new_with_label (_("Show Quick Connect on top of the Menu"));
    gtk_widget_show (applet_quick_ontop_check);
    gtk_table_attach_defaults (GTK_TABLE (table), applet_quick_ontop_check, 0, 2, 0, 1);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (applet_quick_ontop_check), grdc_pref.applet_quick_ontop);

    applet_hide_count_check = gtk_check_button_new_with_label (_("Hide Total Count in Group Expander"));
    gtk_widget_show (applet_hide_count_check);
    gtk_table_attach_defaults (GTK_TABLE (table), applet_hide_count_check, 0, 2, 1, 2);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (applet_hide_count_check), grdc_pref.applet_hide_count);

    /* Show the default tab */
    if (default_tab > 0) gtk_notebook_set_current_page (GTK_NOTEBOOK (notebook), default_tab);

    /* Run the dialog */
    response = gtk_dialog_run (GTK_DIALOG (dialog));

    switch (response)
    {
        case GTK_RESPONSE_OK:
            grdc_pref.separated_process = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (separated_process_check));
            grdc_pref.save_view_mode = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (save_view_mode_check));
            grdc_pref.invisible_toolbar = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (invisible_toolbar_check));

            grdc_pref.default_action = atoi (gtk_combo_box_get_active_text (GTK_COMBO_BOX (default_action_combo)));
            grdc_pref.default_mode = atoi (gtk_combo_box_get_active_text (GTK_COMBO_BOX (default_mode_combo)));
            grdc_pref.scale_quality = atoi (gtk_combo_box_get_active_text (GTK_COMBO_BOX (scale_quality_combo)));

            grdc_pref.sshtunnel_port = atoi (gtk_entry_get_text (GTK_ENTRY (sshtunnel_port_entry)));
            if (grdc_pref.sshtunnel_port <= 0) grdc_pref.sshtunnel_port = DEFAULT_SSHTUNNEL_PORT;

            grdc_pref.recent_maximum = atoi (gtk_entry_get_text (GTK_ENTRY (recent_maximum_entry)));
            if (grdc_pref.recent_maximum < 0) grdc_pref.recent_maximum = 0;

            g_free (grdc_pref.groups);
            grdc_pref.groups = grdc_string_list_get_text (GRDC_STRING_LIST (groups_list));

            g_free (grdc_pref.resolutions);
            s = grdc_string_list_get_text (GRDC_STRING_LIST (resolutions_list));
            if (s[0] == '\0') s = g_strdup (default_resolutions);
            grdc_pref.resolutions = s;

            grdc_pref.applet_quick_ontop = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (applet_quick_ontop_check));
            grdc_pref.applet_hide_count = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (applet_hide_count_check));

            grdc_pref_save ();
            ret = TRUE;
            break;

        default:
            ret = FALSE;
            break;
    }

    gtk_widget_destroy (dialog);
    return ret;
}

