/*
 * Grdc - GTK+/Gnome Remote Desktop Client
 * Copyright (C) 2009 - Vic Lee 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <unistd.h>
#include <signal.h>
#include "config.h"
#ifdef HAVE_PTHREAD
#include <pthread.h>
#endif
#include "grdcpublic.h"
#include "grdcfile.h"
#include "grdcplug.h"
#include "grdcplug_rdp.h"

G_DEFINE_TYPE (GrdcPlugRdp, grdc_plug_rdp, GRDC_TYPE_PLUG)

static void
grdc_plug_rdp_plug_added (GtkSocket *socket, GrdcPlugRdp *gp_rdp)
{
    grdc_plug_emit_signal (GRDC_PLUG (gp_rdp), "connect");
}

static void
grdc_plug_rdp_plug_removed (GtkSocket *socket, GrdcPlugRdp *gp_rdp)
{
}

static void
grdc_plug_rdp_set_error (GrdcPlugRdp *gp_rdp)
{
    gchar *buffer;
    gint len;

    buffer = GRDC_PLUG (gp_rdp)->error_message;
    lseek (gp_rdp->error_fd, 0, SEEK_SET);
    len = read (gp_rdp->error_fd, buffer, MAX_ERROR_LENGTH);
    buffer[len] = '\0';

    GRDC_PLUG (gp_rdp)->has_error = TRUE;
}

static void
grdc_plug_rdp_child_exit (GPid pid, gint status, gpointer data)
{
    GrdcPlugRdp *gp_rdp;

    gp_rdp = GRDC_PLUG_RDP (data);
    g_spawn_close_pid (pid);

    if (status != 0 && status != SIGTERM)
    {
        grdc_plug_rdp_set_error (gp_rdp);
    }
    close (gp_rdp->output_fd);
    close (gp_rdp->error_fd);

    grdc_plug_emit_signal (GRDC_PLUG (gp_rdp), "disconnect");
}

static gboolean
grdc_plug_rdp_main (GrdcPlugRdp *gp_rdp)
{
    GrdcPlug *gp = GRDC_PLUG (gp_rdp);
    GrdcFile *grdcfile = gp->grdc_file;
    gchar *argv[50];
    gchar buf[200];
    gchar *host;
    gint argc;
    gint i;
    gint advargc = 0;
    gchar **advargv = NULL;
    GError *error = NULL;
    gboolean ret;

    if (grdcfile->arguments && grdcfile->arguments[0] != '\0')
    {
        if (!g_shell_parse_argv (grdcfile->arguments, &advargc, &advargv, &error))
        {
            g_print ("%s\n", error->message);
            /* We simply ignore argument error. */
            advargv = NULL;
        }
    }

    host = grdc_plug_start_tunnel (gp, 3389);
    if (host == NULL)
    {
        gp_rdp->thread = 0;
        return FALSE;
    }

    argc = 0;
    argv[argc++] = g_strdup ("rdesktop");

    if (advargv)
    {
        for (i = 0; i < advargc; i++)
        {
            argv[argc++] = g_strdup (advargv[i]);
        }
        g_strfreev (advargv);
    }

    if (grdcfile->username && grdcfile->username[0] != '\0')
    {
        argv[argc++] = g_strdup ("-u");
        argv[argc++] = g_strdup (grdcfile->username);
    }

    if (grdcfile->domain && grdcfile->domain[0] != '\0')
    {
        argv[argc++] = g_strdup ("-d");
        argv[argc++] = g_strdup (grdcfile->domain);
    }

    if (grdcfile->password && grdcfile->password[0] != '\0')
    {
        argv[argc++] = g_strdup ("-p");
        argv[argc++] = g_strdup (grdcfile->password);
    }

    if (grdcfile->colordepth > 0)
    {
        argv[argc++] = g_strdup ("-a");
        g_snprintf (buf, sizeof (buf), "%i", grdcfile->colordepth);
        argv[argc++] = g_strdup (buf);
    }

    if (grdcfile->clientname && grdcfile->clientname[0] != '\0')
    {
        argv[argc++] = g_strdup ("-n");
        argv[argc++] = g_strdup (grdcfile->clientname);
    }

    if (grdcfile->keymap && grdcfile->keymap[0] != '\0')
    {
        argv[argc++] = g_strdup ("-k");
        argv[argc++] = g_strdup (grdcfile->keymap);
    }

    if (grdcfile->bitmapcaching)
    {
        argv[argc++] = g_strdup ("-P");
    }

    if (grdcfile->compression)
    {
        argv[argc++] = g_strdup ("-z");
    }

    if (grdcfile->console)
    {
        argv[argc++] = g_strdup ("-0");
    }

    if (grdcfile->exec && grdcfile->exec[0] != '\0')
    {
        argv[argc++] = g_strdup ("-s");
        argv[argc++] = g_strdup (grdcfile->exec);
    }

    if (grdcfile->execpath && grdcfile->execpath[0] != '\0')
    {
        argv[argc++] = g_strdup ("-c");
        argv[argc++] = g_strdup (grdcfile->execpath);
    }

    switch (grdcfile->sharefolder)
    {
    case 1:
        argv[argc++] = g_strdup ("-r");
        argv[argc++] = g_strdup_printf ("disk:home=%s", g_get_home_dir ());
        break;
    case 2:
        argv[argc++] = g_strdup ("-r");
        argv[argc++] = g_strdup ("disk:root=/");
        break;
    }

    argv[argc++] = g_strdup ("-g");
    g_snprintf (buf, sizeof (buf), "%ix%i", grdcfile->resolution_width, grdcfile->resolution_height);
    argv[argc++] = g_strdup (buf);

    gp->width = grdcfile->resolution_width;
    gp->height = grdcfile->resolution_height;

    argv[argc++] = g_strdup ("-X");
    g_snprintf (buf, sizeof (buf), "%i", gp_rdp->socket_id);
    argv[argc++] = g_strdup (buf);

    argv[argc++] = host;
    argv[argc++] = NULL;

    ret = g_spawn_async_with_pipes (
        NULL, /* working_directory: take current */
        argv, /* argv[0] is the executable, parameters start from argv[1], end with NULL */
        NULL, /* envp: take the same as parent */
        G_SPAWN_SEARCH_PATH | G_SPAWN_DO_NOT_REAP_CHILD, /* flags */
        NULL, /* child_setup: function to be called before exec() */
        NULL, /* user_data: parameter for child_setup function */
        &gp_rdp->pid, /* child_pid */
        NULL, /* standard_input */
        &gp_rdp->output_fd, /* standard_output */
        &gp_rdp->error_fd, /* standard_error */
        &error);

    for (i = 0; i < argc; i++) g_free (argv[i]);

    if (ret)
    {
        g_child_watch_add (gp_rdp->pid, grdc_plug_rdp_child_exit, gp_rdp);
        gp_rdp->thread = 0;
        return TRUE;
    }
    else
    {
        g_snprintf (gp->error_message, MAX_ERROR_LENGTH, "%s", error->message);
        gp->has_error = TRUE;
        gp_rdp->thread = 0;
        return FALSE;
    }
}

#ifdef HAVE_LIBSSH
static gpointer
grdc_plug_rdp_main_thread (gpointer data)
{
    pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL);

    CANCEL_ASYNC
    if (!grdc_plug_rdp_main (GRDC_PLUG_RDP (data)))
    {
        IDLE_ADD ((GSourceFunc) grdc_plug_close_connection, data);
    }
    return NULL;
}
#endif

static gboolean
grdc_plug_rdp_open_connection (GrdcPlug *gp)
{
    GrdcPlugRdp *gp_rdp = GRDC_PLUG_RDP (gp);
    GrdcFile *grdcfile = gp->grdc_file;

    gtk_widget_set_size_request (GTK_WIDGET (gp), grdcfile->resolution_width, grdcfile->resolution_height);
    gp_rdp->socket_id = gtk_socket_get_id (GTK_SOCKET (gp_rdp->socket));

#ifdef HAVE_LIBSSH

    if (grdcfile->ssh_enabled)
    {
        if (pthread_create (&gp_rdp->thread, NULL, grdc_plug_rdp_main_thread, gp))
        {
            g_snprintf (gp->error_message, MAX_ERROR_LENGTH, "%s",
                "Failed to initialize pthread. Falling back to non-thread mode...");
            gp->has_error = TRUE;
            gp_rdp->thread = 0;
            return FALSE;
        }
        return TRUE;
    }
    else
    {
        return grdc_plug_rdp_main (gp_rdp);
    }

#else

    return grdc_plug_rdp_main (gp_rdp);

#endif
}

static gboolean
grdc_plug_rdp_close_connection (GrdcPlug *gp)
{
    GrdcPlugRdp *gp_rdp = GRDC_PLUG_RDP (gp);

#ifdef HAVE_PTHREAD
    if (gp_rdp->thread)
    {
        pthread_cancel (gp_rdp->thread);
        pthread_join (gp_rdp->thread, NULL);
    }
#endif

    if (gp_rdp->pid)
    {
        /* If pid exists, "disconnect" signal will be emitted in the child_exit callback */
        kill (gp_rdp->pid, SIGTERM);
    }
    else
    {
        grdc_plug_emit_signal (gp, "disconnect");
    }
    return FALSE;
}

static const gpointer
grdc_plug_rdp_query_feature (GrdcPlug *gp, GrdcPlugFeature feature)
{
    return NULL;
}

static void
grdc_plug_rdp_call_feature (GrdcPlug *gp, GrdcPlugFeature feature, const gpointer data)
{
}

static void
grdc_plug_rdp_class_init (GrdcPlugRdpClass *klass)
{
    klass->parent_class.open_connection = grdc_plug_rdp_open_connection;
    klass->parent_class.close_connection = grdc_plug_rdp_close_connection;
    klass->parent_class.query_feature = grdc_plug_rdp_query_feature;
    klass->parent_class.call_feature = grdc_plug_rdp_call_feature;
}

static void
grdc_plug_rdp_destroy (GtkWidget *widget, gpointer data)
{
}

static void
grdc_plug_rdp_init (GrdcPlugRdp *gp_rdp)
{
    gp_rdp->socket = gtk_socket_new ();
    gtk_widget_show (gp_rdp->socket);
    g_signal_connect (G_OBJECT (gp_rdp->socket), "plug-added",
        G_CALLBACK (grdc_plug_rdp_plug_added), gp_rdp);
    g_signal_connect (G_OBJECT (gp_rdp->socket), "plug-removed",
        G_CALLBACK (grdc_plug_rdp_plug_removed), gp_rdp);
    gtk_container_add (GTK_CONTAINER (gp_rdp), gp_rdp->socket);

    g_signal_connect (G_OBJECT (gp_rdp), "destroy", G_CALLBACK (grdc_plug_rdp_destroy), NULL);

    gp_rdp->socket_id = 0;
    gp_rdp->pid = 0;
    gp_rdp->output_fd = 0;
    gp_rdp->error_fd = 0;
    gp_rdp->thread = 0;
}

GtkWidget*
grdc_plug_rdp_new (void)
{
    return GTK_WIDGET (g_object_new (GRDC_TYPE_PLUG_RDP, NULL));
}

