/*
 * Grdc - GTK+/Gnome Remote Desktop Client
 * Copyright (C) 2009 - Vic Lee 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA.
 */
 
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include "grdcinitdialog.h"

G_DEFINE_TYPE (GrdcInitDialog, grdc_init_dialog, GTK_TYPE_DIALOG)

static void
grdc_init_dialog_class_init (GrdcInitDialogClass *klass)
{
}

static void
grdc_init_dialog_destroy (GrdcInitDialog *dialog, gpointer data)
{
    g_free (dialog->title);
    g_free (dialog->status);
    g_free (dialog->password);
}

static void
grdc_init_dialog_init (GrdcInitDialog *dialog)
{
    g_signal_connect (G_OBJECT (dialog), "destroy", G_CALLBACK (grdc_init_dialog_destroy), NULL);

    dialog->image = NULL;
    dialog->content_vbox = NULL;
    dialog->status_label = NULL;
    dialog->mode = GRDC_INIT_MODE_CONNECTING;
    dialog->title = NULL;
    dialog->status = NULL;
    dialog->password = NULL;
    dialog->save_password = FALSE;
}

static void
grdc_init_dialog_connecting (GrdcInitDialog *dialog)
{
    gtk_label_set_text (GTK_LABEL (dialog->status_label), (dialog->status ? dialog->status : dialog->title));
    gtk_image_set_from_stock (GTK_IMAGE (dialog->image), GTK_STOCK_DIALOG_INFO, GTK_ICON_SIZE_DIALOG);
    gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog), GTK_RESPONSE_OK, FALSE);

    dialog->mode = GRDC_INIT_MODE_CONNECTING;
}

GtkWidget*
grdc_init_dialog_new (GtkWindow *parent, const gchar *title_format, ...)
{
    GrdcInitDialog *dialog;
    GtkWidget *hbox;
    GtkWidget *widget;
    va_list args;

    dialog = GRDC_INIT_DIALOG (g_object_new (GRDC_TYPE_INIT_DIALOG, NULL));

    if (parent)
    {
        gtk_window_set_transient_for (GTK_WINDOW (dialog), parent);
    }

    va_start (args, title_format);
    dialog->title = g_strdup_vprintf (title_format, args);
    va_end (args);
    gtk_window_set_title (GTK_WINDOW (dialog), dialog->title);

    gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
    gtk_dialog_add_buttons (GTK_DIALOG (dialog),
        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
        GTK_STOCK_OK, GTK_RESPONSE_OK,
        NULL);
    gtk_dialog_set_alternative_button_order (GTK_DIALOG (dialog),
        GTK_RESPONSE_OK, GTK_RESPONSE_CANCEL, -1);

    gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);

    gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER_ALWAYS);
    gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);

    /**** Create the dialog content from here ****/

    /* Create top-level hbox */
    hbox = gtk_hbox_new (FALSE, 4);
    gtk_widget_show (hbox);
    gtk_container_set_border_width (GTK_CONTAINER (hbox), 15);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), hbox, TRUE, TRUE, 0);

    /* Icon */
    widget = gtk_image_new_from_stock (GTK_STOCK_DIALOG_INFO, GTK_ICON_SIZE_DIALOG);
    gtk_widget_show (widget);
    gtk_box_pack_start (GTK_BOX (hbox), widget, TRUE, TRUE, 4);
    dialog->image = widget;

    /* Create vbox for other dialog content */
    widget = gtk_vbox_new (FALSE, 4);
    gtk_widget_show (widget);
    gtk_box_pack_start (GTK_BOX (hbox), widget, TRUE, TRUE, 4);
    dialog->content_vbox = widget;

    /* Entries */
    widget = gtk_label_new (dialog->title);
    gtk_widget_show (widget);
    gtk_box_pack_start (GTK_BOX (dialog->content_vbox), widget, TRUE, TRUE, 4);
    dialog->status_label = widget;

    grdc_init_dialog_connecting (dialog);

    return GTK_WIDGET (dialog);
}

void
grdc_init_dialog_set_status (GrdcInitDialog *dialog, const gchar *status_format, ...)
{
    va_list args;

    if (status_format)
    {
        if (dialog->status) g_free (dialog->status);

        va_start (args, status_format);
        dialog->status = g_strdup_vprintf (status_format, args);
        va_end (args);

        gtk_label_set_text (GTK_LABEL (dialog->status_label), dialog->status);
    }
}

void
grdc_init_dialog_set_status_temp (GrdcInitDialog *dialog, const gchar *status_format, ...)
{
    va_list args;
    gchar *s;

    if (status_format)
    {
        va_start (args, status_format);
        s = g_strdup_vprintf (status_format, args);
        va_end (args);

        gtk_label_set_text (GTK_LABEL (dialog->status_label), s);
        g_free (s);
    }
}

gint
grdc_init_dialog_authpwd (GrdcInitDialog *dialog, const gchar *label, gboolean allow_save)
{
    GtkWidget *table;
    GtkWidget *password_entry;
    GtkWidget *save_password_check;
    GtkWidget *widget;
    gint ret;
    gchar *s;

    gtk_label_set_text (GTK_LABEL (dialog->status_label), (dialog->status ? dialog->status : dialog->title));

    /* Create table */
    table = gtk_table_new (2, 2, FALSE);
    gtk_widget_show (table);
    gtk_table_set_row_spacings (GTK_TABLE (table), 8);
    gtk_table_set_col_spacings (GTK_TABLE (table), 8);

    /* Icon */
    gtk_image_set_from_stock (GTK_IMAGE (dialog->image), GTK_STOCK_DIALOG_AUTHENTICATION, GTK_ICON_SIZE_DIALOG);

    /* Entries */
    widget = gtk_label_new (label);
    gtk_widget_show (widget);
    gtk_table_attach (GTK_TABLE (table), widget, 0, 1, 0, 1, 0, 0, 0, 0);

    password_entry = gtk_entry_new_with_max_length (100);
    gtk_widget_show (password_entry);
    gtk_table_attach_defaults (GTK_TABLE (table), password_entry, 1, 2, 0, 1);
    gtk_entry_set_visibility (GTK_ENTRY (password_entry), FALSE);
    gtk_entry_set_activates_default (GTK_ENTRY (password_entry), TRUE);

    s = g_strdup_printf (_("Save %s"), label);
    save_password_check = gtk_check_button_new_with_label (s);
    g_free (s);
    gtk_widget_show (save_password_check);
    gtk_table_attach_defaults (GTK_TABLE (table), save_password_check, 0, 2, 1, 2);
    if (allow_save)
    {
        if (dialog->save_password) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (save_password_check), TRUE);
    }
    else
    {
        gtk_widget_set_sensitive (save_password_check, FALSE);
    }

    /* Pack it into the dialog */
    gtk_box_pack_start (GTK_BOX (dialog->content_vbox), table, TRUE, TRUE, 4);

    gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog), GTK_RESPONSE_OK, TRUE);

    gtk_widget_grab_focus (password_entry);

    dialog->mode = GRDC_INIT_MODE_AUTHPWD;

    /* Now run it */
    ret = gtk_dialog_run (GTK_DIALOG (dialog));
    if (ret == GTK_RESPONSE_OK)
    {
        dialog->password = g_strdup (gtk_entry_get_text (GTK_ENTRY (password_entry)));
        dialog->save_password = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (save_password_check));
    }

    gtk_container_remove (GTK_CONTAINER (dialog->content_vbox), table);

    grdc_init_dialog_connecting (dialog);

    return ret;
}

