/*
 * Grdc - GTK+/Gnome Remote Desktop Client
 * Copyright (C) 2009 - Vic Lee 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <stdlib.h>
#include "config.h"
#ifdef HAVE_LIBAVAHI_UI
#include <avahi-ui/avahi-ui.h>
#endif
#include "grdcpixmaps.h"
#include "grdcpublic.h"
#include "grdcpref.h"
#include "grdcconnectionwindow.h"
#include "grdcprefdialog.h"
#include "grdcfile.h"
#include "grdcssh.h"
#include "grdcscaler.h"
#include "grdcfileeditor.h"

#define COLUMN_WIDTH_1 130
#define COLUMN_ALIGN_1 0.8
#define COLUMN_WIDTH_2 280

static const gpointer protocol_list[] =
{
    "RDP", N_("RDP - Windows Terminal Service"), GINT_TO_POINTER (GRDC_PIXMAP_TYPE_RDP),
#ifdef HAVE_LIBVNCCLIENT
    "VNC", N_("VNC - Virtual Network Computing"), GINT_TO_POINTER (GRDC_PIXMAP_TYPE_VNC),
    "VNCI", N_("VNC - Incoming Connection"), GINT_TO_POINTER (GRDC_PIXMAP_TYPE_VNC),
#endif /* HAVE_LIBVNCCLIENT */
#ifdef HAVE_LIBSSH
    "SFTP", N_("SFTP - Secure File Transfer"), GINT_TO_POINTER (GRDC_PIXMAP_TYPE_SSH),
#endif /* HAVE_LIBSSH */
    NULL
};

static const gpointer sharefolder_list[] =
{
    "0", N_("None"),
    "1", N_("My Home Folder"),
    "2", N_("System Root Folder"),
    NULL
};

/* Copied from tsclient... */
static const gchar *rdp_keymap_list = "\
ar,be,\
bg,br,cf,ch,common,\
cs,cz,da,de,de-ch,dk,dv,\
en-dv,en-gb,en-us,es,et,fi,\
fo,fr,fr-be,fr-ca,fr-ch,\
gk,gr,he,hr,hu,is,it,\
ja,ko,la,lt,lv,mk,\
nl,no,pl,po,pt,pt-br,\
ro,ru,sf,sl,sp,su,\
sv,th,tr,uk,us,usl,\
usr,yu";

static const gchar *charset_list = "\
ASCII,BIG5,\
CP437,CP720,CP737,CP775,CP850,CP852,CP855,\
CP857,CP858,CP862,CP866,CP874,CP1125,CP1250,\
CP1251,CP1252,CP1253,CP1254,CP1255,CP1256,\
CP1257,CP1258,\
EUC-JP,EUC-KR,GBK,\
ISO-8859-1,ISO-8859-2,ISO-8859-3,ISO-8859-4,\
ISO-8859-5,ISO-8859-6,ISO-8859-7,ISO-8859-8,\
ISO-8859-9,ISO-8859-10,ISO-8859-11,ISO-8859-12,\
ISO-8859-13,ISO-8859-14,ISO-8859-15,ISO-8859-16,\
SJIS";

typedef struct _GrdcFileEditor
{
    GtkWindow *parent;
    GrdcFile *grdc_file;
    GtkWidget *editor_dialog;

    GtkWidget *name_entry;
    GtkWidget *group_combo;
    GtkWidget *protocol_combo;

    GtkWidget *config_frame;
    GtkWidget *config_container;

    GtkWidget *server_combo;
    GtkWidget *listenport_entry;
    GtkWidget *username_entry;
    GtkWidget *password_entry;
    GtkWidget *domain_entry;
    GtkWidget *clientname_entry;
    GtkWidget *resolution_auto_radio;
    GtkWidget *resolution_custom_radio;
    GtkWidget *resolution_custom_combo;
    GtkWidget *colordepth_combo;
    GtkWidget *quality_combo;
    GtkWidget *keymap_combo;
    GtkWidget *gkeymap_combo;
    GtkWidget *exec_entry;
    GtkWidget *execpath_entry;
    GtkWidget *arguments_entry;
    GtkWidget *sharefolder_combo;
    GtkWidget *scaler_widget;

    GtkWidget *bitmapcaching_check;
    GtkWidget *compression_check;
    GtkWidget *showcursor_check;
    GtkWidget *viewonly_check;
    GtkWidget *console_check;
    GtkWidget *disableserverinput_check;

    GtkWidget *ssh_enabled_check;
    GtkWidget *ssh_server_default_radio;
    GtkWidget *ssh_server_custom_radio;
    GtkWidget *ssh_server_entry;
    GtkWidget *ssh_auth_password_radio;
    GtkWidget *ssh_auth_publickey_radio;
    GtkWidget *ssh_username_entry;
    GtkWidget *ssh_privatekey_entry;
    GtkWidget *ssh_charset_combo;

    gboolean quit_when_close;
} GrdcFileEditor;

#ifdef HAVE_LIBAVAHI_UI
static gchar*
grdc_file_editor_browse_avahi (GrdcFileEditor *gfe, const gchar *servicetypes)
{
    GtkWidget *dialog;
    gchar *host;

    dialog = aui_service_dialog_new (_("Choose a Remote Desktop Server"),
        GTK_WINDOW (gfe->editor_dialog),
        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
        GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
        NULL);
    gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (gfe->editor_dialog));
    aui_service_dialog_set_resolve_service (AUI_SERVICE_DIALOG (dialog), TRUE);
    aui_service_dialog_set_resolve_host_name (AUI_SERVICE_DIALOG (dialog), TRUE);
    aui_service_dialog_set_browse_service_types (AUI_SERVICE_DIALOG (dialog), servicetypes, NULL);
    if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
    {
        host = g_strdup_printf ("%s:%i",
            aui_service_dialog_get_host_name (AUI_SERVICE_DIALOG (dialog)),
            aui_service_dialog_get_port (AUI_SERVICE_DIALOG (dialog)));
    }
    else
    {
        host = NULL;
    }

    gtk_widget_destroy (dialog);

    return host;
}

static void
grdc_file_editor_browse_avahi_vnc (GtkWidget *button, GrdcFileEditor *gfe)
{
    gchar *host;

    host = grdc_file_editor_browse_avahi (gfe, "_rfb._tcp");
    if (host == NULL) return;

    gtk_entry_set_text (GTK_ENTRY (gtk_bin_get_child (GTK_BIN (gfe->server_combo))), host);
    
    g_free (host);
}
#endif

static void
grdc_file_editor_on_realize (GtkWidget *widget, gpointer user_data)
{
    GrdcFileEditor *gfe;
    GtkWidget *defaultwidget;

    gfe = (GrdcFileEditor*) user_data;

    defaultwidget = (gfe->grdc_file->filename == NULL ?
        (gfe->server_combo ? gfe->server_combo : gfe->listenport_entry) : 
        gfe->name_entry);

    if (GTK_IS_EDITABLE (defaultwidget))
    {
        gtk_editable_select_region (GTK_EDITABLE (defaultwidget), 0, -1);
    }
    gtk_widget_grab_focus (defaultwidget);
}

static void
grdc_file_editor_resolution_radio_on_toggled (GtkToggleButton *togglebutton, GrdcFileEditor *gfe)
{
    gtk_widget_set_sensitive (gfe->resolution_custom_combo, 
        gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gfe->resolution_custom_radio)));
}

static void
grdc_file_editor_create_notebook_container (GrdcFileEditor *gfe)
{
    /* Create the notebook */
    gfe->config_container = gtk_notebook_new ();
    gtk_container_add (GTK_CONTAINER (gfe->config_frame), gfe->config_container);
    gtk_container_set_border_width (GTK_CONTAINER (gfe->config_container), 4);
    gtk_widget_show (gfe->config_container);
}

static GtkWidget*
grdc_file_editor_create_notebook_tab (GrdcFileEditor *gfe,
    const gchar *stock_id, const gchar *label, gint rows, gint cols)
{
    GtkWidget *tablabel, *tabbody;
    GtkWidget *table;
    GtkWidget *widget;

    tablabel = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (tablabel);

    widget = gtk_image_new_from_stock (stock_id, GTK_ICON_SIZE_MENU);
    gtk_box_pack_start (GTK_BOX (tablabel), widget, FALSE, FALSE, 0);
    gtk_widget_show (widget);

    widget = gtk_label_new (label);
    gtk_box_pack_start (GTK_BOX (tablabel), widget, FALSE, FALSE, 0);
    gtk_widget_show (widget);

    tabbody = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (tabbody);
    gtk_notebook_append_page (GTK_NOTEBOOK (gfe->config_container), tabbody, tablabel);

    table = gtk_table_new (rows, cols, FALSE);
    gtk_widget_show (table);
    gtk_table_set_row_spacings (GTK_TABLE (table), 8);
    gtk_table_set_col_spacings (GTK_TABLE (table), 8);
    gtk_container_set_border_width (GTK_CONTAINER (table), 15);
    gtk_box_pack_start (GTK_BOX (tabbody), table, FALSE, FALSE, 0);

    return table;
}

#ifdef HAVE_LIBSSH
static void
grdc_file_editor_ssh_server_custom_radio_on_toggled (GtkToggleButton *togglebutton, GrdcFileEditor *gfe)
{
    gtk_widget_set_sensitive (gfe->ssh_server_entry,
        gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gfe->ssh_enabled_check)) &&
            (gfe->ssh_server_custom_radio == NULL ||
            gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gfe->ssh_server_custom_radio)))
    );
}

static void
grdc_file_editor_ssh_auth_publickey_radio_on_toggled (GtkToggleButton *togglebutton, GrdcFileEditor *gfe)
{
    gboolean b;
    gchar *s;

    b = ((!gfe->ssh_enabled_check ||
        gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gfe->ssh_enabled_check))) &&
        gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gfe->ssh_auth_publickey_radio)));
    gtk_widget_set_sensitive (gfe->ssh_privatekey_entry, b);

    if (b)
    {
        s = (gchar*) gtk_entry_get_text (GTK_ENTRY (gfe->ssh_privatekey_entry));
        if (s == NULL || s[0] == '\0')
        {
            s = grdc_ssh_find_identity ();
            if (s)
            {
                gtk_entry_set_text (GTK_ENTRY (gfe->ssh_privatekey_entry), s);
                g_free (s);
            }
        }
    }
}

static void
grdc_file_editor_ssh_enabled_check_on_toggled (GtkToggleButton *togglebutton, GrdcFileEditor *gfe)
{
    gboolean enabled = TRUE;

    if (gfe->ssh_enabled_check)
    {
        enabled = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gfe->ssh_enabled_check));
        if (gfe->ssh_server_default_radio) gtk_widget_set_sensitive (gfe->ssh_server_default_radio, enabled);
        if (gfe->ssh_server_custom_radio) gtk_widget_set_sensitive (gfe->ssh_server_custom_radio, enabled);
        grdc_file_editor_ssh_server_custom_radio_on_toggled (NULL, gfe);
        gtk_widget_set_sensitive (gfe->ssh_charset_combo, enabled);
        gtk_widget_set_sensitive (gfe->ssh_username_entry, enabled);
        gtk_widget_set_sensitive (gfe->ssh_auth_password_radio, enabled);
        gtk_widget_set_sensitive (gfe->ssh_auth_publickey_radio, enabled);
    }
    grdc_file_editor_ssh_auth_publickey_radio_on_toggled (NULL, gfe);

    if (enabled && gtk_entry_get_text (GTK_ENTRY (gfe->ssh_username_entry)) [0] == '\0')
    {
        gtk_entry_set_text (GTK_ENTRY (gfe->ssh_username_entry), g_get_user_name ());
    }
}
#endif

/* ssh_type: 1=port forward, 2=forward listen, 3=sftp */
static void
grdc_file_editor_create_ssh_tab (GrdcFileEditor *gfe, gint ssh_type)
{
#ifdef HAVE_LIBSSH
    GtkWidget *table;
    GtkWidget *frame;
    GtkWidget *frametable;
    GtkWidget *widget;
    gchar buf[200];
    gchar *s;
    gint row = 0;

    /* forward_listen not yet supported in libssh */
    if (ssh_type == 2) return;

    /* The SSH tab (implementation) */
    if (ssh_type == 3)
    {
        table = grdc_file_editor_create_notebook_tab (gfe, GTK_STOCK_DIALOG_AUTHENTICATION,
            "SFTP", 2, 1);
    }
    else
    {
        table = grdc_file_editor_create_notebook_tab (gfe, GTK_STOCK_DIALOG_AUTHENTICATION,
            "SSH", 3, 1);

        widget = gtk_check_button_new_with_label (_("Enable SSH Tunnel"));
        gtk_widget_show (widget);
        gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 0, 1);
        g_signal_connect (G_OBJECT (widget), "toggled",
            G_CALLBACK (grdc_file_editor_ssh_enabled_check_on_toggled), gfe);
        gfe->ssh_enabled_check = widget;
        row++;
    }

    /* SSH Server frame */
    frame = gtk_frame_new (_("SSH Server"));
    gtk_widget_show (frame);
    gtk_table_attach_defaults (GTK_TABLE (table), frame, 0, 1, row, row + 1);
    row++;

    frametable = gtk_table_new (3, 2, FALSE);
    gtk_widget_show (frametable);
    gtk_table_set_row_spacings (GTK_TABLE (frametable), 4);
    gtk_table_set_col_spacings (GTK_TABLE (frametable), 4);
    gtk_container_set_border_width (GTK_CONTAINER (frametable), 4);
    gtk_container_add (GTK_CONTAINER (frame), frametable);

    switch (ssh_type)
    {
    case 1: /* port forward */
        g_snprintf (buf, sizeof (buf), _("Same Server at Port %i"), DEFAULT_SSH_PORT);
        widget = gtk_radio_button_new_with_label (NULL, buf);
        gtk_widget_show (widget);
        gtk_table_attach_defaults (GTK_TABLE (frametable), widget, 0, 2, 0, 1);
        gfe->ssh_server_default_radio = widget;

        widget = gtk_radio_button_new_with_label_from_widget (
            GTK_RADIO_BUTTON (gfe->ssh_server_default_radio), _("Custom"));
        gtk_widget_show (widget);
        gtk_table_attach (GTK_TABLE (frametable), widget, 0, 1, 1, 2, GTK_FILL, 0, 0, 0);
        g_signal_connect (G_OBJECT (widget), "toggled",
            G_CALLBACK (grdc_file_editor_ssh_server_custom_radio_on_toggled), gfe);
        gfe->ssh_server_custom_radio = widget;

        widget = gtk_entry_new_with_max_length (100);
        gtk_widget_show (widget);
        gtk_table_attach_defaults (GTK_TABLE (frametable), widget, 1, 2, 1, 2);
        gfe->ssh_server_entry = widget;
        break;

    case 2: /* forward listen */
        gfe->ssh_server_default_radio = NULL;
        gfe->ssh_server_custom_radio = NULL;
        
        widget = gtk_entry_new_with_max_length (100);
        gtk_widget_show (widget);
        gtk_table_attach_defaults (GTK_TABLE (frametable), widget, 0, 2, 0, 2);
        gfe->ssh_server_entry = widget;
        break;
    
    case 3: /* sftp */
        gfe->ssh_server_default_radio = NULL;
        gfe->ssh_server_custom_radio = NULL;

        widget = gtk_label_new (_("Server"));
        gtk_widget_show (widget);
        gtk_table_attach (GTK_TABLE (frametable), widget, 0, 1, 0, 2, GTK_FILL, 0, 0, 0);
        
        s = grdc_pref_get_recent ("SFTP");
        widget = grdc_public_create_combo_entry (s, gfe->grdc_file->server, TRUE);
        gtk_widget_show (widget);
        gtk_entry_set_activates_default (GTK_ENTRY (gtk_bin_get_child (GTK_BIN (widget))), TRUE);
        gfe->server_combo = widget;
        g_free (s);
        gtk_table_attach_defaults (GTK_TABLE (frametable), widget, 1, 2, 0, 2);
        break;
    }

    widget = gtk_label_new (_("Character Set"));
    gtk_widget_show (widget);
    gtk_table_attach (GTK_TABLE (frametable), widget, 0, 1, 2, 3, GTK_FILL, 0, 0, 0);

    widget = grdc_public_create_combo_entry (charset_list, gfe->grdc_file->ssh_charset, FALSE);
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (frametable), widget, 1, 2, 2, 3);
    gfe->ssh_charset_combo = widget;

    /* SSH Authentication frame */
    frame = gtk_frame_new (_("SSH Authentication"));
    gtk_widget_show (frame);
    gtk_table_attach_defaults (GTK_TABLE (table), frame, 0, 1, row, row + 1);

    frametable = gtk_table_new (3, 2, FALSE);
    gtk_widget_show (frametable);
    gtk_table_set_row_spacings (GTK_TABLE (frametable), 4);
    gtk_table_set_col_spacings (GTK_TABLE (frametable), 4);
    gtk_container_set_border_width (GTK_CONTAINER (frametable), 4);
    gtk_container_add (GTK_CONTAINER (frame), frametable);

    widget = gtk_label_new (_("User Name"));
    gtk_widget_show (widget);
    gtk_table_attach (GTK_TABLE (frametable), widget, 0, 1, 0, 1, 0, 0, 0, 0);

    widget = gtk_entry_new_with_max_length (100);
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (frametable), widget, 1, 2, 0, 1);
    gfe->ssh_username_entry = widget;

    widget = gtk_radio_button_new_with_label (NULL, _("Password"));
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (frametable), widget, 0, 2, 1, 2);
    gfe->ssh_auth_password_radio = widget;

    widget = gtk_radio_button_new_with_label_from_widget (
        GTK_RADIO_BUTTON (gfe->ssh_auth_password_radio), _("Identity File"));
    gtk_widget_show (widget);
    gtk_table_attach (GTK_TABLE (frametable), widget, 0, 1, 2, 3, GTK_FILL, 0, 0, 0);
    g_signal_connect (G_OBJECT (widget), "toggled",
        G_CALLBACK (grdc_file_editor_ssh_auth_publickey_radio_on_toggled), gfe);
    gfe->ssh_auth_publickey_radio = widget;

    widget = gtk_entry_new_with_max_length (100);
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (frametable), widget, 1, 2, 2, 3);
    gfe->ssh_privatekey_entry = widget;

    /* Set the values */
    switch (ssh_type)
    {
    case 1:
    case 2:
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (gfe->ssh_enabled_check), gfe->grdc_file->ssh_enabled);

        if (ssh_type == 1)
        {
            gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (
                gfe->grdc_file->ssh_server && gfe->grdc_file->ssh_server[0] != '\0' ?
                gfe->ssh_server_custom_radio : gfe->ssh_server_default_radio), TRUE);
        }
        gtk_entry_set_text (GTK_ENTRY (gfe->ssh_server_entry),
            gfe->grdc_file->ssh_server ? gfe->grdc_file->ssh_server : "");
        break;

    case 3:
        break;
    }

    gtk_entry_set_text (GTK_ENTRY (gfe->ssh_username_entry),
        gfe->grdc_file->ssh_username ? gfe->grdc_file->ssh_username : "");
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (
        gfe->grdc_file->ssh_auth == SSH_AUTH_PUBLICKEY ?
        gfe->ssh_auth_publickey_radio : gfe->ssh_auth_password_radio), TRUE);
    gtk_entry_set_text (GTK_ENTRY (gfe->ssh_privatekey_entry),
        gfe->grdc_file->ssh_privatekey ? gfe->grdc_file->ssh_privatekey : "");

    grdc_file_editor_ssh_enabled_check_on_toggled (NULL, gfe);
#endif
}

static void
grdc_file_editor_create_rdp_config (GrdcFileEditor *gfe)
{
    GtkWidget *table;
    GtkWidget *widget;
    GtkWidget *hbox;
    gchar *s;

    grdc_file_editor_create_notebook_container (gfe);

    /* The Basic tab */
    table = grdc_file_editor_create_notebook_tab (gfe, GTK_STOCK_INFO, _("Basic"), 7, 2);

    widget = gtk_label_new (_("Server"));
    gtk_widget_show (widget);
    gtk_misc_set_alignment (GTK_MISC (widget), COLUMN_ALIGN_1, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 0, 1);

    s = grdc_pref_get_recent ("RDP");
    widget = grdc_public_create_combo_entry (s, gfe->grdc_file->server, TRUE);
    gtk_widget_show (widget);
    gtk_entry_set_activates_default (GTK_ENTRY (gtk_bin_get_child (GTK_BIN (widget))), TRUE);
    gfe->server_combo = widget;
    g_free (s);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 0, 1);

    widget = gtk_label_new (_("User Name"));
    gtk_widget_show (widget);
    gtk_widget_set_size_request (widget, COLUMN_WIDTH_1, -1);
    gtk_misc_set_alignment (GTK_MISC (widget), COLUMN_ALIGN_1, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 1, 2);

    widget = gtk_entry_new_with_max_length (100);
    gtk_widget_show (widget);
    gtk_widget_set_size_request (widget, COLUMN_WIDTH_2, -1);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 1, 2);
    if (gfe->grdc_file->username) gtk_entry_set_text (GTK_ENTRY (widget), gfe->grdc_file->username);
    gfe->username_entry = widget;

    widget = gtk_label_new (_("Password"));
    gtk_widget_show (widget);
    gtk_misc_set_alignment (GTK_MISC (widget), COLUMN_ALIGN_1, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 2, 3);

    widget = gtk_entry_new_with_max_length (100);
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 2, 3);
    gtk_entry_set_visibility (GTK_ENTRY (widget), FALSE);
    if (gfe->grdc_file->password) gtk_entry_set_text (GTK_ENTRY (widget), gfe->grdc_file->password);
    gfe->password_entry = widget;

    widget = gtk_label_new (_("Domain"));
    gtk_widget_show (widget);
    gtk_misc_set_alignment (GTK_MISC (widget), COLUMN_ALIGN_1, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 3, 4);

    widget = gtk_entry_new_with_max_length (100);
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 3, 4);
    if (gfe->grdc_file->domain) gtk_entry_set_text (GTK_ENTRY (widget), gfe->grdc_file->domain);
    gfe->domain_entry = widget;

    widget = gtk_label_new (_("Resolution"));
    gtk_widget_show (widget);
    gtk_misc_set_alignment (GTK_MISC (widget), COLUMN_ALIGN_1, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 4, 5);

    widget = gtk_radio_button_new_with_label (NULL, _("Use Client Resolution"));
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 4, 5);
    gfe->resolution_auto_radio = widget;

    hbox = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (hbox);
    gtk_table_attach_defaults (GTK_TABLE (table), hbox, 1, 2, 5, 6);

    widget = gtk_radio_button_new_with_label_from_widget (
        GTK_RADIO_BUTTON (gfe->resolution_auto_radio), _("Custom"));
    gtk_widget_show (widget);
    gtk_box_pack_start (GTK_BOX (hbox), widget, FALSE, FALSE, 0);
    g_signal_connect (G_OBJECT (widget), "toggled", 
        G_CALLBACK (grdc_file_editor_resolution_radio_on_toggled), gfe);
    gfe->resolution_custom_radio = widget;

    widget = grdc_public_create_combo_text_d (grdc_pref.resolutions, gfe->grdc_file->resolution, NULL);
    gtk_widget_show (widget);
    gtk_box_pack_start (GTK_BOX (hbox), widget, TRUE, TRUE, 0);
    gfe->resolution_custom_combo = widget;

    if (!gfe->grdc_file->resolution || gfe->grdc_file->resolution[0] == '\0' ||
        g_strcmp0 (gfe->grdc_file->resolution, "AUTO") == 0)
    {
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (gfe->resolution_auto_radio), TRUE);
        gtk_widget_set_sensitive (gfe->resolution_custom_combo, FALSE);
    }
    else
    {
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (gfe->resolution_custom_radio), TRUE);
    }

    widget = gtk_label_new (_("Color Depth"));
    gtk_widget_show (widget);
    gtk_misc_set_alignment (GTK_MISC (widget), COLUMN_ALIGN_1, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 6, 7);

    widget = grdc_public_create_combo_mapint (colordepth_list, gfe->grdc_file->colordepth, FALSE);
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 6, 7);
    gfe->colordepth_combo = widget;

    /* The Advanced tab */
    table = grdc_file_editor_create_notebook_tab (gfe, GTK_STOCK_DIALOG_WARNING, _("Advanced"), 8, 2);

    widget = gtk_label_new (_("Client Name"));
    gtk_widget_show (widget);
    gtk_widget_set_size_request (widget, COLUMN_WIDTH_1, -1);
    gtk_misc_set_alignment (GTK_MISC (widget), COLUMN_ALIGN_1, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 0, 1);

    widget = gtk_entry_new_with_max_length (100);
    gtk_widget_set_size_request (widget, COLUMN_WIDTH_2, -1);
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 0, 1);
    if (gfe->grdc_file->clientname) gtk_entry_set_text (GTK_ENTRY (widget), gfe->grdc_file->clientname);
    gfe->clientname_entry = widget;

    widget = gtk_label_new (_("Keyboard Mapping"));
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 1, 2);

    widget = grdc_public_create_combo_entry (rdp_keymap_list, gfe->grdc_file->keymap, FALSE);
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 1, 2);
    gfe->keymap_combo = widget;

    widget = gtk_label_new (_("Share Folder"));
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 2, 3);

    widget = grdc_public_create_combo_mapint (sharefolder_list, gfe->grdc_file->sharefolder, FALSE);
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 2, 3);
    gfe->sharefolder_combo = widget;

    widget = gtk_label_new (_("Startup Program"));
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 3, 4);

    widget = gtk_entry_new_with_max_length (300);
    gtk_widget_set_size_request (widget, COLUMN_WIDTH_2, -1);
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 3, 4);
    if (gfe->grdc_file->exec) gtk_entry_set_text (GTK_ENTRY (widget), gfe->grdc_file->exec);
    gfe->exec_entry = widget;

    widget = gtk_label_new (_("Startup Path"));
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 4, 5);

    widget = gtk_entry_new_with_max_length (300);
    gtk_widget_set_size_request (widget, COLUMN_WIDTH_2, -1);
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 4, 5);
    if (gfe->grdc_file->execpath) gtk_entry_set_text (GTK_ENTRY (widget), gfe->grdc_file->execpath);
    gfe->execpath_entry = widget;

    widget = gtk_check_button_new_with_label (_("Attach to console (Windows 2003 or newer)"));
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 2, 5, 6);
    if (gfe->grdc_file->console) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), TRUE);
    gfe->console_check = widget;

    hbox = gtk_hbox_new (TRUE, 0);
    gtk_widget_show (hbox);
    gtk_table_attach_defaults (GTK_TABLE (table), hbox, 0, 2, 6, 7);

    widget = gtk_check_button_new_with_label (_("Enable bitmap caching"));
    gtk_widget_show (widget);
    gtk_box_pack_start (GTK_BOX (hbox), widget, TRUE, TRUE, 0);
    if (gfe->grdc_file->bitmapcaching) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), TRUE);
    gfe->bitmapcaching_check = widget;

    widget = gtk_check_button_new_with_label (_("Enable compression"));
    gtk_widget_show (widget);
    gtk_box_pack_start (GTK_BOX (hbox), widget, TRUE, TRUE, 0);
    if (gfe->grdc_file->compression) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), TRUE);
    gfe->compression_check = widget;

    widget = gtk_label_new (_("Advanced Arguments"));
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 7, 8);

    widget = gtk_entry_new_with_max_length (300);
    gtk_widget_set_size_request (widget, COLUMN_WIDTH_2, -1);
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 7, 8);
    if (gfe->grdc_file->arguments) gtk_entry_set_text (GTK_ENTRY (widget), gfe->grdc_file->arguments);
    gfe->arguments_entry = widget;

    /* The SSH tab */
    grdc_file_editor_create_ssh_tab (gfe, 1);
}

static void
grdc_file_editor_create_vnc_config (GrdcFileEditor *gfe, gboolean incoming)
{
    GtkWidget *table;
    GtkWidget *widget;
    GtkWidget *hbox;
    gchar *keys;
    gchar *s;

    grdc_file_editor_create_notebook_container (gfe);

    /* The Basic tab */
    table = grdc_file_editor_create_notebook_tab (gfe, GTK_STOCK_INFO, _("Basic"), 7, 2);

    if (incoming)
    {
        widget = gtk_label_new (_("Listen on Port"));
        gtk_widget_show (widget);
        gtk_misc_set_alignment (GTK_MISC (widget), COLUMN_ALIGN_1, 0.5);
        gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 0, 1);

        s = g_strdup_printf ("%i", gfe->grdc_file->listenport);
        widget = gtk_entry_new_with_max_length (5);
        gtk_widget_show (widget);
        gtk_entry_set_activates_default (GTK_ENTRY (widget), TRUE);
        gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 0, 1);
        gtk_entry_set_text (GTK_ENTRY (widget), s);
        gfe->listenport_entry = widget;
        g_free (s);
    }
    else
    {
        widget = gtk_label_new (_("Server"));
        gtk_widget_show (widget);
        gtk_misc_set_alignment (GTK_MISC (widget), COLUMN_ALIGN_1, 0.5);
        gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 0, 1);

        s = grdc_pref_get_recent ("VNC");
        widget = grdc_public_create_combo_entry (s, gfe->grdc_file->server, TRUE);
        gtk_widget_show (widget);
        gtk_entry_set_activates_default (GTK_ENTRY (gtk_bin_get_child (GTK_BIN (widget))), TRUE);
        gfe->server_combo = widget;
        g_free (s);
#ifdef HAVE_LIBAVAHI_UI
        hbox = gtk_hbox_new (FALSE, 0);
        gtk_widget_show (hbox);
        gtk_box_pack_start (GTK_BOX (hbox), widget, TRUE, TRUE, 0);

        widget = gtk_button_new_with_label ("...");
        gtk_widget_set_tooltip_text (widget, _("Browse the network to find a VNC server"));
        gtk_widget_show (widget);
        gtk_box_pack_start (GTK_BOX (hbox), widget, FALSE, FALSE, 0);
        g_signal_connect (G_OBJECT (widget), "clicked", G_CALLBACK (grdc_file_editor_browse_avahi_vnc), gfe);

        gtk_table_attach_defaults (GTK_TABLE (table), hbox, 1, 2, 0, 1);
#else
        gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 0, 1);
#endif
    }

    widget = gtk_label_new (_("Password"));
    gtk_widget_show (widget);
    gtk_widget_set_size_request (widget, COLUMN_WIDTH_1, -1);
    gtk_misc_set_alignment (GTK_MISC (widget), COLUMN_ALIGN_1, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 1, 2);

    widget = gtk_entry_new_with_max_length (100);
    gtk_widget_show (widget);
    gtk_widget_set_size_request (widget, COLUMN_WIDTH_2, -1);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 1, 2);
    gtk_entry_set_visibility (GTK_ENTRY (widget), FALSE);
    if (gfe->grdc_file->password) gtk_entry_set_text (GTK_ENTRY (widget), gfe->grdc_file->password);
    gfe->password_entry = widget;

    widget = gtk_label_new (_("Color Depth"));
    gtk_widget_show (widget);
    gtk_misc_set_alignment (GTK_MISC (widget), COLUMN_ALIGN_1, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 2, 3);

    widget = grdc_public_create_combo_mapint (colordepth_list, gfe->grdc_file->colordepth, FALSE);
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 2, 3);
    gfe->colordepth_combo = widget;

    widget = gtk_label_new (_("Quality"));
    gtk_widget_show (widget);
    gtk_misc_set_alignment (GTK_MISC (widget), COLUMN_ALIGN_1, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 3, 4);

    widget = grdc_public_create_combo_mapint (quality_list, gfe->grdc_file->quality, FALSE);
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 3, 4);
    gfe->quality_combo = widget;

    widget = gtk_check_button_new_with_label (_("Show remote cursor"));
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 2, 4, 5);
    if (gfe->grdc_file->showcursor) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), TRUE);
    gfe->showcursor_check = widget;

    widget = gtk_check_button_new_with_label (_("View Only"));
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 2, 5, 6);
    if (gfe->grdc_file->viewonly) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), TRUE);
    gfe->viewonly_check = widget;

    widget = gtk_check_button_new_with_label (_("Disable Server Input"));
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 2, 6, 7);
    if (gfe->grdc_file->disableserverinput) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), TRUE);
    gfe->disableserverinput_check = widget;

    /* The Advanced tab */
    table = grdc_file_editor_create_notebook_tab (gfe, GTK_STOCK_DIALOG_WARNING, _("Advanced"), 3, 2);

    widget = gtk_label_new (_("Horizontal Scale"));
    gtk_widget_show (widget);
    gtk_widget_set_size_request (widget, COLUMN_WIDTH_1, -1);
    gtk_table_attach (GTK_TABLE (table), widget, 0, 1, 0, 1, 0, 0, 0, 0);

    widget = gtk_label_new (_("Vertical Scale"));
    gtk_widget_show (widget);
    gtk_table_attach (GTK_TABLE (table), widget, 0, 1, 1, 2, 0, 0, 0, 0);

    widget = grdc_scaler_new ();
    gtk_widget_show (widget);
    gtk_widget_set_size_request (widget, COLUMN_WIDTH_2, -1);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 0, 2);
    grdc_scaler_set (GRDC_SCALER (widget), gfe->grdc_file->hscale, gfe->grdc_file->vscale, gfe->grdc_file->aspectscale);
    gfe->scaler_widget = widget;

    widget = gtk_label_new (_("Keyboard Mapping"));
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 2, 3);

    keys = grdc_pref_keymap_groups ();
    widget = grdc_public_create_combo_text_d (keys, gfe->grdc_file->gkeymap, _("<None>"));
    g_free (keys);
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 2, 3);
    gfe->gkeymap_combo = widget;

    /* The SSH tab */
    grdc_file_editor_create_ssh_tab (gfe, (incoming ? 2 : 1));
}

static void
grdc_file_editor_create_sftp_config (GrdcFileEditor *gfe)
{
    grdc_file_editor_create_notebook_container (gfe);
    grdc_file_editor_create_ssh_tab (gfe, 3);
}

static void
grdc_file_editor_protocol_combo_on_changed (GtkComboBox *combo, GrdcFileEditor *gfe)
{
    gchar *protocol;

    if (gfe->config_container)
    {
        gtk_container_remove (GTK_CONTAINER (gfe->config_frame), gfe->config_container);
        gfe->config_container = NULL;
    }

    gfe->server_combo = NULL;
    gfe->listenport_entry = NULL;
    gfe->username_entry = NULL;
    gfe->password_entry = NULL;
    gfe->domain_entry = NULL;
    gfe->clientname_entry = NULL;
    gfe->resolution_auto_radio = NULL;
    gfe->resolution_custom_radio = NULL;
    gfe->resolution_custom_combo = NULL;
    gfe->colordepth_combo = NULL;
    gfe->quality_combo = NULL;
    gfe->keymap_combo = NULL;
    gfe->gkeymap_combo = NULL;
    gfe->exec_entry = NULL;
    gfe->execpath_entry = NULL;
    gfe->arguments_entry = NULL;
    gfe->sharefolder_combo = NULL;
    gfe->scaler_widget = NULL;

    gfe->bitmapcaching_check = NULL;
    gfe->compression_check = NULL;
    gfe->showcursor_check = NULL;
    gfe->viewonly_check = NULL;
    gfe->console_check = NULL;
    gfe->disableserverinput_check = NULL;

    gfe->ssh_enabled_check = NULL;
    gfe->ssh_server_default_radio = NULL;
    gfe->ssh_server_custom_radio = NULL;
    gfe->ssh_server_entry = NULL;
    gfe->ssh_username_entry = NULL;
    gfe->ssh_auth_password_radio = NULL;
    gfe->ssh_auth_publickey_radio = NULL;
    gfe->ssh_privatekey_entry = NULL;
    gfe->ssh_charset_combo = NULL;

    protocol = gtk_combo_box_get_active_text (combo);
    if (protocol)
    {
        if (strncmp (protocol, "VNC", 3) == 0)
        {
            grdc_file_editor_create_vnc_config (gfe, (protocol[3] == 'I'));
        }
        else if (g_strcmp0 (protocol, "SFTP") == 0)
        {
            grdc_file_editor_create_sftp_config (gfe);
        }
        else /* RDP */
        {
            grdc_file_editor_create_rdp_config (gfe);
        }
        g_free (protocol);
    }
}

static void
grdc_file_editor_update (GrdcFileEditor *gfe)
{
    gchar *s;

    g_free (gfe->grdc_file->name);
    gfe->grdc_file->name = g_strdup (gtk_entry_get_text (GTK_ENTRY (gfe->name_entry)));

    g_free (gfe->grdc_file->group);
    gfe->grdc_file->group = (gfe->group_combo ?
        gtk_combo_box_get_active_text (GTK_COMBO_BOX (gfe->group_combo)) : NULL);

    g_free (gfe->grdc_file->protocol);
    gfe->grdc_file->protocol = gtk_combo_box_get_active_text (GTK_COMBO_BOX (gfe->protocol_combo));

    g_free (gfe->grdc_file->server);
    gfe->grdc_file->server = (gfe->server_combo ?
        gtk_combo_box_get_active_text (GTK_COMBO_BOX (gfe->server_combo)) : NULL);

    if (gfe->username_entry)
    {
        g_free (gfe->grdc_file->username);
        gfe->grdc_file->username = g_strdup (gtk_entry_get_text (GTK_ENTRY (gfe->username_entry)));
    }

    g_free (gfe->grdc_file->password);
    gfe->grdc_file->password = (gfe->password_entry ?
        g_strdup (gtk_entry_get_text (GTK_ENTRY (gfe->password_entry))) : NULL);

    if (gfe->domain_entry)
    {
        g_free (gfe->grdc_file->domain);
        gfe->grdc_file->domain = g_strdup (gtk_entry_get_text (GTK_ENTRY (gfe->domain_entry)));
    }

    if (gfe->clientname_entry)
    {
        g_free (gfe->grdc_file->clientname);
        gfe->grdc_file->clientname = g_strdup (gtk_entry_get_text (GTK_ENTRY (gfe->clientname_entry)));
    }

    if (gfe->resolution_auto_radio)
    {
        g_free (gfe->grdc_file->resolution);
        gfe->grdc_file->resolution = (
            gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gfe->resolution_auto_radio)) ?
                g_strdup ("AUTO") :
                gtk_combo_box_get_active_text (GTK_COMBO_BOX (gfe->resolution_custom_combo)));
    }

    if (gfe->listenport_entry)
    {
        gfe->grdc_file->listenport = atoi (
            gtk_entry_get_text (GTK_ENTRY (gfe->listenport_entry)));
    }

    if (gfe->colordepth_combo)
    {
        s = gtk_combo_box_get_active_text (GTK_COMBO_BOX (gfe->colordepth_combo));
        gfe->grdc_file->colordepth = atoi (s);
        g_free (s);
    }

    if (gfe->quality_combo)
    {
        s = gtk_combo_box_get_active_text (GTK_COMBO_BOX (gfe->quality_combo));
        gfe->grdc_file->quality = atoi (s);
        g_free (s);
    }

    if (gfe->keymap_combo)
    {
        g_free (gfe->grdc_file->keymap);
        gfe->grdc_file->keymap = gtk_combo_box_get_active_text (GTK_COMBO_BOX (gfe->keymap_combo));
    }

    if (gfe->gkeymap_combo)
    {
        g_free (gfe->grdc_file->gkeymap);
        gfe->grdc_file->gkeymap = gtk_combo_box_get_active_text (GTK_COMBO_BOX (gfe->gkeymap_combo));
    }

    if (gfe->sharefolder_combo)
    {
        s = gtk_combo_box_get_active_text (GTK_COMBO_BOX (gfe->sharefolder_combo));
        gfe->grdc_file->sharefolder = atoi (s);
        g_free (s);
    }

    if (gfe->scaler_widget)
    {
        gfe->grdc_file->hscale = GRDC_SCALER (gfe->scaler_widget)->hscale;
        gfe->grdc_file->vscale = GRDC_SCALER (gfe->scaler_widget)->vscale;
        gfe->grdc_file->aspectscale = GRDC_SCALER (gfe->scaler_widget)->aspectscale;
    }

    if (gfe->ssh_charset_combo)
    {
        g_free (gfe->grdc_file->ssh_charset);
        gfe->grdc_file->ssh_charset = gtk_combo_box_get_active_text (GTK_COMBO_BOX (gfe->ssh_charset_combo));
    }

    if (gfe->exec_entry)
    {
        g_free (gfe->grdc_file->exec);
        gfe->grdc_file->exec = g_strdup (gtk_entry_get_text (GTK_ENTRY (gfe->exec_entry)));
    }

    if (gfe->execpath_entry)
    {
        g_free (gfe->grdc_file->execpath);
        gfe->grdc_file->execpath = g_strdup (gtk_entry_get_text (GTK_ENTRY (gfe->execpath_entry)));
    }

    if (gfe->arguments_entry)
    {
        g_free (gfe->grdc_file->arguments);
        gfe->grdc_file->arguments = g_strdup (gtk_entry_get_text (GTK_ENTRY (gfe->arguments_entry)));
    }

    if (gfe->bitmapcaching_check)
    {
        gfe->grdc_file->bitmapcaching = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gfe->bitmapcaching_check));
    }
    if (gfe->compression_check)
    {
        gfe->grdc_file->compression = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gfe->compression_check));
    }
    if (gfe->showcursor_check)
    {
        gfe->grdc_file->showcursor = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gfe->showcursor_check));
    }
    if (gfe->viewonly_check)
    {
        gfe->grdc_file->viewonly = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gfe->viewonly_check));
    }
    if (gfe->console_check)
    {
        gfe->grdc_file->console = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gfe->console_check));
    }
    if (gfe->disableserverinput_check)
    {
        gfe->grdc_file->disableserverinput = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gfe->disableserverinput_check));
    }

    if (g_strcmp0 (gfe->grdc_file->protocol, "SFTP") == 0)
    {
        gfe->grdc_file->ssh_enabled = TRUE;
    }
    else
    {
        gfe->grdc_file->ssh_enabled = (gfe->ssh_enabled_check ?
            gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gfe->ssh_enabled_check)) : FALSE);
    }
    g_free (gfe->grdc_file->ssh_username);
    gfe->grdc_file->ssh_username = (gfe->grdc_file->ssh_enabled ?
        g_strdup (gtk_entry_get_text (GTK_ENTRY (gfe->ssh_username_entry))) : NULL);
    g_free (gfe->grdc_file->ssh_server);
    gfe->grdc_file->ssh_server = (gfe->grdc_file->ssh_enabled && gfe->ssh_server_entry &&
        (gfe->ssh_server_custom_radio == NULL ||
        gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gfe->ssh_server_custom_radio))) ?
        g_strdup (gtk_entry_get_text (GTK_ENTRY (gfe->ssh_server_entry))) : NULL);
    gfe->grdc_file->ssh_auth = (
        gfe->ssh_auth_publickey_radio &&
        gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gfe->ssh_auth_publickey_radio)) ?
        SSH_AUTH_PUBLICKEY : SSH_AUTH_PASSWORD);
    g_free (gfe->grdc_file->ssh_privatekey);
    gfe->grdc_file->ssh_privatekey = (
        gfe->grdc_file->ssh_auth == SSH_AUTH_PUBLICKEY ?
        g_strdup (gtk_entry_get_text (GTK_ENTRY (gfe->ssh_privatekey_entry))) : NULL);
}

static void
grdc_file_editor_set_default (GtkWidget *button, GrdcFileEditor *gfe)
{
    GrdcFile *gf;
    GtkWidget *dialog;

    grdc_file_editor_update (gfe);

    gf = grdc_file_dup (gfe->grdc_file);

    g_free (gf->filename);
    gf->filename = g_strdup (grdc_pref_file);

    /* Clear properties that should never be default */
    g_free (gf->name); gf->name = NULL;
    g_free (gf->server); gf->server = NULL;
    g_free (gf->password); gf->password = NULL;

    grdc_file_save (gf);

    grdc_file_free (gf);

    dialog = gtk_message_dialog_new (GTK_WINDOW (gfe->editor_dialog),
        GTK_DIALOG_MODAL, GTK_MESSAGE_INFO, GTK_BUTTONS_OK,
        _("Default settings saved."));
    gtk_dialog_run (GTK_DIALOG (dialog));
    gtk_widget_destroy (dialog);
}

static void
grdc_file_editor_open_groups (GtkWidget *button, GrdcFileEditor *gfe)
{
    if (grdc_pref_dialog_run (GTK_WINDOW (gfe->editor_dialog), GRDC_PREF_GROUPS_TAB))
    {
        grdc_public_load_combo_text_d (gfe->group_combo, grdc_pref.groups, gfe->grdc_file->group, _("<None>"));
    }
}

static void
grdc_file_editor_init (GrdcFileEditor *gfe)
{
    GtkWidget *frame;
    GtkWidget *table;
    GtkWidget *hbox;
    GtkWidget *widget;

    /* Create the editor dialog */
    gfe->editor_dialog = gtk_dialog_new_with_buttons (_("Remote Desktop Preference"),
        gfe->parent, GTK_DIALOG_MODAL,
        GTK_STOCK_SAVE, GTK_RESPONSE_APPLY,
        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
        GTK_STOCK_CONNECT, GTK_RESPONSE_OK,
        NULL);
    gtk_dialog_set_alternative_button_order (GTK_DIALOG (gfe->editor_dialog),
        GTK_RESPONSE_OK, GTK_RESPONSE_APPLY, GTK_RESPONSE_CANCEL, -1);
    gtk_dialog_set_default_response (GTK_DIALOG (gfe->editor_dialog), GTK_RESPONSE_OK);
    gtk_dialog_set_has_separator (GTK_DIALOG (gfe->editor_dialog), FALSE);
    gtk_window_set_resizable (GTK_WINDOW (gfe->editor_dialog), FALSE);
    if (gfe->grdc_file->filename == NULL)
    {
        gtk_dialog_set_response_sensitive (GTK_DIALOG (gfe->editor_dialog), GTK_RESPONSE_APPLY, FALSE);
    }

    g_signal_connect (G_OBJECT (gfe->editor_dialog),
        "realize",G_CALLBACK (grdc_file_editor_on_realize), gfe);

    /* The Set As Default button */
    widget = gtk_button_new_with_label (_("Default"));
    gtk_widget_show (widget);
    gtk_button_set_image (GTK_BUTTON (widget), gtk_image_new_from_stock (GTK_STOCK_PREFERENCES, GTK_ICON_SIZE_BUTTON));
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (gfe->editor_dialog)->action_area), widget, FALSE, TRUE, 0);
    gtk_button_box_set_child_secondary (GTK_BUTTON_BOX (GTK_DIALOG (gfe->editor_dialog)->action_area), widget, TRUE);

    g_signal_connect (G_OBJECT (widget), "clicked", G_CALLBACK (grdc_file_editor_set_default), gfe);

    /* Create the Profile frame on the top (for name and protocol) */
    frame = gtk_frame_new (_("Profile"));
    gtk_widget_show (frame);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (gfe->editor_dialog)->vbox), frame, TRUE, TRUE, 2);

    table = gtk_table_new (3, 2, FALSE);
    gtk_widget_show (table);
    gtk_table_set_row_spacings (GTK_TABLE (table), 4);
    gtk_table_set_col_spacings (GTK_TABLE (table), 4);
    gtk_container_set_border_width (GTK_CONTAINER (table), 8);
    gtk_container_add (GTK_CONTAINER (frame), table);

    /* Profile: Name */
    widget = gtk_label_new (_("Name"));
    gtk_widget_show (widget);
    gtk_widget_set_size_request (widget, 100, -1);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 0, 1);

    widget = gtk_entry_new_with_max_length (100);
    gtk_widget_show (widget);
    gtk_widget_set_size_request (widget, 300, -1);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 0, 1);
    gfe->name_entry = widget;
    if (gfe->grdc_file->filename == NULL)
    {
        gtk_widget_set_sensitive (widget, FALSE);
        gtk_entry_set_text (GTK_ENTRY (widget), _("Quick Connect"));
    }
    else
    {
        if (gfe->grdc_file->name) gtk_entry_set_text (GTK_ENTRY (widget), gfe->grdc_file->name);
    }

    /* Profile: Group */
    if (gfe->grdc_file->filename == NULL)
    {
        gfe->group_combo = NULL;
    }
    else
    {
        widget = gtk_label_new (_("Group"));
        gtk_widget_show (widget);
        gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 1, 2);

        hbox = gtk_hbox_new (FALSE, 0);
        gtk_widget_show (hbox);
        gtk_table_attach_defaults (GTK_TABLE (table), hbox, 1, 2, 1, 2);

        gfe->group_combo = grdc_public_create_combo_text_d (grdc_pref.groups, gfe->grdc_file->group, _("<None>"));
        gtk_widget_show (gfe->group_combo);
        gtk_box_pack_start (GTK_BOX (hbox), gfe->group_combo, TRUE, TRUE, 0);

        widget = gtk_button_new_with_label ("...");
        gtk_widget_show (widget);
        gtk_box_pack_start (GTK_BOX (hbox), widget, FALSE, FALSE, 0);
        g_signal_connect (G_OBJECT (widget), "clicked", G_CALLBACK (grdc_file_editor_open_groups), gfe);
    }

    /* Profile: Protocol */
    widget = gtk_label_new (_("Protocol"));
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 0, 1, 2, 3);

    widget = grdc_public_create_combo_map (protocol_list, gfe->grdc_file->protocol, TRUE);
    gtk_widget_show (widget);
    gtk_table_attach_defaults (GTK_TABLE (table), widget, 1, 2, 2, 3);
    g_signal_connect (G_OBJECT (widget), "changed", G_CALLBACK (grdc_file_editor_protocol_combo_on_changed), gfe);
    gfe->protocol_combo = widget;

    /* Create the Preference frame */
    frame = gtk_frame_new (_("Settings"));
    gtk_widget_show (frame);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (gfe->editor_dialog)->vbox), frame, TRUE, TRUE, 2);
    gfe->config_frame = frame;

    gfe->config_container = NULL;

    grdc_file_editor_protocol_combo_on_changed (GTK_COMBO_BOX (gfe->protocol_combo), gfe);
}

static gint
grdc_file_editor_run (GrdcFileEditor *gfe)
{
    gint ret;
    GrdcFile *gf;

    ret = gtk_dialog_run (GTK_DIALOG (gfe->editor_dialog));
    switch (ret)
    {
        case GTK_RESPONSE_OK:
            grdc_file_editor_update (gfe);
            gf = grdc_file_dup (gfe->grdc_file);
            /* Put server into name for Quick Connect */
            if (gf->filename == NULL)
            {
                g_free (gf->name);
                gf->name = g_strdup (gf->server);
            }
            grdc_connection_window_open_struct (gfe->parent, gf, gfe->quit_when_close);
            break;
        case GTK_RESPONSE_APPLY:
            grdc_file_editor_update (gfe);
            break;
        default:
            break;
    }
    return ret;
}

static gint
grdc_file_editor_main (GrdcFileEditor *gfe)
{
    gint ret;

    grdc_file_editor_init (gfe);
    ret = grdc_file_editor_run (gfe);
    if (ret == GTK_RESPONSE_OK || ret == GTK_RESPONSE_APPLY)
    {
        grdc_file_save (gfe->grdc_file);
    }
    gtk_widget_destroy (gfe->editor_dialog);
    return ret;
}

gint
grdc_file_editor_create (GtkWindow *parent, gboolean quit_when_close)
{
    return grdc_file_editor_create_full (parent, quit_when_close, NULL, NULL);
}

gint
grdc_file_editor_create_full (GtkWindow *parent, gboolean quit_when_close,
    const gchar *server, const gchar *protocol)
{
    GrdcFileEditor editor;
    gint ret;

    editor.parent = parent;
    editor.grdc_file = grdc_file_new ();
    if (server) { g_free (editor.grdc_file->server); editor.grdc_file->server = g_strdup (server); }
    if (protocol) { g_free (editor.grdc_file->protocol); editor.grdc_file->protocol = g_strdup (protocol); }
    editor.quit_when_close = quit_when_close;

    ret = grdc_file_editor_main (&editor);

    grdc_file_free (editor.grdc_file);

    return ret;
}

gint
grdc_file_editor_open (GtkWindow *parent, gchar *filename, gboolean quit_when_close)
{
    GrdcFile *grdcfile;
    GrdcFileEditor editor;
    GtkWidget *dialog;
    gint ret;

    grdcfile = grdc_file_load (filename);
    if (grdcfile)
    {
        editor.parent = parent;
        editor.grdc_file = grdcfile;
        editor.quit_when_close = quit_when_close;

        ret = grdc_file_editor_main (&editor);

        grdc_file_free (editor.grdc_file);

        return ret;
    }
    else
    {
        dialog = gtk_message_dialog_new (parent, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
            _("File %s not found."), filename);
        gtk_dialog_run (GTK_DIALOG (dialog));
        gtk_widget_destroy (dialog);
        return GTK_RESPONSE_CANCEL;
    }
}

gint
grdc_file_editor_open_temp (GtkWindow *parent, gboolean quit_when_close)
{
    return grdc_file_editor_open_temp_full (parent, quit_when_close, NULL, NULL);
}

gint
grdc_file_editor_open_temp_full (GtkWindow *parent, gboolean quit_when_close,
    const gchar *server, const gchar *protocol)
{
    GrdcFileEditor editor;
    gint ret;

    editor.parent = parent;
    editor.grdc_file = grdc_file_new_temp ();
    if (server) { g_free (editor.grdc_file->server); editor.grdc_file->server = g_strdup (server); }
    if (protocol) { g_free (editor.grdc_file->protocol); editor.grdc_file->protocol = g_strdup (protocol); }
    editor.quit_when_close = quit_when_close;

    ret = grdc_file_editor_main (&editor);

    grdc_file_free (editor.grdc_file);

    return ret;
}

