/*
 * Grdc - GTK+/Gnome Remote Desktop Client
 * Copyright (C) 2009 - Vic Lee 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <stdlib.h>
#include "grdcpublic.h"
#include "grdcpref.h"
#include "grdcpixmaps.h"
#include "grdccrypt.h"
#include "grdcfile.h"

#define MIN_WINDOW_WIDTH 10
#define MIN_WINDOW_HEIGHT 10

#define DEFAULT_LISTEN_PORT 5500

const gpointer colordepth_list[] =
{
    "8", N_("256 Colors"),
    "15", N_("High Color (15 bit)"),
    "16", N_("High Color (16 bit)"),
    "24", N_("True Color (24 bit)"),
    NULL
};

const gpointer quality_list[] =
{
    "0", N_("Poor (Fastest)"),
    "1", N_("Medium"),
    "2", N_("Good"),
    "9", N_("Best (Slowest)"),
    NULL
};

GrdcFile*
grdc_file_new_temp (void)
{
    GrdcFile *gf;

    /* Try to load from the preference file for default settings first */
    gf = grdc_file_load (grdc_pref_file);

    if (!gf)
    {
        gf = (GrdcFile*) g_malloc (sizeof (GrdcFile));

        gf->filename = NULL;
        gf->name = NULL;
        gf->group = NULL;
        gf->server = NULL;
        gf->protocol = g_strdup ("RDP");
        gf->username = NULL;
        gf->password = NULL;
        gf->domain = NULL;
        gf->clientname = NULL;
        gf->resolution = g_strdup ("AUTO");
        gf->keymap = NULL;
        gf->gkeymap = NULL;
        gf->exec = NULL;
        gf->execpath = NULL;
        gf->arguments = NULL;
        gf->colordepth = 8;
        gf->quality = 0;
        gf->listenport = DEFAULT_LISTEN_PORT;
        gf->sharefolder = 0;
        gf->hscale = 0;
        gf->vscale = 0;

        gf->bitmapcaching = FALSE;
        gf->compression = FALSE;
        gf->showcursor = FALSE;
        gf->viewonly = FALSE;
        gf->console = FALSE;
        gf->disableserverinput = FALSE;
        gf->aspectscale = FALSE;

        gf->ssh_enabled = FALSE;
        gf->ssh_server = NULL;
        gf->ssh_auth = SSH_AUTH_PASSWORD;
        gf->ssh_username = NULL;
        gf->ssh_privatekey = NULL;
        gf->ssh_charset = NULL;
    }

    g_free (gf->name);
    gf->name = g_strdup (_("New Connection"));
    g_free (gf->filename);
    gf->filename = NULL;

    gf->viewmode = grdc_pref.default_mode;
    gf->scale = FALSE;
    gf->keyboard_grab = FALSE;
    gf->window_width = 640;
    gf->window_height = 480;
    gf->window_maximize = FALSE;
    gf->toolbar_opacity = 0;

    return gf;
}

GrdcFile*
grdc_file_new (void)
{
    GrdcFile *gf;
    GTimeVal gtime;
    gchar buf[MAX_PATH_LEN];

    g_get_current_time (&gtime);
    g_snprintf (buf, MAX_PATH_LEN, "%s/.grdc/%li%03li.grdc", g_get_home_dir (),
        gtime.tv_sec, gtime.tv_usec / 1000);

    gf = grdc_file_new_temp ();
    gf->filename = g_strdup (buf);

    return gf;
}

GrdcFile*
grdc_file_load (const gchar *filename)
{
    GKeyFile *gkeyfile;
    GrdcFile *gf;
    gchar *s;

    gkeyfile = g_key_file_new ();

    if (!g_key_file_load_from_file (gkeyfile, filename, G_KEY_FILE_NONE, NULL)) return NULL;

    if (g_key_file_has_key (gkeyfile, "grdc", "name", NULL))
    {
        gf = (GrdcFile*) g_malloc (sizeof (GrdcFile));

        gf->filename = g_strdup (filename);
        gf->name = g_key_file_get_string (gkeyfile, "grdc", "name", NULL);
        gf->group = g_key_file_get_string (gkeyfile, "grdc", "group", NULL);
        gf->server = g_key_file_get_string (gkeyfile, "grdc", "server", NULL);
        gf->protocol = g_key_file_get_string (gkeyfile, "grdc", "protocol", NULL);
        gf->username = g_key_file_get_string (gkeyfile, "grdc", "username", NULL);

        s = g_key_file_get_string (gkeyfile, "grdc", "password", NULL);
        gf->password = grdc_crypt_decrypt (s);
        g_free (s);

        gf->domain = g_key_file_get_string (gkeyfile, "grdc", "domain", NULL);
        gf->clientname = g_key_file_get_string (gkeyfile, "grdc", "clientname", NULL);
        gf->resolution = g_key_file_get_string (gkeyfile, "grdc", "resolution", NULL);
        gf->keymap = g_key_file_get_string (gkeyfile, "grdc", "keymap", NULL);
        gf->gkeymap = g_key_file_get_string (gkeyfile, "grdc", "gkeymap", NULL);
        gf->exec = g_key_file_get_string (gkeyfile, "grdc", "exec", NULL);
        gf->execpath = g_key_file_get_string (gkeyfile, "grdc", "execpath", NULL);
        gf->arguments = g_key_file_get_string (gkeyfile, "grdc", "arguments", NULL);

        gf->colordepth = g_key_file_get_integer (gkeyfile, "grdc", "colordepth", NULL);
        gf->quality = g_key_file_get_integer (gkeyfile, "grdc", "quality", NULL);
        gf->listenport = (g_key_file_has_key (gkeyfile, "grdc", "listenport", NULL) ?
            g_key_file_get_integer (gkeyfile, "grdc", "listenport", NULL) : DEFAULT_LISTEN_PORT);
        gf->sharefolder = g_key_file_get_integer (gkeyfile, "grdc", "sharefolder", NULL);
        gf->hscale = g_key_file_get_integer (gkeyfile, "grdc", "hscale", NULL);
        gf->vscale = g_key_file_get_integer (gkeyfile, "grdc", "vscale", NULL);

        gf->bitmapcaching = g_key_file_get_boolean (gkeyfile, "grdc", "bitmapcaching", NULL);
        gf->compression = g_key_file_get_boolean (gkeyfile, "grdc", "compression", NULL);
        gf->showcursor = g_key_file_get_boolean (gkeyfile, "grdc", "showcursor", NULL);
        gf->viewonly = g_key_file_get_boolean (gkeyfile, "grdc", "viewonly", NULL);
        gf->console = g_key_file_get_boolean (gkeyfile, "grdc", "console", NULL);
        gf->disableserverinput = g_key_file_get_boolean (gkeyfile, "grdc", "disableserverinput", NULL);
        gf->aspectscale = g_key_file_get_boolean (gkeyfile, "grdc", "aspectscale", NULL);

        gf->ssh_enabled = g_key_file_get_boolean (gkeyfile, "grdc", "ssh_enabled", NULL);
        gf->ssh_server = g_key_file_get_string (gkeyfile, "grdc", "ssh_server", NULL);
        gf->ssh_auth = g_key_file_get_integer (gkeyfile, "grdc", "ssh_auth", NULL);
        gf->ssh_username = g_key_file_get_string (gkeyfile, "grdc", "ssh_username", NULL);
        gf->ssh_privatekey = g_key_file_get_string (gkeyfile, "grdc", "ssh_privatekey", NULL);
        gf->ssh_charset = g_key_file_get_string (gkeyfile, "grdc", "ssh_charset", NULL);

        gf->viewmode = g_key_file_get_integer (gkeyfile, "grdc", "viewmode", NULL);
        gf->scale = g_key_file_get_boolean (gkeyfile, "grdc", "scale", NULL);
        gf->keyboard_grab = g_key_file_get_boolean (gkeyfile, "grdc", "keyboard_grab", NULL);
        gf->window_width = MAX (MIN_WINDOW_WIDTH, g_key_file_get_integer (gkeyfile, "grdc", "window_width", NULL));
        gf->window_height = MAX (MIN_WINDOW_HEIGHT, g_key_file_get_integer (gkeyfile, "grdc", "window_height", NULL));
        gf->window_maximize = (g_key_file_has_key (gkeyfile, "grdc", "window_maximize", NULL) ?
            g_key_file_get_boolean (gkeyfile, "grdc", "window_maximize", NULL) : TRUE);
        gf->toolbar_opacity = MIN (TOOLBAR_OPACITY_LEVEL,
            MAX (0, g_key_file_get_integer (gkeyfile, "grdc", "toolbar_opacity", NULL)));
    }
    else
    {
        gf = NULL;
    }

    g_key_file_free (gkeyfile);

    return gf;
}

void
grdc_file_save (GrdcFile *gf)
{
    GKeyFile *gkeyfile;
    gchar *s;
    gchar *content;
    gsize length;

    if (gf->filename == NULL) return;

    gkeyfile = g_key_file_new ();

    g_key_file_load_from_file (gkeyfile, gf->filename, G_KEY_FILE_NONE, NULL);

    g_key_file_set_string (gkeyfile, "grdc", "name", (gf->name ? gf->name : ""));
    g_key_file_set_string (gkeyfile, "grdc", "group", (gf->group ? gf->group : ""));
    g_key_file_set_string (gkeyfile, "grdc", "server", (gf->server ? gf->server : ""));
    g_key_file_set_string (gkeyfile, "grdc", "protocol", (gf->protocol ? gf->protocol : ""));
    g_key_file_set_string (gkeyfile, "grdc", "username", (gf->username ? gf->username : ""));

    s = grdc_crypt_encrypt (gf->password);
    g_key_file_set_string (gkeyfile, "grdc", "password", (s ? s : ""));
    g_free (s);

    g_key_file_set_string (gkeyfile, "grdc", "domain", (gf->domain ? gf->domain : ""));
    g_key_file_set_string (gkeyfile, "grdc", "clientname", (gf->clientname ? gf->clientname : ""));
    g_key_file_set_string (gkeyfile, "grdc", "resolution", (gf->resolution ? gf->resolution : ""));
    g_key_file_set_string (gkeyfile, "grdc", "keymap", (gf->keymap ? gf->keymap : ""));
    g_key_file_set_string (gkeyfile, "grdc", "gkeymap", (gf->gkeymap ? gf->gkeymap : ""));
    g_key_file_set_string (gkeyfile, "grdc", "exec", (gf->exec ? gf->exec : ""));
    g_key_file_set_string (gkeyfile, "grdc", "execpath", (gf->execpath ? gf->execpath : ""));
    g_key_file_set_string (gkeyfile, "grdc", "arguments", (gf->arguments ? gf->arguments : ""));

    g_key_file_set_integer (gkeyfile, "grdc", "colordepth", gf->colordepth);
    g_key_file_set_integer (gkeyfile, "grdc", "quality", gf->quality);
    g_key_file_set_integer (gkeyfile, "grdc", "listenport", gf->listenport);
    g_key_file_set_integer (gkeyfile, "grdc", "sharefolder", gf->sharefolder);
    g_key_file_set_integer (gkeyfile, "grdc", "hscale", gf->hscale);
    g_key_file_set_integer (gkeyfile, "grdc", "vscale", gf->vscale);

    g_key_file_set_boolean (gkeyfile, "grdc", "bitmapcaching", gf->bitmapcaching);
    g_key_file_set_boolean (gkeyfile, "grdc", "compression", gf->compression);
    g_key_file_set_boolean (gkeyfile, "grdc", "showcursor", gf->showcursor);
    g_key_file_set_boolean (gkeyfile, "grdc", "viewonly", gf->viewonly);
    g_key_file_set_boolean (gkeyfile, "grdc", "console", gf->console);
    g_key_file_set_boolean (gkeyfile, "grdc", "disableserverinput", gf->disableserverinput);
    g_key_file_set_boolean (gkeyfile, "grdc", "aspectscale", gf->aspectscale);

    g_key_file_set_boolean (gkeyfile, "grdc", "ssh_enabled", gf->ssh_enabled);
    g_key_file_set_string (gkeyfile, "grdc", "ssh_server", (gf->ssh_server ? gf->ssh_server : ""));
    g_key_file_set_integer (gkeyfile, "grdc", "ssh_auth", gf->ssh_auth);
    g_key_file_set_string (gkeyfile, "grdc", "ssh_username", (gf->ssh_username ? gf->ssh_username : ""));
    g_key_file_set_string (gkeyfile, "grdc", "ssh_privatekey", (gf->ssh_privatekey ? gf->ssh_privatekey : ""));
    g_key_file_set_string (gkeyfile, "grdc", "ssh_charset", (gf->ssh_charset ? gf->ssh_charset : ""));

    g_key_file_set_integer (gkeyfile, "grdc", "viewmode", gf->viewmode);
    g_key_file_set_boolean (gkeyfile, "grdc", "scale", gf->scale);
    g_key_file_set_boolean (gkeyfile, "grdc", "keyboard_grab", gf->keyboard_grab);
    g_key_file_set_integer (gkeyfile, "grdc", "window_width", gf->window_width);
    g_key_file_set_integer (gkeyfile, "grdc", "window_height", gf->window_height);
    g_key_file_set_boolean (gkeyfile, "grdc", "window_maximize", gf->window_maximize);
    g_key_file_set_integer (gkeyfile, "grdc", "toolbar_opacity", gf->toolbar_opacity);

    content = g_key_file_to_data (gkeyfile, &length, NULL);
    g_file_set_contents (gf->filename, content, length, NULL);

    g_key_file_free (gkeyfile);
    g_free (content);
}

void
grdc_file_free (GrdcFile *grdcfile)
{
    if (grdcfile == NULL) return;

    g_free (grdcfile->filename);
    g_free (grdcfile->name);
    g_free (grdcfile->group);
    g_free (grdcfile->server);
    g_free (grdcfile->protocol);
    g_free (grdcfile->username);
    g_free (grdcfile->password);
    g_free (grdcfile->domain);
    g_free (grdcfile->clientname);
    g_free (grdcfile->resolution);
    g_free (grdcfile->keymap);
    g_free (grdcfile->gkeymap);
    g_free (grdcfile->exec);
    g_free (grdcfile->execpath);
    g_free (grdcfile->arguments);

    g_free (grdcfile->ssh_server);
    g_free (grdcfile->ssh_username);
    g_free (grdcfile->ssh_privatekey);
    g_free (grdcfile->ssh_charset);
    g_free (grdcfile);
}

GrdcFile*
grdc_file_dup (GrdcFile *grdcfile)
{
    GrdcFile *gf;

    gf = (GrdcFile*) g_malloc (sizeof (GrdcFile));
    gf->filename = g_strdup (grdcfile->filename);
    gf->name = g_strdup (grdcfile->name);
    gf->group = g_strdup (grdcfile->group);
    gf->server = g_strdup (grdcfile->server);
    gf->protocol = g_strdup (grdcfile->protocol);
    gf->username = g_strdup (grdcfile->username);
    gf->password = g_strdup (grdcfile->password);
    gf->domain = g_strdup (grdcfile->domain);
    gf->clientname = g_strdup (grdcfile->clientname);
    gf->resolution = g_strdup (grdcfile->resolution);
    gf->keymap = g_strdup (grdcfile->keymap);
    gf->gkeymap = g_strdup (grdcfile->gkeymap);
    gf->exec = g_strdup (grdcfile->exec);
    gf->execpath = g_strdup (grdcfile->execpath);
    gf->arguments = g_strdup (grdcfile->arguments);

    gf->colordepth = grdcfile->colordepth;
    gf->quality = grdcfile->quality;
    gf->listenport = grdcfile->listenport;
    gf->sharefolder = grdcfile->sharefolder;
    gf->hscale = grdcfile->hscale;
    gf->vscale = grdcfile->vscale;

    gf->bitmapcaching = grdcfile->bitmapcaching;
    gf->compression = grdcfile->compression;
    gf->showcursor = grdcfile->showcursor;
    gf->viewonly = grdcfile->viewonly;
    gf->console = grdcfile->console;
    gf->disableserverinput = grdcfile->disableserverinput;
    gf->aspectscale = grdcfile->aspectscale;

    gf->ssh_enabled = grdcfile->ssh_enabled;
    gf->ssh_server = g_strdup (grdcfile->ssh_server);
    gf->ssh_auth = grdcfile->ssh_auth;
    gf->ssh_username = g_strdup (grdcfile->ssh_username);
    gf->ssh_privatekey = g_strdup (grdcfile->ssh_privatekey);
    gf->ssh_charset = g_strdup (grdcfile->ssh_charset);

    gf->viewmode = grdcfile->viewmode;
    gf->scale = grdcfile->scale;
    gf->keyboard_grab = grdcfile->keyboard_grab;
    gf->window_width = grdcfile->window_width;
    gf->window_height = grdcfile->window_height;
    gf->window_maximize = grdcfile->window_maximize;
    gf->toolbar_opacity = grdcfile->toolbar_opacity;

    return gf;
}

void
grdc_file_update_screen_resolution (GrdcFile *grdcfile)
{
    GdkScreen *screen;
    gchar *tmp, *pos;


    if (grdcfile->resolution == NULL ||
        g_strcmp0 (grdcfile->resolution, "AUTO") == 0 ||
        g_strrstr (grdcfile->resolution, "x") == NULL)
    {
        screen = gdk_screen_get_default ();
        grdcfile->resolution_width = gdk_screen_get_width (screen);
        grdcfile->resolution_height = gdk_screen_get_height (screen);
    }
    else
    {
        tmp = g_strdup (grdcfile->resolution);
        pos = g_strrstr (tmp, "x");
        *pos++ = '\0';
        grdcfile->resolution_width = MAX (100, MIN (4096, atoi (tmp)));
        grdcfile->resolution_height = MAX (100, MIN (4096, atoi (pos)));
    }
}

GdkPixbuf*
grdc_file_get_icon (GrdcFile *grdcfile, gint size)
{
    gint type;

    if (g_strcmp0 (grdcfile->protocol, "SFTP") == 0)
    {
        type = GRDC_PIXMAP_TYPE_SSH;
    }
    else
    {
        type = (strncmp (grdcfile->protocol, "VNC", 3) == 0 ?
            (grdcfile->ssh_enabled ? GRDC_PIXMAP_TYPE_VNC_SSH : GRDC_PIXMAP_TYPE_VNC) :
            (grdcfile->ssh_enabled ? GRDC_PIXMAP_TYPE_RDP_SSH : GRDC_PIXMAP_TYPE_RDP));
    }
    return grdc_pixmap (type, size);
}

gboolean
grdc_file_is_incoming (GrdcFile *grdcfile)
{
    return (strncmp (grdcfile->protocol, "VNC", 3) == 0 && grdcfile->protocol[3] == 'I');
}

