/**
 * GMyth Library
 *
 * @file gmyth/gmyth_scheduler.h
 *
 * @brief <p> The scheduler encapsulates all functions for browsing, scheduling
 * and modifying the recorded content.
 *
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia.
 * @author Alexsandro Jose Virginio dos Santos <alexsandro.santos@indt.org.br>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef __GMYTH_SCHEDULER_H__
#define __GMYTH_SCHEDULER_H__

#include <glib-object.h>
#include <time.h>

#include "gmyth_common.h"
#include "gmyth_query.h"
#include "gmyth_backendinfo.h"

G_BEGIN_DECLS

typedef enum {
    GMYTH_SCHEDULE_ONE_OCCURRENCE,
    GMYTH_SCHEDULE_ALL_OCCURRENCES,
    GMYTH_SCHEDULE_EXCEPTION
} GMythScheduleType;


#define GMYTH_SCHEDULER_TYPE               (gmyth_scheduler_get_type ())
#define GMYTH_SCHEDULER(obj)               (G_TYPE_CHECK_INSTANCE_CAST ((obj), GMYTH_SCHEDULER_TYPE, GMythScheduler))
#define GMYTH_SCHEDULER_CLASS(klass)       (G_TYPE_CHECK_CLASS_CAST ((klass), GMYTH_SCHEDULER_TYPE, GMythSchedulerClass))
#define IS_GMYTH_SCHEDULER(obj)            (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GMYTH_SCHEDULER_TYPE))
#define IS_GMYTH_SCHEDULER_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST ((klass), GMYTH_SCHEDULER_TYPE))
#define GMYTH_SCHEDULER_GET_CLASS(obj)     (G_TYPE_INSTANCE_GET_CLASS ((obj), GMYTH_SCHEDULER_TYPE, GMythSchedulerClass))

typedef struct _GMythScheduler GMythScheduler;
typedef struct _GMythSchedulerClass GMythSchedulerClass;

struct _GMythSchedulerClass {
    GObjectClass    parent_class;
};

struct _GMythScheduler {
    GObject         parent;

    unsigned long   recordid;
    unsigned long   type;
    unsigned long   search;
    GString        *profile;

    long            dupin;
    long            dupmethod;
    long            autoexpire;
    short int       autotranscode;
    long            transcoder;

    short int       autocommflag;
    short int       autouserjob1;
    short int       autouserjob2;
    short int       autouserjob3;
    short int       autouserjob4;

    long            startoffset;
    long            endoffset;
    long            maxepisodes;
    long            maxnewest;

    long            recpriority;
    GString        *recgroup;
    GString        *playgroup;

    long            prefinput;
    short int       inactive;

    GString        *search_type;
    GString        *search_what;

    GMythQuery     *msqlquery;
    GMythBackendInfo *backend_info;
};

typedef struct {
    guint           schedule_id;
    gint            channel_id;
    GString        *program_id;

    GTimeVal       *start_time;
    GTimeVal       *end_time;

    GString        *seriesid;
    GString        *title;
    GString        *subtitle;
    GString        *description;
    GString        *category;

    GMythScheduleType type;

    gint parentid;

} ScheduleInfo;

typedef struct {
    guint           record_id;
    gint            channel_id;
    GString        *program_id;

    GTimeVal       *start_time;
    GTimeVal       *end_time;

    GString         *seriesid;
    GString        *title;
    GString        *subtitle;
    GString        *description;
    GString        *category;

    GString        *basename;

    guint64         filesize;

} RecordedInfo;


GType               gmyth_scheduler_get_type                (void);

GMythScheduler*     gmyth_scheduler_new                     (void);
gboolean            gmyth_scheduler_connect                 (GMythScheduler * scheduler,
                                                             GMythBackendInfo * backend_info);
gboolean            gmyth_scheduler_connect_with_timeout    (GMythScheduler * scheduler,
                                                             GMythBackendInfo * backend_info,
                                                             guint timeout);
gboolean            gmyth_scheduler_disconnect              (GMythScheduler * scheduler);
gint                gmyth_scheduler_get_schedule_list       (GMythScheduler * scheduler,
                                                             GList ** sched_list);
gint                gmyth_scheduler_get_recorded_list       (GMythScheduler * scheduler,
                                                             GList ** rec_list);
RecordedInfo*       gmyth_scheduler_get_recorded_info       (GMythScheduler *scheduler,
                                                             const char *basename);
gboolean            gmyth_scheduler_was_recorded_before     (GMythScheduler* scheduler,
                                                             gint channel_id,
                                                             time_t start_time);
gboolean            gmyth_scheduler_reactivate_schedule     (GMythScheduler* scheduler,
                                                             gint channel_id,
                                                             time_t start_time);
GMythProgramInfo*   gmyth_scheduler_get_recorded_on_time    (GMythScheduler* scheduler,
                                                             guint channel_id);
GMythProgramInfo*   gmyth_scheduler_get_recorded            (GMythScheduler * scheduler,
                                                             GString * channel,
                                                             GTimeVal * starttime);
ScheduleInfo*       gmyth_scheduler_add_schedule_program    (GMythScheduler * scheduler,
                                                             GMythProgramInfo *program,
                                                             GMythScheduleType type);
gint                gmyth_scheduler_add_schedule            (GMythScheduler * scheduler,
                                                             ScheduleInfo * schedule_info);
gboolean            gmyth_scheduler_add_schedule_full       (GMythScheduler * scheduler,
                                                             ScheduleInfo * schedule_info,
                                                             GMythScheduleType type);
gboolean            gmyth_scheduler_add_exception           (GMythScheduler *scheduler,
                                                             gint schedule_id,
                                                             ScheduleInfo *exception_info);
gboolean            gmyth_scheduler_delete_schedule         (GMythScheduler * scheduler,
                                                             gint record_id);
gint                gmyth_scheduler_delete_recorded         (GMythScheduler * scheduler,
                                                             gint record_id);
gboolean            gmyth_scheduler_stop_recording          (GMythScheduler * scheduler,
                                                             gint channel_id);
void                gmyth_scheduler_recorded_info_get_preview(RecordedInfo * info,
                                                             GByteArray * data);


void                gmyth_recorded_info_free                (RecordedInfo * info);
void                gmyth_schedule_info_free                (ScheduleInfo * info);
void                gmyth_recorded_info_list_free           (GList * list);
void                gmyth_schedule_info_list_free           (GList * list);

G_END_DECLS
#endif                          /* __GMYTH_SCHEDULER_H__ */
