/*
 * Galaxium Messenger
 * Copyright (C) 2003-2007 Philippe Durand <draekz@gmail.com>
 * Copyright (C) 2007 Ben Motmans <ben.motmans@gmail.com>
 * 
 * License: GNU General Public License (GPL)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.IO;
using System.Collections.Generic;

using Anculus.Core;
using Galaxium.Core;

namespace Galaxium.Protocol
{
	public abstract class AbstractConversation : IConversation, IComparable<AbstractConversation>
	{
		public event EventHandler<ConversationEventArgs> Established;
		public event EventHandler<ConversationEventArgs> Closed;
		
		public event EventHandler<TextMessageEventArgs> MessageReceived;
		
		public event EventHandler<ContactActionEventArgs> ContactJoined;
		public event EventHandler<ContactActionEventArgs> ContactLeft;
		public event EventHandler<ConversationEventArgs> AllContactsLeft;
		
		protected Guid _uid;
		protected bool _active = false;
		protected ISession _session;
		protected IContact _primaryContact;
		protected ContactCollection _contacts;
		
		protected IConversationLog _log;
		private EventHandler _logAttachHandler;
		
		protected AbstractConversation (IContact primaryContact, ISession session)
		{
			ThrowUtility.ThrowIfNull ("session", session);
			
			_uid = Guid.NewGuid();
			_primaryContact = primaryContact;
			_session = session;
			_contacts = new ContactCollection ();
			
			_logAttachHandler = new EventHandler (LoggingEnabledChanged);
			ConversationLogUtility.LoggingEnabledChanged += _logAttachHandler;
			
			RequestLoggerIfNeeded (ConversationLogUtility.EnableEventLogging);
		}
		
		~AbstractConversation ()
		{
			ConversationLogUtility.LoggingEnabledChanged -= _logAttachHandler;
			
			if (_log != null)
				_log.Close();
		}
		
		public Guid UniqueIdentifier
		{
			get { return _uid; }
		}
		
		public ISession Session
		{
			get { return _session; }
		}
		
		public virtual bool Active
		{
			get { return _active; }
			set { _active = value; }
		}
		
		public IConversationLog ConversationLog
		{
			get { return _log; }
		}
		
		//FIXME: this should be internal
		public ContactCollection ContactCollection
		{
			get { return _contacts; }
		}
		
		public IContact PrimaryContact
		{
			get { return _primaryContact; }
		}
		
		public virtual bool IsPrivateConversation
		{
			get { return _contacts.Count < 2; }
		}
		
		public virtual bool IsChannelConversation
		{
			get { return false; }
		}
		
		public abstract void Close ();
		
		public abstract void InviteContact (IContact contact);
		
		public int CompareTo(AbstractConversation conversation)
		{
			return conversation.UniqueIdentifier.CompareTo(this.UniqueIdentifier);
		}
		
		public int CompareTo(IConversation conversation)
		{
			return conversation.UniqueIdentifier.CompareTo(this.UniqueIdentifier);
		}
		
		public virtual void AddContact (IContact contact)
		{
			if (contact == null)
				throw new ArgumentNullException ("contact");
			
			_contacts.Add (contact);
			
			OnContactJoined (new ContactActionEventArgs (Session, this, contact));
		}
		
		public virtual void RemoveContact (IContact contact)
		{
			if (contact == null)
				throw new ArgumentNullException ("contact");
			
			_contacts.Remove (contact);
			
			OnContactLeft (new ContactActionEventArgs (Session, this, contact));
			if (_contacts.Count == 0)
				OnAllContactsLeft (new ConversationEventArgs (this));
		}
		
		protected virtual void OnEstablished (ConversationEventArgs args)
		{
			if (Established != null)
				Established (this, args);
		}
		
		protected virtual void OnClosed (ConversationEventArgs args)
		{
			Active=  false;
			
			if (Closed != null)
				Closed (this, args);
		}
		
		protected virtual void OnMessageReceived (TextMessageEventArgs args)
		{
			if (MessageReceived != null)
				MessageReceived (this, args);
		}
		
		protected virtual void OnContactJoined (ContactActionEventArgs args)
		{
			if (ContactJoined != null)
				ContactJoined (this, args);
		}
		
		protected virtual void OnContactLeft (ContactActionEventArgs args)
		{
			if (_contacts.Count < 1)
				Active = false;
			
			if (ContactLeft != null)
				ContactLeft (this, args);
		}
		
		protected virtual void OnAllContactsLeft (ConversationEventArgs args)
		{
			Active = false;
			
			if (AllContactsLeft != null)
				AllContactsLeft (this, args);
		}
		
		protected virtual void LogMessage (ITextMessage msg)
		{
			if (_primaryContact != null && _primaryContact.Logging)
			{
				try
				{
					if (RequestLoggerIfNeeded (false))
						_log.LogMessage (msg);
				}
				catch (Exception ex)
				{
					Anculus.Core.Log.Error (ex, "Exception thrown whilst logging message");
				}
			}
		}
		
		protected virtual void LogEvent (DateTime timestamp, string description)
		{
			if (_primaryContact != null && _primaryContact.Logging)
			{
				try
				{
					if (RequestLoggerIfNeeded (true))
						_log.LogEvent (timestamp, description);
				}
				catch (Exception ex)
				{
					Anculus.Core.Log.Error (ex, "Exception thrown whilst logging message");
				}
			}
		}
		
		protected virtual bool RequestLoggerIfNeeded (bool logEvent)
		{
			if (!ConversationLogUtility.EnableLogging)
				return false;
			
			if (logEvent && !ConversationLogUtility.EnableEventLogging)
				return false;
			
			if (_log == null)
				_log = ConversationLogUtility.GetConversationLog (this);
			
			return true;
		}
		
		private void LoggingEnabledChanged (object sender, EventArgs args)
		{
			if (!ConversationLogUtility.EnableLogging)
			{
				_log.Close ();
				_log = null;
			}
		}
	}
}