/*
 * Galaxium Messenger
 * Copyright (C) 2007 Ben Motmans <ben.motmans@gmail.com>
 * 
 * License: GNU General Public License (GPL)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.IO;
using System.Diagnostics;
using System.Collections.Generic;

using Mono.Addins;

using Gtk;

using Anculus.Core;

using Galaxium.Core;
using Galaxium.Gui;

namespace Galaxium.Gui.GtkGui
{
	public static class GtkUtility
	{
		private static bool _initialized;
		private static bool _enableSwf;
		
		public static bool EnableSwf
		{
			get { return _enableSwf; }
		}
		
		public static void Initialize ()
		{
			if (!_initialized)
			{
				Log.Debug ("Initializing gtk utility...");
				
				IconUtility.Initialize ();
			
				try
				{
					Swfdec.Swfdec.Initialize ();
					_enableSwf = true;
				}
				catch (DllNotFoundException ex)
				{
					Log.Info ("Unable to find {0}, swf playback disabled", ex.Message);
				}
				
				_initialized = true;
			}
		}
		
		public static void CheckThreading ()
		{
#if DEBUG
			if (System.Threading.Thread.CurrentThread.Name != CoreUtility.ApplicationName)
			{
				StackTrace trace = new StackTrace(1, true);
				Log.Fatal ("Not executing in main thread:\n"+trace.ToString());
			}
#endif
		}
		
		public static AccelKey ParseAccelKey (string accelKey)
		{
			if (String.IsNullOrEmpty (accelKey))
				return AccelKey.Zero;
			
			Gdk.Key key = Gdk.Key.VoidSymbol;
			
			string[] split = accelKey.Split ('+');
			
			if (split.Length == 1)
			{
				//it's a key alone, eg F7
				if (TryParseKey (split[0], out key))
					return new AccelKey (key, Gdk.ModifierType.None, AccelFlags.Visible);
			}
			else if (split.Length == 2)
			{
				//modifier+key
				Gdk.ModifierType modifiers = ParseModifier (split[0]);
				
				if (TryParseKey (split[1], out key))
					return new AccelKey (key, modifiers, AccelFlags.Visible);
			}
			else if (split.Length == 3)
			{
				//modifier+modifier+key
				Gdk.ModifierType modifiers = ParseModifier (split[0]);
				
				modifiers |= ParseModifier (split[1]);
				
				if (TryParseKey (split[2], out key))
					return new AccelKey (key, modifiers, AccelFlags.Visible);
			}
			return AccelKey.Zero;
		}
		
		public static bool TryParseKey (string keyString, out Gdk.Key key)
		{
			try
			{
				key = (Gdk.Key)Enum.Parse (typeof (Gdk.Key), keyString);
				return true;
			}
			catch (ArgumentException)
			{
				
			}
			
			key = Gdk.Key.VoidSymbol;
			
			return false;
		}
		
		public static Gdk.ModifierType ParseModifier (string modifierString)
		{
			if (String.IsNullOrEmpty (modifierString))
				return Gdk.ModifierType.None;
			
			if (String.Compare (modifierString, "control", true) == 0 || String.Compare (modifierString, "control", true) == 0)
				return Gdk.ModifierType.ControlMask;
			else if (String.Compare (modifierString, "shift", true) == 0)
				return Gdk.ModifierType.ShiftMask;
			else if (String.Compare (modifierString, "alt", true) == 0)
				return Gdk.ModifierType.Mod1Mask;
			
			return Gdk.ModifierType.None;
		}
	}
}