/*
 * Copyright (C) 2009 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as 
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include "mail-server.h"

G_DEFINE_TYPE (MailServer, mail_server, INDICATE_TYPE_SERVER);

#define MAIL_SERVER_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  MAIL_TYPE_SERVER, \
  MailServerPrivate))

struct _MailServerPrivate
{
  EShell *shell;
  guint count;
};

enum
{
  PROP_0,

  PROP_COUNT
};

/* Forwards */
static gboolean mail_server_get_indicator_count (IndicateServer  *server,
                                                 guint           *count, 
                                                 GError          **error);
static gboolean mail_server_get_indicator_property (IndicateServer *server,
                                                    guint           id,
                                                    gchar          *type,
                                                    gchar         **value,
                                                    GError        **error);

static gboolean mail_server_show_indicator_to_user (IndicateServer *server,
                                                    guint           id,
                                                    GError        **error);

static void really_present_window (GtkWindow *window);

/* GObject stuff */
static void
mail_server_get_property (GObject    *object,
                        guint       prop_id,
                        GValue     *value,
                        GParamSpec *pspec)
{
  MailServer *server = MAIL_SERVER (object);

  switch (prop_id)
  {
    case PROP_COUNT:
      g_value_set_uint (value, server->priv->count);
      break;
    
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}

static void
mail_server_set_property (GObject      *object,
                        guint         prop_id,
                        const GValue *value,
                        GParamSpec   *pspec)
{
  MailServer *server = MAIL_SERVER (object);

  switch (prop_id)
  {
    case PROP_COUNT:
      mail_server_set_message_count (server, g_value_get_uint (value));
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


static void
mail_server_finalize (GObject *obj)
{
  MailServer *server = MAIL_SERVER (obj);
  MailServerPrivate *priv;

  g_return_if_fail (MAIL_IS_SERVER (server));
  priv = server->priv;

  G_OBJECT_CLASS (mail_server_parent_class)->finalize (obj);
}

static void
mail_server_class_init (MailServerClass *klass)
{
  GParamSpec   *pspec;
  GObjectClass *obj_class         = G_OBJECT_CLASS (klass);
  IndicateServerClass * ind_class = INDICATE_SERVER_CLASS (klass);
    
  obj_class->finalize     = mail_server_finalize;
  obj_class->set_property = mail_server_set_property;
  obj_class->get_property = mail_server_get_property;

  ind_class->get_indicator_count    = mail_server_get_indicator_count;
  ind_class->get_indicator_property = mail_server_get_indicator_property;
  ind_class->show_indicator_to_user = mail_server_show_indicator_to_user;

  /* Property */
  pspec = g_param_spec_uint ("count",
                             "count",
                             "count",
                             0, G_MAXUINT, 0,
                             G_PARAM_READWRITE);
  g_object_class_install_property (obj_class, PROP_COUNT, pspec); 

  g_type_class_add_private (obj_class, sizeof (MailServerPrivate));
}

static void
mail_server_init (MailServer *server)
{
  MailServerPrivate *priv;
       
  priv = server->priv = MAIL_SERVER_GET_PRIVATE (server);

  indicate_server_set_desktop_file(INDICATE_SERVER(server), "/usr/share/applications/evolution-mail.desktop");
  indicate_server_set_type(INDICATE_SERVER(server), "message.mail");
}

/*
 * Public Functions
 */

MailServer * 
mail_server_get_default (void)
{
  static MailServer *server = NULL;
  
  if (!server)
    server = g_object_new (MAIL_TYPE_SERVER, 
                           NULL);
  return server;
}

guint
mail_server_get_message_count (MailServer *server)
{
  g_return_val_if_fail (MAIL_IS_SERVER (server), 0);

  return server->priv->count;
}

void
mail_server_set_message_count (MailServer *server, 
                               guint       count)
{
  MailServerPrivate *priv;
  gint i, diff;
  gboolean added = TRUE;

  g_return_if_fail (MAIL_IS_SERVER (server));
  priv = server->priv;
  
  g_debug ("MAIL SERVER: Count changed: %d", count);

  if (priv->count >= count)
  {
    added = FALSE;
    diff = priv->count - count;
  }
  else
  {
    diff = count - priv->count;
  }

  server->priv->count = count;
  
  for (i = 0; i < diff; i++)
    added ? indicate_server_emit_indicator_added (INDICATE_SERVER (server),
                                                  i + 1, "message")
          : indicate_server_emit_indicator_removed (INDICATE_SERVER (server), 
                                                    diff - i, "message");
}

/*
 * Indicate Server overrides
 */
static gboolean 
mail_server_get_indicator_count (IndicateServer  *server,
                                 guint           *count, 
                                 GError         **error)
{
  g_return_val_if_fail (MAIL_IS_SERVER (server), FALSE);

  *count = MAIL_SERVER (server)->priv->count;
  return TRUE;
}

static gboolean 
mail_server_get_indicator_property (IndicateServer *server,
                                    guint           id,
                                    gchar          *type,
                                    gchar         **value,
                                    GError        **error)
{
  g_return_val_if_fail (MAIL_IS_SERVER (server), FALSE);

  if (g_strcmp0 (type, "subtype") == 0)
  {
    *value = g_strdup ("email");
    return TRUE;
  }

  if (g_strcmp0 (type, "desktop") == 0)
  {
    /* We should probably do something clever here */
    *value = g_strdup ("/usr/share/applications/evolution-mail.desktop");
    return TRUE;
  }

  g_warning ("Mail Server does not support the \"%s\" property", type);

  return FALSE;
}

static gboolean 
mail_server_show_indicator_to_user (IndicateServer *server,
                                    guint           id,
                                    GError        **error)
{
#define MAIL_ICON "evolution-mail"
  MailServerPrivate *priv;

  g_return_val_if_fail (MAIL_IS_SERVER (server), FALSE);
  priv = MAIL_SERVER (server)->priv;

  g_debug ("MAIL SERVER: Show Evolution to user");

  if (priv->shell)
  {
    GList *w;
    GtkWindow *mail_window = NULL;

    for (w = priv->shell->priv->windows; w; w = w->next)
    {
      GtkWindow *window = w->data;

      if (GTK_IS_WINDOW (window))
      {
        if (g_strcmp0 (MAIL_ICON, gtk_window_get_icon_name (window)) == 0)
        {
          mail_window = window;
        }
      }
      else
      {
        g_warning ("No open windows: Wait, we're not a mac :-/");
        return FALSE;
      }
    }

    if (!GTK_IS_WINDOW (mail_window))
    {
      if (priv->shell->priv->windows && priv->shell->priv->windows->data)
        mail_window = priv->shell->priv->windows->data;
      else
        return FALSE;
    }

    really_present_window (mail_window);
    mail_server_set_message_count (MAIL_SERVER (server), 0);
  }
  else
  {
    g_warning ("Cannot show window, no shell");
    return FALSE;
  }

  return TRUE;
}

void
mail_server_set_e_shell (MailServer *server,
                         EShell     *shell)
{
  g_return_if_fail (MAIL_IS_SERVER (server));

  server->priv->shell = shell;
}


/*
 * Taken from libtomboy, (C) 2008 Novell, LGPL v2 or later
 */

static void
tomboy_window_override_user_time (GtkWindow *window)
{
	guint32 ev_time = gtk_get_current_event_time();

	if (ev_time == 0) {
		/* 
		 * FIXME: Global keypresses use an event filter on the root
		 * window, which processes events before GDK sees them.
		 */
		//ev_time = tomboy_keybinder_get_current_event_time ();
    ev_time = GDK_CURRENT_TIME;
	}
	if (ev_time == 0) {
		gint ev_mask = gtk_widget_get_events (GTK_WIDGET(window));
		if (!(ev_mask & GDK_PROPERTY_CHANGE_MASK)) {
			gtk_widget_add_events (GTK_WIDGET (window),
					       GDK_PROPERTY_CHANGE_MASK);
		}

		/* 
		 * NOTE: Last resort for D-BUS or other non-interactive
		 *       openings.  Causes roundtrip to server.  Lame. 
		 */
		ev_time = gdk_x11_get_server_time (GTK_WIDGET(window)->window);
	}

	gdk_x11_window_set_user_time (GTK_WIDGET(window)->window, ev_time);
}

static void 
really_present_window (GtkWindow *window)
{
	if (!GTK_WIDGET_REALIZED (window))
		gtk_widget_realize (GTK_WIDGET (window));

	tomboy_window_override_user_time (window);

	gtk_window_present (window);
}

