/*
 *  Copyright (c) 2007 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Value.h"

#include "Debug.h"
#include "Type.h"

using namespace GTLCore;

struct Value::Private {
  union {
    bool b;
    float f;
    int i;
    unsigned int ui;
  } value;
  const Type* type;
};

Value::Value() : d(new Private)
{
  d->type = Type::Undefined;
}

Value::Value(float v) : d(new Private)
{
  d->value.f = v;
  d->type = Type::Float;
}

Value::Value(bool v) : d(new Private)
{
  d->value.b = v;
  d->type = Type::Boolean;
}

Value::Value(int v) : d(new Private)
{
  d->value.i = v;
  d->type = Type::Integer32;
}

Value::Value(unsigned int v) : d(new Private)
{
  d->value.ui = v;
  d->type = Type::UnsignedInteger32;
}


Value::Value(const Value& rhs) : d(new Private(*rhs.d))
{
}
Value Value::operator=(const Value& rhs)
{
  *d = *rhs.d;
  return *this;
}
Value::~Value()
{
  delete d;
}

#define RETURN_AS( _type_ ) \
  switch( d->type->dataType()) \
  { \
    case Type::BOOLEAN: \
      return (_type_)d->value.b; \
    case Type::FLOAT: \
      return (_type_)d->value.f; \
    case Type::INTEGER32: \
      return (_type_)d->value.i; \
    case Type::UNSIGNED_INTEGER32: \
      return (_type_)d->value.ui; \
    default: \
      GTL_ABORT("Can't convert to that type."); \
      return (_type_)0; \
 }

float Value::asFloat() const
{
  RETURN_AS(float);
}

bool Value::asBoolean() const
{
  RETURN_AS(bool);
}

int Value::asInt32() const
{
  RETURN_AS(int);
}

unsigned int Value::asUnsignedInt32() const
{
  RETURN_AS(unsigned int);
}

const Type* Value::type() const
{
  return d->type;
}

void Value::setInt32(int v)
{
  d->value.i = v;
  d->type = Type::Integer32;
}
