/* DefaultCellListCellRenderer.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.lists;

import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics2D;
import java.awt.Insets;
import java.awt.Rectangle;
import javax.swing.Icon;
import javax.swing.JLabel;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;

/**
 * Default implementation of {@link CellListCellRenderer}. Renders a 
 * cell to look like a horizontally
 * centered label, with caption below the icon, except 
 * that the selection background and focus indicator are restricted to
 * the text part of the label.
 * <p>Works best with list whose elements implement the interface 
 * {@link CellListCellValue}.
 * <p>Extension classes override method {@link #prepare prepare} to
 * determine what text and what icon should be displayed.
 */
public class DefaultCellListCellRenderer implements CellListCellRenderer {
   
    /**
     * Default constructor.
     */
    public DefaultCellListCellRenderer() {
        setFont (UIManager.getFont ("Label.font"));
        setForeground (UIManager.getColor ("Label.foreground"));
        setSelectionBackground (UIManager.getColor("List.selectionBackground"));
        setSelectionForeground (UIManager.getColor("List.selectionForeground"));
	setBorderSelectionColor(UIManager.getColor("Tree.selectionBorderColor"));
        
        setIconBackground (null);
            
        setInsets (new Insets (3,5,3,5));
    }

    
    /**
     * {@inheritDoc}<p>
     *
     * If {@code value} is of type {@link CellListCellValue}, this 
     * implementation returns the tool tip text provided by that interface.
     * Otherwise it returns {@code null}.
     */
    public String getToolTipText(CellList list, Object value, int index) {
        if (value instanceof CellListCellValue) {
            return ((CellListCellValue)value).getCellToolTipText ();
        }
        else
            return null;
    }

    /**
     * Caption to be displayed.
     */
    private String text;
    
    /**
     * Return the caption to be displayed.
     */
    public String getText ()  {

        return this.text;
    }

    /**
     * Set the caption to be displayed.
     */
    public void setText (String text)  {
        this.text = text;
    }
    

    /**
     * Holds value of property foreground.
     */
    private Color foreground;
    
    /**
     * The color which is used to display the caption.
     * @return Value of property foreGround.
     */
    public Color getForeground()  {

        return this.foreground;
    }


    /**
     * Set the color to be used for the caption.
     * @param foreground New value of property foreGround.
     */
    public final void setForeground(java.awt.Color foreground)  {

        this.foreground = foreground;
    }

    /**
     * Font used for the caption.
     */
    private Font font;

    /**
     * Return the font used for the caption.
     */
    public Font getFont() {
        return this.font;
    }

    /**
     * Set the font used for the caption.
     */
    public final void setFont(Font font) {
        this.font = font;
    }
    

    /**
     * Holds value of property icon.
     */
    private Icon icon;
    
    /**
     * Icon to be displayed.
     * @return Value of property icon.
     */
    public Icon getIcon () {
        return this.icon;
    }

    /**
     * Set the icon to be displayed.
     * @param icon New value of property icon.
     */
    public void setIcon (Icon icon) {
        this.icon = icon;
    }

    //
    private int iconTextGap = 4;
    
    /**
     * Return the gap between text and icon
     */
    public int getIconTextGap () {
        return this.iconTextGap;
    }
    
    /**
     * Set the gap between text and icon
     */
    public void setIconTextGap (int iconTextGap) {
        this.iconTextGap = iconTextGap;
    }
    
    //
    private Insets insets;
    
    /**
     * Return the insets of this cell.
     */
    public Insets getInsets () {
        return this.insets;
    }
    
    /**
     * Set the insets of this cell.
     */
    public final void setInsets (Insets insets) {
        this.insets = insets;
    }

    /**
     * Holds value of property selectionForeground.
     */
    private Color selectionForeground;
    

    /**
     * Return the text color for selections.
     * @return Value of property selectionForeground.
     */
    public Color getSelectionForeground() {
        return this.selectionForeground;
    }

    /**
     * Set the text color for selections.
     * @param selectionForeground New value of property selectionForeground.
     */
    public final void setSelectionForeground(Color selectionForeground) {
        this.selectionForeground = selectionForeground;
    }
    
    /**
     * Holds value of property selectionBackground.
     */
    private Color selectionBackground;
    

    /**
     * Return the background color for selections.
     * @return Value of property selectionBackground.
     */
    public Color getSelectionBackground() {
        return this.selectionBackground;
    }

    /**
     * Set the background color for selections.
     * @param selectionBackground New value of property selectionBackground.
     */
    public final void setSelectionBackground(Color selectionBackground) {
        this.selectionBackground = selectionBackground;
    }
    
    /**
     * Holds value of property iconBackground.
     */
    private Color iconBackground;
    

    /**
     * Return the background color for icons.
     * @return Value of property iconBackground.
     */
    public Color getIconBackground() {
        return this.iconBackground;
    }

    /**
     * Set the background color for icons.
     * @param iconBackground New value of property iconBackground.
     */
    public final void setIconBackground(Color iconBackground) {
        this.iconBackground = iconBackground;
    }

    //
    private Color borderSelectionColor;
    
    /**
     * Return the color which is used to draw the focus indicator.
     */
    public Color getBorderSelectionColor () {
        return this.borderSelectionColor;
    }
    
    /**
     * Set the color which is used to draw the focus indicator.
     */
    public final void setBorderSelectionColor (Color borderSelectionColor) {
        this.borderSelectionColor = borderSelectionColor;
    }

    /**
     * Prepare the renderer so that the given element can be displayed.
     * Extensions of this class should override this method (rather
     * than {@link #paintElement paintElement}) 
     * to set the correct text and icon.
     * <p>This implementation distinguishes three cases:
     * <ul>
     * <li>If {@code value} is of type {@link CellListCellValue}, 
     * icon and text
     * are set according to the information provided by that interface.
     * </li>
     * <li>Otherwise, if {@code value} is of type {@link Icon}, that
     * icon is used, without text caption.</li>
     * <li>Otherwise, no icon is displayed and the text is set to 
     * <tt>value.toString()</tt>.</li>
     * </ul>
     * @param list The cell list to which the element belongs
     * @param value The corresponding value returned from the model
     * @param index The index of the cell in the list
     * @param isSelected True if the specified cell was selected.
     * @param cellHasFocus True if the specified cell has the focus.
     */
    protected void prepare(CellList list, Object value, 
        int index, boolean isSelected, boolean cellHasFocus) {
        if (value instanceof CellListCellValue) {
            CellListCellValue clcv = (CellListCellValue)value;
            setText (clcv.getCellText());
            if (isSelected) {
                setIcon (clcv.getCellSelectedIcon());
            }
            else {
                setIcon (clcv.getCellIcon());                
            }
        }
        else if (value instanceof Icon) {
            setIcon ((Icon)value);
            setText (null);
        }
        else {
            setText (value.toString());
            setIcon (null);
        }
    }
    
    //
    private final Rectangle viewR = new Rectangle ();
    private final Rectangle iconR = new Rectangle ();
    private final Rectangle textR = new Rectangle ();
    
    /**
     * {@inheritDoc} The element is painted to look like a horizontally
     * centered label, with caption below the icon, except 
     * that the selection background and focus indicator are restricted to
     * the text part of the label.<p>
     * This implementation first calls {@link #prepare prepare} to initialize
     * the text and icon property from the value parameter. Extensions
     * of this class are advised to override {@code prepare} 
     * instead of this method.
     */
    public void paintElement(Graphics2D g2, CellList list, Object value, 
            int index, boolean isSelected, boolean cellHasFocus) {

        prepare (list, value, index, isSelected, cellHasFocus);

        viewR.x = insets.left;
        viewR.y = insets.top;
        viewR.width = list.getCellWidth () - insets.left - insets.right;
        viewR.height = list.getCellHeight () - insets.top - insets.bottom;
        iconR.x = 0;
        iconR.y = 0;
        iconR.width = 0;
        iconR.height = 0;
        textR.x = 0;
        textR.y = 0;
        textR.width = 0;
        textR.height = 0;
        
        FontMetrics fm = g2.getFontMetrics (font);
        String finalText = SwingUtilities.layoutCompoundLabel (
                fm,
                text, 
                icon,
                JLabel.BOTTOM, JLabel.CENTER, JLabel.BOTTOM, JLabel.CENTER, 
                viewR, iconR, textR, iconTextGap);
        
        // paint Text
        if (finalText != null && finalText.length() > 0) {
            g2.setFont (font);
            if (isSelected) {
                g2.setColor(selectionBackground);
                g2.fill (textR);
                g2.setColor (selectionForeground);
            }
            else {
                g2.setColor (foreground);
            }
            g2.drawString (finalText, textR.x, textR.y + fm.getAscent());
        }
        
        // paint Focus around text
        if (cellHasFocus) {
            g2.setColor(borderSelectionColor);
            g2.drawRect(textR.x - 1, textR.y - 1, textR.width + 1, textR.height + 1);
        }
        
        // paint Icon
        if (icon != null) {
            if (iconBackground != null) {
                g2.setColor(iconBackground);
                g2.fill (iconR);
            }
            Rectangle iconClip = iconR.intersection(g2.getClipBounds());
            g2.setClip (iconClip); 
            icon.paintIcon (list, g2, iconR.x, iconR.y);
        }
        
    }

}
