package org.jboss.cache.eviction;


import org.jboss.cache.CacheSPI;
import org.jboss.cache.DefaultCacheFactory;
import org.jboss.cache.Fqn;
import org.jboss.cache.util.TestingUtil;
import org.jboss.cache.factories.UnitTestCacheConfigurationFactory;
import org.jboss.cache.config.Configuration;
import org.jboss.cache.config.EvictionConfig;
import org.jboss.cache.config.EvictionRegionConfig;
import org.jboss.cache.lock.IsolationLevel;
import static org.testng.AssertJUnit.*;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

import java.util.List;
import java.util.ArrayList;

/**
 * Unit tests for LRU Policy.
 *
 * @author Ben Wang, Feb 11, 2004
 * @author Daniel Huang - dhuang@jboss.org
 * @version $Revision: 4880 $
 */
@Test(groups = {"functional"})
public class NullEvictionPolicyTest
{
   CacheSPI<Object, Object> cache_;

   @BeforeMethod(alwaysRun = true)
   public void setUp() throws Exception
   {
      cache_ = null;
   }

   @AfterMethod(alwaysRun = true)
   public void tearDown() throws Exception
   {
      if (cache_ != null)
      {
         cache_.stop();
         cache_.destroy();
      }      
   }





   /**
    * Builds a cache was null eviction by default and in "/test" region,
    * LRU in "/lru" region.  Does a mix of puts/reads/removes, wakes for
    * eviction thread to kick in, checks that nothing was evicted from the
    * null policy regions but was from lru region.
    */
   public void testEviction()
   {      
      Configuration config = UnitTestCacheConfigurationFactory.createConfiguration(Configuration.CacheMode.LOCAL, true);
      EvictionConfig evConfig = config.getEvictionConfig();
      evConfig.setWakeupIntervalSeconds(1);
      evConfig.setDefaultEventQueueSize(200000);
      evConfig.setDefaultEvictionPolicyClass("org.jboss.cache.eviction.NullEvictionPolicy");
      List<EvictionRegionConfig> regionConfigs = new ArrayList<EvictionRegionConfig>();
      regionConfigs.add(buildEvictionRegionConfig("/_default_"));
      regionConfigs.add(buildEvictionRegionConfig("/test"));
      regionConfigs.add(UnitTestCacheConfigurationFactory.buildLruEvictionRegionConfig("/lru", 10000,1));
      evConfig.setEvictionRegionConfigs(regionConfigs);
      config.setTransactionManagerLookupClass("org.jboss.cache.transaction.DummyTransactionManagerLookup");
      config.setIsolationLevel(IsolationLevel.SERIALIZABLE);
      cache_ = (CacheSPI)new DefaultCacheFactory().createCache(config);

      String dfltRootStr = "/a/";
      String testRootStr = "/test/";
      String lruRootStr = "/lru/";
      
      for (int i = 0; i < 20; i++)
      {                 
            Fqn dflt = Fqn.fromString(dfltRootStr + i);
            Fqn test = Fqn.fromString(testRootStr + i);
            Fqn lru = Fqn.fromString(lruRootStr + i);
            cache_.put(dflt, "key", "value");
            cache_.put(test, "key", "value");
            cache_.put(lru, "key", "value");
      }
      TestingUtil.sleepThread(3500);
      for (int i = 0; i < 20; i++)
      {
         Fqn dflt = Fqn.fromString(dfltRootStr + i);
         Fqn test = Fqn.fromString(testRootStr + i);
         Fqn lru = Fqn.fromString(lruRootStr + i);

            assertEquals("value", cache_.get(dflt, "key"));
            assertEquals("value", cache_.get(test, "key"));
            assertNull(cache_.get(lru, "key"));
      }
   }

   private EvictionRegionConfig buildEvictionRegionConfig(String regionName)
   {
      EvictionRegionConfig evRegConfig = new EvictionRegionConfig();
      evRegConfig.setRegionName(regionName);
      NullEvictionPolicyConfig nullEvictionPolicyConfig = new NullEvictionPolicyConfig();
      evRegConfig.setEvictionPolicyConfig(nullEvictionPolicyConfig);
      return evRegConfig;
   }

}
