require 'stringio'
require 'webrick/httpservlet/abstract'

# Classes under CKAdapter are interface between CGIKit and web servers.
# 
# The role of adapter is to provide the consistent interface to 
# CKApplication. The difference of the relationship between 
# CGIKit and a web server is aboserbed by the obejct. 
module CKAdapter
	class CGI
		attr_accessor :headers, :params
		attr_reader :input, :output, :error

		CR  = "\r"
		LF  = "\n"
		EOL = CR + LF

		def initialize
			@headers = ENV
			@input   = $stdin
			@output  = $stdout
			@error   = $stderr
			_initialize_query
		end

		def create_request
			CKRequest.new( headers, params )
		end

		def create_response
			CKResponse.new headers
		end

		def run( request, response, &block )
			if request then params.update request end
			ckrequest  = create_request
			ckresponse = response || create_response

			if block_given?
				block.call( ckrequest, ckresponse )
			end

			print ckresponse.to_s
		end

		private

		# Copied from cgi.rb.
		# Other interfaces should override _parse* methods.
		def _initialize_query
			if ("POST" == @headers['REQUEST_METHOD']) and
				 %r|\Amultipart/form-data.*boundary=\"?([^\";,]+)\"?|n.match(@headers['CONTENT_TYPE'])
				boundary = $1.dup
				@params = _parse_multipart(boundary, Integer(@headers['CONTENT_LENGTH']))
			else
				case @headers['REQUEST_METHOD']
				when "HEAD" then query = _parse_head
				when "GET"  then query = _parse_get
				when "POST" then query = _parse_post
				else             query = _parse_else end
				@params = CKRequest.parse_query_string query
			end
		end

		def _parse_get
			@headers['QUERY_STRING'] or ""
		end

		alias :_parse_head :_parse_get

		def _parse_post
			@input.binmode
			@input.read(Integer(@headers['CONTENT_LENGTH'])) or ''
		end

		def _parse_else
			require "shellwords"

			string = unless ARGV.empty?
				ARGV.join(' ')
			else
				if STDIN.tty?
					STDERR.print(
						%|(offline mode: enter name=value pairs on standard input)\n|
					)
				end
				readlines.join(' ').gsub(/\n/n, '')
			end.gsub(/\\=/n, '%3D').gsub(/\\&/n, '%26')

			words = Shellwords.shellwords(string)

			if words.find{|x| /=/n.match(x) }
				words.join('&')
			else
				words.join('+')
			end
		end

		def _parse_multipart( boundary, content_length )
			params   = Hash.new([])
			boundary = "--" + boundary
			buf      = ""
			bufsize  = 10 * 1024

			# start multipart/form-data
			@input.binmode
			boundary_size   = boundary.size + EOL.size
			content_length -= boundary_size
			status          = @input.read boundary_size
			if status == nil then
				raise EOFError, "no content body"
			elsif boundary + EOL != status
				raise EOFError, "bad content body"
			end

			until content_length == -1
				head = nil
				data = CKByteData.new

				until head and /#{boundary}(?:#{EOL}|--)/n.match(buf)
					if (not head) and /#{EOL}#{EOL}/n.match(buf)
						buf = buf.sub(/\A((?:.|\n)*?#{EOL})#{EOL}/n) do
						head = $1.dup
						""
					end
					next
				end

				if head and ( (EOL + boundary + EOL).size < buf.size )
					data << buf[0 ... (buf.size - (EOL + boundary + EOL).size)]
					buf[0 ... (buf.size - (EOL + boundary + EOL).size)] = ""
				end

				if bufsize < content_length then
					c = @input.read(bufsize) or ''
				else
					c = @input.read(content_length) or ''
				end
				buf += c
				content_length -= c.size
			end

			buf = buf.sub(/\A((?:.|\n)*?)(?:#{EOL})?#{boundary}(#{EOL}|--)/n) do
				data << $1
				if "--" == $2
					content_length = -1
					end
					""
				end

				/Content-Disposition:.* filename="?([^\";]*)"?/ni.match(head)
				filename = ($1 || "").dup
				if /Mac/ni.match(@headers['HTTP_USER_AGENT']) and
					/Mozilla/ni.match(@headers['HTTP_USER_AGENT']) and
					(not /MSIE/ni.match(@headers['HTTP_USER_AGENT']))
					filename = CKUtilities.unescape_url filename
				end
				data.path = filename

				/Content-Type: ([^\r\n]*)/ni.match(head)
				if $1 then
					data.content_type = $1.dup
				else
					data = data.to_s
				end

				/Content-Disposition:.* name="?([^\";]*)"?/ni.match(head)
				name = $1.dup

				if params.has_key? name then
					params[name].push data
				else
					params[name] = [data]
				end
			end

			params
		end
	end


	class ModRuby < CGI
		def run( request, response, &block )
			if request then params.update request end
			ckrequest  = create_request
			ckresponse = response || create_response

			if block_given?
				block.call( ckrequest, ckresponse )
			end

			print ckresponse.status_line
			print ckresponse.to_s
		end
	end


	class Template < CGI
		def initialize
			@headers = ENV
			@input   = StringIO.new
			@output  = StringIO.new
			@error   = StringIO.new
		end

		def run( request, response, &block )
			ckrequest  = request || create_request
			ckresponse = response || create_response

			if block_given?
				block.call( ckrequest, ckresponse )
			end

			ckresponse
		end
	end


	# The adapter and WEBrick::CGIKitServlet are for WEBrick
	# writtend by Ken-ichi HASHIMOTO and Shinya Kawaji. Thanks!
	class WEBrick
		GENERAL_HEADER = %w[
			Cache-Control
			Connection
			Date
			Pragma
			Trailer
			Transfet-Encoding
			Upgrade
			Via
			Warning
		]

		RESPONSE_HEADER = %w[
			Accept-Ranges
			Age
			ETag
			Location
			Proxy-Authenticate
			Retry-After
			Server
			Vary
			WWW-Authenticate
		]

		ENTITY_HEADER = %w[
			Allow
			Content-Encoding
			Content-Language
			Content-Length
			Content-Location
			Content-MD5
			Content-Range
			Content-Disposition
			Expires
			Last-Modified
		]

		LEAVE_HEADER = {}
		(GENERAL_HEADER + RESPONSE_HEADER + ENTITY_HEADER).each{|key|
			LEAVE_HEADER[key] = true
		}

		def run(request, response)
			# set request
			headers = request.meta_vars
			form_values = Hash.new([])
			request.query.each{|key, val|
				form_values[key] = val.list
			}

			ckrequest  = CKRequest.new(headers, form_values)
			ckresponse = CKResponse.new(headers)

			# run
			if block_given?
				yield(ckrequest, ckresponse)
			end

			# set response
			response.request_http_version = ckresponse.http_version
			response.status = ckresponse.status
			ckresponse.headers.each{|key, val|
				if LEAVE_HEADER[key]
					response[key] = val
				end
			}
			response['Content-Type'] = ckresponse.headers['Content-Type']
			if ckresponse.encoding
				response['Content-Type'] << "; charset=#{ckresponse.encoding}"
			end
			ckresponse.cookies.each{|cookie|
				response.cookies << cookie
			}
			response.body = ckresponse.content
		end
	end

end


module WEBrick
	module CGIKitServlet
		class ApplicationHandler < HTTPServlet::AbstractServlet
			def initialize(server, app)
				@app = app
				@app.interface = CKAdapter::WEBrick
			end

			def do_GET(req, res)
				@app.run(req, res)
			end
			alias_method :do_POST, :do_GET
		end

		class PathHandler < ApplicationHandler
			def initialize(server, component_path)
				app = CKApplication.new
				app.component_path = component_path
				super(server, app)
			end
		end

		class HashHandler < ApplicationHandler
			def initialize(server, hash)
				app = CKApplication.new
				hash.each{|key, val|
					app.__send__("#{key}=", val)
				}
				super(server, app)
			end
		end
	end
end


