;;; cgikit.el --- supports CGIKit framework

;; $Id: cgikit.el,v 1.4 2004/04/15 15:24:56 john Exp $

;; Copyright (C) 2003  Free Software Foundation, Inc.

;; Author: rubikitch <rubikitch@ruby-lang.org>
;; Keywords: languages, convenience

;; This file is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This file is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; Supports Ruby-CGI programming with CGIKit.

;;; Bugs:
;; search bug

;;; Code:


;;;; customize
(defvar cgikit-find-files-function 'cgikit-find-files-1)

(defvar cgikit-embedded-ckd nil)

(defvar cgikit-buffer-name-format "[%s]%s")

;;;; templates
(defun cgikit-html-template (component-name)
  "Template of *.html"
  '("" . ""))

(defun cgikit-rb-template (component-name)
  "Template of *.rb"
  `(,(format "class %s < CKComponent
" component-name)
    .
    "

  def init
  end
end
"))
  
(defun cgikit-ckd-template (component-name)
  "Template of *.ckd"
  '("" . ""))

(defvar cgikit-main-template
  "Template of main program."
  "#!/usr/local/bin/ruby

require 'cgikit'

app = CKApplication.new
app.run
")


;; filename
(defun cgikit-html-file-name (dir)
  (concat dir (cgikit-component-name dir) ".html"))
(defun cgikit-rb-file-name (dir)
  (concat dir (cgikit-component-name dir) ".rb"))
(defun cgikit-ckd-file-name (dir)
  (concat dir (cgikit-component-name dir) ".ckd"))


;; component name
;; (cgikit-component-name "/a/examples/Examples/ConditionalPage")
;; (cgikit-component-name "/a/examples/Examples/ConditionalPage/")
;; (cgikit-component-name "~/src/cgikit/examples/Examples/ConditionalPage/ConditionalPage.html")

(defun cgikit-component-name (file-or-dir)
  (setq dir (if (file-regular-p file-or-dir)
                (file-name-directory file-or-dir)
              file-or-dir))
  (setq dir (if (string= (substring dir -1) "/")
                (substring dir 0 -1)
              dir))
  (file-name-nondirectory dir))



;; read from minibuffer
;; (cgikit-read-directory "Test: ")
(defun cgikit-read-directory (prompt)
  (let ((dir (read-file-name prompt nil default-directory nil)))
    (if (string= (substring dir -1) "/")
        dir
      (concat dir "/"))))

;; (cgikit-read-component-name "dir: ")
(defun cgikit-read-component-name (prompt)
  (let ((dirs (delq nil (mapcar (lambda (fn)
                                  (and (not (member fn '("." "..")))
                                       (file-directory-p fn)
                                       (list (file-name-nondirectory fn))))
                                (directory-files ".." t))))
        (completion-ignore-case t))
    (completing-read prompt dirs)))

;; cgikit-find-directory
;; dir MUST end with '/'
(defun cgikit-find-directory (dir)
  (interactive (list (cgikit-read-directory "CGIKit component directory: ")))
  (setq dir (expand-file-name dir))
  (cond ((file-directory-p dir)
         (cgikit-find-files dir))
        ((file-regular-p dir)
         (error "must be directory"))
        (t
         (make-directory dir)
         (cgikit-find-files dir))))

;; cgikit-find-other-component
(defun cgikit-find-other-component (component)
  (interactive (list (cgikit-read-component-name "Other component: ")))
  (cgikit-find-directory
   (concat (file-name-directory (substring default-directory 0 -1)) component "/")))

;; find-file
(defun cgikit-find-files (dir)
  (cgikit-make-help-table)
  (funcall cgikit-find-files-function dir))


;; +--------------------+
;; |    template        |
;; |                    |
;; +--------------------+
;; | code       | bind  |
;; |            |       |
;; +--------------------+
(defun cgikit-find-files-1 (dir)
  (delete-other-windows)
  (split-window)
  (cgikit-find-html-file dir)
  (other-window 1)
  (cgikit-find-rb-file dir)
  (unless (cgikit-embedded-ckd-p)
    (split-window-horizontally (- (frame-width) 40))
    (other-window 1)
    (cgikit-find-ckd-file dir))
  (other-window 1)
  )


(defun cgikit-find-file (file template-func)
  (unless (file-exists-p file)
    (let ((buf (find-file-noselect file))
          (template (funcall template-func (cgikit-component-name (file-name-directory file)))))
      (set-buffer buf)
      (insert (car template))
      (save-excursion
        (insert (cdr template)))))
  (find-file file)
  (rename-buffer (cgikit-buffer-name (buffer-file-name))))


; (cgikit-buffer-name "~/public_html/fm/MainPage/MainPage.html")
; (cgikit-buffer-name "~/public_html/fm/Components/MainPage/MainPage.html")
; (cgikit-buffer-name "~/dat/fm/cgikit/components/MainPage/MainPage.html")

(defun cgikit-buffer-name (filename)
  (let ((component (cgikit-component-name (file-name-directory filename)))
        (app (cgikit-application-name (file-name-directory filename))))
    (format cgikit-buffer-name-format app  (file-name-nondirectory filename))))

(defun cgikit-application-name (dir)
  (catch 'exit
    (dolist (x (cdr (reverse (split-string dir "/"))))
      (unless (member x '("elements" "components" "Components" "cgikit"))
        (throw 'exit x)))))

(defun cgikit-find-html-file (dir)
  (cgikit-find-file (cgikit-html-file-name dir) 'cgikit-html-template)
  (cgikit-html-mode 1))
(defun cgikit-find-rb-file (dir)
  (cgikit-find-file (cgikit-rb-file-name dir) 'cgikit-rb-template)
  (cgikit-rb-mode 1))
(defun cgikit-find-ckd-file (dir)
  (cgikit-find-file (cgikit-ckd-file-name dir) 'cgikit-ckd-template)
  (cgikit-ckd-mode 1))


;; switch to window
(defun cgikit-switch-to-html-window ()
  (interactive)
  (select-window (cgikit-find-window-from-filename "\.html$")))
(defun cgikit-switch-to-rb-window ()
  (interactive)
  (select-window (cgikit-find-window-from-filename "\.rb$")))
(defun cgikit-switch-to-ckd-window ()
  (interactive)
  (select-window (cgikit-find-window-from-filename "\.ckd$")))



;; cgikit-html minor mode
(easy-mmode-define-minor-mode cgikit-html-mode
 "Minor mode to edit CGIKit template."
 nil
 " CK"
 '(("\M-e" . cgikit-html-goto-binding)
   ("\M-p" . cgikit-html-previous-cgikit-tag)
   ("\M-n" . cgikit-html-next-cgikit-tag)
   ("\C-c\C-k" . cgikit-html-insert-cgikit-tag)
   ("\C-c\C-f" . cgikit-find-other-component)
   ("\C-c\C-d" . cgikit-find-directory)
   ))

(defun cgikit-html-insert-cgikit-tag (element-name)
  (interactive "sElement Name: ")
  (if (string-match "/$" element-name)
      (insert "<cgikit name=" element-name ">")
    (insert "<cgikit name=" element-name ">\n")
    (save-excursion
      (insert "</cgikit><!-- /" element-name " -->"))))

(defun cgikit-find-window-from-filename (pattern)
  (let (x buf)
    (save-excursion
      (walk-windows (lambda (win)
                      (setq buf (window-buffer win))
                      (set-buffer buf)
                      (when (string-match pattern (buffer-file-name))
                        (setq x win)))))
    x))

(defun cgikit-html-get-element-name-at-point ()
  (save-excursion
    (let ((case-fold-search t)
          (p (point))
          b e element-name)
      (when (and (or (eq (char-before) ?>) (search-forward ">" nil t))
                 (re-search-backward "<cgikit +name=[\'\"]?\\([^\'\"/ ]+\\)[ \'\"/]*?>" nil t))
        (setq b (match-beginning 0)
              e (match-end 0)
              element-name (match-string 1))
        (and (<= b p) (<= p e) element-name)))))
    
    

(defun cgikit-embedded-ckd-p ()
  (save-window-excursion
    (cgikit-switch-to-rb-window)
    cgikit-embedded-ckd))

(defun cgikit-html-goto-binding (element-name)
  "Go to the binding of this CGIKIT tag."
  (interactive
   (list (cgikit-html-get-element-name-at-point)))
  (unless (null element-name)
    (cond ((cgikit-embedded-ckd-p)
           (cgikit-switch-to-rb-window)
           (goto-char (point-min))
           (unless (re-search-forward (concat "^=begin ckd\n" element-name " ") nil t)
             (cgikit-rb-make-new-binding element-name)))
          (t
           (cgikit-switch-to-ckd-window)
           (goto-char (point-min))
           (unless (re-search-forward (concat "^" element-name " ") nil t)
             (cgikit-ckd-make-new-binding element-name))))
    (recenter 0)))

(defun cgikit-html-search-cgikit-tag (search-func)
  (let ((case-fold-search t))
    (funcall search-func "<cgikit +name=" nil t)))

(defun cgikit-html-next-cgikit-tag ()
  "Go to the next CGIKIT tag."
  (interactive)
  (cgikit-html-search-cgikit-tag 're-search-forward))

(defun cgikit-html-previous-cgikit-tag ()
  "Go to the previous CGIKIT tag."
  (interactive)
  (if (save-excursion
        (forward-char -1)
        (cgikit-html-search-cgikit-tag 're-search-backward))
      (goto-char (match-end 0))))


;; cgikit-rb minor mode
(easy-mmode-define-minor-mode cgikit-rb-mode
 "Minor mode to edit CGIKit rb file."
 nil
 " CK"
 '(("\M-e" . cgikit-ckd-goto-tag)
   ("\M-p" . cgikit-ckd-previous-binding)
   ("\M-n" . cgikit-ckd-next-binding)
   ("\M-c" . cgikit-ckd-help-at-point)
   ("\M-h" . cgikit-help-display)
   ("\C-c\C-f" . cgikit-find-other-component)
   ("\C-c\C-d" . cgikit-find-directory)
   ))

(defun cgikit-rb-generate-ckd ()
  (when (and cgikit-rb-mode (cgikit-embedded-ckd-p))
    (save-excursion
    (let ((rbfile (file-name-nondirectory buffer-file-name))
          (ckdfile (concat (substring buffer-file-name 0 -3) ".ckd"))
          (content (with-output-to-string
                     (let (s e)
                       (goto-char (point-min))
                       (while (re-search-forward "^=begin +ckd\n" nil t)
                         (setq s (match-end 0))
                         (re-search-forward "^=end\n" nil t)
                         (setq e (match-beginning 0))
                         (princ (buffer-substring-no-properties s e))
                         (princ "\n"))))))
      (set-buffer (find-file-noselect ckdfile))
      (erase-buffer)
      (insert
       "# This file is auto-generated by " rbfile "! DON'T EDIT!!!\n"
       content "\n")
      (save-buffer)
      (kill-buffer (current-buffer))))))
          
(add-hook 'after-save-hook 'cgikit-rb-generate-ckd)
                               
                       
  
;; cgikit-ckd minor mode
(easy-mmode-define-minor-mode cgikit-ckd-mode
 "Minor mode to edit CGIKit binding."
 nil
 " CK"
 '(("\M-e" . cgikit-ckd-goto-tag)
   ("\M-p" . cgikit-ckd-previous-binding)
   ("\M-n" . cgikit-ckd-next-binding)
   ("\M-c" . cgikit-ckd-help-at-point)
   ("\M-h" . cgikit-help-display)
   ("\C-c\C-f" . cgikit-find-other-component)
   ("\C-c\C-d" . cgikit-find-directory)
   ))

(setq cgikit-ckd-binding-regexp "^\\(.+\\) +: +\\([A-Za-z]+\\).*{")

(defun cgikit-ckd-get-binding-at-point ()
  "Get '(element-name . kind-of-element) at point."
  (save-excursion
    (end-of-line)
    (and (re-search-backward cgikit-ckd-binding-regexp nil t)
         (cons (match-string 1) (match-string 2)))))
  
(defun cgikit-ckd-goto-tag (element-name)
  (interactive
   (list (car (cgikit-ckd-get-binding-at-point))))
  (when element-name
    (cgikit-switch-to-html-window)
    (goto-char (point-min))
    (let ((case-fold-search t))
      (re-search-forward (concat "<cgikit +name=[\'\"]?" element-name "[ \'\"/]*>")))))
   
(defun cgikit-ckd-search-binding (search-func)
  (funcall search-func cgikit-ckd-binding-regexp nil t))

(defun cgikit-ckd-next-binding ()
  "Go to the next CGIKIT tag."
  (interactive)
  (if (save-excursion
        (forward-char 1)
        (cgikit-ckd-search-binding 're-search-forward))
      (goto-char (match-beginning 0))))

(defun cgikit-ckd-previous-binding ()
  "Go to the previous CGIKIT tag."
  (interactive)
  (cgikit-ckd-search-binding 're-search-backward))
 
(defun cgikit-ckd-make-new-binding (element-name)
  (interactive "sElement: ")
  (goto-char (point-max))
  (unless (eq (char-before) ?\n)
    (insert "\n"))
  (insert "\n" element-name " : CK")
  (save-excursion
    (insert " {\n\n}\n")))

(defun cgikit-rb-make-new-binding (element-name)
  (interactive "sElement: ")
  (when (and (re-search-forward "^class .+ CKComponent" nil t)
             (re-search-forward "^end" nil t))
    (goto-char (match-beginning 0))
    (insert "\n=begin ckd\n" element-name " : CK")
    (save-excursion
      (insert " {\n\n}\n=end\n  def " element-name "\n\n  end\n\n"))))


;; cgikit-insert-main-template
(defun cgikit-insert-main-template ()
  (interactive)
  (insert cgikit-main-template))
(provide 'cgikit)


;; help table
(setq cgikit-help-buffer " *CGIKit Help*")

(defun cgikit-make-help-table ()
  (unless (get-buffer cgikit-help-buffer)
    (set-buffer (get-buffer-create cgikit-help-buffer))
    (insert "
CKBrowser : escape list selected values multiple size enabled
CKCheckbox : checked value selection enabled
CKComponent : 
CKConditional : condition negate
CKContent : 
CKFileUpload : data file enabled
CKForm : action method enctype href target query fileupload
CKFrame : name page src value
CKGenericElement : tag enabled string option form_value form_values invoke_action
CKHyperlink : action enabled href page string target secure query
CKImage : alt border width height file src data mime
CKPopUpButton : escape list default selected values enabled
CKRadioButton : name checked value selection enabled
CKRepetition : count list item index
CKResetButton : value
CKString : value escape empty
CKSubmitButton : action value enabled
CKText : value columns rows enabled
CKTextField : type value size maxlength enabled
")))

(defun cgikit-momentary-help-display (buf)
  (save-window-excursion
    (when (minibuffer-window-active-p (selected-window))
      (other-window 1))
    (switch-to-buffer " *CGIKit Element help*" t)
    (delete-other-windows)
    (momentary-string-display (with-current-buffer buf (buffer-string))
                              (window-start))
    (kill-buffer " *CGIKit Element help*")))

(defun cgikit-help-display (element-name)
  (interactive (list (cdr (cgikit-ckd-get-binding-at-point))))
  (with-temp-buffer
    (insert (cdr (assoc element-name cgikit-help-alist)))
    (cgikit-momentary-help-display (current-buffer))))


(defun cgikit-ckd-help-at-point ()
  (interactive)
  (let* ((kind-of-element (cdr (cgikit-ckd-get-binding-at-point))))
    (save-excursion
      (cgikit-make-help-table)
      (set-buffer cgikit-help-buffer)
      (goto-char (point-min))
      (unless (search-forward kind-of-element nil t)
        (error "%s:no such kind of element." kind-of-element))
      (message (buffer-substring (point-at-bol) (point-at-eol))))))

;; invade globals
(defun cgikit-invade-global-namespace ()
  "Define aliases about cgikit.el"
  (defalias 'ckf 'cgikit-find-directory)
  (defalias 'ckc 'cgikit-find-other-component)
  )




;;; cgikit.el ends here
