import sys, os, xmlrpclib, Queue, threading, getopt, logging

import gobject

import hotwire.command
from hotwire.async import QueueIterator, CancellableQueueIterator, MiniThreadPool
from hotwire.minion import MinionEvent
import hotwire.logutil
import hotwire.util

class MinionHotwireSubcontext(object):
    """A Hotwire context which also holds a channel id; essentially
    lets us push events generated by pipeline commands back into the
    event queue for the master."""
    def __init__(self, chanid, event_queue, mainctx):
        self._mainctx = mainctx
        self.__chanid = chanid
        self.__eventqueue = event_queue
        super(MinionHotwireSubcontext, self).__init__()

    def chdir(self, dir):
        d = self._mainctx.chdir(dir)
        self.__eventqueue.put(MinionEvent(-1, 'Cwd', d))
        return d

    def start_download(self, arg):
        self.__eventqueue.put(MinionEvent(-1, 'StartDownload', str(arg)))

    def open_term(self, cmd):
        self.__eventqueue.put(MinionEvent(-1, 'ExecTerminal', cmd, self.get_cwd()))

    def __getattribute__(self, name):
        try:
            return object.__getattribute__(self, name)
        except AttributeError:
            mainctx = object.__getattribute__(self, '_mainctx')
            return object.__getattribute__(mainctx, name)

class MinionImpl(object):
    def __init__(self):
        super(MinionImpl, self).__init__()
        self.__pipeline_counter = 0
        self.__active_pipelines = {}
        self.__eventqueue = Queue.Queue()
        self.__eventqueue_iter = QueueIterator(self.__eventqueue)
        self.__maincontext = hotwire.command.HotwireContext()
        # initial Cwd notification
        self.__eventqueue.put(MinionEvent(-1, 'Cwd', self.__maincontext.get_cwd()))

    def create_pipeline(self, chanid, pipeline_str):
        context = MinionHotwireSubcontext(chanid, self.__eventqueue, self.__maincontext) 
        pipeline = hotwire.command.Pipeline.parse(pipeline_str, context=context)
        pipeline.connect("exception", lambda p,cmd,e: self.__eventqueue.put(MinionEvent(chanid, 'PipelineException', e))) 
        self.__pipeline_counter += 1
        self.__active_pipelines[self.__pipeline_counter] = pipeline
        pipeline.set_output_queue(self.__eventqueue, lambda obj: self.__map_pipeline_event(self.__pipeline_counter, chanid, obj))
        pipeline.execute()
        self.__eventqueue.put(MinionEvent(chanid, 'PipelineCreated', self.__pipeline_counter))

    def __map_pipeline_event(self, pipeline_id, chanid, obj):
        if obj:
            return MinionEvent(chanid, 'PipelineObject', pipeline_id, obj.__class__.__name__, str(obj))
        else:
            del self.__active_pipelines[pipeline_id]
            return MinionEvent(chanid, 'PipelineComplete', pipeline_id)

    def close(self):
        self.__eventqueue_iter.cancel()

    def get_events(self):
        return self.__eventqueue_iter 

# We have this hack right now because for some reason sessions aren't
# being closed when the master quits
noinput_exit_timer = 0
def noinput_exit_timer_reset(loop):
    global noinput_exit_timer
    if noinput_exit_timer:
        gobject.source_remove(noinput_exit_timer)
    noinput_exit_timer = gobject.timeout_add(60 * 1000, lambda: loop.quit())

def main():
    gobject.threads_init()
    loop = gobject.MainLoop()

    try:
        opts, args = getopt.getopt(sys.argv[1:], "d", ["debug", "thread-debug"])
    except getopt.GetoptError:
        sys.exit(2)
    debug = False
    for o, v in opts:
        if o in ('-d', '--debug'):
            debug = True
        elif o in ('--thread-debug',):
            hotwire.util.start_thread_dump_task(7000, sys.stderr)

    default_log_level = logging.ERROR
    if debug:
        default_log_level = logging.DEBUG
    hotwire.logutil.init(default_log_level, [], 'hotwire.')

    minime = MinionImpl()
    def event_thread_main():
        for event in minime.get_events():
            sys.stderr.write('writing event %s\n' % (event,))
            result_xml = xmlrpclib.dumps((event.chanid, event.name, event.args))
            sys.stdout.write('%d\n' % (len(result_xml),))
            sys.stdout.write(result_xml)
            sys.stdout.flush()
    MiniThreadPool.getInstance().run(event_thread_main)

    noinput_exit_timer_reset(loop)

    def stdin_reader():
        try:
            while True:
                reqmeta = sys.stdin.readline()
                reqlen = int(reqmeta)
                request = sys.stdin.read(reqlen)
                noinput_exit_timer_reset(loop)
                (params, meth) = xmlrpclib.loads(request)
                result = None
                if hasattr(minime, meth):
                    gobject.idle_add(lambda meth, params: getattr(minime, meth)(*params),
                                     meth, params)
        finally:
            loop.quit()
    
    MiniThreadPool.getInstance().run(stdin_reader)
        
    sys.stderr.write("running loop\n")
    #hotwire.util.start_thread_dump_task(5000, sys.stderr)
    loop.run()
    sys.stderr.write("loop quit\n")
    os._exit(0)

if __name__ == '__main__':
    main()
