/***************************************************************************
 *   Copyright (C) 2005 by Adam Treat                                      *
 *   treat@kde.org                                                         *
 *                                                                         *
 *   Copyright (C) 2004 by Scott Wheeler                                   *
 *   wheeler@kde.org                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <kaction.h>
#include <kdebug.h>

#include <qdialog.h>
#include <qlayout.h>

#include "datatablesplitter.h"
#include "datatable.h"
#include "advancedsearchdialog.h"
#include "searchwidget.h"
#include "datatablesearch.h"
#include "actioncollection.h"

using namespace ActionCollection;

DataTableSplitter::DataTableSplitter( QWidget *parent, const char *name ) :
        QSplitter( Qt::Horizontal, parent, name ),
        m_dataTableBox( 0 ),
        m_searchWidget( 0 ),
        m_dataTableStack( 0 )
{
    setupActions();
    setupLayout();
    readConfig();
}

DataTableSplitter::~DataTableSplitter()
{
    saveConfig();

    delete m_dataTableBox;
}

void DataTableSplitter::setupActions()
{}

void DataTableSplitter::setupLayout()
{
    setOpaqueResize( false );

    // Create a splitter to go between the dataTables and the editor.

    QSplitter *editorSplitter = new QSplitter( Qt::Vertical, this, "editorSplitter" );

    // Create the dataTable and the editor.

    QWidget *top = new QWidget( editorSplitter );
    QVBoxLayout *topLayout = new QVBoxLayout( top );

    m_dataTableStack = new QWidgetStack( top, "dataTableStack" );
    m_dataTableStack->addWidget( new QWidget() );

    // Create the DataTableBox

    m_dataTableBox = new DataTableBox( this, m_dataTableStack, "dataTableBox" );
    connect( m_dataTableBox, SIGNAL( selectionChanged() ),
             this, SLOT( slotDataTableChanged() ) );
    connect( m_dataTableBox, SIGNAL( advancedSearchClicked() ),
             this, SLOT( slotAdvancedSearchClicked() ) );

    moveToFirst( m_dataTableBox );

    m_searchWidget = new SearchWidget( top, "searchWidget" );

    //UGLY ASS TODO.  The Search Widget should be a child of the dataTable
    connect( m_dataTableBox, SIGNAL( searchRemoved() ),
             this, SLOT( slotSearchRemoved() ) );
    connect( m_dataTableBox, SIGNAL( searchEdited() ),
             this, SLOT( slotSearchEdited() ) );
    connect( m_dataTableBox, SIGNAL( searchInvoked( const QString & ) ),
             m_searchWidget, SLOT( invokeSearch( const QString & ) ) );

    connect( m_searchWidget, SIGNAL( signalQueryChanged() ),
             this, SLOT( slotShowSearchResults() ) );
    connect( m_searchWidget, SIGNAL( signalClearAdvancedQuery() ),
             this, SLOT( slotClearAdvancedSearchResults() ) );
    connect( m_searchWidget, SIGNAL( signalSearchListChanged( const DataTableSearch & ) ),
             this, SLOT( slotAdvancedSearchChanged( const DataTableSearch & ) ) );

    topLayout->addWidget( m_searchWidget );
    topLayout->addWidget( m_dataTableStack );
}

void DataTableSplitter::readConfig()
{
    KConfigGroup config( KGlobal::config(), "Splitter" );

    QValueList<int> splitterSizes = config.readIntListEntry( "DataTableSplitterSizes" );
    if ( splitterSizes.isEmpty() )
    {
        splitterSizes.append( 100 );
        splitterSizes.append( 640 );
    }
    setSizes( splitterSizes );

}

void DataTableSplitter::saveConfig()
{
    KConfigGroup config( KGlobal::config(), "Splitter" );
    config.writeEntry( "DataTableSplitterSizes", sizes() );
}

void DataTableSplitter::slotShowSearchResults()
{
    DataTableList dataTables;
    dataTables.append( visibleDataTable() );
    DataTableSearch search = m_searchWidget->createSearch( dataTables );
    visibleDataTable() ->setSearch( search );
}

void DataTableSplitter::slotClearAdvancedSearchResults()
{
    DataTableSearch search;
    DataTableList dataTables = m_dataTableBox->dataTablesInCurrentTree();

    DataTableList::Iterator it = dataTables.begin();
    for ( ; it != dataTables.end(); ++it )
    {
        ( *it ) ->setSearchFilter( QString::null );
        search.execQuery( ( *it ) , ( *it ) ->defaultFilter() );
        ( *it )->setSearch( search );
    }

    m_searchWidget->setSearch( search );

    slotDataTableChanged(); //update
}

void DataTableSplitter::slotAdvancedSearchClicked()
{
    AdvancedSearchDialog::Result r =
        AdvancedSearchDialog( m_dataTableBox->dataTablesInCurrentTree() ).exec();

    if ( r.result == QDialog::Rejected )
        return;

    if ( !r.dataTableName.isEmpty() )
        m_dataTableBox ->appendSearch( r.search );

    slotAdvancedSearchChanged( r.search );
}

void DataTableSplitter::slotSearchEdited()
{
    AdvancedSearchDialog::Result r =
        AdvancedSearchDialog( m_dataTableBox->dataTablesInCurrentTree(),
                              m_dataTableBox->rootOfCurrentDataTable()->search() ).exec();

    if ( r.result == QDialog::Rejected )
        return;

    if ( !r.dataTableName.isEmpty() )
        m_dataTableBox ->appendSearch( r.search );

    slotAdvancedSearchChanged( r.search );
}

void DataTableSplitter::slotSearchRemoved()
{
    m_searchWidget->searchRemoved();
    slotDataTableChanged(); //update
}

void DataTableSplitter::slotAdvancedSearchChanged( const DataTableSearch &search )
{
    DataTable *dataTable = m_dataTableBox->dataTableByName( search.parentName() );

    if ( !dataTable )
        return;

    dataTable ->setSearch( search );
    m_searchWidget->setSearch( search );

    slotDataTableChanged(); //update
}

void DataTableSplitter::slotDataTableChanged()
{
    if ( m_dataTableBox->hasSelection() )
        m_searchWidget->setSearchList( m_dataTableBox ->searchList() );

    if ( !visibleDataTable() || !m_dataTableBox->hasSelection() )
    {
        m_searchWidget->setEnabled( false );
        return ;
    }

    m_searchWidget->setEnabled( true );
    m_searchWidget->setSearchList( m_dataTableBox ->searchList() );

    bool simple = m_dataTableBox->rootOfCurrentDataTable()->search().searchLevel() == DataTableSearch::Simple;
    action( "edit_search" ) ->setEnabled( !simple );
    action( "remove_search" ) ->setEnabled( !simple );
    m_searchWidget->setLineEnabled( simple );

    if ( simple )
        m_searchWidget->setSearch( visibleDataTable() ->search() );
    else
        m_searchWidget->setSearch( m_dataTableBox->rootOfCurrentDataTable() ->search() );
}

#include "datatablesplitter.moc"
