/***************************************************************************
 *   Copyright (C) 2005 by Adam Treat                                      *
 *   treat@kde.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "datafield.h"

#include <qregexp.h>
#include <qsqlrecord.h>
#include <qdatetime.h>

#include <kdebug.h>

DataRelation::~DataRelation()
{
    DataFieldList::iterator it = m_fieldList.begin();
    for ( ; it != m_fieldList.end(); ++it )
        delete ( *it );
}

QString DataRelation::table() const
{
    return m_table;
}

void DataRelation::setTable( const QString &table )
{
    m_table = table;
    DataFieldList::iterator it;
    for ( it = m_fieldList.begin(); it != m_fieldList.end(); ++it )
        ( *it )->setTable( table );
}

QString DataRelation::key() const
{
    return m_key;
}

void DataRelation::setKey( const QString &key )
{
    m_key = key;
}

QString DataRelation::fullKey() const
{
    return m_table + "." + m_key;
}

QString DataRelation::field()
{
    return getPrimaryField()->name();
}

void DataRelation::setField( const QString &field )
{
    getPrimaryField()->setName( field );
}

QString DataRelation::fullField()
{
    return getPrimaryField()->fullName();
}

QString DataRelation::constraint()
{
    return m_constraint;
}

void DataRelation::setConstraint( const QString &constraint )
{
    m_constraint = constraint;
}

QVariant::Type DataRelation::type()
{
    return getPrimaryField()->type();
}

void DataRelation::setType( const QVariant::Type type )
{
    getPrimaryField()->setType( type );
}

int DataRelation::number()
{
    return getPrimaryField()->number();
}

void DataRelation::setNumber( int num )
{
    getPrimaryField()->setNumber( num );
}

int DataRelation::editorWidth()
{
    return getPrimaryField()->editorWidth();
}

void DataRelation::setEditorWidth( int num )
{
    getPrimaryField()->setEditorWidth( num );
}

DataRelation* DataRelation::dataRelation( const QString &key )
{
    DataFieldList::iterator it;
    for ( it = m_fieldList.begin(); it != m_fieldList.end(); ++it )
        if ( ( *it ) ->name() == key )
            return ( *it ) ->relation();

    return 0;
}

void DataRelation::addDataRelation( const QString &name, const QString &targetTable,
                                    const QString &targetKey, const QString &targetField,
                                    const QString &targetConstraint,
                                    const QVariant::Type targetType, int number,
                                    int editorWidth )
{
    DataField * f = dataField( name );
    DataRelation *r = new DataRelation();
    r->setTable( targetTable );
    r->setKey( targetKey );
    r->setConstraint( targetConstraint );
    r->addDataField( targetField, targetType, number, editorWidth, false, true, false, false );

    f->setRelation( r );
}

DataFieldList DataRelation::fieldList()
{
    return m_fieldList;
}

DataField* DataRelation::dataField( const QString &key )
{
    DataFieldList::iterator it;
    for ( it = m_fieldList.begin(); it != m_fieldList.end(); ++it )
        if ( ( *it ) ->name() == key )
            return ( *it );

    return 0;
}

void DataRelation::addDataField( const QString &name, const QVariant::Type type,
                                 int number, int editorWidth, bool foreign, bool primary,
                                 bool reporter, bool hidden )
{
    DataField * f = new DataField();
    f->setName( name );
    f->setTable( table() );
    f->setType( type );
    f->setNumber( number == -1 ? m_fieldList.count() : number );
    f->setEditorWidth( editorWidth );
    f->setForeign( foreign );
    f->setPrimary( primary );
    f->setReporter( reporter );
    f->setHidden( hidden, false );
    m_fieldList.append( f );
}

void DataRelation::removeDataField( DataField *field )
{
    m_fieldList.remove( field );
    delete field;
    field = 0;
}

DataField* DataRelation::getReportField()
{
    for ( DataFieldList::Iterator it = m_fieldList.begin(); it != m_fieldList.end(); ++it )
    {
        if ( ( *it )->reporter() )
            return ( *it );
    }
    return getPrimaryField();
}

DataField* DataRelation::getPrimaryField()
{
    for ( DataFieldList::Iterator it = m_fieldList.begin(); it != m_fieldList.end(); ++it )
    {
        if ( ( *it )->primary() )
            return ( *it );
    }
    return 0;
}

QString DataField::name() const
{
    return m_name;
}

void DataField::setName( const QString &name )
{
    m_name = name;
}

QString DataField::fullName() const
{
    return m_table + "." + m_name;
}

QString DataField::table() const
{
    return m_table;
}

void DataField::setTable( const QString &table )
{
    m_table = table;
}

QVariant::Type DataField::type() const
{
    //     if ( relation() )
    //         return relation()->type();
    //     else
    return m_type;
}

void DataField::setType( const QVariant::Type type )
{
    m_type = type;
}

int DataField::isRequired() const
{
    return m_required;
}

void DataField::setRequired( int isRequired )
{
    m_required = isRequired;
}

bool DataField::calculated() const
{
    return m_calculated;
}

void DataField::setCalculated( bool calculated )
{
    m_calculated = calculated;
}

bool DataField::isVirtual() const
{
    return m_virtual;
}

void DataField::setVirtual( bool isVirtual )
{
    m_virtual = isVirtual;
}

QString DataField::equation() const
{
    return m_equation;
}

void DataField::setEquation( const QString &equation )
{
    m_equation = equation;
}

QString DataField::label() const
{
    if ( !m_label.isEmpty() )
        return m_label;
    else
        return m_name;
}

void DataField::setLabel( const QString &label )
{
    m_label = label;
}

int DataField::number() const
{
    return m_number;
}

void DataField::setNumber( int number )
{
    m_number = number;
}

int DataField::width() const
{
    return m_width;
}

void DataField::setWidth( int width )
{
    m_width = width;
}

int DataField::editorWidth() const
{
    return m_editorWidth;
}

void DataField::setEditorWidth( int width )
{
    m_editorWidth = width;
}

bool DataField::foreign() const
{
    return m_foreign;
}

void DataField::setForeign( bool foreign )
{
    m_foreign = foreign;

    if ( !m_foreign && m_relation )
        m_relation = 0;
}

bool DataField::primary() const
{
    return m_primary;
}

void DataField::setPrimary( bool primary )
{
    m_primary = primary;
}

bool DataField::reporter() const
{
    return m_reporter;
}

void DataField::setReporter( bool reporter )
{
    m_reporter = reporter;
}

bool DataField::hidden() const
{
    return m_hidden;
}

void DataField::setHidden( bool hidden, bool invalidateNumber )
{
    m_hidden = hidden;

    if ( m_hidden && invalidateNumber )
        m_number = -1;
}

bool DataField::resizable() const
{
    return m_resizable;
}

void DataField::setResizable( bool resizable )
{
    m_resizable = resizable;
}

bool DataField::sortable() const
{
    return m_sortable;
}

void DataField::setSortable( bool sortable )
{
    m_sortable = sortable;
}

DataRelation* DataField::relation() const
{
    return m_relation;
}

void DataField::setRelation( DataRelation* relation )
{
    m_relation = relation;
}

static QStringList listFromRx( const QString str, QRegExp rx, int cap )
{
    QStringList list;
    int pos = 0;
    while ( pos >= 0 )
    {
        pos = rx.search( str, pos );
        if ( pos >= 0 )
        {
            list.append( rx.cap( cap ) );
            pos += rx.matchedLength();
        }
    }
    return list;
}

QVariant DataField::calculateField( DataRecordList currentRecords )
{
    if ( !m_calculated || m_equation.isEmpty() || !currentRecords.count() )
        return QVariant();

    //TODO Eventually we'll need to replace this simplistic grammar with a real
    //scripting engine...

    /// DataKiosk Simple Equation Grammar

    /// Possible Variables:
    ///    [current_date]
    ///    [current_time]
    ///    [current_datetime]
    ///    [f:table.field]

    /// Possible Constants:
    ///    [#:integer]
    ///    ["some string"]

    /// Possible Operators:
    ///    '+' -- addition -- valid for numbers and strings
    ///    '-' -- subtraction -- valid for numbers, times, dates, and datetimes
    ///    '*' -- multiplication -- valid for numbers
    ///    '/' -- division -- valid for numbers

    /// Note: Order of operation is _always_ left to right.
    /// Note: For [times,dates,datetimes] the only valid operator is '-'
    ///       which will compute the [days,times] between the two.
    /// Note: For strings the only valid operator is the '+'
    ///       which will concat the two strings
    /// Example: [current_date] - [f:table.some_other_date] + [" days"]

//     QString equation = "[current_date] - [f:TRANSFER_DATE] + [\" days\"]";
//     kdDebug() << "field: " << label() <<  " equation: " << equation() << endl;

    QRegExp varconsRx( "\\[(current_date|current_time|current_datetime|f:\\w+\\.\\w+|#:\\d+|\"[^\"]*\")\\]" );
    QRegExp opsRx( "\\+|\\-|\\*|/" );

    QStringList vars = listFromRx( equation(), varconsRx, 1 );
    QStringList ops = listFromRx( equation().replace( varconsRx, ""), opsRx, 0 );

//     kdDebug() << "vars: " << vars << endl;
//     kdDebug() << "ops: " << ops << endl;

    QValueList<QVariant> values;
    QStringList::iterator it = vars.begin();
    for ( ; it != vars.end(); ++it )
    {
        if ( *it == "current_date" )
            values.append( QVariant( QDate::currentDate() ) );
        else if ( *it == "current_time" )
            values.append( QVariant( QTime::currentTime() ) );
        else if ( *it == "current_datetime" )
            values.append( QVariant( QDateTime::currentDateTime() ) );
        else if ( ( *it ).startsWith("f:") )
        {
            QVariant value;
            QString full = ( *it ).right( ( *it ).length() - 2 );
            QStringList lst = QStringList::split( '.', full );
            const QSqlRecord *record = currentRecords[ lst[ 0 ] ];
            if ( record )
            {
                const QSqlField *field = record->field( lst[ 1 ] );
                if ( field )
                {
                    value = field->value();
                    values.append( value );
                }
            }
        }
        else if ( ( *it ).startsWith("#:") )
            values.append( QVariant( ( *it ).right( ( *it ).length() - 2 ).toInt() ) );
        else if ( ( *it ).startsWith("\"") )
            values.append( QVariant( ( *it ).mid( 1, ( *it ).length() - 2 ) ) );
    }

    QVariant result;
    for ( uint i = 0; i < values.count(); ++i )
    {
        if ( !result.isValid() )
        {
            result = values[0];
            continue;
        }
        QString op = ops[ i - 1 ];

        switch ( values[i].type() )
        {
        case QVariant::UInt:
            if ( op == "+" )
                result = result.toUInt() + values[i].toUInt();
            else if ( op == "-" )
                result = result.toUInt() - values[i].toUInt();
            else if ( op == "*" )
                result = ( result.toDouble() * values[i].toDouble() );
            else if ( op == "/" )
                result = ( result.toDouble() / values[i].toDouble() );
            break;
        case QVariant::Int:
            if ( op == "+" )
                result = result.toInt() + values[i].toInt();
            else if ( op == "-" )
                result = result.toInt() - values[i].toInt();
            else if ( op == "*" )
                result = ( result.toDouble() * values[i].toDouble() );
            else if ( op == "/" )
                result = ( result.toDouble() / values[i].toDouble() );
            break;
        case QVariant::LongLong:
            if ( op == "+" )
                result = result.toLongLong() + values[i].toLongLong();
            else if ( op == "-" )
                result = result.toLongLong() - values[i].toLongLong();
            else if ( op == "*" )
                result = ( result.toDouble() * values[i].toDouble() );
            else if ( op == "/" )
                result = ( result.toDouble() / values[i].toDouble() );
            break;
        case QVariant::ULongLong:
            if ( op == "+" )
                result = result.toULongLong() + values[i].toULongLong();
            else if ( op == "-" )
                result = result.toULongLong() - values[i].toULongLong();
            else if ( op == "*" )
                result = ( result.toDouble() * values[i].toDouble() );
            else if ( op == "/" )
                result = ( result.toDouble() / values[i].toDouble() );
            break;
        case QVariant::Double:
            if ( op == "+" )
                result = result.toDouble() + values[i].toDouble();
            else if ( op == "-" )
                result = result.toDouble() - values[i].toDouble();
            else if ( op == "*" )
                result = ( result.toDouble() * values[i].toDouble() );
            else if ( op == "/" )
                result = ( result.toDouble() / values[i].toDouble() );
            break;
        case QVariant::String:
            if ( op == "+" )
                result = result.toString() + values[i].toString();
            break;
        case QVariant::Time:
            if ( op == "-" )
                result = values[i].toTime().secsTo( result.toTime() );
            break;
        case QVariant::Date:
            if ( op == "-" )
                result = values[i].toDate().daysTo( result.toDate() );
            break;
        case QVariant::DateTime:
            if ( op == "-" )
                result = values[i].toDateTime().daysTo( result.toDateTime() );
            break;
        case QVariant::Invalid:
        case QVariant::Bool:
        case QVariant::CString:
        case QVariant::Pixmap:
        case QVariant::Palette:
        case QVariant::ColorGroup:
        case QVariant::Color:
        case QVariant::Font:
        case QVariant::Brush:
        case QVariant::Bitmap:
        case QVariant::Cursor:
        case QVariant::Map:
        case QVariant::StringList:
        case QVariant::Rect:
        case QVariant::Size:
        case QVariant::IconSet:
        case QVariant::Point:
        case QVariant::PointArray:
        case QVariant::Region:
        case QVariant::SizePolicy:
        case QVariant::ByteArray:
        default:
            break;
        }
    }

    return result;
}

