/*
 * Danpei -- a GTK+ based Image Viewer
 * Copyright (C) 2001-2003 Shinji Moiino
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/* edit_menu.c */

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <gdk/gdkx.h>

#include "config.h"
#include "dialog.h"
#include "edit_menu.h"
#include "image_cache.h"
#include "intl.h"
#include "main.h"
#include "thumbnail.h"
#include "thumbnail_table.h"
#include "typedefs.h"
#include "version.h"
#include "utils.h"

/* Static function declarations. */
static void edit_menu_init_clipboard_structure (ClipBoard*  );

static void edit_menu_free_clipboard_structure (ClipBoard*  );

static void edit_menu_add_thumbnail_list       (TopLevel* ,
                                                gchar*      );

static void edit_menu_cb_dialog_ok             (GtkWidget*,
                                                gpointer    );

/* Function definitions. */
/*
 * @edit_menu_cb_select_all
 *
 *  Set all thumbnails selected.
 *
 */
void edit_menu_cb_select_all(GtkWidget *widget,
                             gpointer  data   ,
                             guint     action   ) {
  TopLevel  *tp;
  Thumbnail *thumb;
  gint      wkgint1, wkgint2;

  /* Initialize the local variables. */
  tp      = (TopLevel*)data;
  wkgint1 = wkgint2 = 0;
  
  thumb = tp->thumbnail_table.top_thumbnail;
  wkgint2 = tp->thumbnail_table.start_pos;
  for (wkgint1 = 0; wkgint1 < wkgint2; wkgint1++) {
    thumb = thumb->next;
  }

  wkgint2 = tp->thumbnail_table.end_pos - tp->thumbnail_table.start_pos + 1;
  for (wkgint1 = 0; wkgint1 < wkgint2; wkgint1++) {
    if (thumb == NULL) {
      break;
     }
    else {
      if (thumb->disabled == FALSE) {
        thumbnail_set_shadow(thumb, TRUE);
        thumbnail_count_selected_thumbnails(tp, 1);
      }
      thumb = thumb->next;
    }
  }

  return;
}

/*
 * @edit_menu_cb_unselect_all
 *
 *  Set all thumbnails unselected.
 *
 */
void edit_menu_cb_unselect_all(GtkWidget *widget,
                               gpointer  data   ,
                               guint     action   ) {
  TopLevel  *tp;
  Thumbnail *thumb;
  gint      wkgint1, wkgint2;

  /* Initialize the local variables. */
  tp      = (TopLevel*)data;
  wkgint1 = wkgint2 = 0;
  
  thumb = tp->thumbnail_table.top_thumbnail;
  wkgint2 = tp->thumbnail_table.start_pos;
  for (wkgint1 = 0; wkgint1 < wkgint2; wkgint1++) {
    thumb = thumb->next;
  }

  wkgint2 = tp->thumbnail_table.end_pos - tp->thumbnail_table.start_pos + 1;
  for (wkgint1 = 0; wkgint1 < wkgint2; wkgint1++) {
    if (thumb == NULL) {
      break;
     }
    else {
      if (thumb->disabled == FALSE) {
        thumbnail_set_shadow(thumb, FALSE);
        thumbnail_count_selected_thumbnails(tp, -1);
      }
      thumb = thumb->next;
    }
  }

  return;
}

/*
 * @edit_menu_cb_copy
 *
 *  Copy all selected thumbnails to the clip board.
 *
 */
void edit_menu_cb_copy(GtkWidget *widget,
                       gpointer  data   ,
                       guint     action   ) {
  TopLevel  *tp;
  Thumbnail *thumb;
  ClipBoard *clip1 , *clip2;
  gint      wkgint1, wkgint2;

  /* Initialize the local variables. */
  tp      = (TopLevel*)data;
  thumb   = NULL;
  clip1   = clip2   = NULL;
  wkgint1 = wkgint2 = 0;

  /* Destroy the current clipboard. */
  edit_menu_destroy_clipboard(&(tp->clipboard));

  thumb = tp->thumbnail_table.top_thumbnail;
  wkgint2 = tp->thumbnail_table.start_pos;
  for (wkgint1 = 0; wkgint1 < wkgint2; wkgint1++) {
    thumb = thumb->next;
  }

  /* Create a new clipboard. */
  tp->clipboard.type = EDIT_COPY;
  tp->clipboard.num  = 0;

  wkgint2 = tp->thumbnail_table.end_pos - tp->thumbnail_table.start_pos + 1;
  for (wkgint1 = 0; wkgint1 < wkgint2; wkgint1++) {
    if (thumb == NULL) {
      break;
    }
    else {
      if ((thumb->selected == TRUE) && (thumb->disabled == FALSE)) {
         clip1 = (ClipBoard*)malloc(sizeof(ClipBoard));
         if (clip1 == NULL) {
           /* Out of memory. */
           fprintf(stderr, "danpei: Out of memory.\n");
           fprintf(stderr, "        edit_menu.c: error -- 01.\n");
           gtk_exit(-1);
         }
         edit_menu_init_clipboard_structure(clip1);
         clip1->path = (gchar*)malloc(sizeof(gchar) *
                                      strlen(thumb->path) + 1);
         if (clip1->path == NULL) {
           /* Out of memory. */
           fprintf(stderr, "danpei: Out of memory.\n");
           fprintf(stderr, "        edit_menu.c: error -- 02.\n");
           gtk_exit(-1);
         }
         clip1->filename = (gchar*)malloc(sizeof(gchar) *
                                          strlen(thumb->filename) + 1);
         if (clip1->filename == NULL) {
           /* Out of memory. */
           fprintf(stderr, "danpei: Out of memory.\n");
           fprintf(stderr, "        edit_menu.c: error -- 03.\n");
           gtk_exit(-1);
         }
         strcpy(clip1->path    , thumb->path);
         strcpy(clip1->filename, thumb->filename); 

         if (tp->clipboard.top == NULL) {
           tp->clipboard.top = clip1;
           clip1->next = NULL;
         }
         else {
           clip2->next = clip1;
           clip1->next = NULL;
           clip2 = clip1;
         }
         clip2 = clip1;
         (tp->clipboard.num)++;
      }
      thumb = thumb->next;
    }
  }

  toplevel_set_menu_sensitive(tp, STATUS_CLIPBOARD);

  return;
}

/*
 * @edit_menu_cb_cut
 *
 *  Copy all selected thumbnails to the clip board.
 *  -- at the time pasted, these thumbnails are removed.
 *
 */
void edit_menu_cb_cut(GtkWidget *widget,
                      gpointer  data   ,
                      guint     action   ) {
  TopLevel  *tp;
  Thumbnail *thumb;
  ClipBoard *clip1 , *clip2;
  gint      wkgint1, wkgint2;

  /* Initialize the local variables. */
  tp      = (TopLevel*)data;
  thumb   = NULL;
  clip1   = clip2   = NULL;
  wkgint1 = wkgint2 = 0;

  /* Destroy the current clipboard. */
  edit_menu_destroy_clipboard(&(tp->clipboard));

  thumb = tp->thumbnail_table.top_thumbnail;
  wkgint2 = tp->thumbnail_table.start_pos;
  for (wkgint1 = 0; wkgint1 < wkgint2; wkgint1++) {
    thumb = thumb->next;
  }

  /* Create a new clipboard. */
  tp->clipboard.type = EDIT_CUT;
  tp->clipboard.num  = 0;

  wkgint2 = tp->thumbnail_table.end_pos - tp->thumbnail_table.start_pos + 1;
  for (wkgint1 = 0; wkgint1 < wkgint2; wkgint1++) {
    if (thumb == NULL) {
      break;
    }
    else {
      if ((thumb->selected == TRUE) && (thumb->disabled == FALSE)) {
         clip1 = (ClipBoard*)malloc(sizeof(ClipBoard));
         if (clip1 == NULL) {
           /* Out of memory. */
           fprintf(stderr, "danpei: Out of memory.\n");
           fprintf(stderr, "        edit_menu.c: error -- 04.\n");
           gtk_exit(-1);
         }
         edit_menu_init_clipboard_structure(clip1);
         clip1->path = (gchar*)malloc(sizeof(gchar) *
                                      strlen(thumb->path) + 1);
         if (clip1->path == NULL) {
           /* Out of memory. */
           fprintf(stderr, "danpei: Out of memory.\n");
           fprintf(stderr, "        edit_menu.c: error -- 05.\n");
           gtk_exit(-1);
         }
         clip1->filename = (gchar*)malloc(sizeof(gchar) *
                                          strlen(thumb->filename) + 1);
         if (clip1->filename == NULL) {
           /* Out of memory. */
           fprintf(stderr, "danpei: Out of memory.\n");
           fprintf(stderr, "        edit_menu.c: error -- 06.\n");
           gtk_exit(-1);
         }
         strcpy(clip1->path    , thumb->path);
         strcpy(clip1->filename, thumb->filename); 

         if (tp->clipboard.top == NULL) {
           tp->clipboard.top = clip1;
           clip1->next = NULL;
         }
         else {
           clip2->next = clip1;
           clip1->next = NULL;
           clip2 = clip1;
         }
         clip2 = clip1;
         (tp->clipboard.num)++;
      }
      thumb = thumb->next;
    }
  }

  toplevel_set_menu_sensitive(tp, STATUS_CLIPBOARD);

  return;
}

/*
 * @edit_menu_cb_paste
 *
 *
 *
 */
void edit_menu_cb_paste(GtkWidget *widget,
                        gpointer  data   ,
                        guint     action    ) {
  TopLevel        *tp;
  Thumbnail       *thumb;
  ClipBoard       *clip1, *clip2;
  NodeInfo        *nodeinfo;
  GtkCTreeNode    *node;
  gchar           *src, *dest;
  gint            wkgint1, wkgint2;
  gint            prev_start_pos, prev_end_pos;
  gboolean        is_src, is_dest;
  AppDialogResult ret;
  gchar           *progressbar_msg[] = { "     "                         ,
                                         N_("  Copying files now ...  ") ,
                                         "     "                         ,
                                         ""                                };

  /* Initialize the local variables. */
  tp             = (TopLevel*)data;
  nodeinfo       = NULL;
  thumb          = NULL;
  clip1          = clip2        = NULL;
  src            = dest         = NULL;
  wkgint1        = wkgint2      = 0;
  prev_start_pos = prev_end_pos = -1;
  is_src         = is_dest      = FALSE;
  ret            = APP_RET_CANCEL;

  if ((tp->clipboard.type != EDIT_CUT ) &&
      (tp->clipboard.type != EDIT_COPY)    ) { return; }

  node = gtk_ctree_node_nth(GTK_CTREE(tp->dir_tree.ctree), 
                            GTK_CLIST(tp->dir_tree.ctree)->focus_row);
  if (node == NULL) { return; }

  nodeinfo = gtk_ctree_node_get_row_data(GTK_CTREE(tp->dir_tree.ctree), 
                                         node);
  if (nodeinfo == NULL) { 
    return; 
  }

  if (strcmp((tp->clipboard.top)->path, 
              tp->thumbnail_table.dir_name) == 0) {
    is_src = TRUE;
  }
  else {
    is_src = FALSE;
  }

  if (strcmp(nodeinfo->path, tp->thumbnail_table.dir_name) == 0) {
    is_dest = TRUE;
  }
  else {
    is_dest = FALSE;
  }

  if ((is_src == TRUE) && (is_dest == TRUE)) ;
  else {
    tp->progressbar_dialog.window = GTK_WINDOW(tp->window);
    dialog_progressbar_dialog_create(progressbar_msg,
                                     &(tp->progressbar_dialog));
    tp->all_yes = FALSE;
    wkgint1 = 0;
    wkgint2 = tp->clipboard.num;
    clip1 = clip2 = tp->clipboard.top;
    while (clip1 != NULL) {
      /* Update the progressbar dialog. */
      if (tp->progressbar_dialog.is_cancel == TRUE) {
        /** dialog_progressbar_dialog_destroy(&(tp->progressbar_dialog)); **/
        clip1 = NULL;
        break;
      }
      dialog_progressbar_dialog_update(&(tp->progressbar_dialog),
                                       wkgint1, wkgint2);

      src = (gchar*)malloc(sizeof(gchar) * 
                           (strlen(clip1->path)     +
                            strlen("/")             +
                            strlen(clip1->filename) + 1));
      if (src == NULL) {
        /* Out of memory. */
        fprintf(stderr, "danpei: Out of memory.\n");
        fprintf(stderr, "        edit_menu.c: error -- 07.\n");
        gtk_exit(-1);
      }
      else {
        sprintf(src, "%s/%s", clip1->path, clip1->filename);
      }

      dest = (gchar*)malloc(sizeof(gchar) * 
                            (strlen(nodeinfo->path)  +
                             strlen("/")             +
                             strlen(clip1->filename) + 1));
      if (dest == NULL) {
        /* Out of memory. */
        fprintf(stderr, "danpei: Out of memory.\n");
        fprintf(stderr, "        edit_menu.c: error -- 08.\n");
        gtk_exit(-1);
      }
      else {
        sprintf(dest, "%s/%s", nodeinfo->path, clip1->filename);
      }

      ret = APP_RET_OK;
      if (access(dest, F_OK) == 0) {
        clip1->status = CLIP_OVERWRITTEN;
        if ((tp->app_option.dialog.overwrite == TRUE) &&
            (tp->all_yes == FALSE)                       ) {
          ret = dialog_message_dialog_create(FILE_OVERWRITTEN, 
                                             clip1->filename, 
                                             APP_OK_CANCEL_SKIP_ALL_DIALOG,
                                             APP_CANCEL_BUTTON   ,
                                             GTK_WINDOW(tp->window));
        }
      }

      if ((ret == APP_RET_OK) || (ret == APP_RET_OK_ALL)) {
        if (ret == APP_RET_OK_ALL) { tp->all_yes = TRUE; }
        if(edit_menu_copy_file(dest, src, tp) == TRUE) {
          if (tp->app_option.image_cache.cache_on == TRUE) {
            /* Copy the cache file, src to dest. */
            if (clip1->status == CLIP_OVERWRITTEN) {
              /* When overwritten, remove the cache file of the old image. */
              image_cache_delete(&(tp->cache), dest);
            }
            image_cache_copy(&(tp->cache), src, dest);
          }
          if (clip1->status == CLIP_NO_OPERATED) {
            clip1->status = CLIP_PASTED;
          }

          if (tp->clipboard.type == EDIT_CUT) {
            if (remove(src) == 0) {
              if (tp->app_option.image_cache.cache_on == TRUE) {
                image_cache_delete(&(tp->cache), src);
              }
              if (is_src == TRUE) {
                thumb = thumbnail_get_from_path(tp, src);
                if (thumb != NULL) {
                  thumbnail_change_link(&(tp->thumbnail_table), thumb); 
                  if (widget != NULL) {
                    if (GTK_IS_WIDGET(thumb->ev_box) == TRUE) {
                      gtk_widget_destroy(thumb->ev_box);
                    }
                  }
                  thumbnail_free_structure(thumb);
                  thumbnail_count_selected_thumbnails(tp, -1);
                }
              }
            }
            else {
              dialog_message_dialog_create(COPY_INSTEAD, src ,
                                           APP_OK_ONLY_DIALOG,
                                           APP_OK_BUTTON     ,
                                           GTK_WINDOW(tp->window));
            }
          }
        }
        else {
          clip1->status = CLIP_NO_OPERATED;
        }
      }
      else { 
        if (ret == APP_RET_CANCEL) {
          free(src); free(dest);
          break;
        }
      }
      free(src); free(dest);
      wkgint1++;
      clip1 = clip1->next;
    }

    dialog_progressbar_dialog_destroy(&(tp->progressbar_dialog));

    /* Refresh the current page, if needed. */
    if ((is_src == TRUE) && (tp->clipboard.type == EDIT_CUT) &&
        (tp->app_option.auto_refresh == TRUE)                   ) {
      thumbnail_table_refresh(tp);
    }

    if (is_dest == TRUE) {
      /* Add clipboard to current thumbnail table. */
      edit_menu_add_thumbnail_list(tp, nodeinfo->path);

      /* Refresh the current page, if needed. */
      if (tp->app_option.auto_refresh == TRUE) {
        prev_start_pos = tp->thumbnail_table.start_pos;
        prev_end_pos   = tp->thumbnail_table.end_pos;
        thumbnail_table_set_position_after_refresh(tp,
                                                   prev_start_pos,
                                                   prev_end_pos    );
        if ((tp->thumbnail_table.start_pos == prev_start_pos) &&
            (tp->thumbnail_table.end_pos   == prev_end_pos  )    ) ;
        else {
          thumbnail_table_display_current_page(tp, -1, -1);
        }
      }
    }

    /* If the operation is EDIT_CUT, then destroy the clipboard. */
    if (tp->clipboard.type == EDIT_CUT) {
      edit_menu_destroy_clipboard(&(tp->clipboard));
      toplevel_set_menu_sensitive(tp, STATUS_NO_CLIPBOARD);
    }

    tp->all_yes = FALSE;
  }

  return;
}

/*
 * @edit_menu_cb_rotate_left
 *
 *  Rotate selected thumbnails left and overwrite original data.
 *
 */
void edit_menu_cb_rotate_left(GtkWidget *widget,
                              gpointer  data   ,
                              guint     action    ) {
  TopLevel       *tp;
  Thumbnail      *thumb;
  gchar          *com, *filename;
  gint           wkgint1, wkgint2;
  gchar          *progressbar_msg[] = { "     "                  ,
                                        N_("  Rotate now ...  ") ,
                                        "     "                  ,
                                        ""                         };

  /* Initialize the local variables. */
  tp      = (TopLevel*)data;
  thumb   = NULL;
  com     = filename = NULL;
  wkgint1 = wkgint2  = 0;

  thumb   = tp->thumbnail_table.top_thumbnail;
  wkgint2 = tp->thumbnail_table.selected_thumbnails;

  /* Create the progressbar dialog. */
  tp->progressbar_dialog.window = GTK_WINDOW(tp->window);
  dialog_progressbar_dialog_create(progressbar_msg,
                                   &(tp->progressbar_dialog));

  while (thumb != NULL) {
    if ((thumb->selected == TRUE) && (thumb->disabled == FALSE)) {
      /* Update the progressbar dialog. */
      if (tp->progressbar_dialog.is_cancel == TRUE) {
        thumb = NULL;
        break;
      }
      dialog_progressbar_dialog_update(&(tp->progressbar_dialog),
                                       wkgint1, wkgint2);

      filename = (gchar*)malloc(sizeof(gchar) *
                                (strlen(thumb->path)     +
                                 strlen("/")             +
                                 strlen(thumb->filename) + 1));
      if (filename == NULL) {
        /* Out of memory. */
        fprintf(stderr, "danpei: Out of memory.\n");
        fprintf(stderr, "        edit_menu.c: error -- 13.\n");
        gtk_exit(-1);
      }
      else {
        sprintf(filename, "%s/%s", thumb->path, thumb->filename);
      }

      if (access(filename, W_OK) == 0) {
        com = (gchar*)malloc(sizeof(gchar) *
                             (strlen(tp->app_option.path.rotate_left)  +
                              strlen(" '")                             +
                              strlen(filename)                         +
                              strlen("' '")                            +
                              strlen(filename)                         +
                              strlen("' &")                            + 1));
        if (com == NULL) {
          /* Out of memory. */
          fprintf(stderr, "danpei: Out of memory.\n");
          fprintf(stderr, "        edit_menu.c: error -- 14.\n");
          gtk_exit(-1);
        }
        else {
          sprintf(com, "%s '%s' '%s'",
                  tp->app_option.path.rotate_left, filename, filename);
          system(com);
          free(com);
          gtk_widget_set_sensitive(thumb->entry, TRUE);
          if (tp->app_option.image_cache.cache_on == TRUE) {
            image_cache_delete(&(tp->cache), filename);
          }
          if (thumb->pixmap != NULL) {
            gdk_pixmap_unref(thumb->pixmap);
          }
          thumb->pixmap = NULL;
          gtk_widget_destroy(thumb->icon_box);
          thumbnail_create_pixmap(thumb, tp);
          thumbnail_create_icon_area(thumb, tp);
        }
      }
      else {
        /* Display a message dialog since we are not allowed to overwrite
         * original data.
         */
        switch (errno) {
          case ENOENT:
            dialog_error_dialog_create(FILE_NOT_EXISTS, filename,
                                       GTK_WINDOW(tp->window));
            /* That file no longer exists, so make that thumbnail 
             * disabled and delete from the cache.
             */
            if (tp->app_option.image_cache.cache_on == TRUE) {
              image_cache_delete(&(tp->cache), filename);
            }
            thumbnail_set_sensitive(thumb, FALSE);
            thumbnail_count_selected_thumbnails(tp, -1);
            break;
          case EACCES:
          case EPERM :
            dialog_error_dialog_create(PERMISSION_DENIED, filename,
                                       GTK_WINDOW(tp->window));
            gtk_widget_set_sensitive(thumb->entry, FALSE);
            break;
          case EROFS: 
            dialog_error_dialog_create(READ_ONLY_FS, filename,
                                       GTK_WINDOW(tp->window));
            gtk_widget_set_sensitive(thumb->entry, FALSE);
            break;
          default:
            dialog_error_dialog_create(ROTATION_ERROR, filename,
                                       GTK_WINDOW(tp->window));
            break;
        }
      }
      free(filename);
      wkgint1++;
    }
    thumb = thumb->next;
  }
  dialog_progressbar_dialog_destroy(&(tp->progressbar_dialog));

  return;
}

/*
 * @edit_menu_cb_rotate_right
 *
 *
 *
 */
void edit_menu_cb_rotate_right(GtkWidget *widget,
                               gpointer  data   ,
                               guint     action    ) {
  TopLevel  *tp;
  Thumbnail *thumb;
  gchar     *com, *filename;
  gint      wkgint1, wkgint2;
  gchar     *progressbar_msg[] = { "     "                         ,
                                   N_("  Rotate now ...  ") ,
                                   "     "                         ,
                                   ""                                };

  /* Initialize the local variables. */
  tp      = (TopLevel*)data;
  thumb   = NULL;
  com     = filename = NULL;
  wkgint1 = wkgint2  = 0;

  thumb   = tp->thumbnail_table.top_thumbnail;
  wkgint2 = tp->thumbnail_table.selected_thumbnails;

  /* Create the progressbar dialog. */
  tp->progressbar_dialog.window = GTK_WINDOW(tp->window);
  dialog_progressbar_dialog_create(progressbar_msg,
                                   &(tp->progressbar_dialog));

  while (thumb != NULL) {
    /* Check cancel. */
    if (tp->progressbar_dialog.is_cancel == TRUE) {
      thumb = NULL;
      break;
    }
    if ((thumb->selected == TRUE) && (thumb->disabled == FALSE)) {
      /* Update the progressbar dialog. */
      dialog_progressbar_dialog_update(&(tp->progressbar_dialog),
                                       wkgint1, wkgint2);

      filename = (gchar*)malloc(sizeof(gchar) *
                                (strlen(thumb->path)     +
                                 strlen("/")             +
                                 strlen(thumb->filename) + 1));
      if (filename == NULL) {
        /* Out of memory. */
        fprintf(stderr, "danpei: Out of memory.\n");
        fprintf(stderr, "        edit_menu.c: error -- 15.\n");
        gtk_exit(-1);
      }
      else {
        sprintf(filename, "%s/%s", thumb->path, thumb->filename);
      }
 
      if (access(filename, W_OK) == 0) {
        com = (gchar*)malloc(sizeof(gchar) *
                             (strlen(tp->app_option.path.rotate_right) +
                              strlen(" '")                             +
                              strlen(filename)                         +
                              strlen("' '")                            +
                              strlen(filename)                         +
                              strlen("' &")                            + 1));
        if (com == NULL) {
          /* Out of memory. */
          fprintf(stderr, "danpei: Out of memory.\n");
          fprintf(stderr, "        edit_menu.c: error -- 16.\n");
          gtk_exit(-1);
        }
        else {
          sprintf(com, "%s '%s' '%s'",
                  tp->app_option.path.rotate_right, filename, filename);
          system(com);
          free(com);
          gtk_widget_set_sensitive(thumb->entry, TRUE);
          if (tp->app_option.image_cache.cache_on == TRUE) {
            image_cache_delete(&(tp->cache), filename);
          }
          if (thumb->pixmap != NULL) {
            gdk_pixmap_unref(thumb->pixmap);
          }
          thumb->pixmap = NULL;
          gtk_widget_destroy(thumb->icon_box);
          thumbnail_create_pixmap(thumb, tp);
          thumbnail_create_icon_area(thumb, tp);
        }
      }
      else {
        /* Display a message dialog since we are not allowed to overwrite
         * original data.
         */
        switch (errno) {
          case ENOENT:
            dialog_error_dialog_create(FILE_NOT_EXISTS, filename,
                                       GTK_WINDOW(tp->window));
            /* That file no longer exists, so make that thumbnail 
             * disabled and delete from the cache.
             */
            if (tp->app_option.image_cache.cache_on == TRUE) {
              image_cache_delete(&(tp->cache), filename);
            }
            thumbnail_set_sensitive(thumb, FALSE);
            thumbnail_count_selected_thumbnails(tp, -1);
            break;
          case EACCES:
          case EPERM :
            dialog_error_dialog_create(PERMISSION_DENIED, filename,
                                       GTK_WINDOW(tp->window));
            gtk_widget_set_sensitive(thumb->entry, FALSE);
            break;
          case EROFS: 
            dialog_error_dialog_create(READ_ONLY_FS, filename,
                                       GTK_WINDOW(tp->window));
            gtk_widget_set_sensitive(thumb->entry, FALSE);
            break;
          default:
            dialog_error_dialog_create(ROTATION_ERROR, filename,
                                       GTK_WINDOW(tp->window));
            break;
        }
      }
      free(filename);
      wkgint1++;
    }
    thumb = thumb->next;
  }
  dialog_progressbar_dialog_destroy(&(tp->progressbar_dialog));

  return;
}

/*
 * @edit_menu_cb_show_clipboard
 *
 *  Display the clip board.
 *
 */
void edit_menu_cb_show_clipboard(GtkWidget *widget,
                                 gpointer  data   ,
                                 guint     action   ) {
  TopLevel     *tp;
  ClipBoard    *clip;
  gchar        *path;

  GtkWidget    *dialog;
  GtkWidget    *frame;
  GtkWidget    *vbox1, *vbox2;
  GtkWidget    *hbox1, *hbox2;
  GtkWidget    *scrolled_window;
  GtkWidget    *label;
  GtkWidget    *button;

  /* Initialize the local variables. */
  tp   = (TopLevel*)data;
  clip = NULL;

  dialog = gtk_dialog_new();
  gtk_window_set_title(GTK_WINDOW(dialog), _("Clip Board"));
  gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, FALSE, FALSE);
  gtk_window_set_transient_for(GTK_WINDOW(dialog), GTK_WINDOW(tp->window));
  gtk_signal_connect(GTK_OBJECT(dialog), 
                     "delete-event",
                     GTK_SIGNAL_FUNC(edit_menu_cb_dialog_ok), NULL);

  frame = gtk_frame_new(NULL);
  gtk_container_border_width(GTK_CONTAINER(frame), 8);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), frame, TRUE, TRUE, 0);
  gtk_widget_show(frame);

  vbox1 = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(frame), vbox1);
  gtk_widget_show(vbox1);

  hbox1 = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox1), hbox1, FALSE, FALSE, 0);
  gtk_widget_show(hbox1);
  
  label = gtk_label_new(_("Operation: "));
  gtk_widget_set_usize(label, 90, 20);
  gtk_box_pack_start(GTK_BOX(hbox1), label, FALSE, FALSE, 0);
  gtk_widget_show(label);

  switch (tp->clipboard.type) {
    case EDIT_CUT:
      label = gtk_label_new(_("Cut"));
      break;
    case EDIT_COPY:
      label = gtk_label_new(_("Copy"));
      break;
    default:
      label = gtk_label_new(_("None"));
      break;
  }
  gtk_widget_set_usize(label, 60, 20);
  gtk_box_pack_start(GTK_BOX(hbox1), label, FALSE, FALSE, 0);
  gtk_widget_show(label);

  scrolled_window = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window),
                                 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start(GTK_BOX(vbox1), scrolled_window, TRUE, TRUE, 0);
  gtk_widget_show(scrolled_window);

  vbox2 = gtk_vbox_new(FALSE, 0);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_window),
                                        vbox2);
  gtk_widget_show(vbox2);

  clip = tp->clipboard.top;
  while (clip != NULL) {
    path = (gchar*)malloc(sizeof(gchar*) * 
                          (strlen(clip->path)    +
                           strlen("/")           +
                           strlen(clip->filename)  ) + 1);
    if (path == NULL) {
      /* Out of memory. */
      fprintf(stderr, "danpei: Out of memory.\n");
      fprintf(stderr, "        edit_menu.c: error -- 09.\n");
      gtk_exit(-1);
    }
    else {
      sprintf(path, "%s/%s", clip->path, clip->filename);
    }
    hbox2 = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox2), hbox2, FALSE, FALSE, 0);
    gtk_widget_show(hbox2);

    label = gtk_label_new(path);
    gtk_box_pack_start(GTK_BOX(hbox2), label, FALSE, FALSE, 0);
    gtk_widget_show(label);
    free(path);
    clip = clip->next;
  }

  button = gtk_button_new_with_label(_("   OK   "));
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area), 
                     button, FALSE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(button), 
                     "clicked",
                     GTK_SIGNAL_FUNC(edit_menu_cb_dialog_ok), NULL);
  GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default(button);
  gtk_widget_grab_focus(button);
  gtk_widget_show(button);

  gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);
  gtk_widget_set_usize(dialog, 350, 250);
  gtk_widget_show(dialog);

  gtk_main();

  gtk_window_set_modal(GTK_WINDOW(dialog), FALSE);
  gtk_widget_destroy(dialog);

  return;
}

/*
 * @edit_menu_copy_file
 *
 *
 *
 */
gboolean edit_menu_copy_file(gchar    *dest,
                             gchar    *src ,
                             TopLevel *tp    ) {
  AppErrorNum app_err;
  gint        which;

  /* Initialize the local variables. */
  app_err = APP_ERROR_NONE;
  which   = 0;

  /* Copy file. */
  app_err = utils_copy_file(dest, src, &which);
  if (app_err != APP_ERROR_NONE) {
    switch (app_err) {
      case FILE_NOT_EXISTS:
        if (which == 0) {
          dialog_error_dialog_create(FILE_NOT_EXISTS, src,
                                     GTK_WINDOW(tp->window));
        }
        else {
          dialog_error_dialog_create(FILE_NOT_EXISTS, dest,
                                     GTK_WINDOW(tp->window));
        }
        break;
      case PERMISSION_DENIED:
        if (which == 0) {
          dialog_error_dialog_create(PERMISSION_DENIED, src,
                                     GTK_WINDOW(tp->window));
        }
        else {
          dialog_error_dialog_create(PERMISSION_DENIED, dest,
                                     GTK_WINDOW(tp->window));
        }
        break;
      case TOO_LONG_FILE_NAME:
        if (which == 0) {
          dialog_error_dialog_create(TOO_LONG_FILE_NAME, src,
                                     GTK_WINDOW(tp->window));
        }
        else {
          dialog_error_dialog_create(TOO_LONG_FILE_NAME, dest,
                                     GTK_WINDOW(tp->window));
        }
        break;
      case READ_ONLY_FS:
        dialog_error_dialog_create(READ_ONLY_FS, dest,
                                   GTK_WINDOW(tp->window));
        break;
      case NO_SPACE_LEFT:
        dialog_error_dialog_create(NO_SPACE_LEFT, dest,
                                   GTK_WINDOW(tp->window));
        break;
      case FILE_READ_ERROR:
        dialog_error_dialog_create(FILE_READ_ERROR, src,
                                   GTK_WINDOW(tp->window));
        break;
      case FILE_WRITE_ERROR:
        dialog_error_dialog_create(FILE_WRITE_ERROR, dest,
                                   GTK_WINDOW(tp->window));
        break;
      case OPEN_FILE_ERROR:
        if (which == 0) {
          dialog_error_dialog_create(OPEN_FILE_ERROR, src,
                                     GTK_WINDOW(tp->window));
        }
        else {
          dialog_error_dialog_create(OPEN_FILE_ERROR, dest,
                                     GTK_WINDOW(tp->window));
        }
        break;
      default:
        dialog_error_dialog_create(PASTE_ERROR, dest,
                                   GTK_WINDOW(tp->window));
        break;
    }
    return FALSE;
  }

  return TRUE;
}

/*
 * @edit_menu_destroy_clipboard
 *
 *  Free ManageClipBoard structure object. 
 *
 */
void edit_menu_destroy_clipboard(ManageClipBoard *mc) {
  ClipBoard *clip1, *clip2;

  clip1 = mc->top;
  while (clip1 != NULL) {
    clip2 = clip1->next;
    edit_menu_free_clipboard_structure(clip1);
    clip1 = clip2;
  }

  mc->top  = NULL;
  mc->type = EDIT_NONE;
  mc->num  = 0;

  return;
}


/* Static function definitions. */
/*
 * @edit_menu_init_clipboard_structure
 *
 *  Initialize ClipBoard structure object. 
 *
 */
static void edit_menu_init_clipboard_structure(ClipBoard *clip) {
  clip->path     = NULL;
  clip->filename = NULL;
  clip->status   = CLIP_NO_OPERATED;
  clip->next     = NULL;

  return;
}

/*
 * @edit_menu_free_clipboard_structure
 *
 *  Free ClipBoard structure object. 
 *
 */
static void edit_menu_free_clipboard_structure(ClipBoard *clip) {
  if (clip->path != NULL) {
    free(clip->path);
    clip->path = NULL;
  }
  if (clip->filename != NULL) {
    free(clip->filename);
    clip->filename = NULL;
  }
  
  free(clip);

  return;
}

/*
 * @edit_menu_add_thumbnail_list
 *
 *  Add files in the clipboard to the current thumbnail list.
 *
 */
static void edit_menu_add_thumbnail_list (TopLevel *tp       ,
                                          gchar    *dest_path  ) {
  Thumbnail    *thumb1, *thumb2;
  ClipBoard    *clip;

  /* Initialize th elocal variables. */
  thumb1 = thumb2 = NULL;
  clip   = NULL;

  clip   = tp->clipboard.top;
  thumb2 = tp->thumbnail_table.end_thumbnail;
  while (clip != NULL) {
    if (clip->status == CLIP_PASTED) {
      thumb1 = (Thumbnail*)malloc(sizeof(Thumbnail));
      if (thumb1 == NULL) {
        /* Out of memory. */
        fprintf(stderr, "danpei: Out of memory.\n");
        fprintf(stderr, "        edit_menu.c: error -- 10.\n");
        gtk_exit(-1);
      }
      else {
        thumbnail_init(thumb1);
      }
  
      thumb1->path = strdup(dest_path);
      if (thumb1->path == NULL) {
        /* Out of memory. */
        fprintf(stderr, "danpei: Out of memory.\n");
        fprintf(stderr, "        edit_menu.c: error -- 11.\n");
        gtk_exit(-1);
      }
      thumb1->filename = strdup(clip->filename);
      if (thumb1->filename == NULL) {
        /* Out of memory. */
        fprintf(stderr, "danpei: Out of memory.\n");
        fprintf(stderr, "        edit_menu.c: error -- 12.\n");
        gtk_exit(-1);
      }
  
      if (tp->thumbnail_table.top_thumbnail == NULL) {
        tp->thumbnail_table.top_thumbnail = thumb2 = thumb1;
        thumb1->prev = NULL;
        thumb1->next = NULL;
      }
      else {
        thumb2->next = thumb1;
        thumb1->prev = thumb2;
        thumb1->next = NULL;
        thumb2       = thumb1;
      }
      tp->thumbnail_table.file_num++;
      clip->status = CLIP_NO_OPERATED;
    }
    clip = clip->next;
  }
  tp->thumbnail_table.end_thumbnail = thumb1;

  return;
}

/*
 * @edit_menu_cb_dialog_ok
 *
 *
 *
 */
static void edit_menu_cb_dialog_ok(GtkWidget *widget,
                                   gpointer  data     ) {
  gtk_main_quit();

  return;
}

