
/*
 *  Diverse SLab audio routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


/*
 * These are largely device specific operations for the audio devices. Some of
 * the calls are general, but may eventually be used by several of the SLab
 * applications.
 *
 * The operations for audio control are rather ugly. The application sets up
 * a number of values in the controlBuffer, and the engine intermittanly looks
 * to see if these values are initialised. If so, it acts on the values, and
 * then clears them. The reason we have to use this indirection is that the
 * front end applications do not own the audio device, this is under control
 * of the engine.
 *
 *	Jul 28th 96 - Only LINUX code is active.
 */

/*
 * Audio device structure format definitions.
 */
#include "slabrevisions.h"
#include "slabaudiodev.h"
#include "slabcdefs.h"

#ifdef DEBUG
#include <stdio.h>
#endif
#include <unistd.h>
#include <fcntl.h>
#ifdef SUBFRAGMENT
#if (MACHINE == FreeBSD)
#include <stdlib.h>
#else
#include <malloc.h>
#endif
#endif

#if (MACHINE == FreeBSD)
#include <sys/soundcard.h>
#endif
#ifdef linux
/*
 * This may want to be /usr/lib/oss/include/sys/soundcard.h if someone has
 * purchased the 4Front drivers?
 */
#include <sys/soundcard.h>
audio_buf_info bufferInfo;
#endif
#ifdef SUN
#include <sys/ioccom.h>
#include <sun/audioio.h>
#include <fcntl.h>

audio_info_t audioInfo;
#endif


/*
static char *SLAB_SND_LABELS[SLAB_NRDEVICES] =	SLAB_DEVICE_LABELS;
 * Called once when the device is opened by the engine. Initialises our fd to
 * 44100 16 bits stereo operation. 
 * Changed for 2.30 (or so) to include capability to have no requests to alter
 * the default fragment size. The code will do a simplified audio init, and let\ * the audio IO daemon use the default fragment for IO, decoupled from the block
 * sample size.
 */
int
initAudioDevice2(audioDev, devID, fragSize)
duplexDev *audioDev;
int devID;
{
	int trackfile, i, j, results, data, mode, index;
	int flags;

	/*
	 * The device is basically just opened for the first call of this routine.
	 * This is then called several times afterwards - each time we start the
	 * engine (that is a tape start, not an engine restart). This may be
	 * problematic.
	 *
	 * First set duplex (later check duplex).
	 * Set the fragments.
	 * In this order set bits, channels, speed.
	 */
#ifdef ALSA_SUPPORT
	/*
	 * ALSA is one of the secondary intrusive flags. If this is set we have
	 * already configured what we want from the open request on the device.
	 */
	if ((audioDev->siflags & AUDIO_ALSA) != 0)
		return(0);
#endif

	/*
	 * If we get here we need to call some OSS routine.
	 */
	return(ossAudioInit(audioDev, devID, fragSize));
}

setAudioStart2(audioDev, devID)
duplexDev *audioDev;
{
	int enable;

#ifdef IOCTL_DBG
	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("setAudioStart2(%i)\n", devID);
#endif

#ifdef ALSA_SUPPORT
	if (audioDev->siflags & AUDIO_ALSA)
	{
		alsaDevAudioStart(audioDev);
		return;
	}
#endif
	if (audioDev->fd < 0)
		return;

#ifdef linux
	if ((audioDev->genCaps & SNDCTL_DSP_SETTRIGGER) &&
		(audioDev->Command == 1))
	{
		enable = (PCM_ENABLE_OUTPUT | PCM_ENABLE_INPUT);
#ifdef IOCTL_DBG
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("ioctl(%i, SNDCTL_DSP_SETTRIGGER, &%08x)\n",
				audioDev->fd, enable);
#endif
		ioctl(audioDev->fd, SNDCTL_DSP_SETTRIGGER, &enable);
	}
#endif
}

setAudioStop2(audioDev, devID)
duplexDev *audioDev;
{
	int enable;
#ifdef IOCTL_DBG
	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("setAudioStop2(%i)\n", devID);
#endif

#ifdef ALSA_SUPPORT
	if (audioDev->siflags & AUDIO_ALSA)
	{
#ifdef IOCTL_DBG
		/*
		 * This is a superfluous bit of debug, but it will at least show that
		 * we are recognising the ALSA configuration status. There are no audio
		 * stop commands associated with the SLab ALSA drivers, although we
		 * could use the "pause" calls.
		 */
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("ALSA device, returning from setAudioStop2\n");
#endif
		return;
	}
#endif

	if (audioDev->fd < 0)
		return;

#ifdef linux
	if ((audioDev->genCaps & SNDCTL_DSP_SETTRIGGER) &&
		(audioDev->Command == 1))
	{
		enable = ~(PCM_ENABLE_OUTPUT | PCM_ENABLE_INPUT);
#ifdef IOCTL_DBG
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("ioctl(%i, SNDCTL_DSP_SETTRIGGER, &%08x)\n",
				audioDev->fd, enable);
#endif
		ioctl(audioDev->fd, SNDCTL_DSP_SETTRIGGER, &enable);
#ifdef IOCTL_DBG
		if (audioDev->cflags & SLAB_AUDIODBG)
		{
			ioctl(audioDev->fd, SNDCTL_DSP_GETTRIGGER, &enable);
			printf("ioctl(%i, SNDCTL_DSP_GETTRIGGER, &%08x)\n",
				audioDev->fd, enable);
		}
#endif
	}
#endif
}

audioClose(duplexDev *audioDev)
{
	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("audioClose(%08x, %i, %s)\n", audioDev, audioDev->devID,
			audioDev->devName);

#ifdef ALSA_SUPPORT
	if (audioDev->siflags & AUDIO_ALSA)
	{
		alsaDevClose(audioDev);
		return 0;
	}
#endif
	if (audioDev->fd != -1)
	{
		close(audioDev->fd);
		audioDev->fd = -1;
	}
	if (audioDev->fd2 != -1)
	{
		close(audioDev->fd2);
		audioDev->fd2 = -1;
	}

	return 0;
}

audioOpen(duplexDev *audioDev, int device, int flags)
{
	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("audioOpen(%08x, %i, %i): %s\n", audioDev, device, flags,
			audioDev->devName);

#ifdef ALSA_SUPPORT
	if (audioDev->siflags & AUDIO_ALSA)
		return(alsaDevOpen(audioDev, device, flags, audioDev->fragSize));
	else
#endif
	{
		/*
		 * We need to "mung" the flags.
		 */
		if (flags == SLAB_OWRONLY)
			flags = O_WRONLY;
		else if (flags == SLAB_ORDONLY)
			flags = O_RDONLY;
		else if (flags == SLAB_ORDWR)
			flags = O_RDWR;
		else {
			printf("	WHAT WERE THOSE FLAGS: %x\n", flags);
			/*
			 * We may as well take some default value if we are unsure about the
			 * request.
			 */
			flags = O_RDWR;
		}
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("flags are now %i\n", flags);
		/*
		 * Open the device.
		 */
		if ((audioDev->fd = open(audioDev->devName, flags)) < 0)
		{
			/*
			 * Device open failed. Print an error message and exit.
			 */
			printf("Failed to open audio device \"%s\", flags %i\n",
				audioDev->devName, flags);
			return(-10);
		}
#ifdef IOCTL_DBG
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("Opened audio device \"%s\"\n", audioDev->devName);
#endif
	}

#ifdef AUDIO_API
	/*
	 * We will eventually allow configurable buffer sizes. At the moment we
	 * will take the default value for the main output device, and
	 * configure that on all the other devices.
	 */
	initAudioDevice2(audioDev, device, audioDev->fragSize);
#else
	initAudioDevice(device);
#endif
	return(audioDev->fd);
}

channelStatus(duplexDev *audioDev)
{
#ifdef ALSA_SUPPORT
	if (audioDev->siflags & AUDIO_ALSA)
		return(alsaChannelStatus(audioDev));
	else
#endif
		return(0);
}

headStatus(duplexDev *audioDev)
{
	/*
	 * We do not sync the primary device, we track the others to it.
	 */
	if (audioDev->devID == 0)
		return(0);

#ifdef ALSA_SUPPORT
	if (audioDev->siflags & AUDIO_ALSA)
		return(alsaHeadStatus(audioDev));
	else
#endif
		return(0);
}

