/** @file
 * CPUM - CPU Monitor(/ Manager), Context Structures.
 */

/*
 * Copyright (C) 2006-2019 Oracle Corporation
 *
 * This file is part of VirtualBox Open Source Edition (OSE), as
 * available from http://www.virtualbox.org. This file is free software;
 * you can redistribute it and/or modify it under the terms of the GNU
 * General Public License (GPL) as published by the Free Software
 * Foundation, in version 2 as it comes in the "COPYING" file of the
 * VirtualBox OSE distribution. VirtualBox OSE is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY of any kind.
 *
 * The contents of this file may alternatively be used under the terms
 * of the Common Development and Distribution License Version 1.0
 * (CDDL) only, as it comes in the "COPYING.CDDL" file of the
 * VirtualBox OSE distribution, in which case the provisions of the
 * CDDL are applicable instead of those of the GPL.
 *
 * You may elect to license modified versions of this file under the
 * terms and conditions of either the GPL or the CDDL or both.
 */



#pragma D  depends_on library x86.d




/** @defgroup grp_cpum_ctx  The CPUM Context Structures
 * @ingroup grp_cpum
 * @{
 */

/**
 * Selector hidden registers.
 */
typedef struct CPUMSELREG
{
    /** The selector register. */
    RTSEL       Sel;
    /** Padding, don't use. */
    RTSEL       PaddingSel;
    /** The selector which info resides in u64Base, u32Limit and Attr, provided
     * that CPUMSELREG_FLAGS_VALID is set. */
    RTSEL       ValidSel;
    /** Flags, see CPUMSELREG_FLAGS_XXX. */
    uint16_t    fFlags;

    /** Base register.
     *
     * Long mode remarks:
     *  - Unused in long mode for CS, DS, ES, SS
     *  - 32 bits for FS & GS; FS(GS)_BASE msr used for the base address
     *  - 64 bits for TR & LDTR
     */
    uint64_t    u64Base;
    /** Limit (expanded). */
    uint32_t    u32Limit;
    /** Flags.
     * This is the high 32-bit word of the descriptor entry.
     * Only the flags, dpl and type are used. */
    X86DESCATTR Attr;
} CPUMSELREG;

/** @name CPUMSELREG_FLAGS_XXX - CPUMSELREG::fFlags values.
 * @{ */
inline uint16_t CPUMSELREG_FLAGS_VALID = 0x0001;

inline uint16_t CPUMSELREG_FLAGS_STALE = 0x0002;

inline uint16_t CPUMSELREG_FLAGS_VALID_MASK = 0x0003;

/** @} */

/** Checks if the hidden parts of the selector register are valid. */


/** Old type used for the hidden register part.
 * @deprecated  */
typedef CPUMSELREG CPUMSELREGHID;

/**
 * The sysenter register set.
 */
typedef struct CPUMSYSENTER
{
    /** Ring 0 cs.
     * This value +  8 is the Ring 0 ss.
     * This value + 16 is the Ring 3 cs.
     * This value + 24 is the Ring 3 ss.
     */
    uint64_t    cs;
    /** Ring 0 eip. */
    uint64_t    eip;
    /** Ring 0 esp. */
    uint64_t    esp;
} CPUMSYSENTER;

/** @def CPUM_UNION_NM
 * For compilers (like DTrace) that does not grok nameless unions, we have a
 * little hack to make them palatable.
 */
/** @def CPUM_STRUCT_NM
 * For compilers (like DTrace) that does not grok nameless structs (it is
 * non-standard C++), we have a little hack to make them palatable.
 */


/** @def CPUM_UNION_STRUCT_NM
 * Combines CPUM_UNION_NM and CPUM_STRUCT_NM to avoid hitting the right side of
 * the screen in the compile time assertions.
 */


/** A general register (union). */
typedef union CPUMCTXGREG
{
    /** Natural unsigned integer view. */
    uint64_t            u;
    /** 64-bit view. */
    uint64_t            u64;
    /** 32-bit view. */
    uint32_t            u32;
    /** 16-bit view. */
    uint16_t            u16;
    /** 8-bit view. */
    uint8_t             u8;
    /** 8-bit low/high view.    */
     struct
    {
        /** Low byte (al, cl, dl, bl, ++). */
        uint8_t         bLo;
        /** High byte in the first word - ah, ch, dh, bh. */
        uint8_t         bHi;
    } s  ;
} CPUMCTXGREG;



/**
 * CPU context core.
 *
 * @todo        Eliminate this structure!
 * @deprecated  We don't push any context cores any more in TRPM.
 */
typedef struct CPUMCTXCORE
{
    /** @name General Register.
     * @note  These follow the encoding order (X86_GREG_XXX) and can be accessed as
     *        an array starting a rax.
     * @{ */
    union
    {
        uint8_t         al;
        uint16_t        ax;
        uint32_t        eax;
        uint64_t        rax;
    } rax  ;
    union
    {
        uint8_t         cl;
        uint16_t        cx;
        uint32_t        ecx;
        uint64_t        rcx;
    } rcx  ;
    union
    {
        uint8_t         dl;
        uint16_t        dx;
        uint32_t        edx;
        uint64_t        rdx;
    } rdx  ;
    union
    {
        uint8_t         bl;
        uint16_t        bx;
        uint32_t        ebx;
        uint64_t        rbx;
    } rbx  ;
    union
    {
        uint16_t        sp;
        uint32_t        esp;
        uint64_t        rsp;
    } rsp  ;
    union
    {
        uint16_t        bp;
        uint32_t        ebp;
        uint64_t        rbp;
    } rbp  ;
    union
    {
        uint8_t         sil;
        uint16_t        si;
        uint32_t        esi;
        uint64_t        rsi;
    } rsi  ;
    union
    {
        uint8_t         dil;
        uint16_t        di;
        uint32_t        edi;
        uint64_t        rdi;
    } rdi  ;
    uint64_t            r8;
    uint64_t            r9;
    uint64_t            r10;
    uint64_t            r11;
    uint64_t            r12;
    uint64_t            r13;
    uint64_t            r14;
    uint64_t            r15;
    /** @} */

    /** @name Segment registers.
     * @note These follow the encoding order (X86_SREG_XXX) and can be accessed as
     *       an array starting a es.
     * @{  */
    CPUMSELREG          es;
    CPUMSELREG          cs;
    CPUMSELREG          ss;
    CPUMSELREG          ds;
    CPUMSELREG          fs;
    CPUMSELREG          gs;
    /** @} */

    /** The program counter. */
    union
    {
        uint16_t        ip;
        uint32_t        eip;
        uint64_t        rip;
    } rip  ;

    /** The flags register. */
    union
    {
        X86EFLAGS       eflags;
        X86RFLAGS       rflags;
    } rflags  ;

} CPUMCTXCORE;


/**
 * SVM Host-state area (Nested Hw.virt - VirtualBox's layout).
 *
 * @warning Exercise caution while modifying the layout of this struct. It's
 *          part of VM saved states.
 */
typedef struct SVMHOSTSTATE
{
    uint64_t    uEferMsr;
    uint64_t    uCr0;
    uint64_t    uCr4;
    uint64_t    uCr3;
    uint64_t    uRip;
    uint64_t    uRsp;
    uint64_t    uRax;
    X86RFLAGS   rflags;
    CPUMSELREG  es;
    CPUMSELREG  cs;
    CPUMSELREG  ss;
    CPUMSELREG  ds;
    VBOXGDTR    gdtr;
    VBOXIDTR    idtr;
    uint8_t     abPadding[4];
} SVMHOSTSTATE;
/** Pointer to the SVMHOSTSTATE structure. */
typedef SVMHOSTSTATE *PSVMHOSTSTATE;
/** Pointer to a const SVMHOSTSTATE structure. */
typedef const SVMHOSTSTATE *PCSVMHOSTSTATE;


/**
 * CPU hardware virtualization types.
 */
typedef enum
{
    CPUMHWVIRT_NONE = 0,
    CPUMHWVIRT_VMX,
    CPUMHWVIRT_SVM,
    CPUMHWVIRT_32BIT_HACK = 0x7fffffff
} CPUMHWVIRT;


/**
 * CPU context.
 */
typedef struct CPUMCTX
{
    /** CPUMCTXCORE Part.
     * @{ */

    /** General purpose registers. */
    union /* no tag! */
    {
        /** The general purpose register array view, indexed by X86_GREG_XXX. */
        CPUMCTXGREG     aGRegs[16];

        /** 64-bit general purpose register view. */
         struct /* no tag! */
        {
            uint64_t    rax, rcx, rdx, rbx, rsp, rbp, rsi, rdi, r8, r9, r10, r11, r12, r13, r14, r15;
        } qw  ;
        /** 64-bit general purpose register view. */
         struct /* no tag! */
        {
            uint64_t    r0, r1, r2, r3, r4, r5, r6, r7;
        } qw2  ;
        /** 32-bit general purpose register view. */
         struct /* no tag! */
        {
            uint32_t     eax, u32Pad00,      ecx, u32Pad01,      edx, u32Pad02,      ebx, u32Pad03,
                         esp, u32Pad04,      ebp, u32Pad05,      esi, u32Pad06,      edi, u32Pad07,
                         r8d, u32Pad08,      r9d, u32Pad09,     r10d, u32Pad10,     r11d, u32Pad11,
                        r12d, u32Pad12,     r13d, u32Pad13,     r14d, u32Pad14,     r15d, u32Pad15;
        } dw  ;
        /** 16-bit general purpose register view. */
         struct /* no tag! */
        {
            uint16_t      ax, au16Pad00[3],   cx, au16Pad01[3],   dx, au16Pad02[3],   bx, au16Pad03[3],
                          sp, au16Pad04[3],   bp, au16Pad05[3],   si, au16Pad06[3],   di, au16Pad07[3],
                         r8w, au16Pad08[3],  r9w, au16Pad09[3], r10w, au16Pad10[3], r11w, au16Pad11[3],
                        r12w, au16Pad12[3], r13w, au16Pad13[3], r14w, au16Pad14[3], r15w, au16Pad15[3];
        } w  ;
         struct /* no tag! */
        {
            uint8_t   al, ah, abPad00[6], cl, ch, abPad01[6], dl, dh, abPad02[6], bl, bh, abPad03[6],
                         spl, abPad04[7],    bpl, abPad05[7],    sil, abPad06[7],    dil, abPad07[7],
                         r8l, abPad08[7],    r9l, abPad09[7],   r10l, abPad10[7],   r11l, abPad11[7],
                        r12l, abPad12[7],   r13l, abPad13[7],   r14l, abPad14[7],   r15l, abPad15[7];
        } b  ;
    } g  ;

    /** Segment registers. */
    union /* no tag! */
    {
        /** The segment register array view, indexed by X86_SREG_XXX. */
        CPUMSELREG      aSRegs[6];
        /** The named segment register view. */
         struct /* no tag! */
        {
            CPUMSELREG  es, cs, ss, ds, fs, gs;
        } n  ;
    } s  ;

    /** The program counter. */
    union
    {
        uint16_t        ip;
        uint32_t        eip;
        uint64_t        rip;
    } rip  ;

    /** The flags register. */
    union
    {
        X86EFLAGS       eflags;
        X86RFLAGS       rflags;
    } rflags  ;

    /** @} */ /*(CPUMCTXCORE)*/


    /** @name Control registers.
     * @{ */
    uint64_t            cr0;
    uint64_t            cr2;
    uint64_t            cr3;
    uint64_t            cr4;
    /** @} */

    /** Debug registers.
     * @remarks DR4 and DR5 should not be used since they are aliases for
     *          DR6 and DR7 respectively on both AMD and Intel CPUs.
     * @remarks DR8-15 are currently not supported by AMD or Intel, so
     *          neither do we.
     */
    uint64_t        dr[8];

    /** Padding before the structure so the 64-bit member is correctly aligned.
     * @todo fix this structure!  */
    uint16_t        gdtrPadding[3];
    /** Global Descriptor Table register. */
    VBOXGDTR        gdtr;

    /** Padding before the structure so the 64-bit member is correctly aligned.
     * @todo fix this structure!  */
    uint16_t        idtrPadding[3];
    /** Interrupt Descriptor Table register. */
    VBOXIDTR        idtr;

    /** The task register.
     * Only the guest context uses all the members. */
    CPUMSELREG      ldtr;
    /** The task register.
     * Only the guest context uses all the members. */
    CPUMSELREG      tr;

    /** The sysenter msr registers.
     * This member is not used by the hypervisor context. */
    CPUMSYSENTER    SysEnter;

    /** @name System MSRs.
     * @{ */
    uint64_t        msrEFER;
    uint64_t        msrSTAR;            /**< Legacy syscall eip, cs & ss. */
    uint64_t        msrPAT;             /**< Page attribute table. */
    uint64_t        msrLSTAR;           /**< 64 bits mode syscall rip. */
    uint64_t        msrCSTAR;           /**< Compatibility mode syscall rip. */
    uint64_t        msrSFMASK;          /**< syscall flag mask. */
    uint64_t        msrKERNELGSBASE;    /**< swapgs exchange value. */
    uint64_t        uMsrPadding0;       /**< no longer used (used to hold a copy of APIC base MSR). */
    /** @} */

    /** The XCR0..XCR1 registers. */
    uint64_t                    aXcr[2];
    /** The mask to pass to XSAVE/XRSTOR in EDX:EAX.  If zero we use
     *  FXSAVE/FXRSTOR (since bit 0 will always be set, we only need to test it). */
    uint64_t                    fXStateMask;

    /** Pointer to the FPU/SSE/AVX/XXXX state ring-0 mapping. */
    PX86XSAVEAREA     pXStateR0;
    /** Pointer to the FPU/SSE/AVX/XXXX state ring-3 mapping. */
    RTR3PTR    pXStateR3;
    /** Pointer to the FPU/SSE/AVX/XXXX state raw-mode mapping. */
    RTRCPTR    pXStateRC;
    /** State component offsets into pXState, UINT16_MAX if not present. */
    uint16_t                    aoffXState[64];

    /** 0x2d4 - World switcher flags, CPUMCTX_WSF_XXX. */
    uint32_t                    fWorldSwitcher;
    /** 0x2d8 - Externalized state tracker, CPUMCTX_EXTRN_XXX.
     * Currently only used internally in NEM/win.  */
    uint64_t                    fExtrn;

    /** 0x2e0 - Hardware virtualization state.   */
    struct
    {
        union   /* no tag! */
        {
            struct
            {
                /** 0x2e0 - MSR holding physical address of the Guest's Host-state. */
                uint64_t                uMsrHSavePa;
                /** 0x2e8 - Guest physical address of the nested-guest VMCB. */
                RTGCPHYS                GCPhysVmcb;
                /** 0x2f0 - Cache of the nested-guest VMCB - R0 ptr. */
                PSVMVMCB      pVmcbR0;
                /** 0x2f8 - Cache of the nested-guest VMCB - R3 ptr. */
                RTR3PTR     pVmcbR3;
                /** 0x300 - Guest's host-state save area. */
                SVMHOSTSTATE            HostState;
                /** 0x3b8 - Guest TSC time-stamp of when the previous PAUSE instr. was executed. */
                uint64_t                uPrevPauseTick;
                /** 0x3c0 - Pause filter count. */
                uint16_t                cPauseFilter;
                /** 0x3c2 - Pause filter threshold. */
                uint16_t                cPauseFilterThreshold;
                /** 0x3c4 - Whether the injected event is subject to event intercepts. */
                uint8_t                    fInterceptEvents;
                /** 0x3c5 - Padding. */
                uint8_t                    afPadding[3];
                /** 0x3c8 - MSR permission bitmap - R0 ptr. */
                void *        pvMsrBitmapR0;
                /** 0x3d0 - MSR permission bitmap - R3 ptr. */
                RTR3PTR       pvMsrBitmapR3;
                /** 0x3d8 - IO permission bitmap - R0 ptr. */
                void *        pvIoBitmapR0;
                /** 0x3e0 - IO permission bitmap - R3 ptr. */
                RTR3PTR       pvIoBitmapR3;
                /** 0x3e8 - Host physical address of the nested-guest VMCB.  */
                RTHCPHYS                HCPhysVmcb;
                /** 0x3f0 - Padding. */
                uint64_t                u64Padding0[19];
            } svm;

            struct
            {
                /** 0x2e4 - Guest physical address of the VMXON region. */
                RTGCPHYS                GCPhysVmxon;
                /** 0x2e8 - Guest physical address of the current VMCS pointer. */
                RTGCPHYS                GCPhysVmcs;
                /** 0x2f0 - Guest physical address of the shadow VMCS pointer. */
                RTGCPHYS                GCPhysShadowVmcs;
                /** 0x2f8 - Last emulated VMX instruction/VM-exit diagnostic. */
                VMXVDIAG                enmDiag;
                /** 0x2fc - VMX abort reason. */
                VMXABORT                enmAbort;
                /** 0x300 - VMX abort auxiliary information field. */
                uint32_t                uAbortAux;
                /** 0x304 - Whether the guest is in VMX root mode. */
                uint8_t                    fInVmxRootMode;
                /** 0x305 - Whether the guest is in VMX non-root mode. */
                uint8_t                    fInVmxNonRootMode;
                /** 0x306 - Whether the injected events are subjected to event intercepts.  */
                uint8_t                    fInterceptEvents;
                /** 0x307 - Whether blocking of NMI (or virtual-NMIs) was in effect in VMX non-root
                 *  mode before execution of IRET. */
                uint8_t                    fNmiUnblockingIret;
                /** 0x308 - Cache of the nested-guest current VMCS - R0 ptr. */
                PVMXVVMCS     pVmcsR0;
                /** 0x310 - Cache of the nested-guest curent VMCS - R3 ptr. */
                RTR3PTR    pVmcsR3;
                /** 0X318 - Cache of the nested-guest shadow VMCS - R0 ptr. */
                PVMXVVMCS     pShadowVmcsR0;
                /** 0x320 - Cache of the nested-guest shadow VMCS - R3 ptr. */
                RTR3PTR    pShadowVmcsR3;
                /** 0x328 - Cache of the nested-guest Virtual-APIC page - R0 ptr. */
                void *        pvVirtApicPageR0;
                /** 0x330 - Cache of the nested-guest Virtual-APIC page - R3 ptr. */
                RTR3PTR       pvVirtApicPageR3;
                /** 0x338 - Cache of the nested-guest VMREAD-bitmap - R0 ptr. */
                void *        pvVmreadBitmapR0;
                /** 0x340 - Cache of the nested-guest VMREAD-bitmap - R3 ptr. */
                RTR3PTR       pvVmreadBitmapR3;
                /** 0x348 - Cache of the nested-guest VMWRITE-bitmap - R0 ptr. */
                void *        pvVmwriteBitmapR0;
                /** 0x350 - Cache of the nested-guest VMWRITE-bitmap - R3 ptr. */
                RTR3PTR       pvVmwriteBitmapR3;
                /** 0x358 - The MSR auto-load/store area - R0 ptr. */
                PVMXAUTOMSR   pAutoMsrAreaR0;
                /** 0x360 - The MSR auto-load/store area - R3 ptr. */
                RTR3PTR  pAutoMsrAreaR3;
                /** 0x368 - The MSR bitmap - R0 ptr. */
                void *        pvMsrBitmapR0;
                /** 0x370 - The MSR bitmap - R3 ptr. */
                RTR3PTR       pvMsrBitmapR3;
                /** 0x378 - The I/O bitmap - R0 ptr. */
                void *        pvIoBitmapR0;
                /** 0x380 - The I/O bitmap - R3 ptr. */
                RTR3PTR       pvIoBitmapR3;
                /** 0x388 - Guest TSC timestamp of the first PAUSE instruction that is considered to
                 *  be the first in a loop. */
                uint64_t                uFirstPauseLoopTick;
                /** 0x390 - Guest TSC timestamp of the previous PAUSE instruction. */
                uint64_t                uPrevPauseTick;
                /** 0x398 - Guest TSC timestamp of VM-entry (used for VMX-preemption timer). */
                uint64_t                uVmentryTick;
                /** 0x3a0 - Virtual-APIC write offset (until trap-like VM-exit). */
                uint16_t                offVirtApicWrite;
                /** 0x3a2 - Whether virtual-NMI blocking is in effect. */
                uint8_t                    fVirtNmiBlocking;
                /** 0x3a3 - Padding. */
                uint8_t                 abPadding0[5];
                /** 0x3a8 - Guest VMX MSRs. */
                VMXMSRS                 Msrs;
            } vmx;
        } s  ;

        /** 0x488 - Hardware virtualization type currently in use. */
        CPUMHWVIRT              enmHwvirt;
        /** 0x48c - Global interrupt flag - AMD only (always true on Intel). */
        uint8_t                    fGif;
        uint8_t                    afPadding1[3];
        /** 0x490 - A subset of guest force flags that are saved while running the
         *  nested-guest. */
        uint32_t                fLocalForcedActions;
        uint32_t                fPadding;
        /** 0x498 - Pad to 64 byte boundary. */
        uint8_t                 abPadding0[40];
    } hwvirt;
    /** @} */
} CPUMCTX;



/** @name CPUMCTX_WSF_XXX
 * @{ */
/** Touch IA32_PRED_CMD.IBPB on VM exit. */
inline uint32_t CPUMCTX_WSF_IBPB_EXIT = 1U << 0;

/** Touch IA32_PRED_CMD.IBPB on VM entry. */
inline uint32_t CPUMCTX_WSF_IBPB_ENTRY = 1U << 1;

/** Touch IA32_FLUSH_CMD.L1D on VM entry. */
inline uint32_t CPUMCTX_WSF_L1D_ENTRY = 1U << 2;

/** Flush MDS buffers on VM entry. */
inline uint32_t CPUMCTX_WSF_MDS_ENTRY = 1U << 3;

/** @} */

/** @name CPUMCTX_EXTRN_XXX
 * Used for parts of the CPUM state that is externalized and needs fetching
 * before use.
 *
 * @{ */
/** External state keeper: Invalid.  */
inline uint64_t CPUMCTX_EXTRN_KEEPER_INVALID = 0x0000000000000000;

/** External state keeper: HM. */
inline uint64_t CPUMCTX_EXTRN_KEEPER_HM = 0x0000000000000001;

/** External state keeper: NEM. */
inline uint64_t CPUMCTX_EXTRN_KEEPER_NEM = 0x0000000000000002;

/** External state keeper: REM. */
inline uint64_t CPUMCTX_EXTRN_KEEPER_REM = 0x0000000000000003;

/** External state keeper mask. */
inline uint64_t CPUMCTX_EXTRN_KEEPER_MASK = 0x0000000000000003;


/** The RIP register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_RIP = 0x0000000000000004;

/** The RFLAGS register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_RFLAGS = 0x0000000000000008;


/** The RAX register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_RAX = 0x0000000000000010;

/** The RCX register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_RCX = 0x0000000000000020;

/** The RDX register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_RDX = 0x0000000000000040;

/** The RBX register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_RBX = 0x0000000000000080;

/** The RSP register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_RSP = 0x0000000000000100;

/** The RBP register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_RBP = 0x0000000000000200;

/** The RSI register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_RSI = 0x0000000000000400;

/** The RDI register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_RDI = 0x0000000000000800;

/** The R8 thru R15 register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_R8_R15 = 0x0000000000001000;

/** General purpose registers mask. */
inline uint64_t CPUMCTX_EXTRN_GPRS_MASK = 0x0000000000001ff0;


/** The ES register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_ES = 0x0000000000002000;

/** The CS register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_CS = 0x0000000000004000;

/** The SS register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_SS = 0x0000000000008000;

/** The DS register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_DS = 0x0000000000010000;

/** The FS register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_FS = 0x0000000000020000;

/** The GS register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_GS = 0x0000000000040000;

/** Segment registers (includes CS). */
inline uint64_t CPUMCTX_EXTRN_SREG_MASK = 0x000000000007e000;

/** Converts a X86_XREG_XXX index to a CPUMCTX_EXTRN_xS mask. */


/** The GDTR register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_GDTR = 0x0000000000080000;

/** The IDTR register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_IDTR = 0x0000000000100000;

/** The LDTR register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_LDTR = 0x0000000000200000;

/** The TR register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_TR = 0x0000000000400000;

/** Table register mask. */
inline uint64_t CPUMCTX_EXTRN_TABLE_MASK = 0x0000000000780000;


/** The CR0 register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_CR0 = 0x0000000000800000;

/** The CR2 register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_CR2 = 0x0000000001000000;

/** The CR3 register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_CR3 = 0x0000000002000000;

/** The CR4 register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_CR4 = 0x0000000004000000;

/** Control register mask. */
inline uint64_t CPUMCTX_EXTRN_CR_MASK = 0x0000000007800000;

/** The TPR/CR8 register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_APIC_TPR = 0x0000000008000000;

/** The EFER register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_EFER = 0x0000000010000000;


/** The DR0, DR1, DR2 and DR3 register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_DR0_DR3 = 0x0000000020000000;

/** The DR6 register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_DR6 = 0x0000000040000000;

/** The DR7 register value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_DR7 = 0x0000000080000000;

/** Debug register mask. */
inline uint64_t CPUMCTX_EXTRN_DR_MASK = 0x00000000e0000000;


/** The XSAVE_C_X87 state is kept externally. */
inline uint64_t CPUMCTX_EXTRN_X87 = 0x0000000100000000;

/** The XSAVE_C_SSE, XSAVE_C_YMM, XSAVE_C_ZMM_HI256, XSAVE_C_ZMM_16HI and
 * XSAVE_C_OPMASK state is kept externally. */
inline uint64_t CPUMCTX_EXTRN_SSE_AVX = 0x0000000200000000;

/** The state of XSAVE components not covered by CPUMCTX_EXTRN_X87 and
 * CPUMCTX_EXTRN_SEE_AVX is kept externally. */
inline uint64_t CPUMCTX_EXTRN_OTHER_XSAVE = 0x0000000400000000;

/** The state of XCR0 and XCR1 register values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_XCRx = 0x0000000800000000;



/** The KERNEL GS BASE MSR value is kept externally. */
inline uint64_t CPUMCTX_EXTRN_KERNEL_GS_BASE = 0x0000001000000000;

/** The STAR, LSTAR, CSTAR and SFMASK MSR values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_SYSCALL_MSRS = 0x0000002000000000;

/** The SYSENTER_CS, SYSENTER_EIP and SYSENTER_ESP MSR values are kept externally. */
inline uint64_t CPUMCTX_EXTRN_SYSENTER_MSRS = 0x0000004000000000;

/** The TSC_AUX MSR is kept externally. */
inline uint64_t CPUMCTX_EXTRN_TSC_AUX = 0x0000008000000000;

/** All other stateful MSRs not covered by CPUMCTX_EXTRN_EFER,
 * CPUMCTX_EXTRN_KERNEL_GS_BASE, CPUMCTX_EXTRN_SYSCALL_MSRS,
 * CPUMCTX_EXTRN_SYSENTER_MSRS, and CPUMCTX_EXTRN_TSC_AUX.  */
inline uint64_t CPUMCTX_EXTRN_OTHER_MSRS = 0x0000010000000000;


/** Mask of all the MSRs. */


/** Hardware-virtualization (SVM or VMX) state is kept externally. */
inline uint64_t CPUMCTX_EXTRN_HWVIRT = 0x0000020000000000;


/** Mask of bits the keepers can use for state tracking. */
inline uint64_t CPUMCTX_EXTRN_KEEPER_STATE_MASK = 0xffff000000000000;


/** NEM/Win: Event injection (known was interruption) pending state. */
inline uint64_t CPUMCTX_EXTRN_NEM_WIN_EVENT_INJECT = 0x0001000000000000;

/** NEM/Win: Inhibit maskable interrupts (VMCPU_FF_INHIBIT_INTERRUPTS). */
inline uint64_t CPUMCTX_EXTRN_NEM_WIN_INHIBIT_INT = 0x0002000000000000;

/** NEM/Win: Inhibit non-maskable interrupts (VMCPU_FF_BLOCK_NMIS). */
inline uint64_t CPUMCTX_EXTRN_NEM_WIN_INHIBIT_NMI = 0x0004000000000000;

/** NEM/Win: Mask. */
inline uint64_t CPUMCTX_EXTRN_NEM_WIN_MASK = 0x0007000000000000;


/** HM/SVM: Inhibit maskable interrupts (VMCPU_FF_INHIBIT_INTERRUPTS). */
inline uint64_t CPUMCTX_EXTRN_HM_SVM_INT_SHADOW = 0x0001000000000000;

/** HM/SVM: Nested-guest interrupt pending (VMCPU_FF_INTERRUPT_NESTED_GUEST). */
inline uint64_t CPUMCTX_EXTRN_HM_SVM_HWVIRT_VIRQ = 0x0002000000000000;

/** HM/SVM: Mask. */
inline uint64_t CPUMCTX_EXTRN_HM_SVM_MASK = 0x0003000000000000;


/** HM/VMX: Guest-interruptibility state (VMCPU_FF_INHIBIT_INTERRUPTS,
 *  VMCPU_FF_BLOCK_NMIS). */
inline uint64_t CPUMCTX_EXTRN_HM_VMX_INT_STATE = 0x0001000000000000;

/** HM/VMX: Mask. */
inline uint64_t CPUMCTX_EXTRN_HM_VMX_MASK = 0x0001000000000000;


/** All CPUM state bits, not including keeper specific ones. */
inline uint64_t CPUMCTX_EXTRN_ALL = 0x000003fffffffffc;

/** All CPUM state bits, including keeper specific ones. */
inline uint64_t CPUMCTX_EXTRN_ABSOLUTELY_ALL = 0xfffffffffffffffc;

/** @} */


/**
 * Additional guest MSRs (i.e. not part of the CPU context structure).
 *
 * @remarks Never change the order here because of the saved stated!  The size
 *          can in theory be changed, but keep older VBox versions in mind.
 */
typedef union CPUMCTXMSRS
{
    struct
    {
        uint64_t    TscAux;             /**< MSR_K8_TSC_AUX */
        uint64_t    MiscEnable;         /**< MSR_IA32_MISC_ENABLE */
        uint64_t    MtrrDefType;        /**< IA32_MTRR_DEF_TYPE */
        uint64_t    MtrrFix64K_00000;   /**< IA32_MTRR_FIX16K_80000 */
        uint64_t    MtrrFix16K_80000;   /**< IA32_MTRR_FIX16K_80000 */
        uint64_t    MtrrFix16K_A0000;   /**< IA32_MTRR_FIX16K_A0000 */
        uint64_t    MtrrFix4K_C0000;    /**< IA32_MTRR_FIX4K_C0000 */
        uint64_t    MtrrFix4K_C8000;    /**< IA32_MTRR_FIX4K_C8000 */
        uint64_t    MtrrFix4K_D0000;    /**< IA32_MTRR_FIX4K_D0000 */
        uint64_t    MtrrFix4K_D8000;    /**< IA32_MTRR_FIX4K_D8000 */
        uint64_t    MtrrFix4K_E0000;    /**< IA32_MTRR_FIX4K_E0000 */
        uint64_t    MtrrFix4K_E8000;    /**< IA32_MTRR_FIX4K_E8000 */
        uint64_t    MtrrFix4K_F0000;    /**< IA32_MTRR_FIX4K_F0000 */
        uint64_t    MtrrFix4K_F8000;    /**< IA32_MTRR_FIX4K_F8000 */
        uint64_t    PkgCStateCfgCtrl;   /**< MSR_PKG_CST_CONFIG_CONTROL */
        uint64_t    SpecCtrl;           /**< IA32_SPEC_CTRL */
        uint64_t    ArchCaps;           /**< IA32_ARCH_CAPABILITIES */
    } msr;
    uint64_t    au64[64];
} CPUMCTXMSRS;
/** Pointer to the guest MSR state. */
typedef CPUMCTXMSRS *PCPUMCTXMSRS;
/** Pointer to the const guest MSR state. */
typedef const CPUMCTXMSRS *PCCPUMCTXMSRS;

/**
 * The register set returned by a CPUID operation.
 */
typedef struct CPUMCPUID
{
    uint32_t uEax;
    uint32_t uEbx;
    uint32_t uEcx;
    uint32_t uEdx;
} CPUMCPUID;
/** Pointer to a CPUID leaf. */
typedef CPUMCPUID *PCPUMCPUID;
/** Pointer to a const CPUID leaf. */
typedef const CPUMCPUID *PCCPUMCPUID;

/** @}  */




