/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/************************************************
*** XML and  HTML highlighter ***
************************************************/

enum {
    /* highlighted: */
    syntax_xml_tag_tag,
    syntax_xml_tag_pi, /* PIs counted as tags */
    syntax_xml_tag_arg,
    syntax_xml_tag_arg_unquot, /* HTML arg without quotes */
    syntax_xml_tag_string, /* STRING == STRING_QUOT || STRING_AMP */
    syntax_xml_tag_entity,
    /* internal: */
    syntax_xml_tag_space_after_eq,
    syntax_xml_tag_string_quot,
    syntax_xml_tag_string_amp
};

const SyntaxTagInfo syntax_xml_tags[] = {
    { TRUE  }, /* syntax_xml_tag_tag */
    { TRUE  }, /* syntax_xml_tag_pi */
    { TRUE  }, /* syntax_xml_tag_arg */
    { TRUE  }, /* syntax_xml_tag_arg_unquot */
    { TRUE  }, /* syntax_xml_tag_string */
    { TRUE  }, /* syntax_xml_tag_entity */
    { FALSE }, /* syntax_xml_tag_space_after_eq */
    { FALSE }, /* syntax_xml_tag_string_quot */
    { FALSE }  /* syntax_xml_tag_string_amp */
};

SyntaxRule syntax_xml_rules[] = {
    {  { 0, 0 }, ssm_string, "&", NULL,
       { (1<<syntax_xml_tag_entity), 0 }. FALSE, FALSE },
    {  { 0, 0 }, ssm_string, ";", NULL,
       { 0, (1<<syntax_xml_tag_entity) }. TRUE,  FALSE },
    {  { 0, (1<<syntax_xml_tag_tag)|(1<<syntax_xml_tag_string) }, ssm_string, "<?", NULL,
       { (1<<syntax_xml_tag_pi), 0 }, FALSE, FALSE },
    {  { 0, (1<<syntax_xml_tag_tag)|(1<<syntax_xml_tag_string) }, ssm_string, "<", NULL,
       { (1<<syntax_xml_tag_tag), 0 }, FALSE, TRUE },

    syntax_tags_on_off_t condition;
    SyntaxStrMatchKind match_kind;
    const gchar *match_str;
    gboolean (*match_func)();
    syntax_tags_on_off_t action;
    gboolean after :1, final :1;
};

/* === XML/HTML ===

ENTITY
STRING, ARG_UNQUOT
ARG
PI (counted as tags)
TAG
--
SPACE_AFTER_EQ, STRING_QUOT, STRING_AMP

ANY:
    "&" - start ENTITY [PASS]
    ";" - end ENTITY [PASS]
~TAG & ~STRING
    "<?" - start PI
    "<" - start TAG
    ">" - end TAG, end ARG_UNQUOT
TAG
    "=" - SPACE_AFTER_EQ
TAG & ~STRING
    "\"" - start STRING, start STRING_QUOT, start ARG
    "'" - start STRING, start STRING_AMP, start ARG
SPACE_AFTER_EQ
    "^[:space:]" - start ARG, start ARG_UNQUOT
ARG_UNQUOT
    "[:space:]" - end ARG_UNQUOT, end ARG
STRING_QUOT
    "\"" - end STRING_QUOT, end STRING, end ARG
STRING_AMP
    "'" - end STRING_AMP, end STRING, end ARG
TAG & ~ARG
    "=" - start SPACE_AFTER_EQ
ARG & ~STRING
    " " - end ARG
*/

static gboolean syntax_has_tag(HighlighterState *state, GtkTextIter *iter, GtkTextTag *tag)
{
    return syntax_tag_is_open(state, tag) || gtk_text_iter_has_tag(iter, tag);
}

static void syntax_xml_step(HighlighterState *state, GtkTextIter **iter)
{
    GtkTextIter next_char;

    next_char = *iter;
    gtk_text_iter_forward_char(&next_char);
    switch(c) {
        case '&':
            syntax_start_tag(state, *iter, syntax_xml_tags[syntax_xml_tag_entity]);
            break;
        case ';':
            syntax_end_tag(state, &next_char, syntax_xml_tags[syntax_xml_tag_entity]);
            break;
    }
    if(syntax_has_tag(*iter, syntax_xml_tags[syntax_xml_tag_string_quot])) {
        if(c == '"') {
            syntax_end_tag(state, &next_char, syntax_xml_tags[syntax_xml_tag_string]);
            syntax_end_tag(state, &next_char, syntax_xml_tags[syntax_xml_tag_string_quot]);
        }
    } else if(syntax_has_tag(*iter, syntax_xml_tags[syntax_xml_tag_string_apos])) {
        if(c == '\'') {
            syntax_end_tag(state, &next_char, syntax_xml_tags[syntax_xml_tag_string]);
            syntax_end_tag(state, &next_char, syntax_xml_tags[syntax_xml_tag_string_apos]);
        }
    } else if(syntax_has_tag(*iter, syntax_xml_tags[syntax_xml_tag_arg_unquot])) {
        if(whitespace(c)) {
            syntax_end_tag(state, *iter, syntax_xml_tags[syntax_xml_tag_arg_unquot]);
        }
    } else if(syntax_has_tag(*iter, syntax_xml_tags[syntax_xml_tag_tag])) {
        /* We are in a tag but not in argument. */
        switch(c) {
            case '=':
                syntax_start_tag(state, &next_char, syntax_xml_tags[syntax_xml_tag_space_after_eq]);
                break;
            case '>':
                syntax_end_tag(state, &next_char, syntax_xml_tags[syntax_xml_tag_tag]);
                syntax_end_tag(state, &next_char, syntax_xml_tags[syntax_xml_tag_space_after_eq]);
                break;
            case '"':
                syntax_start_tag(state, *iter, syntax_xml_tags[syntax_xml_tag_string]);
                break;
        }
        if( syntax_has_tag(*iter, syntax_xml_tags[syntax_xml_tag_space_after_eq])
            && !whitespace(c) )
        {
            syntax_end_tag(state, *iter, syntax_xml_tags[syntax_xml_tag_space_after_eq]);
            syntax_start_tag(state, &next_char, syntax_xml_tags[syntax_xml_tag_arg_unquot]);
        }
    } else { /* not string, not tag */
        c2 = ;dpkg -p flex

        switch(c) {
            case '<':
                syntax_start_tag(state, *iter, syntax_xml_tags[syntax_xml_tag_tag]);
                break;
        }
    }
}
