/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "search.h"
#include "glade_interface.h"

#include <string.h>

#include "glade_support.h"
#include "main.h"

void initialize_search(SearchData *search_data, GtkWidget *text_view)
{
    search_data->str = search_data->replacement = NULL;
    search_data->find_dialog = search_data->replace_dialog = NULL;
    search_data->text_view = text_view;
    search_data->window = gtk_widget_get_toplevel(text_view);
}

void destroy_search(SearchData *search_data)
{
    g_free((gpointer)search_data->str);
    g_free((gpointer)search_data->replacement);
}

gboolean simple_find_predicate(GtkTextIter *start,
                               GtkTextIter *end,
                               void *data)
{
    gboolean ret;
    gchar *slice;
    GtkTextBuffer *buffer;
    const SearchData *search_data;

    search_data = (SearchData*)data;
    *end = *start;
    gtk_text_iter_forward_chars(end, search_data->str_len);
    if(gtk_text_iter_get_offset(end) - gtk_text_iter_get_offset(start)
       != (gint)search_data->str_len
    )
        return FALSE;
    buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(search_data->text_view));
    slice = gtk_text_buffer_get_slice(buffer, start, end, FALSE);
    if(search_data->case_sensitive)
        ret = !g_utf8_collate(search_data->str, slice);
    else {
        gchar *utf8_find_str, *utf8_slice;

        utf8_find_str = g_utf8_casefold(search_data->str, -1);
        utf8_slice = g_utf8_casefold(slice, -1);
        ret = !g_utf8_collate(utf8_find_str, utf8_slice);
        g_free(utf8_find_str);
        g_free(utf8_slice);
    }
    if(search_data->whole_word)
        ret = ret && gtk_text_iter_starts_word(start) && gtk_text_iter_ends_word(end);
    g_free(slice);
    return ret;
}

static void inform_no_more_matches(GtkWidget *window)
{
    GtkWidget *msg_dlg;

    msg_dlg = gtk_message_dialog_new(
        GTK_WINDOW(window),
        (GtkDialogFlags)(GTK_DIALOG_MODAL|GTK_DIALOG_DESTROY_WITH_PARENT),
        GTK_MESSAGE_INFO, GTK_BUTTONS_CLOSE,
        _("No more matches."));
    gtk_window_set_transient_for(GTK_WINDOW(msg_dlg), GTK_WINDOW(window));
    my_disable_window_resize(GTK_WINDOW(msg_dlg));
    gtk_dialog_run(GTK_DIALOG(msg_dlg));
    gtk_widget_destroy(msg_dlg);
};

gboolean find_text(GtkTextIter *iter,
                   GtkTextIter *end,
                   find_predicate_t predicate,
                   void *data)
{
    SearchData *search_data = (SearchData*)data;
    GtkTextBuffer *buffer;
    GtkWidget *window;

    buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(search_data->text_view));
    window = gtk_widget_get_toplevel(search_data->text_view);
    if(search_data->backward) {
        GtkTextIter iter_old = *iter;

        gtk_text_iter_backward_char(iter);
        if(gtk_text_iter_equal(iter, &iter_old)) /* We at the border of buffer. */ {
            inform_no_more_matches(window);
            return FALSE;
        }
    } else {
        if(simple_find_predicate(iter, end, data)) {
            GtkTextIter end2;

            if( gtk_text_buffer_get_selection_bounds(buffer, NULL, &end2) &&
                gtk_text_iter_equal(end, &end2) )
            {
                GtkTextIter iter_old = *iter;

                gtk_text_iter_forward_char(iter);
                if(gtk_text_iter_equal(iter, &iter_old)) /* We at the border of buffer. */ {
                    inform_no_more_matches(window);
                    return FALSE;
                }
            }
        }
    }
    do {
        if(simple_find_predicate(iter, end, data)) {
            GtkTextMark *cursor;
            gtk_text_buffer_move_mark_by_name(buffer, N_("insert"), iter);
            gtk_text_buffer_move_mark_by_name(buffer, N_("selection_bound"), end);
            cursor = gtk_text_buffer_get_insert(buffer);
            gtk_text_view_scroll_mark_onscreen(GTK_TEXT_VIEW(search_data->text_view),
                                               cursor);
            return TRUE;
        }
    } while( search_data->backward ?
             gtk_text_iter_backward_char(iter) :
             gtk_text_iter_forward_char(iter) );
    inform_no_more_matches(window);
    return FALSE;
}

/* Seems not good. Probably should be rewritten. */
int replace_all(find_predicate_t predicate,
                gboolean in_sel_only,
                void *data)
{
    SearchData *search_data;
    GtkTextBuffer *buffer;
    GtkTextIter iter, sel_end_iter, end;
    GtkTextMark *sel_end = NULL;
    int count = 0;

    search_data = (SearchData*)data;
    buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(search_data->text_view));
    if(in_sel_only) {
        gtk_text_buffer_get_selection_bounds(buffer, &iter, &sel_end_iter);
        sel_end = gtk_text_buffer_create_mark(buffer, NULL, &sel_end_iter, FALSE);
    } else
        gtk_text_buffer_get_start_iter(buffer, &iter);
    gtk_text_buffer_begin_user_action(buffer);
    /* TODO: will moving vs creating/deleting the mark significally speedup? */
    for(;;) {
        GtkTextMark *mark;

        if( (*predicate)(&iter, &end, search_data) ) {
            if(in_sel_only) {
                gtk_text_buffer_get_iter_at_mark(buffer, &sel_end_iter, sel_end);
                if(gtk_text_iter_compare(&end, &sel_end_iter) > 0) break;
            }
            mark = gtk_text_buffer_create_mark(buffer, NULL, &iter, FALSE);
            gtk_text_buffer_delete(buffer, &iter, &end);
            gtk_text_buffer_insert(buffer, &iter,
                                   search_data->replacement,
                                   search_data->replacement_len);
            ++count;
            gtk_text_buffer_get_iter_at_mark(buffer, &iter, mark);
            gtk_text_buffer_delete_mark(buffer, mark);
        } else
            if( !gtk_text_iter_forward_char(&iter) ||
                (in_sel_only && gtk_text_iter_equal(&iter, &sel_end_iter))
            )
                break;
    };
    if(in_sel_only) gtk_text_buffer_delete_mark(buffer, sel_end);
    gtk_text_buffer_end_user_action(buffer);
    return count;
}

typedef struct SearchDialogData_
{
    GtkWidget *find_entry, *case_check, *whole_word_check, *backward_check;
    GtkWidget *replace_entry;
} SearchDialogData;

static void get_find_dialog_data(SearchDialogData *dlg_data, GtkWidget *dialog)
{
    dlg_data->find_entry       = lookup_widget(dialog, N_("find_entry"));
    dlg_data->case_check       = lookup_widget(dialog, N_("case_sensitive_check"));
    dlg_data->whole_word_check = lookup_widget(dialog, N_("whole_words_check"));
    dlg_data->backward_check   = lookup_widget(dialog, N_("backward_check"));
}

static void get_replace_dialog_data(SearchDialogData *dlg_data, GtkWidget *dialog)
{
    get_find_dialog_data(dlg_data, dialog);
    dlg_data->replace_entry = lookup_widget(dialog, N_("replace_entry"));
}

static void expand_find_dlg_data(const SearchDialogData *dlg_data,
                                 SearchData *search_data)
{
    g_free((gpointer)search_data->str);
    search_data->str = g_strdup(gtk_entry_get_text(GTK_ENTRY(dlg_data->find_entry)));
    search_data->str_len = g_utf8_strlen(search_data->str, -1);
    search_data->case_sensitive = gtk_toggle_button_get_active(
            GTK_TOGGLE_BUTTON(dlg_data->case_check));
    search_data->whole_word = gtk_toggle_button_get_active(
            GTK_TOGGLE_BUTTON(dlg_data->whole_word_check));
    search_data->backward = gtk_toggle_button_get_active(
            GTK_TOGGLE_BUTTON(dlg_data->backward_check));
}

static void expand_replace_dlg_data(const SearchDialogData *dlg_data,
                                    SearchData *search_data)
{
    expand_find_dlg_data(dlg_data, search_data);
    g_free((gpointer)search_data->replacement);
    search_data->replacement = g_strdup(gtk_entry_get_text(GTK_ENTRY(dlg_data->replace_entry)));
    search_data->replacement_len = strlen(search_data->replacement);
}

static void expand_search_data(SearchData *search_data)
{
    SearchDialogData dlg_data;

    if(search_data->replace_dialog) {
        get_replace_dialog_data(&dlg_data, search_data->replace_dialog);
        expand_replace_dlg_data(&dlg_data, search_data);
    } else if(search_data->find_dialog) {
        get_find_dialog_data(&dlg_data, search_data->find_dialog);
        expand_find_dlg_data(&dlg_data, search_data);
    }
}

void find_next(SearchData *search_data)
{
    GtkTextBuffer *buffer;
    GtkTextIter iter, end;
    GtkTextMark *insert_point;

    expand_search_data(search_data);
    buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(search_data->text_view));
    if(search_data->str && *search_data->str) {
        insert_point = gtk_text_buffer_get_insert(buffer);
        gtk_text_buffer_get_iter_at_mark(buffer, &iter, insert_point);
        (void)find_text(&iter, &end, simple_find_predicate, (void*)search_data);
    }
}

void call_find_dialog(SearchData *search_data)
{
    SearchDialogData dlg_data;

    if(search_data->replace_dialog)
        gtk_widget_destroy(search_data->replace_dialog);
    else if(search_data->find_dialog) {
        gtk_window_present(GTK_WINDOW(search_data->find_dialog));
        return;
    }
    search_data->find_dialog = create_find_dialog();
    get_find_dialog_data(&dlg_data, search_data->find_dialog);
    g_object_add_weak_pointer(G_OBJECT(search_data->find_dialog),
                              (void*)&search_data->find_dialog);
    gtk_widget_set_name(search_data->find_dialog, N_("finddialog"));
    gtk_window_set_transient_for(GTK_WINDOW(search_data->find_dialog),
                                 GTK_WINDOW(search_data->window));
    gtk_widget_grab_focus(dlg_data.find_entry);
    gtk_widget_show(GTK_WIDGET(search_data->find_dialog));
}

void call_replace_dialog(SearchData *search_data)
{
    SearchDialogData dlg_data;

    if(search_data->find_dialog)
        gtk_widget_destroy(search_data->find_dialog);
    else if(search_data->replace_dialog) {
        gtk_window_present(GTK_WINDOW(search_data->replace_dialog));
        return;
    }
    search_data->replace_dialog = create_replace_dialog();
    get_replace_dialog_data(&dlg_data, search_data->replace_dialog);
    gtk_window_set_type_hint(GTK_WINDOW(search_data->replace_dialog),
                             GDK_WINDOW_TYPE_HINT_DIALOG);
    g_object_add_weak_pointer(G_OBJECT(search_data->replace_dialog),
                              (void*)&search_data->replace_dialog);
    gtk_widget_set_name(search_data->replace_dialog, N_("replacedialog"));
    gtk_window_set_transient_for(GTK_WINDOW(search_data->replace_dialog),
                                 GTK_WINDOW(search_data->window));

    get_replace_dialog_data(&dlg_data, search_data->replace_dialog);

    gtk_widget_grab_focus(dlg_data.find_entry);
    gtk_widget_show(GTK_WIDGET(search_data->replace_dialog));
}

void accomplish_find(SearchData *search_data)
{
    GtkTextMark *insert_point;
    GtkTextIter iter, end;

    expand_search_data(search_data);
    if(search_data->str) {
        GtkTextBuffer *buffer;

        buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(search_data->text_view));
        insert_point = gtk_text_buffer_get_insert(buffer);
        gtk_text_buffer_get_iter_at_mark(buffer, &iter, insert_point);
        (void)find_text(&iter, &end, simple_find_predicate, search_data);
    }
}

void accomplish_replace(SearchData *search_data)
{
    GtkTextIter iter, end, end2;
    GtkTextBuffer *buffer;

    expand_search_data(search_data);
    buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(search_data->text_view));
    if(!*search_data->str) return;
    if( gtk_text_buffer_get_selection_bounds(buffer, &iter, &end2) &&
        simple_find_predicate(&iter, &end, search_data) &&
        gtk_text_iter_equal(&end, &end2) )
    {
        gtk_text_buffer_begin_user_action(buffer);
        gtk_text_buffer_delete_selection(buffer, TRUE, TRUE);
        gtk_text_buffer_insert_at_cursor(buffer,
                                         search_data->replacement,
                                         search_data->replacement_len);
        gtk_text_buffer_end_user_action(buffer);
        gtk_text_buffer_get_selection_bounds(buffer, &iter, NULL);
        if(search_data->backward)
            gtk_text_iter_backward_chars(&iter, search_data->replacement_len);
    }
    (void)find_text(&iter, &end, simple_find_predicate, search_data);
}

void accomplish_replace_all(SearchData *search_data,
                            gboolean in_sel_only)
{
    int count;
    GtkWidget *msg_dlg;
    SearchDialogData dlg_data;

    expand_search_data(search_data);
    if(search_data->str) {
        GtkTextBuffer *buffer;
        GtkWidget *window;

        window = gtk_widget_get_toplevel(search_data->text_view);
        buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(search_data->text_view));
        count = replace_all(simple_find_predicate,
                            in_sel_only,
                            search_data);
        msg_dlg = gtk_message_dialog_new(
                GTK_WINDOW(window),
                (GtkDialogFlags)(GTK_DIALOG_MODAL|GTK_DIALOG_DESTROY_WITH_PARENT),
                GTK_MESSAGE_INFO, GTK_BUTTONS_CLOSE,
                _("%d replacements was done."),
                count);
        my_disable_window_resize(GTK_WINDOW(msg_dlg));
        gtk_dialog_run(GTK_DIALOG(msg_dlg));
        gtk_widget_destroy(msg_dlg);
    }
}
