/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "inisettings.h"

#ifdef G_PLATFORM_WIN32
#  include <windows.h>
#endif
#ifdef G_OS_WIN32
#  include <shlobj.h>
#  include <gdk/gdkwin32.h>
#  ifndef SHGFP_TYPE_CURRENT
#    define SHGFP_TYPE_CURRENT 0
#  endif
#endif /* G_OS_WIN32 */

#include "glade_support.h"

IniSettingsFile *main_ini_file = NULL;

static guint preferences_lock_count;

static const char *const history_names[] = {
    N_("history-file1"),
    N_("history-file2"),
    N_("history-file3"),
    N_("history-file4"),
    N_("history-file5"),
    N_("history-file6"),
    N_("history-file7"),
    N_("history-file8"),
    N_("history-file9"),
    N_("history-file10"),
};

static const char *const history_nick_names[] = {
    N_("history:file1"),
    N_("history:file2"),
    N_("history:file3"),
    N_("history:file4"),
    N_("history:file5"),
    N_("history:file6"),
    N_("history:file7"),
    N_("history:file8"),
    N_("history:file9"),
    N_("history:file10"),
};

static const char *const history_format_names[] = {
    N_("history-fmt1"),
    N_("history-fmt2"),
    N_("history-fmt3"),
    N_("history-fmt4"),
    N_("history-fmt5"),
    N_("history-fmt6"),
    N_("history-fmt7"),
    N_("history-fmt8"),
    N_("history-fmt9"),
    N_("history-fmt10"),
};

static const char *const history_format_nicks[] = {
    N_("history:fmt1"),
    N_("history:fmt2"),
    N_("history:fmt3"),
    N_("history:fmt4"),
    N_("history:fmt5"),
    N_("history:fmt6"),
    N_("history:fmt7"),
    N_("history:fmt8"),
    N_("history:fmt9"),
    N_("history:fmt10"),
};

static char *alloc_ini_filename(void)
{
    char *result;
    const char *home;
    const int the_len = 14;
#ifdef G_OS_WIN32
    TCHAR home_buf[MAX_PATH];
#endif /* G_OS_WIN32 */

#if defined(G_OS_WIN32) && defined(HAVE_SHFOLDER_H)
    /* FIXME: doesn't work when UNICODE mode */
#  if 0
    home = SHGetSpecialFolderPath(NULL, home_buf, CSIDL_APPDATA, TRUE)
        ? home_buf : NULL;
#  endif
    (void)SHGetFolderPath(NULL, CSIDL_APPDATA, NULL, 0/*SHGFP_TYPE_DEFAULT*/, home_buf);
    home = home_buf;
#else
    home = getenv(N_("HOME"));
#endif /* G_OS_WIN32 */
    result = g_malloc((home?strlen(home):0)+the_len+1);
    if(home)
        strcpy(result, home);
    else
        *result = N_('\0');
    strcat(result, N_("/.gtk2edit.ini")); /* 14 characters - the_len */
    return result;
}

void preferences_init(void)
{
    gchar *file_name;
    int i;

    g_assert(!main_ini_file);

    file_name = alloc_ini_filename();
    main_ini_file = ini_settings_new(file_name, NULL, NULL); /* FIXME: Error callback */
    g_free(file_name);

    ini_settings_freeze_write(main_ini_file);

    ini_settings_install_property(main_ini_file,
                                  g_param_spec_boolean(N_("first-run"),
                                                       N_("main:first_run"),
                                                       _("Application is ran the first time"),
                                                       TRUE,
                                                       G_PARAM_READWRITE));
    ini_settings_install_property(main_ini_file,
                                  g_param_spec_uint(N_("always-ask-encoding"),
                                                       N_("encoding:always_ask"),
                                                       _("Ask encoding on file open: 0 - only if unspecified, 1 - for cmd line, 2 - always"),
                                                       0, 2,
                                                       0,
                                                       G_PARAM_READWRITE));
    ini_settings_install_property(main_ini_file,
                                  g_param_spec_boolean(N_("wrap-text"),
                                                       N_("main:wrap"),
                                                       _("Wrap text at window margin"),
                                                       TRUE,
                                                       G_PARAM_READWRITE));
    ini_settings_install_property(main_ini_file,
                                  g_param_spec_boolean(N_("autoindent"),
                                                       N_("main:autoindent"),
                                                       _("Autoindent on Enter key"),
                                                       TRUE,
                                                       G_PARAM_READWRITE));
    ini_settings_install_property(main_ini_file,
                                  g_param_spec_boolean(N_("strip-space-at-save"),
                                                       N_("space:strip_space_at_save"),
                                                       _("Strip space at ends of lines when saving"),
                                                       FALSE,
                                                       G_PARAM_READWRITE));
    ini_settings_install_property(main_ini_file,
                                  g_param_spec_uint(N_("max-undo-steps"),
                                                    N_("tech:max_undo_steps"),
                                                    _("Number of undo steps remembered"),
                                                    1,
                                                    10000,
                                                    100,
                                                    G_PARAM_READWRITE));
    ini_settings_install_property(main_ini_file,
                                  g_param_spec_string(N_("text-font"),
                                                      N_("fonts:text_font"),
                                                      _("Font used for text"),
                                                      NULL,
                                                      G_PARAM_READWRITE));

    ini_settings_install_property(main_ini_file,
                                  g_param_spec_string(N_("start-encoding"),
                                                      N_("encoding:name"),
                                                      _("Encoding with which it starts"),
                                                      NULL,
                                                      G_PARAM_READWRITE));
    ini_settings_install_property(main_ini_file,
                                  g_param_spec_boolean(N_("start-encoding-bom"),
                                                       N_("encoding:bom"),
                                                       _("Add BOM by default"),
                                                       TRUE,
                                                       G_PARAM_READWRITE));
    ini_settings_install_property(main_ini_file,
                                  g_param_spec_string(N_("start-encoding-term"),
                                                      N_("encoding:line_term"),
                                                      _("Default line terminator in HEX"),
#ifdef G_OS_WIN32
                                                      "\r\n",
#else
                                                      "\n",
#endif /* G_OS_WIN32 */
                                                      G_PARAM_READWRITE));

#ifdef LIMITED_VERSION
    ini_settings_install_property(main_ini_file,
                                  g_param_spec_string(N_("user-name"),
                                                      N_("user:name"),
                                                      _("User's registrations name"),
                                                      NULL,
                                                      G_PARAM_READWRITE));
    ini_settings_install_property(main_ini_file,
                                  g_param_spec_uint(N_("user-code"),
                                                    N_("user:code"),
                                                    _("User's registrations code"),
                                                    0,
                                                    UINT_MAX,
                                                    0,
                                                    G_PARAM_READWRITE));
#endif /* LIMITED_VERSION */
    ini_settings_install_property(main_ini_file,
                                  g_param_spec_uint(N_("history-count"),
                                                    N_("history:count"),
                                                    _("Files history count"),
                                                    0,
                                                    10,
                                                    0,
                                                    G_PARAM_READWRITE));
    for(i=0; i<10; ++i) {
        ini_settings_install_property(main_ini_file,
                                      g_param_spec_string(history_names[i],
                                                          history_nick_names[i],
                                                          N_("")/*NULL*/,
                                                          NULL,
                                                          G_PARAM_READWRITE));
        ini_settings_install_property(main_ini_file,
                                      g_param_spec_string(history_format_names[i],
                                                          history_format_nicks[i],
                                                          N_("")/*NULL*/,
                                                          NULL,
                                                          G_PARAM_READWRITE));
    }

    ini_settings_thaw_write_no_write(main_ini_file);

    ini_settings_load(main_ini_file);
}

void preferences_reload(void)
{
    ini_settings_load(main_ini_file);
}

gboolean preferences_get_boolean(const gchar *name)
{
    gboolean value = FALSE;

    preferences_reload();
    g_object_get(gtk_settings_get_default(), name, &value, NULL);
    return value;
}

void preferences_set_boolean(const gchar *name, gboolean value)
{
    g_object_set(G_OBJECT(gtk_settings_get_default()), name, value, NULL);
}

gboolean preferences_get_is_first_run(void)
{
    return preferences_get_boolean(N_("first-run"));
}

void preferences_set_is_first_run(gboolean value)
{
    preferences_set_boolean(N_("first-run"), value);
}

gboolean preferences_get_autoindent(void)
{
    return preferences_get_boolean(N_("autoindent"));
}

#ifdef LIMITED_VERSION
gchar *preferences_get_user_name(void)
{
    gchar *str;

    preferences_reload();
    str = NULL;
    g_object_get(gtk_settings_get_default(), N_("user-name"), &str, NULL);
    return str;
}

gint preferences_get_user_code(void)
{
    int count;

    preferences_reload();
    count = 0;
    g_object_get(gtk_settings_get_default(), N_("user-code"), &count, NULL);
    return count;
}
#endif /* LIMITED_VERSION */

gint preferences_history_count(void)
{
    int count;

    /*preferences_reload();*/
    count = 0;
    g_object_get(gtk_settings_get_default(), N_("history-count"), &count, NULL);
    return count;
}

gchar *preferences_history_get(int i)
{
    gchar *str;

    g_assert(i>=0 && i<10); /* Check for history count is not necessary. */

    preferences_reload();
    str = NULL;
    g_object_get(gtk_settings_get_default(), history_names[i], &str, NULL);
    return str;
}

gchar *preferences_history_get_format(int i)
{
    gchar *str;

    g_assert(i>=0 && i<10); /* Check for history count is not necessary. */

    preferences_reload();
    str = NULL;
    g_object_get(gtk_settings_get_default(), history_format_names[i], &str, NULL);
    return str;
}

static void preferences_history_raise_item(int index, const gchar *format_str)
{
    gchar *our_item, *tmp_name_str, *tmp_fmt_str;
    int i;

    our_item = g_strdup(preferences_history_get(index));
    for(i=index; i>0; --i) {
        g_object_get(gtk_settings_get_default(), history_names[i-1], &tmp_name_str, NULL);
        g_object_set(gtk_settings_get_default(), history_names[i], tmp_name_str, NULL);
        g_object_get(gtk_settings_get_default(), history_format_names[i-1], &tmp_fmt_str, NULL);
        g_object_set(gtk_settings_get_default(), history_format_names[i], tmp_fmt_str, NULL);
        g_free(tmp_name_str);
        g_free(tmp_fmt_str);
    }
    g_object_set(gtk_settings_get_default(), history_names[0], our_item, NULL);
    if(format_str)
        g_object_set(gtk_settings_get_default(), history_format_names[0], format_str, NULL);
    g_free(our_item);
}

void preferences_history_add(const gchar *file_name, const gchar *format_str)
{
    int count, old_count, i;
    gchar *item;

    ini_settings_freeze_write(main_ini_file);
    ini_settings_freeze_load (main_ini_file);
    old_count = count = preferences_history_count();
    for(i=0; i<count; ++i) {
        item = preferences_history_get(i);
        if(!item) { /* sanity check for broken INI file */
            count = i;
            break;
        }
        if( !strcmp(file_name, item) ) {
            preferences_history_raise_item(i, format_str);
            g_free(item);
            goto end;
        }
        g_free(item);
    }
    if(count == 10) count = 9;
    for(i=count; i>0; --i) {
        gchar *item_format;

        item        = preferences_history_get       (i-1);
        item_format = preferences_history_get_format(i-1);
        g_object_set(gtk_settings_get_default(), history_names       [i], item      , NULL);
        g_object_set(gtk_settings_get_default(), history_format_names[i], item_format, NULL);
        g_free(item);
        g_free(item_format);
    }
    g_object_set(gtk_settings_get_default(), history_names       [0], file_name , NULL);
    g_object_set(gtk_settings_get_default(), history_format_names[0], format_str, NULL);
    g_object_set(gtk_settings_get_default(), N_("history-count"), count+1, NULL);
    for(i=old_count-1; i>count; --i) { /* Cleanup of superfluous items. */
        g_object_set(gtk_settings_get_default(), history_names  [i-1], NULL, NULL);
        g_object_set(gtk_settings_get_default(), history_format_names[i-1], NULL, NULL);
    }
end:
    ini_settings_thaw_load (main_ini_file);
    ini_settings_thaw_write(main_ini_file);
}
