#include "document.h"

#include "main.h"
#include "glade_support.h"
#include "marshallers.h"

enum {
    PROP_DATA = 1,
    PROP_MODIFIED,
    PROP_FILE_NAME
};

struct FileDocumentClass_ {
    GObjectClass base_class;
    
    void (*format_changed)(FileDocument *document, gpointer format, gpointer user_data);
    void (*save_or_load)  (FileDocument *document, const gchar *file_name, gpointer format, gboolean save);
};

struct FileDocument_ {
    GObject base;

    gchar *file_name;
    GtkWidget *parent_window;
    GtkWidget *file_dialog;
    gpointer doc_data; /* for example GtkTextBuffer* */
    gpointer format;   /* for example encoding data */
    gboolean modified;

    file_document_format_copy_t    format_copy;
    file_document_format_destroy_t format_destroy;

    file_document_callback_t      do_clear;
    file_document_file_callback_t do_save, do_load;

    void (*report_error)(gpointer data, GError *error);
};

static void file_document_class_init(FileDocumentClass *klass);
static void file_document_init(FileDocument *doc);
static void file_document_finalize(GObject *obj);

static void file_document_set_property(GObject      *object,
                                       guint         prop_id,
                                       const GValue *value,
                                       GParamSpec   *pspec);
static void file_document_get_property(GObject      *object,
                                       guint         prop_id,
                                       GValue       *value,
                                       GParamSpec   *pspec);

GtkType file_document_get_type(void)
{
    static GType fd_type = 0;

    if(!fd_type) {
        static const GTypeInfo fd_info = {
            sizeof(FileDocumentClass),
            NULL,               /* base_init */
            NULL,               /* base_finalize */
            (GClassInitFunc)file_document_class_init,
            NULL,               /* class_finalize */
            NULL,               /* class_data */
            sizeof(FileDocument),
            0,              /* n_preallocs */
            (GInstanceInitFunc)file_document_init,
            NULL
        };
        fd_type = g_type_register_static(G_TYPE_OBJECT, N_("FileDocument"), &fd_info, 0);
    }
    return fd_type;
}

static void file_document_class_init(FileDocumentClass *klass)
{
    GObjectClass *gobject_class;

    gobject_class = G_OBJECT_CLASS(klass);

    gobject_class->finalize = file_document_finalize;
    gobject_class->set_property = file_document_set_property;
    gobject_class->get_property = file_document_get_property;

    g_object_class_install_property(gobject_class,
                                    PROP_DATA,
                                    g_param_spec_pointer(N_("user-data"),
                                                         _("User data"),
                                                         _("User data such as a text buffer"),
                                                         G_PARAM_READWRITE));
    g_object_class_install_property(gobject_class,
                                    PROP_MODIFIED,
                                    g_param_spec_boolean(N_("modified"),
                                                         _("Is modified"),
                                                         _("Data is modified and not saved"),
                                                         FALSE,
                                                         G_PARAM_READWRITE));
    g_object_class_install_property(gobject_class,              
                                    PROP_FILE_NAME,
                                    g_param_spec_string(N_("file-name"),
                                                         _("File name"),
                                                         _("File name"),
                                                         NULL,
                                                         G_PARAM_READWRITE));

    g_signal_new(N_("format-changed"),
                 TYPE_FILE_DOCUMENT,
                 G_SIGNAL_RUN_LAST,
                 G_STRUCT_OFFSET(FileDocumentClass, format_changed),
                 0, NULL,
                 g_cclosure_user_marshal_VOID__POINTER,
                 G_TYPE_NONE, 1, G_TYPE_POINTER);
    /* The following signal is used for adding files to recently accessed history.
       TODO: It should be not emitted on simple save or revert (needs API change). */
    g_signal_new(N_("save-or-load"),
                 TYPE_FILE_DOCUMENT,
                 G_SIGNAL_RUN_LAST,
                 G_STRUCT_OFFSET(FileDocumentClass, save_or_load),
                 0, NULL,
                 g_cclosure_user_marshal_VOID__STRING_POINTER_BOOLEAN,
                 G_TYPE_NONE, 3, G_TYPE_STRING, G_TYPE_POINTER, G_TYPE_BOOLEAN);

#if 0
    g_signal_new(N_("save"),
                 TYPE_FILE_DOCUMENT,
                 G_SIGNAL_RUN_FIRST|G_SIGNAL_ACTION,
                 G_STRUCT_OFFSET(FileDocumentClass, save),
                 0, NULL,
                 g_cclosure_user_marshal_VOID__VOID,
                 G_TYPE_NONE, 0);
    g_signal_new(N_("save-as"),
                 TYPE_FILE_DOCUMENT,
                 G_SIGNAL_RUN_FIRST|G_SIGNAL_ACTION,
                 G_STRUCT_OFFSET(FileDocumentClass, save_as),
                 0, NULL,
                 g_cclosure_user_marshal_VOID__VOID,
                 G_TYPE_NONE, 0);
    g_signal_new(N_("open"),
                 TYPE_FILE_DOCUMENT,
                 G_SIGNAL_RUN_FIRST|G_SIGNAL_ACTION,
                 G_STRUCT_OFFSET(FileDocumentClass, open),
                 0, NULL,
                 g_cclosure_user_marshal_VOID__VOID,
                 G_TYPE_NONE, 0);
    g_signal_new(N_("clear"),
                 TYPE_FILE_DOCUMENT,
                 G_SIGNAL_RUN_FIRST|G_SIGNAL_ACTION,
                 G_STRUCT_OFFSET(FileDocumentClass, clear),
                 0, NULL,
                 g_cclosure_user_marshal_VOID__VOID,
                 G_TYPE_NONE, 0);
#endif /* 0 */

#if 0
    g_signal_new(N_("do-clear"),
                 TYPE_FILE_DOCUMENT,
                 G_SIGNAL_RUN_FIRST,
                 G_STRUCT_OFFSET(FileDocumentClass, do_clear),
                 0, NULL,
                 g_cclosure_user_marshal_BOOLEAN__POINTER_POINTER,
                 G_TYPE_BOOLEAN,
                 2, G_TYPE_POINTER, G_TYPE_POINTER);
    g_signal_new(N_("do-save"),
                 TYPE_FILE_DOCUMENT,
                 G_SIGNAL_RUN_FIRST,
                 G_STRUCT_OFFSET(FileDocumentClass, do_save),
                 0, NULL,
                 g_cclosure_user_marshal_BOOLEAN__POINTER_STRING_POINTER,
                 G_TYPE_BOOLEAN,
                 3, G_TYPE_POINTER, G_TYPE_STRING, G_TYPE_POINTER);
    g_signal_new(N_("do-load"),
                 TYPE_FILE_DOCUMENT,
                 G_SIGNAL_RUN_FIRST,
                 G_STRUCT_OFFSET(FileDocumentClass, do_load),
                 0, NULL,
                 g_cclosure_user_marshal_BOOLEAN__POINTER_STRING_POINTER,
                 G_TYPE_BOOLEAN,
                 3, G_TYPE_POINTER, G_TYPE_STRING, G_TYPE_POINTER);
#endif /* 0 */
}

static void file_document_default_format_destroy(gpointer format)
{
    g_free(format);
}

static gpointer file_document_default_format_copy(gpointer format)
{
    g_assert(!format);
    
    return NULL;
}

static void file_document_init(FileDocument *doc)
{
    doc->file_name = NULL;
    doc->parent_window = NULL;
    doc->doc_data = NULL;
    doc->format   = NULL;
    doc->modified = FALSE;
    doc->file_dialog = gtk_file_selection_new(N_(""));
    doc->format_destroy = file_document_default_format_destroy;
    doc->format_copy    = file_document_default_format_copy;
}

FileDocument *file_document_new(GtkWidget *parent_window,
                                gpointer doc_data,
                                file_document_callback_t clear,
                                file_document_file_callback_t save,
                                file_document_file_callback_t load,
                                void (*report_error)(gpointer data, GError *error))
{
    FileDocument *doc;

    doc = FILE_DOCUMENT( g_object_new(TYPE_FILE_DOCUMENT, NULL) );
    file_document_init(doc);
    doc->parent_window = parent_window;
    doc->doc_data = doc_data;
    doc->do_clear = clear;
    doc->do_save  = save;
    doc->do_load  = load;
    doc->report_error = report_error;
    gtk_window_set_transient_for(GTK_WINDOW(doc->file_dialog), GTK_WINDOW(parent_window));
    return doc;
}

static void file_document_set_file_name(FileDocument *document, const gchar *name)
{
    /* Take in account that name may be NULL */
    if( document->file_name == name ||
        ( document->file_name != NULL && name != NULL &&
          !strcmp(document->file_name, name) )
    )
        return;
    g_free(document->file_name);
    document->file_name = g_strdup(name);
    g_object_notify(G_OBJECT(document), N_("file-name"));
}

void file_document_format_destroy(FileDocument *document, gpointer format)
{
    if(format) (*document->format_destroy)(format);
}

gpointer file_document_format_copy(const FileDocument *document, gpointer format)
{
    return format ? (*document->format_copy)(format) : NULL;
}

gboolean file_document_new_file(FileDocument *document, gpointer *format)
{
    gboolean res;
    GError *error = NULL;

    if( !file_document_check_save(document) )
        return TRUE;
    res = (*document->do_clear) (document->doc_data, format, &error);
    file_document_set_format(document, *format);
    file_document_set_file_name(document, NULL);
    file_document_set_modified(document, FALSE);
    if(error) (*document->report_error)(document->doc_data, error);
    return res;
}

gboolean file_document_do_save_file(FileDocument *document,
                                    const gchar *file_name,
                                    gpointer *format)
{
    gboolean res;
    GError *error = NULL;

    res = (*document->do_save) (document->doc_data, file_name, format, &error);
    if(res) {
        file_document_set_format(document, *format);
        file_document_set_file_name(document, file_name);
        file_document_set_modified(document, FALSE);
        g_signal_emit_by_name(document, N_("save-or-load"), file_name, *format, TRUE);
    }
    if(error) (*document->report_error)(document->doc_data, error);
    return res;
}

static
gboolean file_document_do_load_file_no_ask(FileDocument *document,
                                           const gchar *file_name,
                                           gpointer *format)
{
    gboolean res;
    GError *error = NULL;

    res = (*document->do_load) (document->doc_data, file_name, format, &error);
    if(res) {
        file_document_set_format(document, *format);
        g_signal_emit_by_name(document, N_("save-or-load"), file_name, *format, FALSE);
    } else {
        file_document_set_format(document, NULL);
        file_document_set_file_name(document, NULL);
        file_document_set_modified(document, FALSE);
        if(error) (*document->report_error)(document->doc_data, error);
        return FALSE;
    }
    file_document_set_file_name(document, file_name);
    file_document_set_modified(document, FALSE);
    return TRUE;
}

gboolean file_document_do_load_file(FileDocument *document,
                                    const gchar *file_name,
                                    gpointer *format)
{
    if( !file_document_check_save(document) )
        return FALSE;
    return file_document_do_load_file_no_ask(document, file_name, format);
}

gboolean file_document_save(FileDocument *document)
{
    return document->file_name ?
        file_document_do_save_file(document, document->file_name, &document->format) :
        file_document_save_as     (document, &document->format);
}

gboolean file_document_save_as(FileDocument *document, gpointer *format)
{
    gboolean res;

    gtk_window_set_title(GTK_WINDOW(document->file_dialog), _("Save as"));
    res = TRUE;
    if(gtk_dialog_run(GTK_DIALOG(document->file_dialog)) == GTK_RESPONSE_OK) {
        const gchar *file_name;

        file_name = gtk_file_selection_get_filename(GTK_FILE_SELECTION(document->file_dialog));
        res = file_document_do_save_file(document, file_name, format);
    }
    gtk_widget_hide(document->file_dialog);
    return res;
}

gboolean file_document_open(FileDocument *document, gpointer *format)
{
    gboolean res;

    gtk_window_set_title(GTK_WINDOW(document->file_dialog), _("Open file"));
#if 0
    if(document->file_name) {
        gchar *dir;

        dir = g_path_get_dirname(document->file_name);
        gtk_file_selection_set_filename(GTK_FILE_SELECTION(document->file_dialog), dir);
        g_free(dir);
    }
#endif /* 0 */
    res = TRUE;
    if(gtk_dialog_run(GTK_DIALOG(document->file_dialog)) == GTK_RESPONSE_OK) {
        const gchar *file_name;

        file_name = gtk_file_selection_get_filename(GTK_FILE_SELECTION(document->file_dialog));
        res = file_document_do_load_file(document, file_name, format);
    }
    gtk_widget_hide(document->file_dialog);
    return res;
}

gboolean file_document_revert(FileDocument *document)
{
    GtkWidget *dlg;
    gboolean res;

    g_assert(document->file_name);

    dlg = gtk_message_dialog_new(GTK_WINDOW(document->parent_window),
                                 GTK_DIALOG_DESTROY_WITH_PARENT,
                                 GTK_MESSAGE_QUESTION,
                                 GTK_BUTTONS_YES_NO,
                                 _("Are you sure to revert to old file?"));
    my_disable_window_resize(GTK_WINDOW(dlg));
    gtk_dialog_set_default_response(GTK_DIALOG(dlg), GTK_RESPONSE_NO);
    res = TRUE;
    if(gtk_dialog_run(GTK_DIALOG(dlg)) == GTK_RESPONSE_YES) {
        res = file_document_do_load_file_no_ask(document, document->file_name, &document->format);
        if(res) file_document_set_modified(document, FALSE);
    }
    gtk_widget_destroy(dlg);

    return res;
}

gboolean file_document_check_save(FileDocument *document)
{
    GtkWidget *dialog;
    gint response;

    if(!document->modified) return TRUE;

    dialog = gtk_message_dialog_new(GTK_WINDOW(document->parent_window),
                                    GTK_DIALOG_DESTROY_WITH_PARENT,
                                    GTK_MESSAGE_QUESTION,
                                    GTK_BUTTONS_NONE,
                                    _("Unsaved changes. Save?"));
    my_disable_window_resize(GTK_WINDOW(dialog));
    gtk_dialog_add_buttons(GTK_DIALOG(dialog),
                           GTK_STOCK_NO,     GTK_RESPONSE_NO,
                           GTK_STOCK_YES,    GTK_RESPONSE_YES,
                           GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                           NULL);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_YES);
    response = gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog);
    switch(response) {
        case GTK_RESPONSE_YES:
            if(!file_document_save(document)) {
                GtkWidget *save_error_dlg;
                gint save_error_response;

                save_error_dlg = gtk_message_dialog_new(
                        GTK_WINDOW(document->parent_window),
                        GTK_DIALOG_DESTROY_WITH_PARENT,
                        GTK_MESSAGE_QUESTION,
                        GTK_BUTTONS_NONE,
                        _("Error trying to save file.\nContinue (losing the file content)?"));
                my_disable_window_resize(GTK_WINDOW(save_error_dlg));
                gtk_dialog_add_buttons(
                        GTK_DIALOG(save_error_dlg),
                        GTK_STOCK_STOP/*GTK_STOCK_NO*/, GTK_RESPONSE_CANCEL,
                        GTK_STOCK_YES,                  GTK_RESPONSE_ACCEPT,
                        NULL);
                gtk_dialog_set_default_response(GTK_DIALOG(save_error_dlg),
                                                GTK_RESPONSE_CANCEL);
                save_error_response = gtk_dialog_run(GTK_DIALOG(save_error_dlg));
                gtk_widget_destroy(save_error_dlg);
                return save_error_response == GTK_RESPONSE_ACCEPT;
            }
            break;
        case GTK_RESPONSE_NO:
            break;
        default:
            return FALSE;
    }
    return TRUE;
}

const gchar *file_document_get_file_name(FileDocument *document)
{
    return document->file_name;
}

gboolean file_document_get_modified(FileDocument *document)
{
    return document->modified;
}

void file_document_set_modified(FileDocument *document, gboolean modified)
{
    document->modified = modified;
    g_object_notify(G_OBJECT(document), N_("modified"));
}

gpointer file_document_get_data(FileDocument *document)
{
    return document->doc_data;
}

void file_document_set_data(FileDocument *document, gpointer data)
{
    document->doc_data = data;
    g_object_notify(G_OBJECT(document), N_("user-data"));
}

static void file_document_set_property(GObject      *object,
                                       guint         prop_id,
                                       const GValue *value,
                                       GParamSpec   *pspec)
{
    switch(prop_id) {
        case PROP_DATA:
            FILE_DOCUMENT(object)->doc_data = g_value_get_pointer(value);
            break;
        case PROP_MODIFIED:
            FILE_DOCUMENT(object)->modified = g_value_get_boolean(value);
            break;
        case PROP_FILE_NAME:
            g_free(FILE_DOCUMENT(object)->file_name);
            FILE_DOCUMENT(object)->file_name = (gchar*)g_value_get_string(value);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
            break;
    }
}

static void file_document_get_property(GObject      *object,
                                       guint         prop_id,
                                       GValue       *value,
                                       GParamSpec   *pspec)
{
    switch(prop_id) {
        case PROP_DATA:
            g_value_set_pointer(value, FILE_DOCUMENT(object)->doc_data);
            break;
        case PROP_MODIFIED:
            g_value_set_boolean(value, FILE_DOCUMENT(object)->modified);
            break;
        case PROP_FILE_NAME:
            g_value_set_string(value, FILE_DOCUMENT(object)->file_name);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

static void file_document_finalize(GObject *obj)
{
    FileDocument *document;

    document = FILE_DOCUMENT(obj);
    g_free(document->file_name);
    file_document_format_destroy(document, document->format);
    gtk_widget_destroy(document->file_dialog);
}

gpointer *file_document_get_format(FileDocument *document)
{
    return &document->format;
}

gpointer file_document_get_format_copy(FileDocument *document)
{
    return file_document_format_copy(document, document->format);
}

void file_document_set_format(FileDocument *document, gpointer format)
{
    if(format != document->format) {
        if(document->format) (*document->format_destroy)(document->format);
        document->format = file_document_format_copy(document, format);
        g_signal_emit_by_name(document, N_("format-changed"), document->format);
    }
}

void file_document_set_format_operations(FileDocument *document,
                                         file_document_format_destroy_t destroy_func,
                                         file_document_format_copy_t    copy_func)
{
    document->format_destroy = destroy_func ? destroy_func : file_document_default_format_destroy;
    document->format_copy    = copy_func    ? copy_func    : file_document_default_format_copy;
}
