/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "bindprops.h"

#include <string.h>

/* TODO: Make it faster by usage of quarks. */

typedef struct MyGLibBindData_ {
    GObject    *src_obj, *dst_obj;
    GParamSpec *dst_pspec;
    gboolean    strict_validation;
    gulong      handler;
} MyGLibBindData;

static void my_glib_bind_properties_transfer(GObject    *src_obj,
                                             GParamSpec *src_pspec,
                                             GObject    *dst_obj,
                                             GParamSpec *dst_pspec,
                                             gboolean strict_validation)
{
    GValue src_value, dst_value, old_dst_value;
    const gchar *src_name, *dst_name;

    src_name = g_param_spec_get_name(src_pspec);
    dst_name = g_param_spec_get_name(dst_pspec);

    memset(&src_value, 0, sizeof(GValue));
    memset(&dst_value, 0, sizeof(GValue));
    memset(&old_dst_value, 0, sizeof(GValue));
    g_value_init(&src_value, G_PARAM_SPEC_VALUE_TYPE(src_pspec));
    g_value_init(&dst_value, G_PARAM_SPEC_VALUE_TYPE(dst_pspec));
    g_value_init(&old_dst_value, G_PARAM_SPEC_VALUE_TYPE(dst_pspec));

    g_object_get_property(src_obj, src_name, &src_value);
    g_object_get_property(dst_obj, dst_name, &old_dst_value);
    g_param_value_convert(dst_pspec, &src_value, &dst_value, strict_validation);

    g_value_unset(&src_value);

    if(g_param_values_cmp(dst_pspec,
                          &dst_value,
                          &old_dst_value)
    )
        g_object_set_property(dst_obj, dst_name, &dst_value);

    g_value_unset(&dst_value);
    g_value_unset(&old_dst_value);
}

static void my_glib_bind_properties_cb(GObject    *src_obj,
                                       GParamSpec *src_pspec,
                                       gpointer    data)
{
    const MyGLibBindData *bind_data;

    bind_data = (MyGLibBindData*)data;
    g_signal_handlers_block_by_func  (src_obj, my_glib_bind_properties_cb, data);
    my_glib_bind_properties_transfer(src_obj           , src_pspec,
                                     bind_data->dst_obj, bind_data->dst_pspec,
                                     bind_data->strict_validation);
    g_signal_handlers_unblock_by_func(src_obj, my_glib_bind_properties_cb, data);
}

static void disconnect_bind_signal(gpointer data, GObject *where_the_object_was)
{
    MyGLibBindData *bind_data = (MyGLibBindData*)data;

    bind_data->dst_obj = NULL;
    g_signal_handler_disconnect(bind_data->src_obj, bind_data->handler);
}

static void my_glib_bind_free_bind_data(gpointer data, GClosure *closure)
{
    MyGLibBindData *bind_data;

    bind_data = (MyGLibBindData*)data;
    if(bind_data->dst_obj)
        g_object_weak_unref(bind_data->dst_obj, disconnect_bind_signal, bind_data);
    g_free(bind_data);
}

void my_glib_bind_properties2_one_way(GObject    *src_obj,
                                      const char *src_prop,
                                      GObject    *dst_obj,
                                      const char *dst_prop,
                                      gboolean    strict_validation)
{
    gchar *signal_name;
    MyGLibBindData *bind_data;
    GParamSpec *src_pspec, *dst_pspec;
    gulong signal;

    src_pspec = g_object_class_find_property(G_OBJECT_GET_CLASS(src_obj), src_prop);
    dst_pspec = g_object_class_find_property(G_OBJECT_GET_CLASS(dst_obj), dst_prop);

    bind_data = g_malloc(sizeof(MyGLibBindData));
    bind_data->src_obj = src_obj;
    bind_data->dst_obj = dst_obj;
    bind_data->dst_pspec = dst_pspec;
    bind_data->strict_validation = strict_validation;

    my_glib_bind_properties_transfer(src_obj, src_pspec,
                                     dst_obj, dst_pspec,
                                     strict_validation);
    signal_name = g_strconcat("notify::", src_prop, NULL);
    bind_data->handler = g_signal_connect_data(src_obj,
                                               signal_name,
                                               G_CALLBACK(my_glib_bind_properties_cb),
                                               bind_data,
                                               my_glib_bind_free_bind_data,
                                               0/*G_CONNECT_AFTER*/);
    g_free(signal_name);
    g_object_weak_ref(dst_obj, disconnect_bind_signal, bind_data);
}

void my_glib_bind_properties_ring(MyGLibBindProperties data[],
                                  guint                number,
                                  gboolean             strict_validation)
{
    gint i;

    for(i=0; i<(gint)number-1; ++i)
        my_glib_bind_properties2_one_way(data[i  ].object, data[i  ].property,
                                         data[i+1].object, data[i+1].property,
                                         strict_validation);
    /* TODO: Desirable to block data[0] (for less work) */
    my_glib_bind_properties2_one_way(data[number-1].object, data[number-1].property,
                                         data[0   ].object, data[0       ].property,
                                         strict_validation);
}

void my_glib_bind_properties2(GObject    *obj1,
                              const char *prop1,
                              GObject    *obj2,
                              const char *prop2,
                              gboolean    strict_validation)
{
    my_glib_bind_properties2_one_way(obj1, prop1, obj2, prop2, strict_validation);
    my_glib_bind_properties2_one_way(obj2, prop2, obj1, prop1, strict_validation);
}

void my_glib_bind_properties(GObject              *obj,
                             const char           *prop,
                             MyGLibBindProperties  data[],
                             guint                 number,
                             gboolean              strict_validation)
{
    gint i;

    g_object_freeze_notify(obj);
    for(i=0; i<(gint)number; ++i)
        my_glib_bind_properties2(obj,            prop,
                                 data[i].object, data[i].property,
                                 strict_validation);
    g_object_thaw_notify(obj);
}

void my_glib_bind_properties_one_way(GObject              *src_obj,
                                     const char           *src_prop,
                                     MyGLibBindProperties  dst_data[],
                                     guint                 dst_number,
                                     gboolean              strict_validation)
{
    gint i;

    for(i=0; i<(gint)dst_number; ++i)
        my_glib_bind_properties2_one_way(src_obj,            src_prop,
                                         dst_data[i].object, dst_data[i].property,
                                         strict_validation);
}
