# Copyright (C) 2012 Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import sys
import imp
import time
import gettext
import logging
from threading import Thread
from datetime import datetime
from os.path import dirname, join, isdir, isfile, exists

import dbus
import gobject

from sugar.env import get_profile_path
from sugar_network.toolkit import Option


plugins = Option(
        'space separated list of plugins to enable',
        type_cast=Option.list_cast, type_repr=Option.list_repr,
        default=[], name='plugins')

blacklisted_deviceicon = Option(
        'space separated list of python files, from deviceicon directory, '
        'to disable corresponding device icons',
        type_cast=Option.list_cast, type_repr=Option.list_repr,
        default=[], name='deviceicon')

blacklisted_cpsection = Option(
        'space separated list of directory names, from cpsection directory, '
        'to disable corresponding Control Panel sections',
        type_cast=Option.list_cast, type_repr=Option.list_repr,
        default=[], name='cpsection')

blacklisted_globalkey = Option(
        'space separated list of python files, from globalkey directory, '
        'to disable corresponding global keys',
        type_cast=Option.list_cast, type_repr=Option.list_repr,
        default=[], name='globalkey')

modules = []

_logger = logging.getLogger('plugins')
_enabled_modules = []


def init():
    all_modules = []

    for plugins_srcroot in [
            join(get_profile_path(), 'plugins'),
            dirname(__file__),
            ]:
        if exists(plugins_srcroot):
            break

    for name in os.listdir(plugins_srcroot):
        path = join(plugins_srcroot, name)
        if not isdir(path):
            continue
        locale_path = join(path, 'locale')
        if exists(locale_path):
            gettext.bindtextdomain('sugar-plugin-%s' % name, locale_path)
        fp, modir, desc = imp.find_module(name, [plugins_srcroot])
        try:
            mod = imp.load_module('jarabe.plugins.%s' % name, fp, modir, desc)
        except Exception, error:
            _logger.warning('Cannot import %r plugin: %s', path, error)
            continue
        if set(['ORDER', 'init', 'start']) - set(dir(mod)):
            _logger.warning('Skip misconfigured %r plugin', path)
            continue
        _logger.info('Found %r plugin in %r', name, path)
        mod.name = name
        if hasattr(mod, 'TITLE'):
            modules.append(mod)
        Option.seek('sugar-plugin-%s' % name, mod)
        all_modules.append(mod)

    Option.seek('shell', [plugins])
    Option.seek('blacklist', [
        blacklisted_deviceicon,
        blacklisted_cpsection,
        blacklisted_globalkey,
        ])
    # Load options only after importing all plugins to let them expose options
    Option.load([
        '/etc/sweets.d',
        '/etc/sweets.conf',
        '~/.config/sweets/config',
        join(get_profile_path(), 'sweets.conf'),
        ])

    for mod in all_modules:
        if not hasattr(mod, 'TITLE') or mod.name in plugins.value:
            _enabled_modules.append(mod)
        else:
            _logger.info('Plugin %r disabled', mod.name)

    modules.sort(lambda x, y: cmp(x.ORDER, y.ORDER))
    _enabled_modules.sort(lambda x, y: cmp(x.ORDER, y.ORDER))

    binding = []
    for mod in _enabled_modules:
        if hasattr(mod, 'binding'):
            binding.extend(mod.binding())
    binding = '\n'.join(binding)

    binding_path = join(get_profile_path(), 'plugins.binding')
    if exists(binding_path) != bool(binding) or \
            binding and file(binding_path).read() != binding:
        if binding:
            with file(binding_path, 'w') as f:
                f.write(binding)
        elif exists(binding_path):
            os.unlink(binding_path)
        _logger.info('Restart shell to source newly create bindings')
        file(join(get_profile_path(), 'restart'), 'w').close()
        exit(0)

    for mod in _enabled_modules:
        _logger.info('Initialize %r plugin', mod.name)
        mod.init()


def start():
    for mod in _enabled_modules:
        _logger.info('Start %r plugin', mod.name)
        mod.start()


def get(name):
    for mod in _enabled_modules:
        if mod.name == name:
            return mod


class Schedule(object):

    def __init__(self, name, timeout, callback):
        self._name = name
        self._timeout = timeout
        self._callback = callback
        self._schedule = None
        self._checkpoint_path = \
                join(get_profile_path(), 'plugins-%s.checkpoint' % name)

        if not exists(self._checkpoint_path):
            file(self._checkpoint_path, 'w').close()
        _logger.debug('Register %r schedule', name)

    def start(self):
        if self._schedule:
            return
        _logger.debug('Start scheduling %r', self._name)
        self._schedule = True
        gobject.idle_add(self._schedule_cb)

    def stop(self):
        if not self._schedule:
            return
        _logger.debug('Stop scheduling %r', self._name)
        if self._schedule is not True:
            gobject.source_remove(self._schedule)
        self._schedule = None

    def _schedule_cb(self):
        checkpoint = int(os.stat(self._checkpoint_path).st_mtime)
        ts = int(time.time())

        if checkpoint > ts:
            timeout = max(1, checkpoint - ts)
            _logger.debug('Checkpoint for %r schedule expires in %s second(s)',
                    self._name, timeout)
            self._schedule = gobject.timeout_add_seconds(
                    timeout, self._schedule_cb)
            return

        _logger.debug('Trigger %r schedule', self._name)

        thread = Thread(target=self._schedule_thread)
        thread.daemon = True
        thread.start()

    def _schedule_thread(self):
        try:
            self._callback()
        except Exception:
            _logger.exception('Failed to process %r', self._name)

        checkpoint = int(time.time()) + self._timeout
        os.utime(self._checkpoint_path, (checkpoint, checkpoint))

        _logger.debug('New checkpoint for %r schedule will be at %s',
                self._name, datetime.fromtimestamp(checkpoint))

        timeout = max(1, checkpoint - int(time.time()))
        self._schedule = gobject.timeout_add_seconds(
                timeout, self._schedule_cb)
