# Copyright (C) 2012 Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import sys
import imp
import time
import gettext
import logging
from datetime import datetime
from os.path import dirname, join, isdir, isfile, exists

import dbus
import gobject

from sugar.env import get_profile_path
from sugar_network.toolkit import Option


plugins = Option(
        'space separated list of plugins to enable',
        type_cast=Option.list_cast, type_repr=Option.list_repr,
        default=[], name='plugins')

blacklisted_deviceicon = Option(
        'space separated list of python files, from deviceicon directory, '
        'to disable corresponding device icons',
        type_cast=Option.list_cast, type_repr=Option.list_repr,
        default=[], name='deviceicon')

blacklisted_cpsection = Option(
        'space separated list of directory names, from cpsection directory, '
        'to disable corresponding Control Panel sections',
        type_cast=Option.list_cast, type_repr=Option.list_repr,
        default=[], name='cpsection')

blacklisted_globalkey = Option(
        'space separated list of python files, from globalkey directory, '
        'to disable corresponding global keys',
        type_cast=Option.list_cast, type_repr=Option.list_repr,
        default=[], name='globalkey')

modules = []

_SCHEDULE_DELAY = 60
_NM_STATE_CONNECTED = [3, 70]
_NM_STATE_DISCONNECTED = [4]

_logger = logging.getLogger('plugins')
_enabled_modules = []
_schedules = []


def init():
    all_modules = []

    for plugins_srcroot in [
            join(get_profile_path(), 'plugins'),
            dirname(__file__),
            ]:
        if exists(plugins_srcroot):
            break

    for name in os.listdir(plugins_srcroot):
        path = join(plugins_srcroot, name)
        if not isdir(path):
            continue
        locale_path = join(path, 'locale')
        if exists(locale_path):
            gettext.bindtextdomain('sugar-plugin-%s' % name, locale_path)
        fp, modir, desc = imp.find_module(name, [plugins_srcroot])
        try:
            mod = imp.load_module('jarabe.plugins.%s' % name, fp, modir, desc)
        except Exception, error:
            _logger.warning('Cannot import %r plugin: %s', path, error)
            continue
        if set(['ORDER', 'init', 'start']) - set(dir(mod)):
            _logger.warning('Skip misconfigured %r plugin', path)
            continue
        _logger.info('Found %r plugin in %r', name, path)
        mod.name = name
        if hasattr(mod, 'TITLE'):
            modules.append(mod)
        Option.seek('sugar-plugin-%s' % name, mod)
        all_modules.append(mod)

    Option.seek('shell', [plugins])
    Option.seek('blacklist', [
        blacklisted_deviceicon,
        blacklisted_cpsection,
        blacklisted_globalkey,
        ])
    # Load options only after importing all plugins to let them expose options
    Option.load([
        '/etc/sweets.d',
        '/etc/sweets.conf',
        '~/.config/sweets/config',
        join(get_profile_path(), 'sweets.conf'),
        ])

    for mod in all_modules:
        if not hasattr(mod, 'TITLE') or mod.name in plugins.value:
            _enabled_modules.append(mod)
        else:
            _logger.info('Plugin %r does not contain CP sub-section', mod.name)

    modules.sort(lambda x, y: cmp(x.ORDER, y.ORDER))
    _enabled_modules.sort(lambda x, y: cmp(x.ORDER, y.ORDER))

    binding = []
    for mod in _enabled_modules:
        if hasattr(mod, 'binding'):
            binding.extend(mod.binding())
    binding = '\n'.join(binding)

    binding_path = join(get_profile_path(), 'plugins.binding')
    if exists(binding_path) != bool(binding) or \
            binding and file(binding_path).read() != binding:
        if binding:
            with file(binding_path, 'w') as f:
                f.write(binding)
        elif exists(binding_path):
            os.unlink(binding_path)
        _logger.info('Restart shell to source newly create bindings')
        file(join(get_profile_path(), 'restart'), 'w').close()
        exit(0)

    for mod in _enabled_modules:
        _logger.info('Initialize %r plugin', mod.name)
        mod.init()
        if hasattr(mod, 'online_schedule'):
            _schedules.append(_Schedule(mod))


def start():
    for mod in _enabled_modules:
        _logger.info('Start %r plugin', mod.name)
        mod.start()
    if _schedules:
        # Do not overload system on startup by running all schedules
        gobject.timeout_add_seconds(_SCHEDULE_DELAY, _connection_manager)


def _connection_manager():
    current_state = [False]

    def connected():
        if current_state[0]:
            return
        current_state[0] = True
        for i in _schedules:
            i.start()

    def disconnected():
        if not current_state[0]:
            return
        current_state[0] = False
        for i in _schedules:
            i.stop()

    def NetworkManagerStateChanged_cb(state):
        if state in _NM_STATE_CONNECTED:
            connected()
        elif state in _NM_STATE_DISCONNECTED:
            disconnected()

    try:
        bus = dbus.SystemBus()
        obj = bus.get_object('org.freedesktop.NetworkManager',
                '/org/freedesktop/NetworkManager')
        props = dbus.Interface(obj, dbus.PROPERTIES_IFACE)
        state = props.Get('org.freedesktop.NetworkManager', 'State')
        if state in _NM_STATE_CONNECTED:
            connected()
        bus.add_signal_receiver(NetworkManagerStateChanged_cb,
                'StateChanged', 'org.freedesktop.NetworkManager')
    except dbus.DBusException:
        _logger.exception('Cannot connect to NetworkManager')
        if _is_online():
            connected()


def _is_online():
    with file('/proc/net/route') as route:
        for line in route.readlines():
            try:
                if int(line.split('\t')[1], 16) == 0:
                    return True
            except ValueError:
                pass
    return False


class _Schedule(object):

    def __init__(self, mod, fallback_delay=0):
        self._mod = mod
        self._started = None

        self._checkpoint_path = \
                join(get_profile_path(), 'plugins-%s.checkpoint' % mod.name)
        if not exists(self._checkpoint_path):
            file(self._checkpoint_path, 'w').close()

        _logger.debug('Register %r schedule', mod.name)

    def start(self):
        if self._started:
            return
        _logger.debug('Start scheduling %r', self._mod.name)
        self._started = True
        gobject.idle_add(self._schedule)

    def stop(self):
        if not self._started:
            return
        _logger.debug('Stop scheduling %r', self._mod.name)
        if self._started is not True:
            gobject.source_remove(self._started)
        self._started = None

    def _schedule(self):
        checkpoint = int(os.stat(self._checkpoint_path).st_mtime)
        ts = int(time.time())

        if checkpoint > ts:
            timeout = max(1, checkpoint - ts)
            _logger.debug('Checkpoint for %r schedule expires in %s second(s)',
                    self._mod.name, timeout)
            gobject.timeout_add_seconds(timeout, self._schedule)
            return

        _logger.debug('Trigger %r schedule', self._mod.name)
        try:
            timeout = self._mod.online_schedule()
        except Exception:
            _logger.exception('Failed to trigger %r, abort schedule',
                    self._mod.name)
            return

        if not timeout:
            _logger.exception('Finish scheduling %r', self._mod.name)
            return

        checkpoint = ts + timeout
        os.utime(self._checkpoint_path, (checkpoint, checkpoint))
        _logger.debug('New checkpoint for %r schedule will be at %s',
                self._mod.name, datetime.fromtimestamp(checkpoint))
        timeout = max(1, checkpoint - int(time.time()))
        gobject.timeout_add_seconds(timeout, self._schedule)
