# Copyright (C) 2012 Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import json
import time
import logging
import subprocess
from os.path import abspath, dirname
from gettext import dgettext

import gtk
import dbus
import gobject
from sugar.graphics.alert import NotifyAlert, ErrorAlert, ConfirmationAlert
from sugar_network import client
from sugar_network.toolkit import Option

from jarabe import plugins


_ = lambda x: dgettext('sugar-plugin-sn', x)

ORDER = 5
TITLE = _('Sugar Network integration')

SN_BROWSER_NAME = 'sugar-network-browser'

_ALERT_SEVERITIES = {
        # severity: (alert_class, alert_message)
        'error': (ErrorAlert, _('Sugar Network error')),
        None: (NotifyAlert, _('Sugar Network')),
        }

_logger = logging.getLogger('plugins.sn')
_bundleregistry = None
_launcher = None
_connection = None
_browser = None

Option.seek('client', [client.discover_server, client.server_mode])


def init():
    import jarabe.model.bundleregistry
    from .bundleregistry import BundleRegistry

    def get_registry():
        global _bundleregistry
        if _bundleregistry is None:
            _bundleregistry = BundleRegistry()
        return _bundleregistry

    jarabe.model.bundleregistry.get_registry = get_registry

    from jarabe.model.shell import Activity
    from jarabe.view import palettes
    from .browser import Palette

    def get_bundle_path(self):
        # pylint: disable-msg=W0212
        if not self._windows:
            return None

        pid = self._windows[0].get_pid()
        with file('/proc/%d/environ' % pid) as f:
            for env in f.read().split('\0'):
                if env.startswith('SUGAR_BUNDLE_PATH='):
                    return env.split('=', 1)[-1]

    Activity.get_bundle_path = get_bundle_path
    palettes.predefined[SN_BROWSER_NAME] = Palette


def start():
    global _launcher

    from jarabe.journal import misc
    from .launcher import Launcher

    _launcher = Launcher()
    misc.launch = _launcher.launch

    from jarabe.model.shell import get_model
    from jarabe.model.bundleregistry import get_registry

    def delayed_start():
        connection().post([], {'event': 'delayed-start'}, cmd='broadcast')
        get_browser()

    def activity_added_cb(model, activity):
        if activity.is_journal():
            shell.disconnect_by_func(activity_added_cb)
            gobject.idle_add(delayed_start)

    shell = get_model()
    shell.connect('activity-added', activity_added_cb)


def control_panel_section():
    section = gtk.VBox()

    server_box = gtk.VBox()
    section.pack_start(server_box)
    master_button = gtk.RadioButton()
    master_button.props.label = _('Connect to master server')
    server_box.pack_start(master_button, expand=False)
    discover_button = gtk.RadioButton(master_button)
    discover_button.props.label = _('Auto discover server in local network')
    server_box.pack_start(discover_button, expand=False)

    if client.discover_server.value:
        discover_button.props.active = True
    else:
        master_button.props.active = True

    server_mode_button = gtk.CheckButton()
    server_mode_button.props.label = _('Behave as a Sugar Network server '
            'providing data from mounted device')
    server_mode_button.props.active = client.server_mode.value
    section.pack_start(server_mode_button, expand=False)

    def master_toggled_cb(button, value):
        client.discover_server.value = value
        Option.save()

    master_button.connect('toggled', master_toggled_cb, False)
    discover_button.connect('toggled', master_toggled_cb, True)

    def server_mode_toggled_cb(button):
        client.server_mode.value = button.props.active
        Option.save()

    server_mode_button.connect('toggled', server_mode_toggled_cb)

    section.show_all()
    return section


def connection():
    global _connection

    if _connection is None:
        ts = time.time()
        subprocess.check_call([
            'sugar-network-client', 'start',
            '--webui',
            '--delayed-start',
            '--replace',
            ])
        _logger.debug('Starting sugar-network-service took %s seconds',
                time.time() - ts)
        _connection = _Connection()

    return _connection


def get_browser():
    global _browser

    if _browser is None:
        from .browser import Browser
        _browser = Browser()

    return _browser


def add_alert(cls, callbacks=None, **kwargs):
    window = get_browser()

    def response_cb(alert, response_id):
        window.remove_alert(alert)
        if callbacks:
            cb = callbacks.get(response_id)
            if cb is not None:
                cb()
        del alert

    if isinstance(cls, basestring):
        cls, title = _ALERT_SEVERITIES.get(cls) or _ALERT_SEVERITIES[None]
        if 'title' not in kwargs:
            kwargs['title'] = title
    alert = cls(**kwargs)
    alert.connect('response', response_cb)

    window.add_alert(alert)
    window.reveal()


class _Connection(gobject.GObject):

    __gsignals__ = {
        'event': (
            gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE,
            [str, gobject.TYPE_PYOBJECT]),
        }

    def __init__(self):
        gobject.GObject.__init__(self)
        self._connection = client.IPCConnection()
        self._subscription = self._connection.subscribe()
        gobject.io_add_watch(self._subscription.fileno(),
                gobject.IO_IN | gobject.IO_HUP, self.__subscription_cb)

    def __getattr__(self, name):
        return getattr(self._connection, name)

    def __subscription_cb(self, source, cb_condition):
        try:
            event = self._subscription.pull()
            if event is None:
                return True
            event_type = event['event']
            if event_type == 'alert':
                add_alert(event['severity'], msg=event['message'])
            elif event_type == 'sync_complete':
                add_alert('info', msg=_('Synchronization completed'))
            elif event_type == 'show_journal':
                from jarabe.journal.journalactivity import get_journal
                if get_journal().show_object(event['uid']):
                    get_journal().reveal()
            elif event_type == 'failure' and event.get('cmd') == 'clone':
                from jarabe.plugins.sn.report import submit
                if event.get('exception') == 'ServiceUnavailable':
                    alert = ErrorAlert
                    msg = _('To proceed further, connect to Sugar Network.')
                else:
                    alert = ConfirmationAlert
                    msg = _('Submit failure report to help developers ' \
                            'track the issue you just encountered?')
                add_alert(alert, msg=msg,
                        title=_('Failed to clone from Sugar Network'),
                        callbacks={gtk.RESPONSE_OK: lambda: submit(event)})
            self.emit('event', event_type, event)
        except Exception:
            _logger.exception('Cannot dispatch %r event', event)
        return True


class _ReportAlert(ErrorAlert):

    def __init__(self):
        icon = Icon(icon_name='transfer-to')
        self.add_button(gtk.RESPONSE_APPLY, _('Submit report'), icon, 0)
        icon.show()
