/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CTrimmingDlg - トリミングダイアログ
*/


#include <stdlib.h>

#include "CTrimmingDlg.h"

#include "CImageRGB16.h"

#include "AXImage.h"
#include "AXLayout.h"
#include "AXLabel.h"
#include "AXLineEdit.h"
#include "AXButton.h"
#include "AXCheckButton.h"
#include "AXScrollView.h"
#include "AXScrollArea.h"
#include "AXScrollBar.h"
#include "AXApp.h"
#include "AXUtil.h"

#include "draw_calc.h"

#include "drawdat.h"
#include "global.h"
#include "strid.h"


//--------------------------

enum
{
    WID_EDIT_X = 100,
    WID_EDIT_Y,
    WID_EDIT_W,
    WID_EDIT_H,
    WID_BT_LEFTTOP,
    WID_BT_RIGHTBOTTOM,
    WID_AREA,
    WID_SCALE_TOP = 200,
};

enum
{
    STRID_TITLE,
    STRID_WHOLE,
    STRID_LEFTTOP,
    STRID_RIGHTBOTTOM,
    STRID_HELP
};

//--------------------------

class CTrimming_area:public AXScrollArea
{
    AXImage         m_img;
    DRAWCANVASINFO  m_dinfo;
    CANVASVIEWPARAM m_vparam;

    int     m_nScale,
            m_fDrag,
            m_nSide;
    AXRectSize  m_rcsBox;
    AXRect      m_rcDrag;
    AXPoint     m_ptTmp;

    enum
    {
        DRAGF_AREA = 1,
        DRAGF_SCROLL,
        DRAGF_MOVESIDE
    };

    void _viewToimg(int *pX,int *pY,int x,int y);
    void _imgToview(int *pX,int *pY,int x,int y);
    int _getNearSide(int x,int y);
    void _drawAreaBox();
    void _drawDragBox();

public:
    CTrimming_area(AXWindow *pParent);

    AXRectSize *getBoxRect() { return &m_rcsBox; }

    void setScale(int scale);
    void setScrollInfo();
    void scrollPoint(BOOL bBottom);
    void changeBoxVal(int type,int val);
    void draw();

    virtual BOOL onPaint(AXHD_PAINT *phd);
    virtual BOOL onSize();
    virtual BOOL onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam);
    virtual BOOL onButtonDown(AXHD_MOUSE *phd);
    virtual BOOL onButtonUp(AXHD_MOUSE *phd);
    virtual BOOL onMouseMove(AXHD_MOUSE *phd);
};

//--------------------------



//**********************************
// CTrimmingDlg
//**********************************


CTrimmingDlg::CTrimmingDlg(AXWindow *pOwner,AXRectSize *prcs)
    : AXDialog(pOwner, WS_DIALOG_NORMAL | WS_BK_FACE)
{
    AXLayout *plTop,*plh,*pl,*plv;
    int i;
    char m[2]={0,0},name[4] = {'X','Y','W','H'};
    LPCSTR szScale[4] = {"x 0.5","x 1","x 4","x 8"};
    AXString str;
    AXScrollView *pview;
    AXCheckButton *pcheck;

    m_prcsRet = prcs;

    //

    _trgroup(strid::GROUP_DLG_TRIMMING);

    setTitle(STRID_TITLE);

    setLayout(plTop = new AXLayoutVert);
    plTop->setSpacing(10);

    //

    plTop->addItem(plh = new AXLayoutHorz(LF_EXPAND_WH, 12));

    //----------- 左側

    plh->addItem(plv = new AXLayoutVert(LF_EXPAND_H, 0));

    //x,y,w,h

    plv->addItem(pl = new AXLayoutMatrix(2, LF_EXPAND_Y));
    pl->setPaddingBottom(10);

    for(i = 0; i < 4; i++)
    {
        m[0] = name[i];
        pl->addItem(new AXLabel(this, 0, LF_CENTER_Y, MAKE_DW4(0,0,6,7), m));
        pl->addItem(m_pedit[i] = new AXLineEdit(this, AXLineEdit::ES_SPIN, LF_CENTER_Y, WID_EDIT_X + i, MAKE_DW4(0,0,0,7)));

        m_pedit[i]->setInit(5, 0, 9999, 0);
    }

    m_pedit[2]->setVal(g_draw->nImgW);
    m_pedit[3]->setVal(g_draw->nImgH);

    //表示倍率

    for(i = 0; i < 5; i++)
    {
        plv->addItem(pcheck = new AXCheckButton(this,
                AXCheckButton::CBS_BUTTON|AXCheckButton::CBS_RADIO, LF_EXPAND_W,
                WID_SCALE_TOP + i, 0, (LPCUSTR)NULL, FALSE));

        if(i == 0)
        {
            pcheck->setText(_str(STRID_WHOLE));
            pcheck->setCheck(TRUE);
        }
        else
        {
            str = szScale[i - 1];
            pcheck->setText(str);
        }
    }

    //左上、右下へ

    plv->addItem(new AXButton(this, 0, LF_EXPAND_W, WID_BT_LEFTTOP, MAKE_DW4(0,15,0,2), STRID_LEFTTOP));
    plv->addItem(new AXButton(this, 0, LF_EXPAND_W, WID_BT_RIGHTBOTTOM, 0, STRID_RIGHTBOTTOM));

    //--------- 右側:表示エリア

    plh->addItem(pview = new AXScrollView(this, AXScrollView::SVS_HORZVERT_FIX|AXScrollView::SVS_SUNKEN, LF_EXPAND_WH));

    pview->setScrollArea(m_parea = new CTrimming_area(pview));
    m_parea->setNotify(this);

    //----------- ヘルプ

    plTop->addItem(new AXLabel(this, AXLabel::LS_BORDER, LF_RIGHT, MAKE_DW4(0,8,0,15), STRID_HELP));

    //----------

    plTop->addItem(createOKCancelButton());

    calcDefSize();
    resize((m_nDefW < 450)? 450: m_nDefW, (m_nDefH < 450)? 450: m_nDefH);

    show();
}

//! 通知

BOOL CTrimmingDlg::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    UINT id = pwin->getItemID();
    int scale[5] = {-1,50,100,400,800};
    AXRectSize *prcs;

    //倍率

    if(id >= WID_SCALE_TOP && id < WID_SCALE_TOP + 5)
    {
        m_parea->setScale(scale[id - WID_SCALE_TOP]);
        return TRUE;
    }

    //

    switch(id)
    {
        //表示エリア
        case WID_AREA:
            prcs = m_parea->getBoxRect();

            m_pedit[0]->setVal(prcs->x);
            m_pedit[1]->setVal(prcs->y);
            m_pedit[2]->setVal(prcs->w);
            m_pedit[3]->setVal(prcs->h);
            break;
        //X,Y,W,H
        case WID_EDIT_X:
        case WID_EDIT_Y:
        case WID_EDIT_W:
        case WID_EDIT_H:
            if(uNotify == AXLineEdit::EN_CHANGE)
                m_parea->changeBoxVal(id - WID_EDIT_X + 10, m_pedit[id - WID_EDIT_X]->getVal());
            break;
        //左上へ
        case WID_BT_LEFTTOP:
            m_parea->scrollPoint(FALSE);
            break;
        //右下へ
        case WID_BT_RIGHTBOTTOM:
            m_parea->scrollPoint(TRUE);
            break;

        //OK
        case 1:
            *m_prcsRet = *(m_parea->getBoxRect());

            if(m_prcsRet->w == g_draw->nImgW && m_prcsRet->h == g_draw->nImgH)
                //変更なし
                endDialog(FALSE);
            else
                endDialog(TRUE);
            break;
        //cancel
        case 2:
            endDialog(FALSE);
            break;
    }

    return TRUE;
}


//*****************************************
// CTrimming_area - 表示エリア
//*****************************************


CTrimming_area::CTrimming_area(AXWindow *pParent)
    : AXScrollArea(pParent, 0)
{
    m_uItemID = WID_AREA;

    m_fDrag  = 0;
    m_nScale = -1;

    m_rcsBox.set(0, 0, g_draw->nImgW, g_draw->nImgH);

    m_vparam.dScale = m_vparam.dScaleDiv = 1;

    m_dinfo.nBaseX  = 0;
    m_dinfo.nBaseY  = 0;
    m_dinfo.dwExCol = 0xc0c0c0;
    m_dinfo.bHRev   = FALSE;
    m_dinfo.pParam  = &m_vparam;
}

//! サイズ変更

BOOL CTrimming_area::onSize()
{
    m_img.recreate(m_nW, m_nH, 32, 32);

    if(m_nScale < 0)
        setScale(m_nScale);
    else
    {
        setScrollInfo();
        draw();
    }

    return TRUE;
}

//! 描画

BOOL CTrimming_area::onPaint(AXHD_PAINT *phd)
{
    m_img.put(m_id, phd->x, phd->y, phd->x, phd->y, phd->w, phd->h);

    return TRUE;
}

//! 通知

BOOL CTrimming_area::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    if(uNotify == SAN_SCROLL_HORZ || uNotify == SAN_SCROLL_VERT)
        draw();

    return TRUE;
}

//! ボタン押し

BOOL CTrimming_area::onButtonDown(AXHD_MOUSE *phd)
{
    if(phd->button == BUTTON_LEFT && m_fDrag == 0)
    {
        if(phd->state & STATE_CTRL)
            //スクロール
            m_fDrag = DRAGF_SCROLL;
        else if(phd->state & STATE_SHIFT)
        {
            //辺移動

            m_fDrag = DRAGF_MOVESIDE;

            m_nSide = _getNearSide(phd->x, phd->y);

            m_rcDrag.set(m_rcsBox);
        }
        else
        {
            //切り取り範囲

            m_fDrag = DRAGF_AREA;

            m_rcDrag.left = m_rcDrag.right = phd->x;
            m_rcDrag.top = m_rcDrag.bottom = phd->y;

            _drawAreaBox();
            _drawDragBox();
            redraw();
        }

        m_ptTmp.x = phd->x;
        m_ptTmp.y = phd->y;

        grabPointer();
    }

    return TRUE;
}

//! ボタン離し

BOOL CTrimming_area::onButtonUp(AXHD_MOUSE *phd)
{
    int x1,y1,x2,y2;

    if(m_fDrag)
    {
        switch(m_fDrag)
        {
            //切り取り範囲
            case DRAGF_AREA:
                _drawDragBox();

                _viewToimg(&x1, &y1, m_rcDrag.left, m_rcDrag.top);
                _viewToimg(&x2, &y2, m_rcDrag.right, m_rcDrag.bottom);

                if(!draw::adjustImgRect(&m_rcsBox, x1, y1, x2, y2))
                    m_rcsBox.set(0, 0, g_draw->nImgW, g_draw->nImgH);

                _drawAreaBox();
                redrawUpdate();

                m_pNotify->onNotify(this, 0, 0);
                break;
            //辺の移動
            case DRAGF_MOVESIDE:
                m_pNotify->onNotify(this, 0, 0);
                break;
        }

        ungrabPointer();
        m_fDrag = 0;
    }

    return TRUE;
}

//! 移動

BOOL CTrimming_area::onMouseMove(AXHD_MOUSE *phd)
{
    int x,y,f;

    switch(m_fDrag)
    {
        //切り取り範囲
        case DRAGF_AREA:
            _drawDragBox();

            m_rcDrag.setMinMax(m_ptTmp.x, m_ptTmp.y, phd->x, phd->y);

            _drawDragBox();
            redrawUpdate();
            break;
        //スクロール
        case DRAGF_SCROLL:
            x = scrH()->getPos() + m_ptTmp.x - phd->x;
            y = scrV()->getPos() + m_ptTmp.y - phd->y;

            f = scrH()->setPos(x);
            f |= scrV()->setPos(y);

            if(f)
            {
                draw();
                redrawUpdate();
            }

            m_ptTmp.x = phd->x;
            m_ptTmp.y = phd->y;
            break;
        //辺の移動
        case DRAGF_MOVESIDE:
            if(m_nSide == 0)
                //左
                x = m_rcDrag.left + (int)((phd->x - m_ptTmp.x) * m_vparam.dScaleDiv);
            else if(m_nSide == 1)
                //上
                x = m_rcDrag.top + (int)((phd->y - m_ptTmp.y) * m_vparam.dScaleDiv);
            else if(m_nSide == 2)
                //右
                x = m_rcDrag.right + (int)((phd->x - m_ptTmp.x) * m_vparam.dScaleDiv);
            else
                //下
                x = m_rcDrag.bottom + (int)((phd->y - m_ptTmp.y) * m_vparam.dScaleDiv);

            changeBoxVal(m_nSide, x);
            break;
    }

    return TRUE;
}


//=======================


//! 倍率セット

void CTrimming_area::setScale(int scale)
{
    AXRectSize rcs;
    int w,h;

    m_nScale = scale;

    if(scale > 0)
        m_vparam.dScale = scale * 0.01;
    else
    {
        //上下左右に余白10pxつける

        if(m_nW < 20 || m_nH < 20)
            w = h = 20;
        else
            w = m_nW - 20, h = m_nH - 20;

        //

        rcs.set(0, 0, g_draw->nImgW, g_draw->nImgH);
        rcs.inBoxKeepAspect(w, h, TRUE);

        //倍率

        m_vparam.dScale = (double)rcs.w / g_draw->nImgW;

        if(m_vparam.dScale < 0.01) m_vparam.dScale = 0.01;
    }

    m_vparam.dScaleDiv = 1.0 / m_vparam.dScale;

    //

    setScrollInfo();
    draw();
}

//! スクロール情報セット

void CTrimming_area::setScrollInfo()
{
    scrH()->setStatus(0, (int)(g_draw->nImgW * m_vparam.dScale + 20 + 0.5), m_nW);
    scrV()->setStatus(0, (int)(g_draw->nImgH * m_vparam.dScale + 20 + 0.5), m_nH);
}

//! 左上/右下へスクロール

void CTrimming_area::scrollPoint(BOOL bBottom)
{
    int x,y;

    if(bBottom)
        x = m_rcsBox.x + m_rcsBox.w - 1, y = m_rcsBox.y + m_rcsBox.h - 1;
    else
        x = m_rcsBox.x, y = m_rcsBox.y;

    x = (int)(x * m_vparam.dScale + 10 - m_nW / 2);
    y = (int)(y * m_vparam.dScale + 10 - m_nH / 2);

    scrH()->setPos(x);
    scrV()->setPos(y);

    draw();
}

//! 範囲の値を指定値に変更
/*
    [0]left [1]top [2]right [3]bottom [10]x [11]y [12]w [13]h
*/

void CTrimming_area::changeBoxVal(int type,int val)
{
    int n;

    _drawAreaBox();

    switch(type)
    {
        //left
        case 0:
            n = m_rcsBox.x + m_rcsBox.w - 1;

            if(val < 0) val = 0; else if(val > n) val = n;

            m_rcsBox.w = n - val + 1;
            m_rcsBox.x = val;
            break;
        //top
        case 1:
            n = m_rcsBox.y + m_rcsBox.h - 1;

            if(val < 0) val = 0; else if(val > n) val = n;

            m_rcsBox.h = n - val + 1;
            m_rcsBox.y = val;
            break;
        //right
        case 2:
            if(val < m_rcsBox.x) val = m_rcsBox.x;
            else if(val >= g_draw->nImgW) val = g_draw->nImgW - 1;

            m_rcsBox.w = val - m_rcsBox.x + 1;
            break;
        //bottom
        case 3:
            if(val < m_rcsBox.y) val = m_rcsBox.y;
            else if(val >= g_draw->nImgH) val = g_draw->nImgH - 1;

            m_rcsBox.h = val - m_rcsBox.y + 1;
            break;
        //x
        case 10:
            if(val < 0) val = 0;
            else if(val + m_rcsBox.w > g_draw->nImgW) val = g_draw->nImgW - m_rcsBox.w;

            m_rcsBox.x = val;
            break;
        //y
        case 11:
            if(val < 0) val = 0;
            else if(val + m_rcsBox.h > g_draw->nImgH) val = g_draw->nImgH - m_rcsBox.h;

            m_rcsBox.y = val;
            break;
        //w
        case 12:
            if(val <= 0) val = 1;
            else if(val > g_draw->nImgW - m_rcsBox.x) val = g_draw->nImgW - m_rcsBox.x;

            m_rcsBox.w = val;
            break;
        //h
        case 13:
            if(val <= 0) val = 1;
            else if(val > g_draw->nImgH - m_rcsBox.y) val = g_draw->nImgH - m_rcsBox.y;

            m_rcsBox.h = val;
            break;
    }

    _drawAreaBox();
    redrawUpdate();
}


//====================


//! イメージ描画

void CTrimming_area::draw()
{
    m_dinfo.rcsDst.set(0, 0, m_nW, m_nH);

    m_dinfo.nScrollX = 10 - scrH()->getPos();
    m_dinfo.nScrollY = 10 - scrV()->getPos();

    if(m_vparam.dScale >= 1)
        g_draw->pimgBlend->drawCanvasNormal(&m_img, m_dinfo);
    else
        g_draw->pimgBlend->drawCanvasScaleDown(&m_img, m_dinfo);

    _drawAreaBox();

    redraw();
}

//! 範囲の矩形を描画

void CTrimming_area::_drawAreaBox()
{
    int x1,y1,x2,y2;

    _imgToview(&x1, &y1, m_rcsBox.x, m_rcsBox.y);
    _imgToview(&x2, &y2, m_rcsBox.x + m_rcsBox.w, m_rcsBox.y + m_rcsBox.h);

    if(m_vparam.dScale >= 1)
        x1--, y1--;

    m_img.boxClip(x1, y1, x2 - x1 + 1, y2 - y1 + 1, AXImage::COL_XOR);
}

//! ドラッグ中の矩形描画

void CTrimming_area::_drawDragBox()
{
    m_img.box(m_rcDrag.left, m_rcDrag.top, m_rcDrag.width(), m_rcDrag.height(), AXImage::COL_XOR);
}

//====================


//! ウィンドウ位置 -> イメージ位置

void CTrimming_area::_viewToimg(int *pX,int *pY,int x,int y)
{
    *pX = (int)((x - m_dinfo.nScrollX) * m_vparam.dScaleDiv);
    *pY = (int)((y - m_dinfo.nScrollY) * m_vparam.dScaleDiv);
}

//! イメージ位置 -> ウィンドウ位置

void CTrimming_area::_imgToview(int *pX,int *pY,int x,int y)
{
    *pX = AXDoubleToInt(x * m_vparam.dScale + m_dinfo.nScrollX);
    *pY = AXDoubleToInt(y * m_vparam.dScale + m_dinfo.nScrollY);
}

//! 一番近い辺を取得

int CTrimming_area::_getNearSide(int x,int y)
{
    int i,d[4],min,type,n;

    _viewToimg(&x, &y, x, y);

    //

    d[0] = m_rcsBox.x - x;
    d[1] = m_rcsBox.y - y;
    d[2] = m_rcsBox.x + m_rcsBox.w - 1 - x;
    d[3] = m_rcsBox.y + m_rcsBox.h - 1 - y;

    //

    min  = ::abs(d[0]);
    type = 0;

    for(i = 1; i < 4; i++)
    {
        n = ::abs(d[i]);

        if(n < min)
        {
            type = i;
            min  = n;
        }
    }

    return type;
}
