/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/

#include "COptionWin.h"

#include "CConfig.h"
#include "CResource.h"
#include "CImage8.h"
#include "CGradList.h"

#include "CMainWin.h"
#include "CValBar.h"
#include "CSelImgBar.h"
#include "CImgPrev.h"
#include "CCustomGradSel.h"
#include "CGradEditDlg.h"

#include "drawdat.h"
#include "draw_main.h"
#include "draw_update.h"

#include "global.h"
#include "strid.h"

#include "AXLayout.h"
#include "AXSpacerItem.h"
#include "AXLabel.h"
#include "AXTab.h"
#include "AXButton.h"
#include "AXCheckButton.h"
#include "AXColorButton.h"
#include "AXArrowButton.h"
#include "AXLineEdit.h"
#include "AXListBox.h"
#include "AXComboBox.h"
#include "AXMenu.h"
#include "AXImage.h"
#include "AXImage32.h"
#include "AXColorDialog.h"
#include "AXMessageBox.h"
#include "AXGC.h"
#include "AXUtilStr.h"
#include "AXAppRes.h"
#include "AXApp.h"

#include "COptionWin_tab.h"



/*!
    @class COptionWin
    @brief オプションウィンドウ
*/


COptionWin *COptionWin::m_pSelf = NULL;


COptionWin::COptionWin(AXWindow *pOwner,UINT addstyle)
    : AXTopWindow(pOwner,
                  WS_HIDE | WS_BK_FACE | WS_TITLE | WS_CLOSE | WS_BORDER | addstyle)
{
    AXLayout *pl;
    int i;

    m_pSelf = this;

    m_pwin = NULL;

    //

    if(g_res->pfontSubWin)
        setFont(g_res->pfontSubWin);

    setTitle("option");

    attachAccel(MAINWIN->getAccel());

    //----------

    setLayout(pl = new AXLayoutVert);

    //タブ

    _trgroup(strid::GROUP_OPTIONWIN);

    pl->addItem(m_ptab = new AXTab(this, AXTab::TABS_TOPTAB | AXTab::TABS_FIT, LF_EXPAND_W, 0, MAKE_DW4(0,0,0,4)));

    for(i = 0; i < 5; i++)
        m_ptab->addItem(strid::OPTWIN_TABNAMETOP + i);

    //初期タブ

    m_ptab->setCurSel(g_conf->btOptWinTab);

    _setTabWindow();
}

//! 表示/非表示

void COptionWin::showToggle()
{
    if(g_conf->isOptionWin())
        showRestore();
    else
        hide();
}

//! ツール変更時

void COptionWin::changeTool()
{
    //タブの選択が「ツール」の場合、タブ内ウィンドウ変更

    if(m_ptab->getCurSel() == 0)
        _setTabWindow();
}

//! タブのウィンドウセット

void COptionWin::_setTabWindow()
{
    //取り外す

    if(m_pwin)
    {
        m_pwin->removeFromLayout();

        delete m_pwin;
        m_pwin = NULL;
    }

    //タブウィンドウ作成

    switch(g_conf->btOptWinTab)
    {
        //ツール
        case 0:
            _createToolTab();
            break;
        //定規
        case 1:
            m_pwin = new COptWin_rule;
            break;
        //テクスチャ
        case 2:
            m_pwin = new COptWin_texture;
            break;
        //入り抜き
        case 3:
            m_pwin = new COptWin_headtail;
            break;
        //設定
        case 4:
            m_pwin = new COptWin_option;
            break;
    }

    //セット

    if(m_pwin)
        m_pLayout->addItem(m_pwin);

    calcDefSize();
    layout();
}

//! ツールタブのウィンドウ作成

void COptionWin::_createToolTab()
{
    switch(g_draw->tool.toolno)
    {
        case draw::TOOL_DOTPEN:
            m_pwin = new COptWin_dotpen;
            break;
        case draw::TOOL_POLYPAINT:
        case draw::TOOL_POLYERASE:
            m_pwin = new COptWin_polypaint;
            break;
        case draw::TOOL_PAINT:
            m_pwin = new COptWin_paint;
            break;
        case draw::TOOL_GRAD:
            m_pwin = new COptWin_grad;
            break;
        case draw::TOOL_MAGICWAND:
            m_pwin = new COptWin_magicwand;
            break;
        case draw::TOOL_SEL:
            m_pwin = new COptWin_sel;
            break;
        case draw::TOOL_MOVE:
            m_pwin = new COptWin_move;
            break;
    }
}


//=========================
// ハンドラ
//=========================


//! 閉じる（非表示）

BOOL COptionWin::onClose()
{
    m_pOwner->onCommand(strid::MENU_VIEW_OPTIONWIN, 0, 0);

    return TRUE;
}

//! 通知

BOOL COptionWin::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    //タブ選択変更

    if(pwin == m_ptab && uNotify == AXTab::TABN_SELCHANGE)
    {
        g_conf->btOptWinTab = m_ptab->getCurSel();

        _setTabWindow();
    }

    return TRUE;
}


//##################################
// 各タブの中身のウィンドウ
//##################################


//*********************************
// COptWin_dotpen : ドットペン
//*********************************


COptWin_dotpen::COptWin_dotpen()
    : AXWindow(OPTIONWIN, WS_BK_FACE, LF_EXPAND_WH)
{
    AXLayout *pl,*pl2;
    DWORD val;

    _trgroup(strid::GROUP_OPTIONWIN);

    val = g_draw->tool.dwDotPenList[g_draw->tool.btDotPenListSel];

    //

    setLayout(pl = new AXLayoutVert(LF_EXPAND_W, 5));
    pl->setSpacing(4);

    //リスト

    pl->addItem(m_pwidList = new COptWin_dotpen_list(this, WID_LIST));

    //タイプ

    pl->addItem(m_pcbType = new AXComboBox(this, 0, LF_EXPAND_W, WID_CB_TYPE, 0));

    m_pcbType->addItemMulTr(strid::OPTWIN_DOTPEN_TYPE_TOP, 3);
    m_pcbType->setCurSel((val >> 16) & 3);

    //サイズ・濃度

    _trgroup(strid::GROUP_WORD);

    pl->addItem(pl2 = new AXLayoutMatrix(2, LF_EXPAND_W));

    pl2->addItem(new AXLabel(this, 0, LF_RIGHT|LF_CENTER_Y, MAKE_DW4(0,0,4,4), strid::WORD_SIZE));
    pl2->addItem(m_pbarSize = new CValBar(this, 0, LF_EXPAND_W|LF_CENTER_Y, WID_BAR_SIZE, MAKE_DW4(0,0,0,4), 0, 1, 32, val & 0xff));

    pl2->addItem(new AXLabel(this, 0, LF_RIGHT|LF_CENTER_Y, MAKE_DW4(0,0,4,0), strid::WORD_DENSITY));
    pl2->addItem(m_pbarOpa = new CValBar(this, 0, LF_EXPAND_W|LF_CENTER_Y, WID_BAR_OPACITY, 0, 0, 1, 100, (val >> 8) & 0xff));
}

//! 通知

BOOL COptWin_dotpen::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    UINT id = pwin->getItemID();
    LPDWORD pDat = g_draw->tool.dwDotPenList + g_draw->tool.btDotPenListSel;

    switch(id)
    {
        //リスト(選択変更)
        case WID_LIST:
            m_pcbType->setCurSel((*pDat >> 16) & 3);
            m_pbarSize->setPos(*pDat & 0xff);
            m_pbarOpa->setPos((*pDat >> 8) & 0xff);
            break;
        //タイプ
        case WID_CB_TYPE:
            if(uNotify == AXComboBox::CBN_SELCHANGE)
            {
                *pDat &= ~0x30000;
                *pDat |= m_pcbType->getCurSel() << 16;
                m_pwidList->draw();
            }
            break;
        //サイズ
        case WID_BAR_SIZE:
            if(!m_pbarSize->isDrag())
            {
                *pDat &= ~0xff;
                *pDat |= m_pbarSize->getPos();
                m_pwidList->draw();
            }
            break;
        //濃度
        case WID_BAR_OPACITY:
            if(!m_pbarOpa->isDrag())
            {
                *pDat &= ~0xff00;
                *pDat |= m_pbarOpa->getPos() << 8;
                m_pwidList->draw();
            }
            break;
    }

    return TRUE;
}

//-------------- リスト

COptWin_dotpen_list::COptWin_dotpen_list(AXWindow *pParent,UINT uID)
    : AXWindow(pParent, 0, LF_FIX_WH, uID, 0)
{
    m_nW = ONEW * draw::DOTPENLIST_NUM + 1;
    m_nH = 7 * 2 + 2 + 6;

    m_img.create(m_nW, m_nH);

    draw();
}

BOOL COptWin_dotpen_list::onPaint(AXHD_PAINT *phd)
{
    m_img.put(m_id);
    return TRUE;
}

//! ボタン押し

BOOL COptWin_dotpen_list::onButtonDown(AXHD_MOUSE *phd)
{
    int no;

    if(phd->button == BUTTON_LEFT)
    {
        //選択変更

        no = phd->x / ONEW;
        if(no >= draw::DOTPENLIST_NUM) no = draw::DOTPENLIST_NUM - 1;

        if(no != g_draw->tool.btDotPenListSel)
        {
            g_draw->tool.btDotPenListSel= no;

            draw();

            getNotify()->onNotify(this, 0, 0);
        }
    }

    return TRUE;
}

//! 描画

void COptWin_dotpen_list::draw()
{
    int i,x,type,len;
    DWORD val;
    char m[16];
    BYTE patThin[7] = {0x20,0x20,0x70,0xf8,0x70,0x20,0x20};
    BYTE patErase[7] = {0xfc,0xfc,0xfc,0xfc,0x84,0x84,0xfc};

    m_img.clear(0xf0f0f0);

    for(i = 0, x = 0; i < draw::DOTPENLIST_NUM; i++, x += ONEW)
    {
        val  = g_draw->tool.dwDotPenList[i];
        type = (val >> 16) & 3;

        //枠
        m_img.box(x, 0, ONEW + 1, m_nH, 0);

        //背景
        if(g_draw->tool.btDotPenListSel == i)
            m_img.fillBox(x + 1, 1, ONEW - 1, m_nH - 2, 0xffcccc);

        //タイプ
        if(type == 1)
            m_img.drawPattern(x + 3, 3, 0, patThin, 5, 7);
        else if(type == 2)
            m_img.drawPattern(x + 3, 3, 0, patErase, 6, 7);

        //サイズ
        len = AXIntToStr(m, val & 0xff);
        m_img.drawNumber(x + 10 + (2 - len) * 5, 3, m, 0);

        //濃度
        len = AXIntToStr(m, (val >> 8) & 0xff);
        m_img.drawNumber(x + 5 + (3 - len) * 5, 3 + 7 + 2, m, 0);
    }

    //選択枠

    m_img.box(g_draw->tool.btDotPenListSel * ONEW, 0, ONEW + 1, m_nH, 0xff0000);

    redraw();
}


//*********************************
// COptWin_move : 移動ツール
//*********************************


COptWin_move::COptWin_move()
    : AXWindow(OPTIONWIN, WS_BK_FACE, 0)
{
    AXLayout *pl;
    int i,type;

    _trgroup(strid::GROUP_OPTIONWIN);

    setLayout(pl = new AXLayoutVert(0, 2));
    pl->setSpacing(4);

    //タイプ

    type = g_draw->tool.optMove & 7;

    for(i = 0; i < 4; i++)
    {
        pl->addItem(new AXCheckButton(this, AXCheckButton::CBS_RADIO, 0,
                        WID_TYPE_TOP + i, 0, strid::OPTWIN_MOVE_TYPE_TOP + i, (type == i)));
    }

    //リンクレイヤ無効

    pl->addItem(new AXCheckButton(this, 0, 0, WID_DISABLE_LINK, MAKE_DW4(0,2,0,0), strid::OPTWIN_SEL_DISABLE_LINK,
            g_draw->tool.optMove & (1<<3)));
}

BOOL COptWin_move::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    UINT id = pwin->getItemID();

    if(id >= WID_TYPE_TOP && id < WID_TYPE_TOP + 4)
    {
        g_draw->tool.optMove &= ~7;
        g_draw->tool.optMove |= id - WID_TYPE_TOP;
    }
    else if(id == WID_DISABLE_LINK)
        g_draw->tool.optMove ^= (1<<3);

    return TRUE;
}


//****************************************
// COptWin_polypaint : 図形塗りつぶし/消去
//****************************************


COptWin_polypaint::COptWin_polypaint()
    : AXWindow(OPTIONWIN, WS_BK_FACE, LF_EXPAND_W)
{
    AXLayout *pl;
    int val;

    m_bErase = (g_draw->tool.toolno == draw::TOOL_POLYERASE);

    val = (m_bErase)? g_draw->tool.optPolyErase: g_draw->tool.optPolyPaint;

    //

    _trgroup(strid::GROUP_WORD);

    setLayout(pl = new AXLayoutMatrix(2));
    pl->setSpacing(4);

    //濃度

    pl->addItem(new AXLabel(this, 0, LF_CENTER_Y, MAKE_DW4(0,0,4,5), strid::WORD_DENSITY));
    pl->addItem(m_pbarOpa = new CValBar(this, 0, LF_EXPAND_W|LF_CENTER_Y, WID_OPACITY, MAKE_DW4(0,0,0,5), 0, 1, 100, val & 0xff));

    //塗り

    if(!m_bErase)
    {
        pl->addItem(new AXLabel(this, 0, LF_CENTER_Y, MAKE_DW4(0,0,4,5), strid::WORD_PIXTYPE));
        pl->addItem(m_pcbPix = new AXComboBox(this, 0, LF_EXPAND_W|LF_CENTER_Y, WID_PIXTYPE, MAKE_DW4(0,0,0,5)));
    }

    //アンチエイリアス

    pl->addItem(new AXSpacerItem);
    pl->addItem(new AXCheckButton(this, 0, 0, WID_ANTIALIAS, 0, strid::WORD_ANTIALIAS, val & (1<<11)));

    //--------

    if(!m_bErase)
    {
        _trgroup(strid::GROUP_PIXTYPE);

        m_pcbPix->addItemMulTr(0, 3);
        m_pcbPix->setCurSel((val >> 8) & 7);
    }
}

BOOL COptWin_polypaint::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    UINT id = pwin->getItemID();

    switch(id)
    {
        //濃度
        case WID_OPACITY:
            if(!m_pbarOpa->isDrag())
            {
                if(m_bErase)
                {
                    g_draw->tool.optPolyErase &= ~0xff;
                    g_draw->tool.optPolyErase |= m_pbarOpa->getPos();
                }
                else
                {
                    g_draw->tool.optPolyPaint &= ~0xff;
                    g_draw->tool.optPolyPaint |= m_pbarOpa->getPos();
                }
            }
            break;
        //塗り
        case WID_PIXTYPE:
            g_draw->tool.optPolyPaint &= ~(0x07 << 8);
            g_draw->tool.optPolyPaint |= (m_pcbPix->getCurSel() << 8);
            break;
        //アンチエイリアス
        case WID_ANTIALIAS:
            if(m_bErase)
                g_draw->tool.optPolyErase ^= (1<<11);
            else
                g_draw->tool.optPolyPaint ^= (1<<11);
            break;
    }

    return TRUE;
}


//*********************************
// COptWin_paint : 塗りつぶし
//*********************************


COptWin_paint::COptWin_paint()
    : AXWindow(OPTIONWIN, WS_BK_FACE, LF_EXPAND_W)
{
    AXLayout *pl;
    DWORD val;

    val = g_draw->tool.optPaint;

    //

    _trgroup(strid::GROUP_OPTIONWIN);

    setLayout(pl = new AXLayoutMatrix(2));
    pl->setSpacing(4);

    //判定

    pl->addItem(new AXLabel(this, 0, LF_CENTER_Y, MAKE_DW4(0,0,4,5), strid::OPTWIN_JUDGE));
    pl->addItem(m_pcbType = new AXComboBox(this, 0, LF_EXPAND_W|LF_CENTER_Y, WID_TYPE, MAKE_DW4(0,0,0,5)));

    m_pcbType->addItemMulTr(strid::OPTWIN_PAINT_TYPE_TOP, 4);
    m_pcbType->setCurSel(val & 7);

    //あまさ

    pl->addItem(new AXLabel(this, 0, LF_CENTER_Y, MAKE_DW4(0,0,4,5), strid::OPTWIN_PAINT_DIFF));
    pl->addItem(m_pbarDiff = new CValBar(this, ((val & 7) == 2)? WS_DISABLE: 0, LF_EXPAND_W|LF_CENTER_Y, WID_DIFF, MAKE_DW4(0,0,0,5), 1, 0, 999, (val >> 13) & 1023));

    //濃度

    pl->addItem(new AXLabel(this, 0, LF_CENTER_Y, MAKE_DW4(0,0,4,5), _string(strid::GROUP_WORD, strid::WORD_DENSITY)));
    pl->addItem(m_pbarOpa = new CValBar(this, 0, LF_EXPAND_W|LF_CENTER_Y, WID_OPACITY, MAKE_DW4(0,0,0,5), 0, 1, 100, (val >> 6) & 127));

    //塗り

    pl->addItem(new AXLabel(this, 0, LF_CENTER_Y, MAKE_DW4(0,0,4,0), _string(strid::GROUP_WORD, strid::WORD_PIXTYPE)));
    pl->addItem(m_pcbPix = new AXComboBox(this, 0, LF_EXPAND_W|LF_CENTER_Y, WID_PIXTYPE, 0));

    _trgroup(strid::GROUP_PIXTYPE);

    m_pcbPix->addItemMulTr(0, 4);
    m_pcbPix->setCurSel((val >> 3) & 7);
}

BOOL COptWin_paint::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    UINT id = pwin->getItemID();

    switch(id)
    {
        //判定
        case WID_TYPE:
            g_draw->tool.optPaint &= ~7;
            g_draw->tool.optPaint |= m_pcbType->getCurSel();

            m_pbarDiff->enable(m_pcbType->getCurSel() != 2);
            break;
        //あまさ
        case WID_DIFF:
            g_draw->tool.optPaint &= ~(1023<<13);
            g_draw->tool.optPaint |= m_pbarDiff->getPos() << 13;
            break;
        //濃度
        case WID_OPACITY:
            g_draw->tool.optPaint &= ~(127<<6);
            g_draw->tool.optPaint |= m_pbarOpa->getPos() << 6;
            break;
        //塗り
        case WID_PIXTYPE:
            g_draw->tool.optPaint &= ~(7<<3);
            g_draw->tool.optPaint |= m_pcbPix->getCurSel() << 3;
            break;
    }

    return TRUE;
}


//*********************************
// COptWin_grad : グラデーション
//*********************************


COptWin_grad::COptWin_grad()
    : AXWindow(OPTIONWIN, WS_BK_FACE, LF_EXPAND_W)
{
    AXLayout *plTop,*pl;
    DWORD val;

    val = g_draw->tool.optGrad;

    //

    _trgroup(strid::GROUP_OPTIONWIN);

    setLayout(plTop = new AXLayoutVert);
    plTop->setSpacing(4);

    //描画色->背景色

    plTop->addItem(m_pckDefault = new AXCheckButton(this, AXCheckButton::CBS_RADIO, 0, WID_CK_DEFAULT, MAKE_DW4(0,0,0,2), strid::OPTWIN_GRAD_DEFAULT, !(val & (1<<11))));

    //カスタム

    plTop->addItem(pl = new AXLayoutHorz(LF_EXPAND_W, 4));

    pl->addItem(m_pckCustom = new AXCheckButton(this, AXCheckButton::CBS_RADIO, LF_CENTER_Y, WID_CK_CUSTOM, 0, strid::OPTWIN_GRAD_CUSTOM, val & (1<<11)));
    pl->addItem(m_pCustom = new CCustomGradSel(this, WID_CB_CUSTOM, (val >> 16) & 255));
    pl->addItem(new AXArrowButton(this, AXArrowButton::ARBTS_DOWN, LF_CENTER_Y, WID_BT_CUSTOM, 0));

    //--------

    plTop->addItem(pl = new AXLayoutMatrix(2, LF_EXPAND_W));
    pl->setPadding(0, 6, 0, 4);

    //濃度

    pl->addItem(new AXLabel(this, 0, LF_CENTER_Y, MAKE_DW4(0,0,4,5), _string(strid::GROUP_WORD, strid::WORD_DENSITY)));
    pl->addItem(m_pbarOpa = new CValBar(this, 0, LF_EXPAND_W|LF_CENTER_Y, WID_OPACITY, MAKE_DW4(0,0,0,5), 0, 1, 100, val & 127));

    //塗り

    pl->addItem(new AXLabel(this, 0, LF_CENTER_Y, MAKE_DW4(0,0,4,0), _string(strid::GROUP_WORD, strid::WORD_PIXTYPE)));
    pl->addItem(m_pcbPix = new AXComboBox(this, 0, LF_EXPAND_W|LF_CENTER_Y, WID_PIXTYPE, 0));

    //---------

    plTop->addItem(pl = new AXLayoutHorz(0, 4));

    pl->addItem(new AXCheckButton(this, 0, 0, WID_REVERSE, 0, strid::OPTWIN_GRAD_REVERSE, val & (1<<12)));
    pl->addItem(new AXCheckButton(this, 0, 0, WID_LOOP, 0, strid::OPTWIN_GRAD_LOOP, val & (1<<13)));

    //------- 塗り

    _trgroup(strid::GROUP_PIXTYPE);

    m_pcbPix->addItemMulTr(0, 4);
    m_pcbPix->setCurSel((val >> 8) & 7);
}

BOOL COptWin_grad::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    UINT id = pwin->getItemID();

    switch(id)
    {
        //描画色->背景色
        case WID_CK_DEFAULT:
            g_draw->tool.optGrad &= ~(1<<11);

            m_pckCustom->setCheck(FALSE);
            break;
        //カスタムチェック
        case WID_CK_CUSTOM:
            g_draw->tool.optGrad |= 1<<11;

            m_pckDefault->setCheck(FALSE);
            break;
        //カスタム選択
        case WID_CB_CUSTOM:
            g_draw->tool.optGrad &= ~(255<<16);
            g_draw->tool.optGrad |= lParam << 16;
            break;
        //カスタムメニュー
        case WID_BT_CUSTOM:
            _runCustomMenu();
            break;
        //濃度
        case WID_OPACITY:
            g_draw->tool.optGrad &= ~255;
            g_draw->tool.optGrad |= m_pbarOpa->getPos();
            break;
        //塗り
        case WID_PIXTYPE:
            g_draw->tool.optGrad &= ~(7<<8);
            g_draw->tool.optGrad |= m_pcbPix->getCurSel() << 8;
            break;
        //反転
        case WID_REVERSE:
            g_draw->tool.optGrad ^= (1<<12);
            break;
        //繰り返し
        case WID_LOOP:
            g_draw->tool.optGrad ^= (1<<13);
            break;
    }

    return TRUE;
}

//! カスタムメニュー実行

void COptWin_grad::_runCustomMenu()
{
    AXMenu *pmenu;
    int id,n;
    AXRectSize rcs;
    WORD wID[] = {
        0, 0xffff, 1,2,3, 0xffff, 4,5
    };

    _trgroup(strid::GROUP_GRADCUSTOMMENU);

    pmenu = new AXMenu;

    pmenu->addTrArray(wID, sizeof(wID)/sizeof(WORD));

    if(!m_pCustom->getSelItem())
    {
        for(n = 0; n < 5; n++)
            pmenu->enable(n, (n == 1));
    }

    getWidget(WID_BT_CUSTOM)->getWindowRect(&rcs);

    id = pmenu->popup(NULL, rcs.x + rcs.w, rcs.y + rcs.h, AXMenu::POPF_RIGHT);

    delete pmenu;

    //--------------

    if(id == -1) return;

    switch(id)
    {
        //編集
        case 0:
            _editCustom();
            break;
        //新規追加
        case 1:
            m_pCustom->addNewGrad();
            break;
        //複製
        case 2:
            m_pCustom->copyGrad();
            break;
        //削除
        case 3:
            m_pCustom->delGrad();
            break;
        //クリップボードにコピー
        case 4:
            m_pCustom->copyClipb();
            break;
        //新規貼り付け
        case 5:
            m_pCustom->pasteClipb();
            break;
    }
}

//! 編集

void COptWin_grad::_editCustom()
{
    CGradEditDlg *pdlg;
    CGradListItem *p;

    p = m_pCustom->getSelItem();
    if(!p) return;

    pdlg = new CGradEditDlg(this, p);

    if(pdlg->runDialog())
    {
        GRADLIST->updateON();

        m_pCustom->draw();
    }
}



//*********************************
// COptWin_magicwand : 自動選択
//*********************************


COptWin_magicwand::COptWin_magicwand()
    : AXWindow(OPTIONWIN, WS_BK_FACE, LF_EXPAND_W)
{
    AXLayout *pl;
    DWORD val;

    val = g_draw->tool.optMagicWand;

    //

    _trgroup(strid::GROUP_OPTIONWIN);

    setLayout(pl = new AXLayoutMatrix(2));
    pl->setSpacing(4);

    //判定

    pl->addItem(new AXLabel(this, 0, LF_CENTER_Y, MAKE_DW4(0,0,4,5), strid::OPTWIN_JUDGE));
    pl->addItem(m_pcbType = new AXComboBox(this, 0, LF_EXPAND_W|LF_CENTER_Y, WID_TYPE, MAKE_DW4(0,0,0,5)));

    m_pcbType->addItemMulTr(strid::OPTWIN_PAINT_TYPE_TOP, 4);
    m_pcbType->setCurSel(val & 7);

    //あまさ

    pl->addItem(new AXLabel(this, 0, LF_CENTER_Y, MAKE_DW4(0,0,4,0), strid::OPTWIN_PAINT_DIFF));
    pl->addItem(m_pbarDiff = new CValBar(this, ((val & 7) == 2)? WS_DISABLE: 0, LF_EXPAND_W|LF_CENTER_Y, WID_DIFF, 0, 1, 0, 999, (val >> 3) & 1023));
}

BOOL COptWin_magicwand::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    UINT id = pwin->getItemID();

    switch(id)
    {
        //判定
        case WID_TYPE:
            g_draw->tool.optMagicWand &= ~7;
            g_draw->tool.optMagicWand |= m_pcbType->getCurSel();

            m_pbarDiff->enable(m_pcbType->getCurSel() != 2);
            break;
        //あまさ
        case WID_DIFF:
            g_draw->tool.optMagicWand &= ~(1023<<3);
            g_draw->tool.optMagicWand |= m_pbarDiff->getPos() << 3;
            break;
    }

    return TRUE;
}


//****************************************
// COptWin_sel : 選択範囲
//****************************************


COptWin_sel::COptWin_sel()
    : AXWindow(OPTIONWIN, WS_BK_FACE, 0)
{
    AXLayout *pl;
    int i;

    _trgroup(strid::GROUP_OPTIONWIN);

    setLayout(pl = new AXLayoutVert(0, 4));
    pl->setSpacing(4);

    for(i = 0; i < 2; i++)
    {
        pl->addItem(new AXCheckButton(this, AXCheckButton::CBS_RADIO, 0, WID_ADD + i, 0,
                strid::OPTWIN_SEL_TYPE_TOP + i, (g_draw->tool.optSel == i)));
    }
}

BOOL COptWin_sel::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    UINT id = pwin->getItemID();

    switch(id)
    {
        case WID_ADD:
        case WID_DEL:
            g_draw->tool.optSel = id - WID_ADD;
            break;
    }

    return TRUE;
}


//*********************************
// COptWin_rule : 定規
//*********************************


COptWin_rule::COptWin_rule()
    : AXWindow(OPTIONWIN, WS_BK_FACE, 0)
{
    AXLayout *pl,*plh;

    _trgroup(strid::GROUP_OPTIONWIN);

    setLayout(pl = new AXLayoutVert(0, 9));
    pl->setSpacing(4);

    //タイプ

    pl->addItem(m_pwidType = new COptWin_rule_type(this));

    //ボタン

    pl->addItem(plh = new AXLayoutHorz(0, 4));

    plh->addItem(new AXButton(this, WS_DISABLE, 0, WID_BTT_ANGLE, 0, strid::OPTWIN_RULE_ANGLE));
    plh->addItem(new AXButton(this, WS_DISABLE, 0, WID_BTT_POS, 0, strid::OPTWIN_RULE_CTPOS));

    _enableBtt();
}

//! 通知

BOOL COptWin_rule::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    if(pwin == m_pwidType)
        //タイプ選択変更（ボタンの有効/無効）
        _enableBtt();
    else if(pwin->getItemID() == WID_BTT_ANGLE)
        _menuAngle();
    else if(pwin->getItemID() == WID_BTT_POS)
        _menuPos();

    return TRUE;
}

//! ボタンの有効/無効

void COptWin_rule::_enableBtt()
{
    int type = g_draw->rule.type;

    enableWidget(WID_BTT_ANGLE,
                 (type == draw::RULETYPE_LINE || type == draw::RULETYPE_GRIDLINE));

    enableWidget(WID_BTT_POS,
                 (type == draw::RULETYPE_CONCLINE || type == draw::RULETYPE_CIRCLE ||
                  type == draw::RULETYPE_ELLIPSE));
}

//! メニュー：角度

void COptWin_rule::_menuAngle()
{
    AXMenu *pm;
    AXString str;
    char m[32];
    int i,angle;
    AXRectSize rcs;
    double dval;

    //現在の角度

    dval = (g_draw->rule.type == draw::RULETYPE_LINE)? g_draw->rule.dAngleLine: g_draw->rule.dAngleGrid;

    angle = (int)(dval * 18000 / 3.141592);
    while(angle < 0) angle += 36000;
    while(angle >= 36000) angle -= 36000;

    //メニュー

    pm = new AXMenu;

    for(i = 0; i < CConfig::RULEANGLE_NUM; i++)
    {
        str = '[';
        str += (i + 1);
        str += "] ";
        AXIntToFloatStr(m, g_conf->wRuleAngle[i], 2);
        str += m;

        pm->add(i, str);
    }

    pm->addSep();

    for(i = 0; i < CConfig::RULEANGLE_NUM; i++)
    {
        AXIntToFloatStr(m, angle, 2);
        str = m;
        str += " -> [";
        str += (i + 1);
        str += ']';

        pm->add(1000 + i, str);
    }

    //

    getWidget(WID_BTT_ANGLE)->getWindowRect(&rcs);

    i = pm->popup(NULL, rcs.x, rcs.y + rcs.h, 0);

    delete pm;

    //処理

    if(i != -1)
    {
        if(i < 1000)
        {
            //呼び出し

            dval = g_conf->wRuleAngle[i] * 3.141592 / 18000.0;

            if(g_draw->rule.type == draw::RULETYPE_LINE)
                g_draw->rule.dAngleLine = dval;
            else
                g_draw->rule.dAngleGrid = dval;
        }
        else
        {
            //登録

            g_conf->wRuleAngle[i - 1000] = angle;
        }
    }
}

//! メニュー:中央位置

void COptWin_rule::_menuPos()
{
    AXMenu *pm;
    AXString str;
    int i;
    AXPoint *ppt;
    AXRectSize rcs;

    //位置

    switch(g_draw->rule.type)
    {
        case draw::RULETYPE_CONCLINE:
            ppt = &g_draw->rule.ptCtConc;
            break;
        case draw::RULETYPE_CIRCLE:
            ppt = &g_draw->rule.ptCtCir;
            break;
        case draw::RULETYPE_ELLIPSE:
            ppt = &g_draw->rule.ptCtEll;
            break;
    }

    //メニュー

    pm = new AXMenu;

    for(i = 0; i < CConfig::RULEPOS_NUM; i++)
    {
        str = '[';
        str += (i + 1);
        str += "] (";
        str += g_conf->sRulePos[i * 2];
        str += ',';
        str += g_conf->sRulePos[i * 2 + 1];
        str += ')';

        pm->add(i, str);
    }

    pm->addSep();

    for(i = 0; i < CConfig::RULEPOS_NUM; i++)
    {
        str = '(';
        str += ppt->x;
        str += ',';
        str += ppt->y;
        str += ") -> [";
        str += (i + 1);
        str += ']';

        pm->add(1000 + i, str);
    }

    getWidget(WID_BTT_POS)->getWindowRect(&rcs);

    i = pm->popup(NULL, rcs.x, rcs.y + rcs.h, 0);

    delete pm;

    //処理

    if(i != -1)
    {
        if(i < 1000)
        {
            //呼び出し

            ppt->x = g_conf->sRulePos[i * 2];
            ppt->y = g_conf->sRulePos[i * 2 + 1];
        }
        else
        {
            //登録

            i -= 1000;

            g_conf->sRulePos[i * 2] = ppt->x;
            g_conf->sRulePos[i * 2 + 1] = ppt->y;
        }
    }
}

//--------------------- タイプ

COptWin_rule_type::COptWin_rule_type(AXWindow *pParent)
    : AXWindow(pParent, 0, LF_FIX_WH)
{
    m_img.create(g_res->pimgRule->getWidth(), g_res->pimgRule->getHeight());

    m_nW = m_img.getWidth();
    m_nH = m_img.getHeight();

    setImage();
}

BOOL COptWin_rule_type::onPaint(AXHD_PAINT *phd)
{
    m_img.put(m_id);
    return TRUE;
}

BOOL COptWin_rule_type::onButtonDown(AXHD_MOUSE *phd)
{
    int n,size,max;

    //選択変更

    size = m_nH - 1;
    max  = (m_nW - 1) / size;

    n = phd->x / size;
    if(n >= max) n = max - 1;

    g_draw->rule.type = n;

    setImage();

    //ボタンの有効/無効などのため親に通知

    m_pParent->onNotify(this, 0, 0);

    return TRUE;
}

void COptWin_rule_type::setImage()
{
    g_res->drawRuleImgSel(g_draw->rule.type);

    m_img.setFromImage32(*(g_res->pimgRule));

    redraw();
}


//**************************************
// COptWin_texture : テクスチャ
//**************************************


COptWin_texture::COptWin_texture()
    : AXWindow(OPTIONWIN, WS_BK_FACE, LF_EXPAND_W)
{
    AXLayout *pl;

    setLayout(pl = new AXLayoutVert(0, 6));
    pl->setSpacing(4);

    //画像選択バー

    pl->addItem(m_pSel = new CSelImgBar(this, 0, LF_EXPAND_W, 0, 0, CSelImgBar::TYPE_OPT_TEXTURE));

    m_pSel->setName(g_conf->strOptTexPath);

    //プレビュー

    pl->addItem(m_pPrev = new CImgPrev(this, 0, 0, 0, 100, 80));

    _setImage();
}

//! 通知

BOOL COptWin_texture::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    if(pwin == m_pSel)
    {
        //選択変更

        g_conf->strOptTexPath = m_pSel->getName();

        draw::loadOptTexImg();

        _setImage();
    }

    return TRUE;
}

//! プレビューイメージセット

void COptWin_texture::_setImage()
{
    if(g_draw->pimg8OptTex->isExist())
        g_draw->pimg8OptTex->drawTexturePrev(m_pPrev->getImg());
    else
        (m_pPrev->getImg())->clear(0x808080);

    m_pPrev->redraw();
}


//*********************************
// COptWin_headtail : 入り抜き
//*********************************


COptWin_headtail::COptWin_headtail()
    : AXWindow(OPTIONWIN, WS_BK_FACE, LF_EXPAND_WH)
{
    AXLayout *pl,*pl2,*plh;
    WORD val;
    int i;
    AXString str;

    val = g_draw->tool.headtail[g_conf->btOptHeadTailType];

    //

    _trgroup(strid::GROUP_OPTIONWIN);

    setLayout(pl = new AXLayoutVert(0, 3));
    pl->setSpacing(5);

    //タイプ

    pl->addItem(plh = new AXLayoutHorz);

    for(i = 0; i < 2; i++)
    {
        plh->addItem(new AXCheckButton(this, AXCheckButton::CBS_RADIO, 0,
                WID_CK_LINE + i, 0, strid::OPTWIN_HT_LINE + i, (g_conf->btOptHeadTailType == i)));
    }

    //---------

    pl->addItem(plh = new AXLayoutHorz(LF_EXPAND_WH, 8));

    //エディット

    plh->addItem(pl2 = new AXLayoutVert(0, 3));

    for(i = 0; i < 2; i++)
    {
        pl2->addItem(new AXLabel(this, 0, 0, 0, strid::OPTWIN_HT_HEAD + i));
        pl2->addItem(m_pedit[i] = new AXLineEdit(this, AXLineEdit::ES_SPIN, LF_EXPAND_W, WID_EDIT_HEAD + i, 0));
    }

    pl2->addItem(new AXButton(this, AXButton::BS_REAL_W, LF_EXPAND_W, WID_BT_REGIST, 0, strid::OPTWIN_HT_REGIST));

    m_pedit[0]->setInit(4, 0, 100, val >> 8);
    m_pedit[1]->setInit(4, 0, 100, val & 0xff);

    //リスト

    plh->addItem(m_pList = new AXListBox(this, AXScrollView::SVS_VERT|AXScrollView::SVS_SUNKEN|AXListBox::LBS_FOCUSOUT_SELOUT,
                                         LF_EXPAND_WH, WID_LIST, 0));

    for(i = 0; i < CConfig::HEADTAIL_NUM; i++)
    {
        str.setInt(g_conf->wHeadTailList[i] >> 8);
        str += "% : ";
        str += (int)(g_conf->wHeadTailList[i] & 0xff);
        str += '%';

        m_pList->addItem(str);
    }
}

//! 通知

BOOL COptWin_headtail::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    UINT id = pwin->getItemID();

    switch(id)
    {
        //直線/ベジェ曲線
        case WID_CK_LINE:
        case WID_CK_BEZIER:
            g_conf->btOptHeadTailType = id - WID_CK_LINE;

            _setVal(g_draw->tool.headtail[id - WID_CK_LINE]);
            break;
        //入り
        case WID_EDIT_HEAD:
            g_draw->tool.headtail[g_conf->btOptHeadTailType] &= 0xff;
            g_draw->tool.headtail[g_conf->btOptHeadTailType] |= (m_pedit[0]->getVal() << 8);
            break;
        //抜き
        case WID_EDIT_TAIL:
            g_draw->tool.headtail[g_conf->btOptHeadTailType] &= 0xff00;
            g_draw->tool.headtail[g_conf->btOptHeadTailType] |= m_pedit[1]->getVal();
            break;
        //登録
        case WID_BT_REGIST:
            _setListVal();
            break;
        //リスト選択
        case WID_LIST:
            if(uNotify == AXListBox::LBN_SELCHANGE)
            {
                int val = g_conf->wHeadTailList[m_pList->getCurSel()];

                g_draw->tool.headtail[g_conf->btOptHeadTailType] = val;

                _setVal(val);
            }
            break;
    }

    return TRUE;
}

//! エディットに値セット

void COptWin_headtail::_setVal(WORD val)
{
    m_pedit[0]->setVal(val >> 8);
    m_pedit[1]->setVal(val & 0xff);
}

//! 現在値をリストにセット

void COptWin_headtail::_setListVal()
{
    AXMenu *pm;
    int i,val;
    AXString str;
    AXRectSize rcs;

    //--------- メニュー

    pm = new AXMenu;

    for(i = 0; i < CConfig::HEADTAIL_NUM; i++)
    {
        val = g_conf->wHeadTailList[i];

        str = '[';
        str += (i + 1);
        str += "] ";
        str += (val >> 8);
        str += ':';
        str += (val & 0xff);

        pm->add(i, str);
    }

    getWidget(WID_BT_REGIST)->getWindowRect(&rcs);

    i = pm->popup(NULL, rcs.x + rcs.w, rcs.y, 0);

    delete pm;

    //--------- セット

    if(i != -1)
    {
        val = (m_pedit[0]->getVal() << 8) | m_pedit[1]->getVal();

        g_conf->wHeadTailList[i] = val;

        //リストテキスト変更

        str.setInt(val >> 8);
        str += "% : ";
        str += (int)(val & 0xff);
        str += '%';

        m_pList->setItemText(i, str);
    }
}


//*********************************
// COptWin_option : 設定
//*********************************


COptWin_option::COptWin_option()
    : AXWindow(OPTIONWIN, WS_BK_FACE, 0)
{
    AXLayout *pl,*plh;
    int i;

    _trgroup(strid::GROUP_OPTIONWIN);

    setLayout(pl = new AXLayoutVert(0, 3));
    pl->setSpacing(4);

    //イメージ背景色

    pl->addItem(new AXLabel(this, 0, 0, 0, strid::OPTWIN_IMGBKCOL));

    pl->addItem(plh = new AXLayoutHorz(0, 4));
    plh->addItem(m_pcbImgBk = new AXComboBox(this, AXComboBox::CBS_OWNERDRAW, 0, WID_CB_IMGBK, 0));
    plh->addItem(new AXButton(this, AXButton::BS_REAL_WH, LF_EXPAND_H, WID_BTT_IMGBK, 0, "..."));

    m_pcbImgBk->setMinWidth(80);
    m_pcbImgBk->setItemH(17);

    for(i = 0; i < CConfig::IMGBKCOL_NUM; i++)
        m_pcbImgBk->addItem((LPCUSTR)NULL);

    m_pcbImgBk->setCurSel(g_conf->btImgBkColSel);
}

//! 通知

BOOL COptWin_option::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    switch(pwin->getItemID())
    {
        //イメージ背景色コンボボックス
        case WID_CB_IMGBK:
            if(uNotify == AXComboBox::CBN_SELCHANGE)
            {
                g_conf->btImgBkColSel = m_pcbImgBk->getCurSel();
                g_conf->dwImgBkCol    = g_conf->dwImgBkColList[g_conf->btImgBkColSel];

                g_draw->view.colImgBk.set(g_conf->dwImgBkCol);

                draw::updateAll();
            }
            break;
        //イメージ背景色、色選択
        case WID_BTT_IMGBK:
            DWORD col;

            col = g_conf->dwImgBkCol;

            if(AXColorDialog::getColor(OPTIONWIN, &col))
            {
                g_conf->dwImgBkCol = col;
                g_conf->dwImgBkColList[g_conf->btImgBkColSel] = col;

                g_draw->view.colImgBk.set(g_conf->dwImgBkCol);

                m_pcbImgBk->redraw();
                draw::updateAll();
            }
            break;
    }

    return TRUE;
}

//! コンボボックス描画

BOOL COptWin_option::onOwnerDraw(AXDrawable *pdraw,AXHD_OWNERDRAW *phd)
{
    AXGC gc;

    //選択

    if(phd->uState & AXWindow::ODS_SELECTED)
        pdraw->drawFillBox(phd->rcs.x, phd->rcs.y, phd->rcs.w, phd->rcs.h, AXAppRes::BACKREVTEXT);

    //色枠

    pdraw->drawBox(phd->rcs.x + 3, phd->rcs.y + 2, phd->rcs.w - 6, phd->rcs.h - 4, AXAppRes::BLACK);

    //色

    gc.createColor(g_conf->dwImgBkColList[phd->nItemNo]);

    pdraw->drawFillBox(phd->rcs.x + 4, phd->rcs.y + 3, phd->rcs.w - 8, phd->rcs.h - 6, gc);

    return TRUE;
}
