/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    COptionDlg - 環境設定ダイアログ
*/


#include "COptionDlg.h"

#include "CConfig.h"
#include "drawdat.h"
#include "cursor.h"

#include "CImage32.h"
#include "CImgPrev.h"

#include "AXLayout.h"
#include "AXLabel.h"
#include "AXListBox.h"
#include "AXLineEdit.h"
#include "AXButton.h"
#include "AXCheckButton.h"
#include "AXComboBox.h"
#include "AXColorButton.h"
#include "AXFileDialog.h"
#include "AXFontDialog.h"

#include "CApp.h"

#include "draw_main.h"

#include "global.h"
#include "strid.h"

#include "COptionDlg_win.h"


//-----------------------

#define WID_LB_MENU     100

struct ENVWORKDAT
{
    int     nInitW,
            nInitH,
            nUndoCnt,
            nStepScaleD,
            nStepScaleU,
            nStepAngle,
            nStepBrushSize,
            nFontDefSize,
            nFontSubWinSize,
            nToolboxType;
    UINT    uUndoMaxBuf;
    DWORD   dwCanvasCol,
            dwCheckCol[2];
    UINT    uFlags;

    BOOL    bChangeCursor;
    AXMem   memDrawCursor;

    AXString    strWinFont,
                strTextureDir,
                strBrushDir;
};

enum
{
    STRID_TITLE = 0,
    STRID_MENU_TOP = 100,

    STRID_INITW = 1000,
    STRID_INITH,
    STRID_UNDOCNT,
    STRID_UNDOBUF,
    STRID_CANVBKCOL,
    STRID_CHECKCOL,
    STRID_TOOLBOXTYPE,

    STRID_FLAGS_TOP = 1100,

    STRID_FONT_FONT = 1200,
    STRID_FONT_DEFSIZE,
    STRID_FONT_SUBWINSIZE,
    STRID_FONT_HELP,

    STRID_STEP_LABEL_TOP = 1300,

    STRID_BRUSHDIR = 1400,
    STRID_TEXTUREDIR,

    STRID_CUR_HOTSPOT = 1500,
    STRID_CUR_LOAD,
    STRID_CUR_DEFAULT,
    STRID_CUR_HELP,
    
    STRID_TOOLBOXTYPE_TOP = 5000
};

//-----------------------


//***************************************
// COptionDlg
//***************************************


COptionDlg::~COptionDlg()
{
    delete m_pdat;
}

COptionDlg::COptionDlg(AXWindow *pOwner)
    : AXDialog(pOwner, WS_DIALOG_NORMAL | WS_BK_FACE)
{
    AXLayout *plTop;

    m_pwin = NULL;

    m_pdat = new ENVWORKDAT;

    //

    _trgroup(strid::GROUP_DLG_ENVOPT);

    setTitle(STRID_TITLE);

    //

    setLayout(plTop = new AXLayoutVert(0, 15));
    plTop->setSpacing(10);

    //

    plTop->addItem(m_playout = new AXLayoutHorz(LF_EXPAND_WH, 10));

    m_playout->addItem(m_plbMenu = new AXListBox(this, AXScrollView::SVS_SUNKEN, LF_EXPAND_H, WID_LB_MENU, 0));

    //メニュー

    m_plbMenu->addItemTrMul(STRID_MENU_TOP, 6);
    m_plbMenu->setAutoWidth(TRUE);
    m_plbMenu->setAutoHeight();
    m_plbMenu->setCurSel(0);

    //ボタン

    plTop->addItem(createOKCancelButton());

    //

    _getConf();

    _setWidget(TRUE);

    //

    calcDefSize();
    resize((m_nDefW < 300)? 300: m_nDefW, m_nDefH);

    show();
}

//! 通知

BOOL COptionDlg::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    switch(pwin->getItemID())
    {
        //メニューリスト
        case WID_LB_MENU:
            if(uNotify == AXListBox::LBN_SELCHANGE)
                _setWidget(FALSE);
            break;

        //OK
        case 1:
            if(m_pwin) m_pwin->getDat(m_pdat);

            endDialog(_setConf());
            break;
        //キャンセル
        case 2:
            endDialog(0);
            break;
    }

    return TRUE;
}

//! 設定データからコピー

void COptionDlg::_getConf()
{
    m_pdat->nInitW          = g_conf->nInitImgW;
    m_pdat->nInitH          = g_conf->nInitImgH;
    m_pdat->nUndoCnt        = g_conf->nUndoCnt;
    m_pdat->uUndoMaxBuf     = g_conf->uUndoMaxBuf;
    m_pdat->dwCanvasCol     = g_conf->dwCanvasBkCol;
    m_pdat->dwCheckCol[0]   = g_conf->dwBkCheckCol[0];
    m_pdat->dwCheckCol[1]   = g_conf->dwBkCheckCol[1];

    m_pdat->uFlags = g_conf->uFlags;

    m_pdat->strWinFont      = g_conf->strFontWin;
    m_pdat->nFontDefSize    = g_conf->sFontSizeDef;
    m_pdat->nFontSubWinSize = g_conf->sFontSizeSubWin;

    m_pdat->nStepScaleD    = g_conf->nScaleDownStep;
    m_pdat->nStepScaleU    = g_conf->nScaleUpStep;
    m_pdat->nStepAngle     = g_conf->nAngleStep;
    m_pdat->nStepBrushSize = g_conf->sDragBrushSizeW;
    m_pdat->nToolboxType   = g_conf->btToolboxType;

    m_pdat->strBrushDir     = g_conf->strUserBrushDir;
    m_pdat->strTextureDir   = g_conf->strUserTexDir;

    m_pdat->memDrawCursor.copy(g_conf->memDrawCursor);
    m_pdat->bChangeCursor = FALSE;
}

//! 設定データへセット
/*!
    @return 更新フラグ
*/

BOOL COptionDlg::_setConf()
{
    int flag = 0;

    //キャンバスを更新するか

    if(g_conf->dwCanvasBkCol != m_pdat->dwCanvasCol ||
       g_conf->dwBkCheckCol[0] != m_pdat->dwCheckCol[0] ||
       g_conf->dwBkCheckCol[1] != m_pdat->dwCheckCol[1])
        flag |= RETF_UPDATE_CANVAS;

    //設定1

    g_conf->nInitImgW       = m_pdat->nInitW;
    g_conf->nInitImgH       = m_pdat->nInitH;
    g_conf->nUndoCnt        = m_pdat->nUndoCnt;
    g_conf->uUndoMaxBuf     = m_pdat->uUndoMaxBuf;
    g_conf->dwCanvasBkCol   = m_pdat->dwCanvasCol;
    g_conf->dwBkCheckCol[0] = m_pdat->dwCheckCol[0];
    g_conf->dwBkCheckCol[1] = m_pdat->dwCheckCol[1];

    draw::setUndoMaxCnt(m_pdat->nUndoCnt);

    g_draw->view.colBkCheck[0].set(m_pdat->dwCheckCol[0]);
    g_draw->view.colBkCheck[1].set(m_pdat->dwCheckCol[1]);
    
    g_conf->btToolboxType = m_pdat->nToolboxType;

    //設定2

    g_conf->uFlags = m_pdat->uFlags;

    //フォント

    g_conf->strFontWin      = m_pdat->strWinFont;
    g_conf->sFontSizeDef    = m_pdat->nFontDefSize;
    g_conf->sFontSizeSubWin = m_pdat->nFontSubWinSize;

    //増減幅

    g_conf->nScaleDownStep  = m_pdat->nStepScaleD;
    g_conf->nScaleUpStep    = m_pdat->nStepScaleU;
    g_conf->nAngleStep      = m_pdat->nStepAngle;
    g_conf->sDragBrushSizeW = m_pdat->nStepBrushSize;

    //ディレクトリ

    g_conf->strUserBrushDir = m_pdat->strBrushDir;
    g_conf->strUserTexDir   = m_pdat->strTextureDir;

    //カーソル

    if(m_pdat->bChangeCursor)
    {
        g_conf->memDrawCursor.copy(m_pdat->memDrawCursor);
        flag |= RETF_UPDATE_CURSOR;
    }

    return flag;
}

//! 設定項目ウィンドウセット

void COptionDlg::_setWidget(BOOL bInit)
{
    //現在のウィンドウ削除

    if(m_pwin)
    {
        m_pwin->getDat(m_pdat);

        m_pwin->removeFromLayout();
        delete m_pwin;

        m_pwin = NULL;
    }

    //作成

    switch(m_plbMenu->getCurSel())
    {
        case 0:
            m_pwin = new COptDlg_opt1(this, m_pdat);
            break;
        case 1:
            m_pwin = new COptDlg_opt2(this, m_pdat);
            break;
        case 2:
            m_pwin = new COptDlg_font(this, m_pdat);
            break;
        case 3:
            m_pwin = new COptDlg_optstep(this, m_pdat);
            break;
        case 4:
            m_pwin = new COptDlg_optdir(this, m_pdat);
            break;
        case 5:
            m_pwin = new COptDlg_optcursor(this, m_pdat);
            break;
   }

    //セット

    if(m_pwin)
    {
        m_playout->addItem(m_pwin);

        if(!bInit)
        {
            //再レイアウト

            calcDefSize();

            if(m_nDefW > m_nW || m_nDefH > m_nH)
                resize((m_nDefW > m_nW)? m_nDefW: m_nW, (m_nDefH > m_nH)? m_nDefH: m_nH);
            else
                layout();
        }
    }
}



//*********************************
// COptDlg_base
//*********************************


COptDlg_base::COptDlg_base(AXWindow *pParent)
    : AXWindow(pParent, WS_BK_FACE, LF_EXPAND_WH)
{

}


//*********************************
// COptDlg_opt1 - 設定1
//*********************************


COptDlg_opt1::COptDlg_opt1(AXWindow *pParent,ENVWORKDAT *p)
    : COptDlg_base(pParent)
{
    AXLayout *pl,*plh;
    int i;

    _trgroup(strid::GROUP_DLG_ENVOPT);

    setLayout(pl = new AXLayoutMatrix(2));

    //エディット

    for(i = 0; i < 4; i++)
    {
        pl->addItem(new AXLabel(this, 0, LF_CENTER_Y, MAKE_DW4(0,0,6,6), STRID_INITW + i));
        pl->addItem(m_pedit[i] = new AXLineEdit(this, AXLineEdit::ES_SPIN, 0, 0, MAKE_DW4(0,0,0,6)));

    }

    m_pedit[0]->setInit(6, 1, draw::IMGSIZE_MAX, p->nInitW);
    m_pedit[1]->setInit(6, 1, draw::IMGSIZE_MAX, p->nInitH);
    m_pedit[2]->setInit(6, 2, 400, p->nUndoCnt);
    m_pedit[3]->setInit(7, 0, 256 * 1024, p->uUndoMaxBuf / 1024);

    //キャンバス背景色

    pl->addItem(new AXLabel(this, 0, LF_CENTER_Y, MAKE_DW4(0,0,6,6), STRID_CANVBKCOL));
    pl->addItem(m_pbtCol[0] = new AXColorButton(this, AXColorButton::CBTS_CHOOSE, LF_CENTER_Y, 0, MAKE_DW4(0,0,0,6), p->dwCanvasCol));

    //チェック柄色

    pl->addItem(new AXLabel(this, 0, LF_CENTER_Y, MAKE_DW4(0,0,6,6), STRID_CHECKCOL));

    pl->addItem(plh = new AXLayoutHorz(0, 5));
    plh->setPaddingBottom(6);

    for(i = 0; i < 2; i++)
    {
        plh->addItem(m_pbtCol[1 + i] = new AXColorButton(this, AXColorButton::CBTS_CHOOSE, LF_CENTER_Y));
        m_pbtCol[1 + i]->setColor(p->dwCheckCol[i]);
    }
    
    //ツールボックスタイプ
    
    pl->addItem(new AXLabel(this, 0, LF_CENTER_Y, MAKE_DW4(0,0,6,0), STRID_TOOLBOXTYPE));
    
    pl->addItem(m_pcbWinType = new AXComboBox(this, 0, LF_CENTER_Y));
    
    m_pcbWinType->addItemMulTr(STRID_TOOLBOXTYPE_TOP, 2);
    m_pcbWinType->setAutoWidth();
    m_pcbWinType->setCurSel(p->nToolboxType);
}

//! 値取得

void COptDlg_opt1::getDat(ENVWORKDAT *p)
{
    p->nInitW       = m_pedit[0]->getVal();
    p->nInitH       = m_pedit[1]->getVal();
    p->nUndoCnt     = m_pedit[2]->getVal();
    p->uUndoMaxBuf  = m_pedit[3]->getVal() * 1024;
    p->dwCanvasCol  = m_pbtCol[0]->getColor();
    p->dwCheckCol[0] = m_pbtCol[1]->getColor();
    p->dwCheckCol[1] = m_pbtCol[2]->getColor();
    p->nToolboxType = m_pcbWinType->getCurSel();
}


//*********************************
// COptDlg_opt2 - 設定2
//*********************************


COptDlg_opt2::COptDlg_opt2(AXWindow *pParent,ENVWORKDAT *p)
    : COptDlg_base(pParent)
{
    AXLayout *pl;
    int i;

    _trgroup(strid::GROUP_DLG_ENVOPT);

    setLayout(pl = new AXLayoutVert(0, 5));

    for(i = 0; i < 3; i++)
    {
        pl->addItem(m_pcheck[i] = new AXCheckButton(this, 0, 0, 0, 0,
                        STRID_FLAGS_TOP + i, p->uFlags & (1 << i)));
    }
}

//! 値取得

void COptDlg_opt2::getDat(ENVWORKDAT *p)
{
    int i;

    p->uFlags = 0;

    for(i = 0; i < 3; i++)
    {
        if(m_pcheck[i]->isChecked())
            p->uFlags |= (1 << i);
    }
}


//*********************************
// COptDlg_font - フォント
//*********************************


COptDlg_font::COptDlg_font(AXWindow *pParent,ENVWORKDAT *p)
    : COptDlg_base(pParent)
{
    AXLayout *plTop,*pl;
    int i,bDefault;

    m_strFont = p->strWinFont;

    bDefault = m_strFont.isEmpty();

    //

    _trgroup(strid::GROUP_DLG_ENVOPT);

    setLayout(plTop = new AXLayoutVert(0, 10));

    //

    plTop->addItem(m_pbtFont = new AXButton(this, 0, LF_EXPAND_W));

    if(bDefault)
        m_pbtFont->setText(_str(STRID_FONT_FONT));
    else
        m_pbtFont->setText(p->strWinFont);

    //

    plTop->addItem(pl = new AXLayoutMatrix(2));

    for(i = 0; i < 2; i++)
    {
        pl->addItem(new AXLabel(this, 0, LF_CENTER_Y, MAKE_DW4(0,0,6,6), STRID_FONT_DEFSIZE + i));
        pl->addItem(m_pedit[i] = new AXLineEdit(this, AXLineEdit::ES_SPIN | (bDefault? WS_DISABLE: 0), 0, 0, MAKE_DW4(0,0,0,6)));
    }

    m_pedit[0]->setInit(4, 1, 100, p->nFontDefSize);
    m_pedit[1]->setInit(4, 1, 100, p->nFontSubWinSize);

    //

    plTop->addItem(new AXLabel(this, 0, 0, 0, STRID_FONT_HELP));
}

//! 通知

BOOL COptDlg_font::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    if(pwin == m_pbtFont)
        _getFontName();

    return TRUE;
}

//! フォント名取得

void COptDlg_font::_getFontName()
{
    AXFontDialog::FONTINFO info;

    m_strFont.getSplit(&info.strName, &info.strStyle, ':');

    if(AXFontDialog::getFontInfo(m_pTopLevel, &info, AXFontDialog::FDFLAG_NOSIZE))
    {
        m_strFont = info.strName;
        m_strFont += ':';
        m_strFont += info.strStyle;

        m_pbtFont->setText(m_strFont);

        m_pedit[0]->enable();
        m_pedit[1]->enable();
    }
}

//! 値取得

void COptDlg_font::getDat(ENVWORKDAT *p)
{
    p->strWinFont = m_strFont;

    p->nFontDefSize    = m_pedit[0]->getVal();
    p->nFontSubWinSize = m_pedit[1]->getVal();
}


//*********************************
// COptDlg_optstep - 増減幅
//*********************************


COptDlg_optstep::COptDlg_optstep(AXWindow *pParent,ENVWORKDAT *p)
    : COptDlg_base(pParent)
{
    AXLayout *pl;
    int i;

    _trgroup(strid::GROUP_DLG_ENVOPT);

    setLayout(pl = new AXLayoutMatrix(2));

    for(i = 0; i < 4; i++)
    {
        pl->addItem(new AXLabel(this, 0, LF_CENTER_Y, MAKE_DW4(0,0,6,6), STRID_STEP_LABEL_TOP + i));
        pl->addItem(m_pedit[i] = new AXLineEdit(this, AXLineEdit::ES_SPIN, 0, 0, MAKE_DW4(0,0,0,6)));
    }

    m_pedit[0]->setInit(5, 1, 100, p->nStepScaleD);
    m_pedit[1]->setInit(5, 1, 800, p->nStepScaleU);
    m_pedit[2]->setInit(5, 1, 180, p->nStepAngle);
    m_pedit[3]->setInit(5, 1, 1000, 1, p->nStepBrushSize);
}

//! 値取得

void COptDlg_optstep::getDat(ENVWORKDAT *p)
{
    p->nStepScaleD    = m_pedit[0]->getVal();
    p->nStepScaleU    = m_pedit[1]->getVal();
    p->nStepAngle     = m_pedit[2]->getVal();
    p->nStepBrushSize = m_pedit[3]->getVal();
}


//*********************************
// COptDlg_optdir - ディレクトリ
//*********************************


COptDlg_optdir::COptDlg_optdir(AXWindow *pParent,ENVWORKDAT *p)
    : COptDlg_base(pParent)
{
    AXLayout *pl,*plh;
    int i;

    _trgroup(strid::GROUP_DLG_ENVOPT);

    setLayout(pl = new AXLayoutVert(0, 5));

    for(i = 0; i < 2; i++)
    {
        pl->addItem(new AXLabel(this, 0, 0, 0, STRID_BRUSHDIR + i));

        pl->addItem(plh = new AXLayoutHorz(LF_EXPAND_W, 3));

        plh->addItem(m_pedit[i] = new AXLineEdit(this, 0, LF_EXPAND_W));
        plh->addItem(new AXButton(this, AXButton::BS_REAL_WH, LF_EXPAND_H, WID_BUTTON + i, 0, "..."));
    }

    m_pedit[0]->setText(p->strBrushDir);
    m_pedit[1]->setText(p->strTextureDir);
}

//! 値取得

void COptDlg_optdir::getDat(ENVWORKDAT *p)
{
    m_pedit[0]->getText(&p->strBrushDir);
    m_pedit[1]->getText(&p->strTextureDir);
}

//! 通知

BOOL COptDlg_optdir::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    UINT id = pwin->getItemID();

    if(id >= WID_BUTTON && id < WID_BUTTON + 2)
    {
        AXString str;

        id -= WID_BUTTON;

        m_pedit[id]->getText(&str);

        if(AXFileDialog::openDir(m_pTopLevel, str, 0, &str))
            m_pedit[id]->setText(str);
    }

    return TRUE;
}


//*********************************
// COptDlg_optcursor : カーソル
//*********************************
/*
    m_memDrawCursor : 空でデフォルト
*/


COptDlg_optcursor::COptDlg_optcursor(AXWindow *pParent,ENVWORKDAT *p)
    : COptDlg_base(pParent)
{
    AXLayout *plTop,*plh,*plv,*pl2;
    LPBYTE pDat;

    m_memDrawCur.copy(p->memDrawCursor);

    pDat = m_memDrawCur;

    //

    _trgroup(strid::GROUP_DLG_ENVOPT);

    setLayout(plTop = new AXLayoutVert(0, 15));

    //------

    plTop->addItem(plh = new AXLayoutHorz(0, 15));

    //プレビュー

    plh->addItem(m_pPrev = new CImgPrev(this, 0, 0, 0, 32, 32));

    _drawPrev();

    //ホットスポット位置

    plh->addItem(plv = new AXLayoutVert(0, 5));

    plv->addItem(new AXLabel(this, 0, 0, 0, STRID_CUR_HOTSPOT));
    plv->addItem(pl2 = new AXLayoutHorz(0, 5));

    pl2->addItem(new AXLabel(this, 0, LF_CENTER_Y, 0, "X"));
    pl2->addItem(m_peditX = new AXLineEdit(this, AXLineEdit::ES_SPIN, 0));

    pl2->addItem(new AXLabel(this, 0, LF_CENTER_Y, 0, "Y"));
    pl2->addItem(m_peditY = new AXLineEdit(this, AXLineEdit::ES_SPIN, 0));

    m_peditX->setInit(4, 0, 31, (pDat)? pDat[2]: 0);
    m_peditY->setInit(4, 0, 31, (pDat)? pDat[3]: 0);

    //-----

    //画像から読み込み/デフォルト

    plTop->addItem(plh = new AXLayoutHorz(0, 5));

    plh->addItem(new AXButton(this, 0, 0, WID_BTT_LOADIMG, 0, STRID_CUR_LOAD));
    plh->addItem(new AXButton(this, 0, 0, WID_BTT_DEFAULT, 0, STRID_CUR_DEFAULT));

    //ヘルプ

    plTop->addItem(new AXLabel(this, AXLabel::LS_BORDER, LF_EXPAND_W, 0, STRID_CUR_HELP));
}

//! 値取得

void COptDlg_optcursor::getDat(ENVWORKDAT *p)
{
    LPBYTE pCur = m_memDrawCur;

    //ホットスポット位置

    if(pCur)
    {
        pCur[2] = m_peditX->getVal();
        pCur[3] = m_peditY->getVal();

        if(pCur[2] > pCur[0]) pCur[2] = pCur[0] - 1;
        if(pCur[3] > pCur[1]) pCur[3] = pCur[1] - 1;
    }

    //

    p->bChangeCursor = TRUE;

    p->memDrawCursor.copy(m_memDrawCur);
}

//! 通知

BOOL COptDlg_optcursor::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    switch(pwin->getItemID())
    {
        //画像から読み込み
        case WID_BTT_LOADIMG:
            _loadImg();
            break;
        //デフォルト
        case WID_BTT_DEFAULT:
            m_memDrawCur.free();

            _drawPrev();
            break;
    }

    return TRUE;
}

//! 画像から読み込み

void COptDlg_optcursor::_loadImg()
{
    AXString str,filter;
    CImage32 img;

    //ファイル名

    CAPP->getFileFilter_normalimg(&filter);

    if(!AXFileDialog::openFile(m_pTopLevel, filter, 0, NULL, 0, &str))
        return;

    //作成

    if(!img.createCursor(str, &m_memDrawCur))
        ((AXTopWindow *)m_pTopLevel)->errMes(strid::GROUP_MESSAGE, strid::MES_FAILED_LOAD);
    else
        _drawPrev();
}

//! カーソルプレビュー描画

void COptDlg_optcursor::_drawPrev()
{
    AXImage *pimg;
    const unsigned char *pCur,*pCurCol,*pCurMask;
    int w,h,ix,iy,pitchCur,xpos,f;

    pimg = m_pPrev->getImg();

    pimg->clear(0xcccccc);

    //カーソルデータ

    pCur = m_memDrawCur;
    if(!pCur) pCur = cursor::getImgDat(cursor::DRAW);

    w = pCur[0];
    h = pCur[1];
    pitchCur = (w + 7) >> 3;

    //描画

    pCurCol  = pCur + 4;
    pCurMask = pCur + 4 + pitchCur * h;

    for(iy = 0; iy < h; iy++)
    {
        for(ix = 0, f = 1, xpos = 0; ix < w; ix++)
        {
            if(pCurMask[xpos] & f)
                pimg->setPixel(ix, iy, (pCurCol[xpos] & f)? 0: 0xffffff);

            f <<= 1;
            if(f == 256) f = 1, xpos++;
        }

        pCurCol  += pitchCur;
        pCurMask += pitchCur;
    }

    m_pPrev->redraw();
}
