/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CMainWin - メインウィンドウ
*/

#include "CMainWin.h"

#include "CCanvasWin.h"
#include "CStatusBar.h"
#include "CToolWin.h"
#include "CBrushWin.h"
#include "COptionWin.h"
#include "CLayerWin.h"
#include "CColorWin.h"
#include "CPrevWin.h"
#include "CImgViewWin.h"
#include "CFilterListWin.h"

#include "COptionDlg.h"
#include "CKeyOptDlg.h"
#include "CDeviceOptDlg.h"
#include "CGridOptDlg.h"
#include "CTrimmingDlg.h"
#include "dialog_canvas.h"

#include "CConfig.h"
#include "CResource.h"
#include "CUndo.h"

#include "drawdat.h"
#include "draw_main.h"
#include "draw_update.h"
#include "draw_layer.h"

#include "AXLayout.h"
#include "AXMenu.h"
#include "AXMenuBar.h"
#include "AXAccelerator.h"
#include "AXToolBar.h"
#include "AXDND.h"
#include "AXAboutDialog.h"
#include "AXStrDialog.h"
#include "AXKey.h"
#include "AXMessageBox.h"
#include "AXMem.h"
#include "AXUtil.h"
#include "AXUtilFile.h"
#include "AXUtilRes.h"
#include "AXApp.h"

#include "global.h"
#include "strid.h"

#include "menudat.h"

//----------------------

#define _APP_VERSION   "AzPainter ver 1.0.7"
#define _APP_COPYRIGHT "Copyright (C) 2013-2016 Azel"

//----------------------


CMainWin *CMainWin::m_pSelf = NULL;


CMainWin::~CMainWin()
{
    if(m_pAccel) delete m_pAccel;
}

CMainWin::CMainWin()
    : AXTopWindow(NULL, WS_TOP_NORMAL | WS_HIDE | WS_DISABLE_IM | WS_DROP)
{
    AXLayout *plTop;
    AXMem mem;

    m_pSelf = this;

    m_uTmpFlags = 0;

    enableDND();

    //メインレイアウト

    setLayout(plTop = new AXLayoutVert);

    //メニュー・ツールバー

    _createMenu();
    _createToolBar();

    //キャンバスとステータスバー

    plTop->addItem(new CCanvasWin(this));
    plTop->addItem(new CStatusBar(this));

    //アクセラレータ

    _setAccelerator();

    //アイコン

    if(AXLoadResAppIcon(&mem, "icon16.png"))
        setAppIcon(mem, TRUE);
}

//! ウィンドウ初期表示

void CMainWin::showMainWin()
{
    int w,h;

    setWinTitle();

    //サイズ変更

    w = g_conf->rcsMainWin.w;
    h = g_conf->rcsMainWin.h;

    if(w == CConfig::WININITVAL) w = 550;
    if(h == CConfig::WININITVAL) h = 450;

    if(g_conf->bMaximized)
        w = 550, h = 450;

    resize(w, h);

    //表示

    draw::initAfterShow();

    STATUSBAR->setLabel_imginfo();
    STATUSBAR->setLabel_help();

    show();

    //位置

    if(g_conf->rcsMainWin.x == CConfig::WININITVAL ||
        g_conf->rcsMainWin.y == CConfig::WININITVAL)
        moveCenter();
    else
        moveInRoot(g_conf->rcsMainWin.x, g_conf->rcsMainWin.y);

    //最大化

    if(g_conf->bMaximized)
    {
        maximize();

        m_uTmpFlags |= TMPFLAG_MAXIMIZE_INIT;
    }
}


//==============================
// ハンドラ
//==============================


//! 閉じる

BOOL CMainWin::onClose()
{
	if(checkImageUpdate())
		axapp->exit();

    return TRUE;
}

//! 表示時

BOOL CMainWin::onMap()
{
    m_uTmpFlags |= TMPFLAG_SHOW;

    return TRUE;
}

//! サイズなど変更時

BOOL CMainWin::onConfigure(AXHD_CONFIGURE *phd)
{
    AXTopWindow::onConfigure(phd);

    /* [初期表示で最大化時]
        初期表示が最大化の場合に、キャンバスサイズからの表示倍率計算が正しく行えるようにする。
        サイズがウィンドウマネージャによって変更された時に実行。 */

    if((m_uTmpFlags & TMPFLAG_MAXIMIZE_INIT) && isMaximized())
    {
        m_uTmpFlags &= ~TMPFLAG_MAXIMIZE_INIT;

        draw::initAfterShow();
    }

    return TRUE;
}

//! D&D

BOOL CMainWin::onDND_Drop(AXDND *pDND)
{
    AXString str;

    if(pDND->getFirstFile(&str))
    {
        pDND->endDrop();

        //

        if(checkImageUpdate())
            loadImage(str);
    }

    return TRUE;
}

//! コマンド
/*!
    キー設定でのショートカットキーやデバイス設定でのコマンドとしても実行される。
*/

BOOL CMainWin::onCommand(UINT uID,ULONG lParam,int from)
{
    //キャンバス操作中は実行しない

    if(!g_draw->isNowOpNone()) return TRUE;

    //ショートカットキーなどのコマンド

    if(uID >= 1000 && uID < 10000)
    {
        _command_ex(uID);
        return TRUE;
    }

    //レイヤ

    if(uID >= strid::MENU_LAYER_TOPID && uID <= strid::MENU_LAYER_ENDID)
        return _cmd_layer(uID);

    //ファイル履歴

    if(uID >= CMDID_MENU_RECENTFILE && uID < CMDID_MENU_RECENTFILE + CConfig::RECFILE_NUM)
    {
        if(AXIsExistFile(g_conf->strRecFile[uID - CMDID_MENU_RECENTFILE]))
        {
            if(checkImageUpdate())
                loadImage(g_conf->strRecFile[uID - CMDID_MENU_RECENTFILE]);
        }

        return TRUE;
    }

    //フィルタ

    if(uID >= strid::MENU_FILTER_TOP && uID < strid::MENU_FILTER_END)
    {
        _command_filter(uID, from);
        return TRUE;
    }

    //指定形式でファイルに出力

    if(uID >= strid::MENU_FILE_FORMAT_APD && uID <= strid::MENU_FILE_FORMAT_PNG_ALPHA)
    {
        exportFile(uID - strid::MENU_FILE_FORMAT_APD);
        return TRUE;
    }

    //-------------

    switch(uID)
    {
        //新規作成
        case strid::MENU_FILE_NEW:
            newImage();
            break;
        //開く
        case strid::MENU_FILE_OPEN:
            if(from == COMFROM_TOOLBAR_DROP)
                _runRecOpenDirMenu();
            else
                openFile();
            break;
        //上書き保存
        case strid::MENU_FILE_SAVE:
            saveFile(FALSE);
            break;
        //別名保存
        case strid::MENU_FILE_SAVERE:
            if(from == COMFROM_TOOLBAR_DROP)
                _runRecSaveDirMenu();
            else
                saveFile(TRUE);
            break;
        //終了
        case strid::MENU_FILE_EXIT:
            onClose();
            break;

        //元に戻す
        case strid::MENU_EDIT_UNDO:
            undoRedo(FALSE);
            break;
        //やり直す
        case strid::MENU_EDIT_REDO:
            undoRedo(TRUE);
            break;
        //塗りつぶし
        case strid::MENU_EDIT_PAINT:
            draw::selPaintErase(FALSE);
            break;
        //消去
        case strid::MENU_EDIT_ERASE:
            draw::selPaintErase(TRUE);
            break;
        //キャンバスサイズ変更
        case strid::MENU_EDIT_RESIZECANVAS:
            resizeCanvas();
            break;
        //キャンバス拡大縮小
        case strid::MENU_EDIT_SCALECANVAS:
            scaleCanvas();
            break;
        //トリミング
        case strid::MENU_EDIT_TRIMMING:
            trimming();
            break;
        //DPI値変更
        case strid::MENU_EDIT_CHANGEDPI:
            changeDPI();
            break;

        //選択解除
        case strid::MENU_SEL_DESELECT:
            draw::selRelease(TRUE);
            break;
        //すべて選択
        case strid::MENU_SEL_ALL:
            draw::selAll();
            break;
        //選択反転
        case strid::MENU_SEL_INVERSE:
            draw::selInverse();
            break;
        //コピー
        case strid::MENU_SEL_COPY:
            draw::selCopy(FALSE);
            break;
        //切り取り
        case strid::MENU_SEL_CUT:
            draw::selCopy(TRUE);
            break;
        //貼り付け
        case strid::MENU_SEL_PASTE:
            draw::selPaste();
            break;
        //1px拡張
        case strid::MENU_SEL_INFLATE:
            draw::selInflateDeflate(TRUE);
            break;
        //1px縮小
        case strid::MENU_SEL_DEFLATE:
            draw::selInflateDeflate(FALSE);
            break;

        //ツールウィンドウ
        case strid::MENU_VIEW_TOOLWIN:
            g_conf->uWinViewFlags ^= CConfig::WINVIEWF_TOOL;
            TOOLWIN->showToggle();
            break;
        //レイヤウィンドウ
        case strid::MENU_VIEW_LAYERWIN:
            g_conf->uWinViewFlags ^= CConfig::WINVIEWF_LAYER;
            LAYERWIN->showToggle();
            break;
        //ブラシウィンドウ
        case strid::MENU_VIEW_BRUSHWIN:
            g_conf->uWinViewFlags ^= CConfig::WINVIEWF_BRUSH;
            BRUSHWIN->showToggle();
            break;
        //オプションウィンドウ
        case strid::MENU_VIEW_OPTIONWIN:
            g_conf->uWinViewFlags ^= CConfig::WINVIEWF_OPTION;
            OPTIONWIN->showToggle();
            break;
        //カラーウィンドウ
        case strid::MENU_VIEW_COLORWIN:
            g_conf->uWinViewFlags ^= CConfig::WINVIEWF_COLOR;
            COLORWIN->showToggle();
            break;
        //プレビューウィンドウ
        case strid::MENU_VIEW_PREVWIN:
            g_conf->uWinViewFlags ^= CConfig::WINVIEWF_PREV;
            PREVWIN->showToggle();
            break;
        //イメージビューウィンドウ
        case strid::MENU_VIEW_IMGVIEWWIN:
            g_conf->uWinViewFlags ^= CConfig::WINVIEWF_IMGVIEW;
            IMGVIEWWIN->showToggle();
            break;
        //フィルタ一覧ウィンドウ
        case strid::MENU_VIEW_FILTERLISTWIN:
            g_conf->uWinViewFlags ^= CConfig::WINVIEWF_FILTERLIST;
            FILTERLISTWIN->showToggle();
            break;
        //全ウィンドウ非表示/復元
        case strid::MENU_VIEW_ALLHIDE:
            allWindowShowToggle();
            break;
        //表示倍率・回転リセット
        case strid::MENU_VIEW_RESETCANVAS:
            draw::resetCanvasView();
            break;
        //キャンバス左右反転表示
        case strid::MENU_VIEW_CANVASHREV:
            draw::canvasHRev();

            m_pToolBar->checkItem(uID, g_draw->isCanvasHRev());
            break;
        //背景チェック柄
        case strid::MENU_VIEW_BKCHECK:
            g_conf->uEtcViewFlags ^= CConfig::ETCVIEWF_BK_CHECK;

            m_pToolBar->checkItem(uID, g_conf->isBkCheck());

            draw::updateAll();
            break;
        //グリッド
        case strid::MENU_VIEW_GRID:
            g_conf->uEtcViewFlags ^= CConfig::ETCVIEWF_GRID;
            draw::updateCanvas();
            break;
        //分割線
        case strid::MENU_VIEW_GRIDSPLIT:
            g_conf->uEtcViewFlags ^= CConfig::ETCVIEWF_GRIDSPLIT;
            draw::updateCanvas();
            break;
        //ツールバー表示
        case strid::MENU_VIEW_TOOLBAR:
            g_conf->uEtcViewFlags ^= CConfig::ETCVIEWF_TOOLBAR;

            m_pToolBar->show(g_conf->isToolBar());

            //ツールバー非表示の場合、メニューバーのセパレータなし
            getMenuBar()->changeStyle(AXMenuBar::MBS_SEP_BOTTOM, g_conf->isToolBar());
            getMenuBar()->calcDefSize();

            layout();
            break;
        //スクロールバー表示
        case strid::MENU_VIEW_SCROLLBAR:
            g_conf->uEtcViewFlags ^= CConfig::ETCVIEWF_SCROLLBAR;
            CANVASWIN->reconfig();
            break;

        //環境設定
        case strid::MENU_OPT_ENV:
            envOption();
            break;
        //デバイス設定
        case strid::MENU_OPT_DEVICE:
            deviceOption();
            break;
        //キー設定
        case strid::MENU_OPT_KEY:
            keyOption();
            break;
        //グリッド設定
        case strid::MENU_OPT_GRID:
            gridOption();
            break;

        //マニュアル
        case strid::MENU_HELP_MANUAL_JA:
            runManual();
            break;
        //情報
        case strid::MENU_HELP_VERSION:
            AXAboutDialog::about(this, _APP_VERSION, _APP_COPYRIGHT);
            break;

        //----------- ツールバーボタン

        //倍率メニュー
        case strid::MAINEX_SCALEMENU:
            _runScaleMenu();
            break;
        //1段階拡大
        case strid::MAINEX_SCALE_UP:
            draw::scaleUpDown(TRUE);
            break;
        //1段階縮小
        case strid::MAINEX_SCALE_DOWN:
            draw::scaleUpDown(FALSE);
            break;
        //左に回転
        case strid::MAINEX_ROTATE_LEFT:
            draw::changeRotate(g_draw->view.nAngle - g_conf->nAngleStep * 100);
            break;
        //右に回転
        case strid::MAINEX_ROTATE_RIGHT:
            draw::changeRotate(g_draw->view.nAngle + g_conf->nAngleStep * 100);
            break;
        //指定形式でファイルに出力
        case strid::MAINEX_EXPORT:
            if(from == COMFROM_TOOLBAR_DROP)
                _runTBExportMenu();
            else
                exportFile(g_conf->btExportType);
            break;
    }

    return TRUE;
}

//! ポップアップメニュー表示時

BOOL CMainWin::onMenuPopup(AXMenuBar *pMenuBar,AXMenu *pMenu,UINT uID)
{
    if(pMenuBar == m_pMenuBar)
    {
        if(uID == strid::MENU_TOP_VIEW)
        {
            pMenu->check(strid::MENU_VIEW_TOOLWIN, g_conf->isToolWin());
            pMenu->check(strid::MENU_VIEW_LAYERWIN, g_conf->isLayerWin());
            pMenu->check(strid::MENU_VIEW_BRUSHWIN, g_conf->isBrushWin());
            pMenu->check(strid::MENU_VIEW_OPTIONWIN, g_conf->isOptionWin());
            pMenu->check(strid::MENU_VIEW_COLORWIN, g_conf->isColorWin());
            pMenu->check(strid::MENU_VIEW_PREVWIN, g_conf->isPrevWin());
            pMenu->check(strid::MENU_VIEW_IMGVIEWWIN, g_conf->isImgViewWin());
            pMenu->check(strid::MENU_VIEW_FILTERLISTWIN, g_conf->isFilterWin());

            pMenu->check(strid::MENU_VIEW_GRID, g_conf->isGrid());
            pMenu->check(strid::MENU_VIEW_BKCHECK, g_conf->isBkCheck());
            pMenu->check(strid::MENU_VIEW_GRID, g_conf->isGrid());
            pMenu->check(strid::MENU_VIEW_GRIDSPLIT, g_conf->isGridSplit());

            pMenu->check(strid::MENU_VIEW_CANVASHREV, g_draw->isCanvasHRev());
            pMenu->check(strid::MENU_VIEW_TOOLBAR, g_conf->isToolBar());
            pMenu->check(strid::MENU_VIEW_SCROLLBAR, g_conf->isScrollBar());
        }
    }

    return TRUE;
}


//==============================
// サブ処理
//==============================


//! ファイルダイアログ時の初期ディレクトリ取得

void CMainWin::getFileInitDir(AXString *pstr)
{
    if(m_strFileName.isEmpty())
        *pstr = g_conf->strSaveDir;
    else
        pstr->path_removeFileName(m_strFileName);
}

//! ウィンドウタイトルのセット

void CMainWin::setWinTitle()
{
    AXString str,str2;

    str = "AzPainter - ";

    //ファイル名

    if(m_strFileName.isEmpty())
        str += "New";
    else
    {
        //ディレクトリ除く
        str2.path_filename(m_strFileName);
        str += str2;
    }

    setTitle(str);
}

//! 保存確認
/*!
    @return FALSE で処理をキャンセル
*/

BOOL CMainWin::checkImageUpdate()
{
    UINT ret;

    if(draw::isChangeImage())
    {
        ret = AXMessageBox::message(this, NULL,
                    _string(strid::GROUP_MESSAGE, strid::MES_SAVE_CHANGEIMAGE),
                    AXMessageBox::SAVE | AXMessageBox::SAVENO | AXMessageBox::CANCEL,
                    AXMessageBox::CANCEL);

        //キャンセル

        if(ret == AXMessageBox::CANCEL) return FALSE;

        //保存する

        if(ret == AXMessageBox::SAVE)
        {
            if(!saveFile(FALSE)) return FALSE;
        }
    }

    return TRUE;
}

//! 編集サイズが変更された後の更新
/*!
    @param pstr ファイル名。NULLで変更なし。空で新規作成
*/

void CMainWin::updateNewCanvas(AXString *pstr)
{
    draw::changeImageSize();
    draw::setScroll_default();

    //------- ファイル変更時

    if(pstr)
    {
        //編集ファイル名・タイトル

        m_strFileName = *pstr;
        setWinTitle();

        //

        draw::changeFile();
    }

    //------- 各更新

    STATUSBAR->setLabel_imginfo();

    draw::changeCanvasWinSize(FALSE);

    PREVWIN->changeImgSize();

    draw::updateAllAndLayer();
}

//! 表示倍率メニュー実行

void CMainWin::_runScaleMenu()
{
    AXMenu *pmenu;
    AXString str;
    AXRect rc;
    int i;

    pmenu = new AXMenu;

    for(i = 25; i <= draw::SCALE_MAX / 10; )
    {
        str.setInt(i);
        str += '%';

        pmenu->add(i, str);

        i += (i < 100)? 25: 100;
    }

    m_pToolBar->getItemRect(strid::MAINEX_SCALEMENU, &rc, TRUE);

    i = pmenu->popup(NULL, rc.left, rc.bottom, 0);

    delete pmenu;

    //倍率変更

    if(i != -1)
        draw::changeScale(i * 10);
}


//==============================
// コマンド処理
//==============================


//! アンドゥ・リドゥ

void CMainWin::undoRedo(BOOL bRedo)
{
    AXRectSize rcs;
    int type,ret;

    //アンドゥ・リドゥ

    CANVASAREA->setCursorWait();

    ret = draw::undoRedo(bRedo, &type, &rcs);

    CANVASAREA->restoreCursorTool();

    if(!ret) return;

    //更新

    switch(type)
    {
        //範囲
        case CUndo::UPDATE_RECT:
            draw::updateRectAndPrev(rcs);
            break;
        //範囲＋レイヤ一覧
        case CUndo::UPDATE_RECTANDLAYER:
            draw::updateRectAndPrev(rcs);
            LAYERWIN->updateLayerAll();
            break;
        //イメージ・キャンバス
        case CUndo::UPDATE_ALL:
            draw::updateAll();
            break;
        //イメージ・キャンバス・レイヤ一覧
        case CUndo::UPDATE_ALLANDLAYER:
            draw::updateAllAndLayer();
            break;
        //レイヤ一覧
        case CUndo::UPDATE_LAYERLIST:
            LAYERWIN->updateLayerAll();
            break;
        //サイズ変更
        case CUndo::UPDATE_RESIZE:
            draw::changeCanvasImgSize(rcs.w, rcs.h);
            updateNewCanvas(NULL);
            break;
    }
}

//! キャンバスサイズ変更

void CMainWin::resizeCanvas()
{
    CResizeCanvasDlg *pdlg;
    int val[3],n,topx,topy;

    //ダイアログ

    val[0] = g_draw->nImgW;
    val[1] = g_draw->nImgH;

    pdlg = new CResizeCanvasDlg(this, val);
    if(!pdlg->runDialog()) return;

    //サイズが同じ

    if(val[0] == g_draw->nImgW && val[1] == g_draw->nImgH)
        return;

    //左上位置

    n = val[2] % 3;

    if(n == 0) topx = 0;
    else if(n == 1) topx = (val[0] - g_draw->nImgW) / 2;
    else topx = val[0] - g_draw->nImgW;

    n = val[2] / 3;

    if(n == 0) topy = 0;
    else if(n == 1) topy = (val[1] - g_draw->nImgH) / 2;
    else topy = val[1] - g_draw->nImgH;

    //実行

    if(!draw::resizeCanvas(val[0], val[1], topx, topy))
        errMes(strid::GROUP_MESSAGE, strid::MES_FAILED);
    else
        updateNewCanvas(NULL);
}

//! キャンバス拡大縮小

void CMainWin::scaleCanvas()
{
    CScaleCanvasDlg *pdlg;
    CScaleCanvasDlg::VALUE val;

    //ダイアログ

    val.nWidth  = g_draw->nImgW;
    val.nHeight = g_draw->nImgH;
    val.nDPI    = g_draw->nImgDPI;
    val.nType   = g_conf->btScaleCanvType;

    pdlg = new CScaleCanvasDlg(this, &val);
    if(!pdlg->runDialog()) return;

    //補間方法保存

    g_conf->btScaleCanvType = val.nType;

    //サイズが同じ

    if(g_draw->nImgW == val.nWidth && g_draw->nImgH == val.nHeight)
        return;

    //処理

    if(!draw::scaleCanvas(val.nWidth, val.nHeight, val.nDPI, val.nType))
        errMes(strid::GROUP_MESSAGE, strid::MES_FAILED);
    else
        updateNewCanvas(NULL);
}

//! トリミング

void CMainWin::trimming()
{
    CTrimmingDlg *pdlg;
    AXRectSize rcs;

    pdlg = new CTrimmingDlg(this, &rcs);

    if(pdlg->runDialog())
    {
        if(!draw::resizeCanvas(rcs.w, rcs.h, -rcs.x, -rcs.y))
            errMes(strid::GROUP_MESSAGE, strid::MES_FAILED);
        else
            updateNewCanvas(NULL);
    }
}

//! DPI値変更

void CMainWin::changeDPI()
{
    AXString str;
    int n;

    _trgroup(strid::GROUP_DLG_CHANGEDPI);

    str.setInt(g_draw->nImgDPI);

    if(AXStrDialog::getString(this, _str((WORD)0), _str(1), &str))
    {
        n = str.toInt();
        if(n < 1) n = 1; else if(n > 10000) n = 10000;

        g_draw->nImgDPI = n;

        STATUSBAR->setLabel_imginfo();
    }
}

//! 環境設定

void CMainWin::envOption()
{
    COptionDlg *pdlg = new COptionDlg(this);
    int ret;

    ret = pdlg->runDialog();

    if(ret & COptionDlg::RETF_UPDATE_CANVAS)
        draw::updateAll();

    if(ret & COptionDlg::RETF_UPDATE_CURSOR)
        CANVASAREA->setCursorTool();
}

//! デバイス設定

void CMainWin::deviceOption()
{
    CDeviceOptDlg *pdlg = new CDeviceOptDlg(this);

    pdlg->runDialog();
}

//! キー設定

void CMainWin::keyOption()
{
    CKeyOptDlg *pdlg = new CKeyOptDlg(this);

    pdlg->runDialog();
}

//! グリッド設定

void CMainWin::gridOption()
{
    CGridOptDlg *pdlg = new CGridOptDlg(this);

    if(pdlg->runDialog())
    {
        if(g_conf->isGrid())
            draw::updateCanvas();
    }
}

//! 全ウィンドウ非表示/復元

void CMainWin::allWindowShowToggle()
{
    int i;
    UINT flag;

    if(g_conf->uEtcViewFlags & CConfig::ETCVIEWF_ALLHIDE)
    {
        //復元（閉じる前と表示状態が違う時のみ）

        for(i = 0, flag = 1; i < 8; i++, flag <<= 1)
        {
            if((g_conf->uWinViewFlags & flag) != (g_conf->uWinViewBkFlags & flag))
                onCommand(strid::MENU_VIEW_TOOLWIN + i, 0, 0);
        }
    }
    else
    {
        //非表示

        g_conf->uWinViewBkFlags = g_conf->uWinViewFlags;

        for(i = 0; i < 8; i++)
        {
            if(g_conf->uWinViewFlags & (1 << i))
                onCommand(strid::MENU_VIEW_TOOLWIN + i, 0, 0);
        }
    }

    g_conf->uEtcViewFlags ^= CConfig::ETCVIEWF_ALLHIDE;
}

//! マニュアル

void CMainWin::runManual()
{
    AXString str;

    axapp->getResourcePath(&str, "manual_ja.html");
    str.prefix("file://");

    AXExeBrowser(str);
}

//! ショートカットキーなどの拡張コマンド

void CMainWin::_command_ex(UINT id)
{
    if(id < 1100)
        //ツール選択 (1000-)
        draw::changeTool(id - 1000);
    else if(id < 1200)
    {
        //描画タイプ変更 (1100-) [ブラシ・ドット描画時]

        if(g_draw->tool.isBrushOrDotPen())
        {
            draw::changeToolSubType(id - 1100);
            TOOLWIN->checkTB_toolsub_pen(id - 1100);
        }
    }
    else if(id < 1300)
    {
        //レイヤ拡張コマンド (1200-)

        switch(id - 1200)
        {
            case 0: draw::layer_curSelUpDown(TRUE); break;  //レイヤ選択 up
            case 1: draw::layer_curSelUpDown(FALSE); break; //レイヤ選択 down
            //カレントレイヤ表示/非表示
            case 2:
                draw::layer_revShow(g_draw->pcurlayer);
                LAYERWIN->updateCurLayer(FALSE);
                break;
        }
    }
    else if(id < 1400)
    {
        //他コマンド (1300-)

        switch(id - 1300)
        {
            case 0: draw::scaleUpDown(TRUE); break;      //1段階拡大
            case 1: draw::scaleUpDown(FALSE); break;     //1段階縮小
            case 2: draw::changeScale(1000); break;      //100%
            case 3: draw::changeRotate(g_draw->view.nAngle - g_conf->nAngleStep * 100); break; //1段階左に回転
            case 4: draw::changeRotate(g_draw->view.nAngle + g_conf->nAngleStep * 100); break; //1段階右に回転
            case 5: draw::changeRotate(0); break;        //回転0
            case 6: BRUSHWIN->toggleRegBrush(); break;   //登録ブラシ/直前ブラシ切替
            //ブラシ選択移動
            case 7:
            case 8:
                BRUSHWIN->moveSelBrush(id - 1307);
                break;
            //描画色/背景色切り替え
            case 9:
                draw::toggleDrawCol();
                COLORWIN->updateDrawColor();
                break;
        }
    }
}


//==============================
// 初期化処理
//==============================


//! メニュー作成

void CMainWin::_createMenu()
{
    AXMenuBar *pbar;

    _trgroup(strid::GROUP_MAINMENU);

    //------ メニューバー

    pbar = new AXMenuBar(this,
                (g_conf->isToolBar())? AXMenuBar::MBS_SEP_BOTTOM: 0,
                LF_EXPAND_W);

    getLayout()->addItem(pbar);

    attachMenuBar(pbar);

    //------ 項目

    pbar->addItemTrMultiple(g_mainmenudat, sizeof(g_mainmenudat)/sizeof(WORD), strid::MENU_FILE_NEW);

    //最近使ったファイル

    m_pmenuRecFile = new AXMenu;
    m_pmenuRecFile->setStrArray(CMDID_MENU_RECENTFILE, g_conf->strRecFile, CConfig::RECFILE_NUM);

    pbar->setItemSubMenu(strid::MENU_FILE_SUB_RECENTFILE, m_pmenuRecFile);
}

//! ツールバー作成

void CMainWin::_createToolBar()
{
    DWORD btt[] = {
        strid::MENU_FILE_NEW, strid::MENU_FILE_OPEN|0x20000, strid::MENU_FILE_SAVE,
        strid::MENU_FILE_SAVERE|0x20000, strid::MAINEX_EXPORT|0x20000, 0xffff,
        strid::MENU_VIEW_CANVASHREV|0x10000, strid::MENU_VIEW_BKCHECK|0x10000,
        strid::MENU_VIEW_GRID|0x10000, 0xffff,
        strid::MAINEX_SCALEMENU, strid::MAINEX_SCALE_UP, strid::MAINEX_SCALE_DOWN, 0xffff,
        strid::MAINEX_ROTATE_LEFT, strid::MAINEX_ROTATE_RIGHT, 0xffff,
        strid::MENU_EDIT_UNDO, strid::MENU_EDIT_REDO, 0xffff,
        strid::MENU_SEL_DESELECT
    };
    int f,imgno;
    UINT i,style;

    //作成

    m_pToolBar = new AXToolBar(this,
            AXToolBar::TBS_TOOLTIP | AXToolBar::TBS_TOOLTIP_TR | (g_conf->isToolBar()? 0: WS_HIDE),
            LF_EXPAND_W);

    m_pToolBar->setImageList(g_res->pilMainTB);
    m_pToolBar->setToolTipTrGroup(strid::GROUP_TOOLTIP_MAIN);

    getLayout()->addItem(m_pToolBar);

    //ボタン

    for(i = 0, imgno = 0; i < sizeof(btt)/sizeof(DWORD); i++)
    {
        f = btt[i];

        if(f == 0xffff)
            m_pToolBar->addItemSep();
        else
        {
            style = (f & 0x10000)? AXToolBar::BF_CHECKBUTTON: AXToolBar::BF_BUTTON;
            if(f & 0x20000) style |= AXToolBar::BF_DROPDOWN;

            m_pToolBar->addItem(f & 0xffff, imgno, f & 0xffff, style, 0);
            imgno++;
        }
    }

    //チェック

    if(g_conf->isBkCheck()) m_pToolBar->checkItem(strid::MENU_VIEW_BKCHECK, TRUE);
    if(g_conf->isGrid()) m_pToolBar->checkItem(strid::MENU_VIEW_GRID, TRUE);
}

//! アクセラレータセット

void CMainWin::_setAccelerator()
{
    UINT i;
    WORD id[] = {
        strid::MENU_FILE_NEW, strid::MENU_FILE_OPEN, strid::MENU_FILE_SAVE, strid::MENU_FILE_SAVERE,
        strid::MENU_EDIT_UNDO, strid::MENU_EDIT_REDO, strid::MENU_EDIT_PAINT,
        strid::MENU_EDIT_ERASE, strid::MENU_EDIT_TRIMMING, strid::MENU_SEL_DESELECT, strid::MENU_SEL_INVERSE,
        strid::MENU_SEL_ALL, strid::MENU_SEL_COPY, strid::MENU_SEL_CUT, strid::MENU_SEL_PASTE,
        strid::MENU_SEL_INFLATE, strid::MENU_SEL_DEFLATE,
        strid::MENU_VIEW_ALLHIDE, strid::MENU_VIEW_RESETCANVAS, strid::MENU_VIEW_CANVASHREV, strid::MENU_VIEW_BKCHECK,
        strid::MENU_VIEW_GRID, strid::MENU_VIEW_GRIDSPLIT,

        strid::MENU_LAYER_NEWLAYER, strid::MENU_LAYER_NEWFOLDER, strid::MENU_LAYER_FROMIMG,
        strid::MENU_LAYER_COPY, strid::MENU_LAYER_DELETE, strid::MENU_LAYER_CLEAR,
        strid::MENU_LAYER_IMGDROP, strid::MENU_LAYER_COMBINE, strid::MENU_LAYER_COMBINESOME,
        strid::MENU_LAYER_VIEW_ALLSHOW, strid::MENU_LAYER_VIEW_ALLHIDE, strid::MENU_LAYER_VIEW_ONLYCUR,
        strid::MENU_LAYER_VIEW_TOGGLECHECK, strid::MENU_LAYER_SAVE,
        strid::MENU_LAYER_CHECK_LINKOFF, strid::MENU_LAYER_CHECK_CHECKOFF, strid::MENU_LAYER_EDIT_HREV,
        strid::MENU_LAYER_EDIT_VREV, strid::MENU_LAYER_VIEW_TOGGLE_NORMAL
    };
    UINT key[] = {
        ACKEY_CTRL|'N', ACKEY_CTRL|'O', ACKEY_CTRL|'S', ACKEY_CTRL|'W',
        ACKEY_CTRL|'Z', ACKEY_CTRL|'Y', KEY_INSERT,
        KEY_DELETE, ACKEY_CTRL|'T', ACKEY_CTRL|'D', ACKEY_CTRL|'I',
        ACKEY_CTRL|'A', ACKEY_CTRL|'C', ACKEY_CTRL|'X', ACKEY_CTRL|'V',
        ACKEY_CTRL|'E', ACKEY_CTRL|'F',
        ACKEY_CTRL|KEY_TAB, ACKEY_CTRL|'R', ACKEY_CTRL|'Q', ACKEY_CTRL|'K',
        ACKEY_CTRL|'G', ACKEY_CTRL|'M',

        ACKEY_ALT|'N', ACKEY_ALT|'F', ACKEY_ALT|'I', ACKEY_ALT|'C', ACKEY_ALT|'D',
        ACKEY_ALT|'X', ACKEY_ALT|'Z', ACKEY_ALT|'B', ACKEY_ALT|'W', ACKEY_ALT|'A',
        ACKEY_ALT|'G', ACKEY_ALT|'U', ACKEY_ALT|'Q', ACKEY_ALT|'S', ACKEY_ALT|'L',
        ACKEY_ALT|'K', ACKEY_ALT|'H', ACKEY_ALT|'V', ACKEY_ALT|'R'
    };

    //

    attachAccel(new AXAccelerator);

    m_pAccel->setDefaultWindow(this);

    for(i = 0; i < sizeof(id)/sizeof(WORD); i++)
        m_pAccel->add(id[i], key[i]);
}
