/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CLayerWin - レイヤウィンドウ
*/


#include "CLayerWin.h"

#include "CLayerWinArea.h"
#include "CValBar.h"

#include "CConfig.h"
#include "CResource.h"
#include "CLayerItem.h"

#include "drawdat.h"
#include "draw_layer.h"
#include "draw_update.h"

#include "global.h"
#include "strid.h"

#include "AXLayout.h"
#include "AXLabel.h"
#include "AXScrollBar.h"
#include "AXToolBar.h"
#include "AXComboBox.h"
#include "AXCheckButton.h"
#include "AXMenu.h"
#include "AXImage.h"
#include "AXDND.h"
#include "AXApp.h"
#include "AXUtilStr.h"


//-------------------------------------

class CLayerWinInfo_amask:public AXWindow
{
    AXImage m_img;

    void _runMenu();

public:
    CLayerWinInfo_amask(AXWindow *pParent,UINT uID);
    void setAMaskType(int no);
    virtual BOOL onPaint(AXHD_PAINT *phd);
    virtual BOOL onButtonDown(AXHD_MOUSE *phd);
};

class CLayerWinInfo:public AXWindow
{
    AXComboBox      *m_pcbBlend;
    AXCheckButton   *m_pckLock;
    CValBar         *m_pbarOpacity;
    CLayerWinInfo_amask *m_pwidAMask;

    enum
    {
        WID_BLEND = 100,
        WID_LOCK,
        WID_OPACITY,
        WID_AMASK
    };

public:
    CLayerWinInfo(AXWindow *pParent);
    void setParam();
    virtual BOOL onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam);
};

//-------------------------------------


//**********************************
// CLayerWin : レイヤウィンドウ
//**********************************


CLayerWin *CLayerWin::m_pSelf = NULL;


CLayerWin::CLayerWin(AXWindow *pOwner,UINT addstyle)
    : AXTopWindow(pOwner,
                  WS_HIDE | WS_TITLE | WS_CLOSE | WS_BORDER | WS_DISABLE_IM | WS_DROP | addstyle)
{
    AXLayout *pl,*plh;

    m_pSelf = this;

    //

    setTitle("layer");

    enableDND();

    attachAccel(((AXTopWindow *)m_pOwner)->getAccel());

    if(g_res->pfontSubWin)
        setFont(g_res->pfontSubWin);

    //レイアウト

    setLayout(pl = new AXLayoutVert);

    pl->addItem(m_pwidInfo = new CLayerWinInfo(this));

    pl->addItem(plh = new AXLayoutHorz(LF_EXPAND_WH, 0));
    plh->addItem(m_pArea = new CLayerWinArea(this));
    plh->addItem(m_pScrV = new AXScrollBar(this, AXScrollBar::SBS_VERT, LF_EXPAND_H));

    _createToolBar();

    //

    m_pArea->setScrollBar(m_pScrV);
    m_pArea->setFont(g_res->pfontSmall);
}

//! 表示/非表示 切り替え

void CLayerWin::showToggle()
{
    if(g_conf->isLayerWin())
        showRestore();
    else
        hide();
}

//! 全体更新

void CLayerWin::updateLayerAll()
{
    m_pwidInfo->setParam();

    m_pArea->setScrollInfo();
    m_pArea->draw();
}

//! 上部の情報部分を更新

void CLayerWin::updateInfoParam()
{
    m_pwidInfo->setParam();
}

//! カレントレイヤのみ更新

void CLayerWin::updateCurLayer(BOOL bInfo)
{
    if(bInfo) m_pwidInfo->setParam();
    m_pArea->draw(g_draw->pcurlayer);
}

//! 指定アイテムのレイヤのみ更新

void CLayerWin::updateLayerItem(CLayerItem *p)
{
    m_pArea->draw(p);
}

//! カレントレイヤが表示される位置にスクロールして更新

void CLayerWin::updateShowCur()
{
    m_pArea->updateShowCur();
}


//=========================
// サブ
//=========================


//! ツールバー作成

void CLayerWin::_createToolBar()
{
    UINT i;
    WORD id[] = {
        strid::MENU_LAYER_NEWLAYER, strid::MENU_LAYER_NEWFOLDER, strid::MENU_LAYER_COPY,
        strid::MENU_LAYER_DELETE, strid::MENU_LAYER_CLEAR, strid::MENU_LAYER_IMGDROP,
        strid::MENU_LAYER_COMBINE, strid::MENU_LAYER_TMP_MOVEUP, strid::MENU_LAYER_TMP_MOVEDOWN
    };

    m_pToolBar = new AXToolBar(this,
            AXToolBar::TBS_TOOLTIP | AXToolBar::TBS_TOOLTIP_TR, LF_EXPAND_W);

    m_pToolBar->setImageList(g_res->pilLayerTB);
    m_pToolBar->setToolTipTrGroup(strid::GROUP_TOOLTIP_LAYER);

    getLayout()->addItem(m_pToolBar);

    for(i = 0; i < sizeof(id)/sizeof(WORD); i++)
        m_pToolBar->addItem(id[i], i, id[i], 0, 0);
}


//=========================
// ハンドラ
//=========================


//! 閉じる（非表示）

BOOL CLayerWin::onClose()
{
    m_pOwner->onCommand(strid::MENU_VIEW_LAYERWIN, 0, 0);

    return TRUE;
}

//! 通知

BOOL CLayerWin::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    if(pwin == m_pScrV && uNotify == AXScrollBar::SBN_SCROLL)
        m_pArea->draw();

    return TRUE;
}

//! コマンド

BOOL CLayerWin::onCommand(UINT uID,ULONG lParam,int from)
{
    m_pOwner->onCommand(uID, lParam, from);

    return TRUE;
}

//! D&D

BOOL CLayerWin::onDND_Drop(AXDND *pDND)
{
    AXString str,fname;
    LPCUSTR p;

    pDND->getFiles(&str);
    pDND->endDrop();

    if(str.isNoEmpty())
    {
        str.replace('\t', 0);

        for(p = str; *p; p += AXUStrLen(p) + 1)
        {
            fname = p;
            draw::layer_fromImg(fname);
        }

        //最後に更新

        draw::updateAllAndLayer();
    }

    return TRUE;
}



//**********************************
// CLayerWinInfo : 上部の情報エリア
//**********************************


CLayerWinInfo::CLayerWinInfo(AXWindow *pParent)
    : AXWindow(pParent, WS_BK_FACE, LF_EXPAND_W)
{
    AXLayout *plTop,*pl;

    _trgroup(strid::GROUP_LAYER);

    setLayout(plTop = new AXLayoutVert(LF_EXPAND_W, 4));
    plTop->setSpacing(4);

    //上段

    plTop->addItem(pl = new AXLayoutHorz(LF_EXPAND_W, 4));

    pl->addItem(m_pcbBlend = new AXComboBox(this, 0, LF_EXPAND_W|LF_CENTER_Y, WID_BLEND, 0));
    pl->addItem(m_pckLock = new AXCheckButton(this, 0, LF_CENTER_Y, WID_LOCK, 0, strid::LAYER_INFO_LOCK, FALSE));

    //下段

    plTop->addItem(pl = new AXLayoutHorz(LF_EXPAND_W, 4));

    pl->addItem(m_pbarOpacity = new CValBar(this, 0, LF_EXPAND_W|LF_CENTER_Y, WID_OPACITY, 0, 0, 1, 100, 100));

    pl->addItem(new AXLabel(this, 0, LF_CENTER_Y, 0, strid::LAYER_INFO_AMASK));
    pl->addItem(m_pwidAMask = new CLayerWinInfo_amask(this, WID_AMASK));

    //

    _trgroup(strid::GROUP_BLENDNAME);

    m_pcbBlend->addItemMulTr(0, draw::BLENDMODE_NUM);
}

//! カレントレイヤのパラメータセット

void CLayerWinInfo::setParam()
{
    CLayerItem *p = g_draw->pcurlayer;
    BOOL bLayer;

    bLayer = !(p->isFolder());

    //合成モード

    m_pcbBlend->setCurSel((bLayer)? p->m_nBlendMode: -1);
    m_pcbBlend->enable(bLayer);

    //ロック

    m_pckLock->setCheck(p->isLock());

    //不透明度

    m_pbarOpacity->setPos((int)(p->m_nOpacity * 100 / 128.0 + 0.5));

    //Aマスク

    m_pwidAMask->setAMaskType((bLayer)? p->m_nAmaskType: 0);
    m_pwidAMask->enable(bLayer);
}

//! 通知

BOOL CLayerWinInfo::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    switch(pwin->getItemID())
    {
        //合成モード
        case WID_BLEND:
            if(uNotify == AXComboBox::CBN_SELCHANGE)
                draw::layer_changeBlendMode(m_pcbBlend->getCurSel());
            break;
        //ロック
        case WID_LOCK:
            draw::layer_revLockFlag(g_draw->pcurlayer);
            break;
        //不透明度
        case WID_OPACITY:
            if(!m_pbarOpacity->isDrag())
                draw::layer_changeOpacity(m_pbarOpacity->getPos());
            break;
        //Aマスク
        case WID_AMASK:
            g_draw->pcurlayer->m_nAmaskType = (int)uNotify;

            m_pwidAMask->setAMaskType(uNotify);
            break;
    }

    return TRUE;
}



//********************************************
// CLayerWinInfo_mask : 上部の情報エリアのAマスク
//********************************************


CLayerWinInfo_amask::CLayerWinInfo_amask(AXWindow *pParent,UINT uID)
    : AXWindow(pParent, 0, LF_FIX_WH, uID, 0)
{
    m_nW = m_nH = 14;

    m_img.create(14, 14);
}

//! タイプ変更

void CLayerWinInfo_amask::setAMaskType(int no)
{
    m_img.blt(0, 0, *(g_res->pimgAMask), no * 14, 0, 14, 14);
    redraw();
}

//! 描画

BOOL CLayerWinInfo_amask::onPaint(AXHD_PAINT *phd)
{
    m_img.put(m_id);
    return TRUE;
}

//! 押し時

BOOL CLayerWinInfo_amask::onButtonDown(AXHD_MOUSE *phd)
{
    if(phd->button == BUTTON_RIGHT ||
       (phd->button == BUTTON_LEFT && (phd->state & STATE_CTRL)) )
       //右 or Ctrl+左でOFF
        getNotify()->onNotify(this, 0, 0);
    else if(phd->button == BUTTON_LEFT)
        //左でメニュー表示
        _runMenu();

    return TRUE;
}

//! メニュー表示

void CLayerWinInfo_amask::_runMenu()
{
    AXMenu *pmenu;
    AXRectSize rcs;
    int id;

    pmenu = new AXMenu;

    _trgroup(strid::GROUP_LAYER);

    pmenu->addTrMul(strid::LAYER_AMASKMENU_TOP, 4);
    pmenu->check(strid::LAYER_AMASKMENU_TOP + g_draw->pcurlayer->m_nAmaskType, TRUE);

    getWindowRect(&rcs);
    id = pmenu->popup(NULL, rcs.x + rcs.w, rcs.y + rcs.h, AXMenu::POPF_RIGHT);

    delete pmenu;

    //

    if(id != -1)
        getNotify()->onNotify(this, id - strid::LAYER_AMASKMENU_TOP, 0);
}
