/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CGradEditDlg - グラデーション編集ダイアログ
*/

#include <string.h>

#include "CGradEditDlg.h"

#include "CGradList.h"
#include "CValBar.h"

#include "AXLayout.h"
#include "AXLabel.h"
#include "AXGroupBox.h"
#include "AXButton.h"
#include "AXCheckButton.h"
#include "AXColorButton.h"
#include "AXArrowButton.h"
#include "AXLineEdit.h"
#include "AXMenu.h"
#include "AXApp.h"
#include "AXAppRes.h"
#include "AXImage.h"

#include "strid.h"


//------------------

#define GPOS_MAX    1024
#define GPOS_BIT    10
#define GRAD_GETPOS(p)  (*((LPWORD)(p)) & 0x3fff)
#define GRAD_GETA(p)    (*(LPWORD)(p + 2))

enum
{
    WID_GRAD_COL = 100,
    WID_GRADCOL_LEFT,
    WID_GRADCOL_RIGHT,
    WID_GRADCOL_MENU,
    WID_GRAD_A,
    WID_GRADA_LEFT,
    WID_GRADA_RIGHT,
    WID_GRADA_MENU,

    WID_EDIT_COLPOS,
    WID_DRAWCOL,
    WID_BACKCOL,
    WID_SPECCOL,
    WID_SPECCOL_BT,

    WID_EDIT_APOS,
    WID_A_BAR,

    WID_CK_LOOP,
    WID_CK_SINGLE
};

//------------------

class CGradEditDlg_edit:public AXWindow
{
public:
    enum
    {
        NOTIFY_UPDATE_VAL,
        NOTIFY_UPDATE_GRAD
    };

protected:
    enum
    {
        UNDERH  = 15,   //グラデーションより下の高さ
        POINTW  = 11,
        POINTHF = 5
    };

    AXMem       m_memDat;
    LPBYTE      m_pCurBuf;
    int         m_nPtCnt,
                m_nCurPos,
                m_fDrag,
                m_bAlpha,
                m_nOneSize;

    AXImage     m_img;

protected:
    void _notify(UINT uNotify);
    void _newPoint(int pos);
    void _evenPoint(int pt);
    void _setPtColCur(int pt);

    int _getWinPosFromGrad(int pos);
    int _getGradPosFromWin(int x);
    int _getPtFromMouse(int mx,int my);
    int _adjustPos(LPBYTE pbuf,int pos);
    void _setPtPos(LPBYTE pbuf,int pos);

    void _drawGrad();
    void _drawGradA();
    void _drawPoint(BOOL bClear);
    void _drawOnePoint(LPBYTE pbuf);

public:
    CGradEditDlg_edit(AXWindow *pParent,UINT uItemID,DWORD dwPadding,BOOL bEditA,LPBYTE pDat);

    LPBYTE getDatBuf() { return (LPBYTE)m_memDat; }
    int getPtCnt() { return m_nPtCnt; }
    LPBYTE getPtBufFromPos(int pos) { return (LPBYTE)m_memDat + pos * m_nOneSize; }
    int getCurPtColType() { return *((LPWORD)m_pCurBuf) >> 14; }
    int getCurPtPos();
    int getCurPtVal();
    int getEnableSplitCnt();

    void setCurPtPos(int pos);
    void setCurPtA(int val);
    void setCurPtColType(int type,DWORD col);
    void setCurPtCol(DWORD col);

    void movPoint(BOOL bRight);
    void delPoint(int pt=-1);
    void setCurPtPosMiddle();
    void setAllPtEven();
    void pointReverse();
    void pointSplit(int cnt);

    void drawAll();

    virtual BOOL onSize();
    virtual BOOL onPaint(AXHD_PAINT *phd);
    virtual BOOL onButtonDown(AXHD_MOUSE *phd);
    virtual BOOL onButtonUp(AXHD_MOUSE *phd);
    virtual BOOL onMouseMove(AXHD_MOUSE *phd);
};

//------------------


//********************************
// CGradEditDlg
//********************************


CGradEditDlg::CGradEditDlg(AXWindow *pOwner,CGradListItem *pItem)
    : AXDialog(pOwner, WS_DIALOG_NORMAL | WS_BK_FACE)
{
    AXLayout *plTop,*pl,*pl2;
    AXGroupBox *pg;
    int i;
    BYTE flag;

    m_pItem = pItem;

    flag = *((LPBYTE)pItem->m_memDat);

    //

    _trgroup(strid::GROUP_DLG_GRADEDIT);

    setTitle(strid::GRADEDIT_TITLE);

    setLayout(plTop = new AXLayoutVert);
    plTop->setSpacing(8);

    //------- 編集

    plTop->addItem(pl = new AXLayoutMatrix(2, LF_EXPAND_W));

    _setwidget_grad(pl, WID_GRAD_COL, FALSE);
    _setwidget_grad(pl, WID_GRAD_A, TRUE);

    //------- 色

    plTop->addItem(pg = new AXGroupBox(this, 0, LF_EXPAND_W, MAKE_DW4(0,4,0,0), _str(strid::GRADEDIT_COLOR)));

    pg->setLayout(pl = new AXLayoutVert(0, 5));
    pl->setSpacing(6);

    //位置

    pl->addItem(pl2 = new AXLayoutHorz(0, 6));

    pl2->addItem(new AXLabel(pg, 0, LF_CENTER_Y, 0, strid::GRADEDIT_POS));
    pl2->addItem(m_peditColPos = new AXLineEdit(pg, AXLineEdit::ES_SPIN, LF_CENTER_Y, WID_EDIT_COLPOS, 0));

    m_peditColPos->setInit(6, 0, 1000, 1, 0);

    //ラジオボタン

    pl->addItem(pl2 = new AXLayoutHorz(0, 4));

    for(i = 0; i < 3; i++)
        pl2->addItem(m_pckColType[i] = new AXCheckButton(pg, AXCheckButton::CBS_RADIO, LF_CENTER_Y, WID_DRAWCOL + i, 0, strid::GRADEDIT_DRAWCOL + i, FALSE));

    //色ボタン

    pl2->addItem(m_pbtCol = new AXColorButton(pg, AXColorButton::CBTS_CHOOSE, LF_CENTER_Y, WID_SPECCOL_BT, 0));

    //--------- 不透明度

    plTop->addItem(pg = new AXGroupBox(this, 0, LF_EXPAND_W, MAKE_DW4(0,6,0,8), _str(strid::GRADEDIT_OPACITY)));

    pg->setLayout(pl = new AXLayoutHorz(0, 5));
    pl->setSpacing(6);

    //位置

    pl->addItem(new AXLabel(pg, 0, LF_CENTER_Y, 0, strid::GRADEDIT_POS));
    pl->addItem(m_peditAPos = new AXLineEdit(pg, AXLineEdit::ES_SPIN, LF_CENTER_Y, WID_EDIT_APOS, 0));

    m_peditAPos->setInit(6, 0, 1000, 1, 0);

    //値

    pl->addItem(new AXLabel(pg, 0, LF_CENTER_Y, 0, strid::GRADEDIT_OPACITY));
    pl->addItem(m_pbarA = new CValBar(pg, 0, LF_EXPAND_W|LF_CENTER_Y, WID_A_BAR, 0, 1, 0, 1000, 0));

    //-------- チェック

    plTop->addItem(pl = new AXLayoutHorz(0, 4));

    for(i = 0; i < 2; i++)
        pl->addItem(m_pckFlag[i] = new AXCheckButton(this, 0, 0, WID_CK_LOOP + i, 0, strid::GRADEDIT_LOOP + i, flag & (1<<i)));

    //-------- ヘルプ

    plTop->addItem(new AXLabel(this, AXLabel::LS_BORDER|AXLabel::LS_BKLIGHT, 0, MAKE_DW4(0,10,0,0), strid::GRADEDIT_HELP));

    //------ OKキャンセル

    plTop->addItem(pl = createOKCancelButton());
    pl->setPaddingTop(15);

    //---------

    _setCurVal_col();
    _setCurVal_a();

    calcDefSize();
    resize((m_nDefW < 450)? 450: m_nDefW, m_nDefH);

    show();
}

//! 編集部分のウィジェット作成

void CGradEditDlg::_setwidget_grad(AXLayout *plTop,UINT idTop,BOOL bAlpha)
{
    AXLayout *plv,*plh;
    CGradEditDlg_edit *pedit;

    plTop->addItem(pedit = new CGradEditDlg_edit(this, idTop, MAKE_DW4(0,0,8,6), bAlpha, m_pItem->m_memDat));
    plTop->addItem(plv = new AXLayoutVert(0, 4));

    if(bAlpha) m_pgradA = pedit; else m_pgradCol = pedit;

    //ボタン

    plv->addItem(plh = new AXLayoutHorz);

    plh->addItem(new AXButton(this, AXButton::BS_REAL_WH, 0, idTop + 1, 0, "<"));
    plh->addItem(new AXButton(this, AXButton::BS_REAL_WH, 0, idTop + 2, 0, ">"));

    plv->addItem(new AXArrowButton(this, AXArrowButton::ARBTS_DOWN, LF_RIGHT, idTop + 3, 0));
}

//! 単色化がONか

BOOL CGradEditDlg::isSingleCol()
{
    return m_pckFlag[1]->isChecked();
}

//! Aの編集データポインタ取得

LPBYTE CGradEditDlg::getAlphaBuf()
{
    return m_pgradA->getDatBuf();
}

//! 色の現在のポイントのデータセット

void CGradEditDlg::_setCurVal_col()
{
    int n;

    //位置

    m_peditColPos->setVal((int)(m_pgradCol->getCurPtPos() * 1000.0 / GPOS_MAX + 0.5));

    //色

    n = m_pgradCol->getCurPtColType();

    m_pckColType[n]->setCheck(TRUE);
    m_pbtCol->setColor(m_pgradCol->getCurPtVal());
}

//! Aの現在のポイントのデータセット

void CGradEditDlg::_setCurVal_a()
{
    //位置
    m_peditAPos->setVal((int)(m_pgradA->getCurPtPos() * 1000.0 / GPOS_MAX + 0.5));

    //値
    m_pbarA->setPos((int)(m_pgradA->getCurPtVal()) * 1000.0 / 1024.0 + 0.5);
}

//! 通知

BOOL CGradEditDlg::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    UINT id = pwin->getItemID();

    switch(id)
    {
        //色編集
        case WID_GRAD_COL:
            _setCurVal_col();
            break;
        //A編集
        case WID_GRAD_A:
            _setCurVal_a();

            if(uNotify == CGradEditDlg_edit::NOTIFY_UPDATE_GRAD)
                m_pgradCol->drawAll();
            break;

        //色 </>
        case WID_GRADCOL_LEFT:
        case WID_GRADCOL_RIGHT:
            m_pgradCol->movPoint((id == WID_GRADCOL_RIGHT));
            _setCurVal_col();
            break;
        //色メニュー
        case WID_GRADCOL_MENU:
            _runGradMenu(FALSE);
            break;

        //A </>
        case WID_GRADA_LEFT:
        case WID_GRADA_RIGHT:
            m_pgradA->movPoint((id == WID_GRADA_RIGHT));
            _setCurVal_a();
            break;
        //Aメニュー
        case WID_GRADA_MENU:
            _runGradMenu(TRUE);
            break;

        //色・位置
        case WID_EDIT_COLPOS:
            if(uNotify == AXLineEdit::EN_CHANGE)
                m_pgradCol->setCurPtPos(m_peditColPos->getVal());
            break;
        //色・タイプ
        case WID_DRAWCOL:
        case WID_BACKCOL:
        case WID_SPECCOL:
            m_pgradCol->setCurPtColType(id - WID_DRAWCOL, m_pbtCol->getColor());
            break;
        //色・指定色カラー
        case WID_SPECCOL_BT:
            if(m_pckColType[2]->isChecked())
                m_pgradCol->setCurPtCol(m_pbtCol->getColor());
            break;
        //A・位置
        case WID_EDIT_APOS:
            if(uNotify == AXLineEdit::EN_CHANGE)
            {
                m_pgradA->setCurPtPos(m_peditAPos->getVal());
                m_pgradCol->drawAll();
            }
            break;
        //A・不透明度
        case WID_A_BAR:
            if(uNotify == CValBar::NOTIFY_CHANGE)
            {
                m_pgradA->setCurPtA(m_pbarA->getPos());
                m_pgradCol->drawAll();
            }
            break;
        //単色化
        case WID_CK_SINGLE:
            m_pgradCol->drawAll();
            break;

        //OK
        case 1:
            _setDat();
            endDialog(TRUE);
            break;
        //キャンセル
        case 2:
            endDialog(FALSE);
            break;
    }

    return TRUE;
}

//! OK時のデータセット

void CGradEditDlg::_setDat()
{
    int colcnt,acnt;
    LPBYTE p;
    BYTE flag;

    colcnt = m_pgradCol->getPtCnt();
    acnt   = m_pgradA->getPtCnt();

    if(!m_pItem->m_memDat.alloc(3 + colcnt * 5 + acnt * 4)) return;

    p = m_pItem->m_memDat;

    //フラグ

    flag = (m_pckFlag[0]->isChecked())? 1: 0;
    if(m_pckFlag[1]->isChecked()) flag |= 2;

    *(p++) = flag;

    //ポイント数

    *(p++) = colcnt;
    *(p++) = acnt;

    //データ

    ::memcpy(p, m_pgradCol->getDatBuf(), colcnt * 5);
    ::memcpy(p + colcnt * 5, m_pgradA->getDatBuf(), acnt * 4);
}

//! 編集メニュー実行

void CGradEditDlg::_runGradMenu(BOOL bAlpha)
{
    AXMenu *pmenu,*psub;
    WORD id[] = {
        1000, 0xffff, 1001, 0xffff, 1002,1003,1004
    };
    int n,cnt;
    AXRectSize rcs;
    AXString str;
    CGradEditDlg_edit *pedit;

    //---------------

    _trgroup(strid::GROUP_DLG_GRADEDIT);

    pmenu = new AXMenu;

    pmenu->addTrArray(id, sizeof(id)/sizeof(WORD));

    //分割

    psub = new AXMenu;

    if(bAlpha)
        cnt = m_pgradA->getEnableSplitCnt();
    else
        cnt = m_pgradCol->getEnableSplitCnt();

    for(n = 2; n <= cnt && n <= 10; n++)
    {
        str.setInt(n);
        psub->add(2000 + n, str);
    }

    pmenu->setSubMenu(1001, psub);

    //

    getWidget(bAlpha? WID_GRADA_MENU: WID_GRADCOL_MENU)->getWindowRect(&rcs);

    n = pmenu->popup(NULL, rcs.x + rcs.w, rcs.y + rcs.h, AXMenu::POPF_RIGHT);

    delete pmenu;

    //------------

    if(n == -1) return;

    pedit = (bAlpha)? m_pgradA: m_pgradCol;

    if(n >= 2000)
        pedit->pointSplit(n - 2000);
    else
    {
        switch(n)
        {
            //ポイント削除
            case 1000:
                pedit->delPoint();
                break;
            //中間位置に移動
            case 1002:
                pedit->setCurPtPosMiddle();
                break;
            //すべて等間隔に
            case 1003:
                pedit->setAllPtEven();
                break;
            //反転
            case 1004:
                pedit->pointReverse();
                break;
        }
    }
}



//*********************************************


//**********************************
// CGradEditDlg_edit
//**********************************


CGradEditDlg_edit::CGradEditDlg_edit(AXWindow *pParent,UINT uItemID,DWORD dwPadding,BOOL bEditA,LPBYTE pDat)
    : AXWindow(pParent, 0, LF_EXPAND_W, uItemID, dwPadding)
{
    m_nMinW = 60;
    m_nMinH = 40;

    m_nCurPos = 0;
    m_fDrag   = 0;
    m_bAlpha  = bEditA;
    m_nOneSize = (bEditA)? 4: 5;

    m_memDat.alloc(CGradListItem::POINT_MAXCNT * m_nOneSize);

    //データコピー

    if(bEditA)
    {
        ::memcpy(m_memDat, pDat + 3 + pDat[1] * 5, pDat[2] * 4);

        m_nPtCnt = pDat[2];
    }
    else
    {
        ::memcpy(m_memDat, pDat + 3, pDat[1] * 5);

        m_nPtCnt = pDat[1];
    }

    //

    m_pCurBuf = m_memDat;
}

//! 描画

BOOL CGradEditDlg_edit::onPaint(AXHD_PAINT *phd)
{
    m_img.put(m_id, 0, 0, 0, 0, m_nW, m_nH);
    return TRUE;
}

//! サイズ変更時

BOOL CGradEditDlg_edit::onSize()
{
    m_img.recreate(m_nW, m_nH, 32, 0);

    drawAll();

    return TRUE;
}

//! ボタン押し

BOOL CGradEditDlg_edit::onButtonDown(AXHD_MOUSE *phd)
{
    if(m_fDrag == 0 && phd->button == BUTTON_LEFT)
    {
        int pt,n;

        //押されたポイント取得

        pt = _getPtFromMouse(phd->x, phd->y);
        if(pt == -2) return TRUE;

        //

        if(pt == -1)
        {
            //ポイントがなければ、新規ポイント作成

            if(phd->state & STATE_ALT) return TRUE;

            n = _getGradPosFromWin(phd->x);

            _newPoint(n);
        }
        else if(phd->state & STATE_CTRL)
            //+Ctrl : 等間隔
            _evenPoint(pt);
        else if(phd->state & STATE_SHIFT)
            //+Shift : 色/Aセット
            _setPtColCur(pt);
        else if(phd->state & STATE_ALT)
            //+Alt : 削除
            delPoint(pt);
        else
        {
            //------ カレント変更

            m_nCurPos = pt;
            m_pCurBuf = getPtBufFromPos(pt);

            _drawPoint(FALSE);
            redraw();

            _notify(NOTIFY_UPDATE_VAL);

            //D&D開始（端の位置は除く）

            n = GRAD_GETPOS(m_pCurBuf);

            if(n != 0 && n != GPOS_MAX)
            {
                m_fDrag = TRUE;
                grabPointer();
            }
        }
    }

    return TRUE;
}

//! ボタン離し

BOOL CGradEditDlg_edit::onButtonUp(AXHD_MOUSE *phd)
{
    if(m_fDrag && phd->button == BUTTON_LEFT)
    {
        m_fDrag = FALSE;
        ungrabPointer();
    }

    return TRUE;
}

//! 移動

BOOL CGradEditDlg_edit::onMouseMove(AXHD_MOUSE *phd)
{
    if(m_fDrag)
    {
        int pos;

        pos = _getGradPosFromWin(phd->x);

        //左と右の間に調整

        pos = _adjustPos(m_pCurBuf, pos);

        //位置変更

        if(GRAD_GETPOS(m_pCurBuf) != pos)
        {
            _setPtPos(m_pCurBuf, pos);

            drawAll();
            _notify(NOTIFY_UPDATE_GRAD);
        }
    }

    return TRUE;
}


//==============================
//
//==============================


//! 通知

void CGradEditDlg_edit::_notify(UINT uNotify)
{
    m_pParent->onNotify(this, uNotify, 0);
}

//! カレント位置の位置を取得

int CGradEditDlg_edit::getCurPtPos()
{
    return GRAD_GETPOS(m_pCurBuf);
}

//! カレント位置の色/Aを取得

int CGradEditDlg_edit::getCurPtVal()
{
    if(m_bAlpha)
        return GRAD_GETA(m_pCurBuf);
    else
        return _RGB(m_pCurBuf[2], m_pCurBuf[3], m_pCurBuf[4]);
}

//! ポイント選択を左右に移動

void CGradEditDlg_edit::movPoint(BOOL bRight)
{
    //両端

    if(!bRight && m_nCurPos == 0) return;
    if(bRight && m_nCurPos == m_nPtCnt - 1) return;

    //移動

    if(bRight)
    {
        m_nCurPos++;
        m_pCurBuf += m_nOneSize;
    }
    else
    {
        m_nCurPos--;
        m_pCurBuf -= m_nOneSize;
    }

    //

    _drawPoint(FALSE);
    redraw();
}

//! カレントと左の間の分割可能数を取得

int CGradEditDlg_edit::getEnableSplitCnt()
{
    int cnt;

    if(m_nCurPos == 0)
        return 0;
    else
    {
        cnt = GRAD_GETPOS(m_pCurBuf) - GRAD_GETPOS(m_pCurBuf - m_nOneSize);

        if(m_nPtCnt + cnt - 1 > CGradListItem::POINT_MAXCNT)
            cnt = CGradListItem::POINT_MAXCNT - (m_nPtCnt - 1);

        return cnt;
    }
}


//==============================
// カレントセット
//==============================


//! カレント位置の位置を変更

void CGradEditDlg_edit::setCurPtPos(int pos)
{
    int nowpos = GRAD_GETPOS(m_pCurBuf);

    if(nowpos != 0 && nowpos != GPOS_MAX)
    {
        pos = _adjustPos(m_pCurBuf, (int)(pos * 1024.0 / 1000.0 + 0.5));

        if(pos != nowpos)
        {
            _setPtPos(m_pCurBuf, pos);

            drawAll();
        }
    }
}

//! カレントの不透明度セット

void CGradEditDlg_edit::setCurPtA(int val)
{
    *((LPWORD)(m_pCurBuf + 2)) = (int)(val * 1024.0 / 1000.0 + 0.5);

    _drawGrad();
    _drawOnePoint(m_pCurBuf);
}

//! カレントの色タイプセット

void CGradEditDlg_edit::setCurPtColType(int type,DWORD col)
{
    *((LPWORD)m_pCurBuf) &= ~0xc000;
    *((LPWORD)m_pCurBuf) |= type << 14;

    //色
    /* 描画色/背景色はプレビュー時用に黒/白にセットしておく */

    if(type == 0)
        m_pCurBuf[2] = m_pCurBuf[3] = m_pCurBuf[4] = 0;
    else if(type == 1)
        m_pCurBuf[2] = m_pCurBuf[3] = m_pCurBuf[4] = 255;
    else
    {
        m_pCurBuf[2] = _GETR(col);
        m_pCurBuf[3] = _GETG(col);
        m_pCurBuf[4] = _GETB(col);
    }

    _drawGrad();
    _drawOnePoint(m_pCurBuf);
}

//! カレントの指定色セット

void CGradEditDlg_edit::setCurPtCol(DWORD col)
{
    m_pCurBuf[2] = _GETR(col);
    m_pCurBuf[3] = _GETG(col);
    m_pCurBuf[4] = _GETB(col);

    _drawGrad();
    _drawOnePoint(m_pCurBuf);
}


//==============================
// ポイント編集
//==============================


//! 新規ポイント作成

void CGradEditDlg_edit::_newPoint(int pos)
{
    LPBYTE p,pNew;
    int newpos = -1,i,ppos,npos,pval,nval;

    if(m_nPtCnt >= CGradListItem::POINT_MAXCNT)
        return;

    //追加する位置
    //※位置が同じ場合は追加しない

    p = m_memDat;

    for(i = 0; i < m_nPtCnt - 1; i++, p += m_nOneSize)
    {
        if(GRAD_GETPOS(p) < pos && pos < GRAD_GETPOS(p + m_nOneSize))
        {
            pNew   = p + m_nOneSize;
            newpos = i + 1;
            break;
        }
    }

    if(newpos == -1) return;

    //セット位置を空ける

    ::memmove(pNew + m_nOneSize, pNew, (m_nPtCnt - newpos) * m_nOneSize);

    //位置セット

    *((LPWORD)pNew) = pos | (m_bAlpha? 0: (2<<14));

    //色/A セット（値は前と次のポイントの中間値）

    ppos = GRAD_GETPOS(pNew - m_nOneSize);
    npos = GRAD_GETPOS(pNew + m_nOneSize);

    if(m_bAlpha)
    {
        //A

        pval = GRAD_GETA(pNew - 4);
        nval = GRAD_GETA(pNew + 4);

        *((LPWORD)(pNew + 2)) = (nval - pval) * (pos - ppos) / (npos - ppos) + pval;
    }
    else
    {
        //色

        for(i = 0; i < 3; i++)
            pNew[2 + i] = (pNew[2+5+i] - pNew[2-5+i]) * (pos - ppos) / (npos - ppos) + pNew[2-5+i];
    }

    //

    m_nPtCnt++;
    m_nCurPos = newpos;
    m_pCurBuf = pNew;

    drawAll();
    _notify(NOTIFY_UPDATE_GRAD);
}

//! ポイント削除
/*!
    @param pt -1で現在の位置
*/

void CGradEditDlg_edit::delPoint(int pt)
{
    LPBYTE p;

    if(pt == -1) pt = m_nCurPos;

    //両端は削除しない

    if(pt == 0 || pt == m_nPtCnt - 1) return;

    //詰める

    p = getPtBufFromPos(pt);

    ::memmove(p, p + m_nOneSize, (m_nPtCnt - pt - 1) * m_nOneSize);

    //

    m_nPtCnt--;

    drawAll();
    _notify(NOTIFY_UPDATE_GRAD);
}

//! 現在ポイントと指定ポイントの間を等間隔に

void CGradEditDlg_edit::_evenPoint(int pt)
{
    LPBYTE pbuf,pbuf2;
    int i,pt1,pt2,poslen,pos1;

    //位置の小さい順

    if(pt < m_nCurPos)
        pt1 = pt, pt2 = m_nCurPos;
    else
        pt1 = m_nCurPos, pt2 = pt;

    //カレントと同じ、または間が一つもない時

    if(pt == m_nCurPos || pt1 + 1 == pt2) return;

    //

    pbuf  = getPtBufFromPos(pt1);
    pbuf2 = getPtBufFromPos(pt2);

    pos1   = GRAD_GETPOS(pbuf);
    poslen = GRAD_GETPOS(pbuf2) - pos1;

    for(i = pt1 + 1, pbuf += m_nOneSize; i < pt2; i++, pbuf += m_nOneSize)
        _setPtPos(pbuf, (i - pt1) * poslen / (pt2 - pt1) + pos1);

    drawAll();
    _notify(NOTIFY_UPDATE_GRAD);
}

//! 指定ポイントにカレントの色/Aをセット

void CGradEditDlg_edit::_setPtColCur(int pt)
{
    LPBYTE p = getPtBufFromPos(pt);

    if(m_bAlpha)
        *((LPWORD)(p + 2)) = *((LPWORD)(m_pCurBuf + 2));
    else
    {
        *((LPWORD)p) &= ~0xc000;
        *((LPWORD)p) |= *((LPWORD)m_pCurBuf) & 0xc000;

        ::memcpy(p + 2, m_pCurBuf + 2, 3);
    }

    drawAll();
    _notify(NOTIFY_UPDATE_GRAD);
}

//! カレントの位置を左右の中央位置に

void CGradEditDlg_edit::setCurPtPosMiddle()
{
    int pos1,pos2;

    if(m_nCurPos == 0 || m_nCurPos == m_nPtCnt - 1)
        return;

    pos1 = GRAD_GETPOS(m_pCurBuf - m_nOneSize);
    pos2 = GRAD_GETPOS(m_pCurBuf + m_nOneSize);

    _setPtPos(m_pCurBuf, pos1 + ((pos2 - pos1) >> 1));

    drawAll();
    _notify(NOTIFY_UPDATE_GRAD);
}

//! すべて等間隔に

void CGradEditDlg_edit::setAllPtEven()
{
    int i,div;
    LPBYTE p;

    p   = getPtBufFromPos(1);
    div = m_nPtCnt - 1;

    for(i = 1; i < m_nPtCnt - 1; i++, p += m_nOneSize)
        _setPtPos(p, GPOS_MAX * i / div);

    drawAll();
    _notify(NOTIFY_UPDATE_GRAD);
}

//! ポイント反転

void CGradEditDlg_edit::pointReverse()
{
    LPBYTE p1,p2;
    int cnt;
    WORD wd;
    BYTE dat[5];

    //入れ替え

    p1  = m_memDat;
    p2  = getPtBufFromPos(m_nPtCnt - 1);
    cnt = m_nPtCnt >> 1;

    for(; cnt > 0; cnt--)
    {
        ::memcpy(dat, p1, m_nOneSize);
        ::memcpy(p1, p2, m_nOneSize);
        ::memcpy(p2, dat, m_nOneSize);

        p1 += m_nOneSize;
        p2 -= m_nOneSize;
    }

    //位置反転

    p1 = m_memDat;

    for(cnt = m_nPtCnt; cnt > 0; cnt--, p1 += m_nOneSize)
    {
        wd = *((LPWORD)p1);

        *((LPWORD)p1) = (GPOS_MAX - (wd & 0x3fff)) | (wd & 0xc000);
    }

    drawAll();
    _notify(NOTIFY_UPDATE_GRAD);
}

//! カレントと左との間を分割
/*!
    @param cnt 分割数
*/

void CGradEditDlg_edit::pointSplit(int cnt)
{
    int curpos,lpos,a1,a2,pos,div,i,j;
    BYTE col1[3],col2[3];
    LPBYTE pbuf;

    cnt--;

    //

    curpos = GRAD_GETPOS(m_pCurBuf);
    lpos   = GRAD_GETPOS(m_pCurBuf - m_nOneSize);

    if(m_bAlpha)
    {
        a1 = GRAD_GETA(m_pCurBuf - m_nOneSize);
        a2 = GRAD_GETA(m_pCurBuf);
    }
    else
    {
        ::memcpy(col1, m_pCurBuf - m_nOneSize + 2, 3);
        ::memcpy(col2, m_pCurBuf + 2, 3);
    }

    //追加分を空ける

    ::memmove(m_pCurBuf + cnt * m_nOneSize, m_pCurBuf, (m_nPtCnt - m_nCurPos) * m_nOneSize);

    //追加分セット

    pbuf = m_pCurBuf;
    div  = cnt + 1;

    for(i = 0; i < cnt; i++, pbuf += m_nOneSize)
    {
        pos = (i + 1) * (curpos - lpos) / div + lpos;

        if(m_bAlpha)
        {
            *((LPWORD)pbuf) = pos;
            *((LPWORD)(pbuf + 2)) = (a2 - a1) * (pos - lpos) / (curpos - lpos) + a1;
        }
        else
        {
            *((LPWORD)pbuf) = pos | (2<<14);

            for(j = 0; j < 3; j++)
                pbuf[2 + j] = (col2[j] - col1[j]) * (pos - lpos) / (curpos - lpos) + col1[j];
        }
    }

    //

    m_nPtCnt  += cnt;
    m_nCurPos += cnt;
    m_pCurBuf += cnt * m_nOneSize;

    drawAll();
    _notify(NOTIFY_UPDATE_GRAD);
}


//==============================
// 計算など
//==============================


//! 指定ポイントの位置セット

void CGradEditDlg_edit::_setPtPos(LPBYTE pbuf,int pos)
{
    *((LPWORD)pbuf) &= 0xc000;
    *((LPWORD)pbuf) |= pos;
}

//! ポイント位置の調整（前後のポイントの間になるように）

int CGradEditDlg_edit::_adjustPos(LPBYTE pbuf,int pos)
{
    int ppos,npos;

    ppos = GRAD_GETPOS(pbuf - m_nOneSize);
    npos = GRAD_GETPOS(pbuf + m_nOneSize);

    if(pos <= ppos)
        pos = ppos + 1;
    else if(pos >= npos)
        pos = npos - 1;

    return pos;
}

//! グラデ位置からウィンドウX位置取得

int CGradEditDlg_edit::_getWinPosFromGrad(int pos)
{
    if(pos == 0)
        return POINTHF;
    else if(pos == GPOS_MAX)
        return m_nW - 1 - POINTHF;
    else
        return ((pos * (m_nW - 1 - POINTW)) >> GPOS_BIT) + POINTHF + 1;
}

//! ウィンドウX位置からグラデ位置取得

int CGradEditDlg_edit::_getGradPosFromWin(int x)
{
    int pos;

    pos = x - (POINTHF + 1);
    pos = pos * GPOS_MAX / (m_nW - POINTHF * 2 - 3);

    return pos;
}

//! マウス位置からポイント位置取得
/*!
    @return [-1]なし [-2]範囲外
*/

int CGradEditDlg_edit::_getPtFromMouse(int mx,int my)
{
    int i,x,y;
    LPBYTE p;

    y = m_nH - UNDERH + 4;

    //カレントを優先判定

    x = _getWinPosFromGrad(GRAD_GETPOS(m_pCurBuf));

    if(mx >= x - POINTHF && mx <= x + POINTHF && my >= y && my < y + UNDERH - 4)
        return m_nCurPos;

    //各ポインタの四角内か

    p = m_memDat;

    for(i = 0; i < m_nPtCnt; i++, p += m_nOneSize)
    {
        x = _getWinPosFromGrad(GRAD_GETPOS(p));

        if(mx >= x - POINTHF && mx <= x + POINTHF && my >= y && my < y + UNDERH - 4)
            return i;
    }

    //範囲外

    if(mx <= POINTHF || mx >= m_nW - POINTHF - 1)
        return -2;

    return -1;
}


//==============================
// 描画
//==============================


//! 全体描画

void CGradEditDlg_edit::drawAll()
{
    //背景

    m_img.clear(axres->colRGB(AXAppRes::FACE));

    //グラデ枠

    m_img.box(POINTHF, 0, m_nW - POINTHF * 2, m_nH - UNDERH, 0);

    //グラデ

    _drawGrad();

    //ポイント

    _drawPoint(FALSE);

    redraw();
}

//! グラデ部分描画

void CGradEditDlg_edit::_drawGrad()
{
    if(m_bAlpha)
        _drawGradA();
    else
    {
        drawGradPrev(&m_img, POINTHF + 1, 1, m_nW - 2 - POINTHF * 2, m_nH - 2 - UNDERH,
            m_memDat, ((CGradEditDlg *)m_pParent)->getAlphaBuf(),
            ((CGradEditDlg *)m_pParent)->isSingleCol());
    }

    redraw();
}

//! グラデ部分描画（A）

void CGradEditDlg_edit::_drawGradA()
{
    LPBYTE pDst,p;
    LPWORD pA;
    int bpp,pitch,ix,iy,w,h,pos,a;

    w = m_nW - 2 - POINTHF * 2;
    h = m_nH - 2 - UNDERH;

    pA = m_memDat;

    pDst  = m_img.getBufPt(POINTHF + 1, 1);
    bpp   = m_img.getBytes();
    pitch = m_img.getPitch();

    for(ix = 0; ix < w; ix++, pDst += bpp)
    {
        pos = (ix << GPOS_BIT) / (w - 1);

        //A

        while(pos > pA[2]) pA += 2;

        a = (pA[3] - pA[1]) * (pos - pA[0]) / (pA[2] - pA[0]) + pA[1];
        a = a * 255 >> 10;

        //縦に描画

        p = pDst;

        for(iy = 0; iy < h; iy++, p += pitch)
            m_img.setPixelBuf(p, a, a, a);
    }
}

//! ポイント描画

void CGradEditDlg_edit::_drawPoint(BOOL bClear)
{
    int i;
    LPBYTE p;

    //クリア

    if(bClear)
        m_img.fillBox(0, m_nH - UNDERH, m_nW, UNDERH, axres->colRGB(AXAppRes::FACE));

    //カレント以外描画

    p = m_memDat;

    for(i = 0; i < m_nPtCnt; i++, p += m_nOneSize)
    {
        if(p != m_pCurBuf)
            _drawOnePoint(p);
    }

    //カレントは最後に描画

    _drawOnePoint(m_pCurBuf);
}

//! 1つのポイント描画

void CGradEditDlg_edit::_drawOnePoint(LPBYTE pbuf)
{
    int x,y;
    DWORD col;

    //描画位置

    x = _getWinPosFromGrad(GRAD_GETPOS(pbuf));
    y = m_nH - UNDERH;

    //枠

    col = (pbuf == m_pCurBuf)? 0xff0000: 0;

    m_img.lineV(x, y, 4, col);
    m_img.box(x - POINTHF, y + 4, POINTW, UNDERH - 4, col);

    //色

    if(m_bAlpha)
    {
        col = GRAD_GETA(pbuf) * 255 >> 10;
        col = (col << 16) | (col << 8) | col;
    }
    else
        col = (pbuf[2] << 16) | (pbuf[3] << 8) | pbuf[4];

    m_img.fillBox(x - POINTHF + 1, y + 5, POINTW - 2, UNDERH - 6, col);
}
