/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CFilterDlg - フィルタダイアログ
*/


#include "CFilterDlg.h"

#include "CFilterBar.h"
#include "CFilterWidget.h"

#include "filterdraw.h"
#include "drawdat.h"
#include "cursor.h"
#include "CConfig.h"
#include "CTileImage.h"
#include "CLayerItem.h"
#include "CFilterConfigList.h"

#include "AXBuf.h"
#include "AXLayout.h"
#include "AXSpacerItem.h"
#include "AXLabel.h"
#include "AXButton.h"
#include "AXLineEdit.h"
#include "AXCheckButton.h"
#include "AXComboBox.h"
#include "AXByteString.h"
#include "AXApp.h"
#include "AXUtilColor.h"
#include "AXTrSysID.h"

#include "drawOpDef.h"
#include "draw_main.h"
#include "draw_update.h"
#include "draw_opfunc.h"

#include "global.h"
#include "strid.h"


//------------------------

enum
{
    TOPF_MASK_PREVTYPE  = 7,
    TOPF_RESET_BUTTON   = 1<<3,
    TOPF_SAVE_DATA      = 1<<4,
    TOPF_GET_CANVASPOS  = 1<<5,

    PREVTYPE_NONE = 0,
    PREVTYPE_INDLG,
    PREVTYPE_CANVAS_COLOR,
    PREVTYPE_CANVAS_COMIC
};

enum
{
    WID_BTT_OK      = 1,
    WID_BTT_CANCEL  = 2,
    WID_BTT_RESET   = 3,
    WID_PREV        = 90,
    WID_CK_PREV,
    WID_CK_CLIP,
    WID_LEVEL,

    WID_BAR_TOP     = 100,
    WID_BAREDIT_TOP = 150,
    WID_COMBO_TOP   = 200,
    WID_CHECK_TOP   = 250
};

enum
{
    STRID_PREVIEW  = 1,
    STRID_CLIPPING = 2,
    STRID_RESET    = 3
};

//------------------------

//! 漫画用プレビュー  [draw::updateRect() で呼ばれる]

void _comicprev(const AXRectSize &rcs)
{
    CTileImage *pimg = ((CFilterDlg *)g_draw->work.pTmp)->getPrevImg();

    if(pimg)
    {
        pimg->blendToRGB16(g_draw->pimgBlend, rcs, 128,
            g_draw->funcBlendCol[g_draw->pcurlayer->m_nBlendMode]);
    }
}

//------------------------


//***********************************
// CFilterDlg
//***********************************


CFilterDlg::~CFilterDlg()
{
    if(m_pimgPrev) delete m_pimgPrev;
}

CFilterDlg::CFilterDlg(AXWindow *pOwner,WORD wTitleID,const BYTE *pDat,FILTERDRAW *pinfo)
    : AXDialog(pOwner, WS_DIALOG_NORMAL | WS_BK_FACE)
{
    AXBuf buf;
    int prevw,prevh;

    m_pinfo     = pinfo;
    m_uMenuID   = wTitleID;
    m_pConfItem = NULL;

    m_pimgPrev  = NULL;
    m_bPreview  = (g_conf->uEtcFlags & CConfig::ETCF_FILTERDLG_PREV)? 1: 0;

    m_nBarCnt   = 0;
    m_nEditCnt  = 0;
    m_nComboCnt = 0;
    m_nCheckCnt = 0;

    m_pwidPrev  = NULL;
    m_pwidLevel = NULL;

    //タイトル

    setTitle(_string(strid::GROUP_MAINMENU, wTitleID));

    //---------- データ

    buf.init((BYTE *)pDat, 1000, AXBuf::ENDIAN_BIG);

    //フラグ (BYTE)

    m_btTopFlags = buf.getBYTE();

    m_nPrevType = m_btTopFlags & TOPF_MASK_PREVTYPE;

    //プレビュー幅・高さ

    if(m_nPrevType == PREVTYPE_INDLG)
    {
        prevw = buf.getWORD();
        prevh = buf.getWORD();
    }

    //---------- 保存データ

    if(m_btTopFlags & TOPF_SAVE_DATA)
        m_pConfItem = FILTERCONF->findID(m_uMenuID);

    //---------- レイアウト&ウィジェット

    _create_widget(&buf, prevw, prevh);

    //-----------

    g_draw->work.nNowOpNo = draw::OPNO_FILTERDLG;
    g_draw->work.pTmp     = this;
    g_draw->work.funcUpdatePrev = NULL;

    //キャンバス位置・デフォルトで中央

    if(m_btTopFlags & TOPF_GET_CANVASPOS)
    {
        m_pinfo->tmpx = g_draw->nImgW >> 1;
        m_pinfo->tmpy = g_draw->nImgH >> 1;
    }

    //プレビュー用準備

    _initPrev();

    //

    calcDefSize();
    resize(m_nDefW, m_nDefH);

    show();
}

//! 閉じる

BOOL CFilterDlg::onClose()
{
    _end(FALSE);
    return TRUE;
}

//! 通知

BOOL CFilterDlg::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    UINT id = pwin->getItemID();
    int n;

    if(id >= WID_BAR_TOP && id < WID_BAR_TOP + MAXNUM_BAR)
    {
        //バー

        if(uNotify == CFilterBar::NOTIFY_CHANGE)
        {
            id -= WID_BAR_TOP;
            n = m_pbar[id]->getPos();
            m_pinfo->valbar[id] = n;
            m_peditBar[id]->setVal(n);
        }
        else
            _prev();
    }
    else if(id >= WID_BAREDIT_TOP && id < WID_BAREDIT_TOP + MAXNUM_BAR)
    {
        //バーエディット

        if(uNotify == AXLineEdit::EN_CHANGE)
        {
            id -= WID_BAREDIT_TOP;
            n = m_peditBar[id]->getVal();
            m_pinfo->valbar[id] = n;
            m_pbar[id]->setPos(n);

            addTimer(0, 500);
        }
    }
    else if(id >= WID_COMBO_TOP && id < WID_COMBO_TOP + MAXNUM_COMBO)
    {
        //コンボボックス

        if(uNotify == AXComboBox::CBN_SELCHANGE)
        {
            id -= WID_COMBO_TOP;
            m_pinfo->valcombo[id] = m_pcombo[id]->getCurSel();

            _prev();
        }
    }
    else if(id >= WID_CHECK_TOP && id < WID_CHECK_TOP + MAXNUM_CHECK)
    {
        //チェックボタン

        m_pinfo->valcheck[id - WID_CHECK_TOP] ^= 1;
        _prev();
    }
    else
    {
        //------ ほか

        switch(id)
        {
            //プレビュー位置変更
            case WID_PREV:
                m_pwidPrev->getDrawRect(&m_pinfo->rs);

                if(m_bPreview)
                    _prev();
                else
                    m_pwidPrev->drawImg(m_pinfo->pimgSrc);
                break;
            //プレビューON/OFF
            case WID_CK_PREV:
                m_bPreview ^= 1;

                g_conf->uEtcFlags ^= CConfig::ETCF_FILTERDLG_PREV;

                _prev();
                break;

            //クリッピング
            case WID_CK_CLIP:
                m_pinfo->bClipping ^= 1;
                _prev();
                break;
            //レベル補正
            case WID_LEVEL:
                m_pwidLevel->getVal(m_pinfo->valbar);
                _prev();
                break;

            //リセット
            case WID_BTT_RESET:
                _reset();
                break;
            //OK
            case WID_BTT_OK:
                _end(TRUE);
                break;
            //キャンセル
            case WID_BTT_CANCEL:
                _end(FALSE);
                break;
        }
    }

    return TRUE;
}

//! タイマー

BOOL CFilterDlg::onTimer(UINT uTimerID,ULONG lParam)
{
    _prev();

    delTimer(uTimerID);

    return TRUE;
}

//! コマンド

BOOL CFilterDlg::onCommand(UINT uID,ULONG lParam,int from)
{
    if(uID == 10000 && (m_btTopFlags & TOPF_GET_CANVASPOS))
    {
        //キャンバスが左クリックされた時

        AXPoint pt;

        g_draw->view.winToimg(&pt, lParam >> 16, lParam & 0xffff);

        m_pinfo->tmpx = pt.x;
        m_pinfo->tmpy = pt.y;

        _prev();
    }

    return TRUE;
}


//==============================
// サブ
//==============================


//! 終了

void CFilterDlg::_end(BOOL bOK)
{
    delTimerAll();

    switch(m_nPrevType)
    {
        //キャンバスプレビュー(カラー操作)
        case PREVTYPE_CANVAS_COLOR:
            if(m_pimgPrev)
            {
                g_draw->pcurlayer->m_pimg = m_pimgCurSave;

                if(!bOK) draw::updateRect(m_rcfPrev);
            }
            break;
        //キャンバスプレビュー(漫画用)
        case PREVTYPE_CANVAS_COMIC:
            if(m_pimgPrev)
            {
                g_draw->work.funcUpdatePrev = NULL;

                draw::updateRect(m_rcfPrev);
            }
            break;
    }

    if(m_btTopFlags & TOPF_SAVE_DATA)
        _saveConfig();

    g_draw->work.nNowOpNo = draw::OPNO_NONE;

    //

    endDialog(bOK);
}

//! 設定保存

void CFilterDlg::_saveConfig()
{
    AXByteString str;
    int i;

    //バー

    for(i = 0; i < m_nBarCnt; i++)
    {
        if(m_cSigBar[i])
        {
            str += m_cSigBar[i];
            str += m_pinfo->valbar[i];
        }
    }

    //チェック

    for(i = 0; i < m_nCheckCnt; i++)
    {
        if(m_cSigCheck[i])
        {
            str += m_cSigCheck[i];
            str += m_pinfo->valcheck[i];
        }
    }

    //コンボボックス

    for(i = 0; i < m_nComboCnt; i++)
    {
        if(m_cSigCombo[i])
        {
            str += m_cSigCombo[i];
            str += m_pinfo->valcombo[i];
        }
    }

    FILTERCONF->setDat(m_uMenuID, str);
}

//! リセット

void CFilterDlg::_reset()
{
    int i;

    //バー

    for(i = 0; i < m_nBarCnt; i++)
    {
        m_pbar[i]->setPos(m_sDefBar[i]);
        m_peditBar[i]->setVal(m_sDefBar[i]);

        m_pinfo->valbar[i] = m_sDefBar[i];
    }

    _prev();
}

//! プレビュー

void CFilterDlg::_prev()
{
    FLAGRECT rcf;

    switch(m_nPrevType)
    {
        //ダイアログ上
        case PREVTYPE_INDLG:
            if(m_bPreview && m_pwidPrev)
            {
                AXWindow::setCursor(cursor::getImgDat(cursor::WAIT));
                axapp->flush();

                m_pimgPrev->freeAllTile();

                //m_pimgPrev に描画

                (*m_pinfo->funcDraw)(*m_pinfo);

                //プレビューイメージに描画

                m_pwidPrev->drawImg(m_pimgPrev);

                AXWindow::unsetCursor();
            }
            break;
        //キャンバスプレビュー（カラー操作用）
        case PREVTYPE_CANVAS_COLOR:
            if(m_bPreview && m_pimgPrev)
            {
                AXWindow::setCursor(cursor::getImgDat(cursor::WAIT));
                axapp->flush();

                (*m_pinfo->funcDraw)(*m_pinfo);

                draw::updateRect(m_rcfPrev);

                AXWindow::unsetCursor();
            }
            break;
        //キャンバスプレビュー(漫画用)
        case PREVTYPE_CANVAS_COMIC:
            if(m_pimgPrev)
            {
                rcf = m_rcfPrev;

                AXWindow::setCursor(cursor::getImgDat(cursor::WAIT));
                axapp->flush();

                //プレビューイメージ・範囲クリア

                if(m_rcfPrev.flag)
                {
                    m_pimgPrev->freeAllTile();
                    m_rcfPrev.clear();
                }

                //描画

                g_draw->drawinfo.rcfDraw.clear();

                (*m_pinfo->funcDraw)(*m_pinfo);

                //更新（前回の範囲と合わせた範囲）

                rcf.combine(g_draw->drawinfo.rcfDraw);

                draw::updateRect(rcf);
                axapp->update();

                //範囲を保存

                m_rcfPrev = g_draw->drawinfo.rcfDraw;

                AXWindow::unsetCursor();
            }
            break;
    }
}


//==============================
// 初期化
//==============================


//! プレビュー初期化

void CFilterDlg::_initPrev()
{
    CTileImage *pimgCur = draw::getCurImg();

    switch(m_nPrevType)
    {
        //ダイアログ内プレビュー
        case PREVTYPE_INDLG:
            draw::setBeforeDraw_filter(TRUE);

            m_pimgPrev = draw::allocTileImage(pimgCur->getColType());
            m_pimgPrev->create(g_draw->nImgW, g_draw->nImgH);

            m_pwidPrev->getDrawRect(&m_pinfo->rs);

            m_pinfo->pimgSrc = pimgCur;
            m_pinfo->pimgDst = m_pimgPrev;

            if(m_bPreview)
                _prev();
            else
                m_pwidPrev->drawImg(m_pinfo->pimgSrc);
            break;
        //キャンバスプレビュー（カラー操作用）
        case PREVTYPE_CANVAS_COLOR:
            /*
                カレントレイヤのイメージを m_pimgPrev に置き換える。
                カレントレイヤイメージをソースにして、m_pimgPrev にプレビュー描画。
                (コピーできなかった場合 or イメージが空の場合はプレビューなし : m_pimgPrev が NULL)
            */

            if(draw::isSelExist())
                m_rcfPrev = g_draw->work.rcfSel;
            else
                pimgCur->getExistImgRectPx(&m_rcfPrev);

            if(m_rcfPrev.flag)
            {
                m_pimgPrev = draw::allocTileImage(pimgCur->getColType());

                if(!m_pimgPrev->copy(*pimgCur))
                {
                    delete m_pimgPrev;
                    m_pimgPrev = NULL;
                }
                else
                {
                    //プレビューあり

                    m_pimgCurSave = pimgCur;

                    g_draw->pcurlayer->m_pimg = m_pimgPrev;

                    m_pinfo->rs.set(m_rcfPrev);
                    m_pinfo->pimgSrc = pimgCur;
                    m_pinfo->pimgDst = m_pimgPrev;
                    m_pinfo->pimgSel = (draw::isSelExist())? g_draw->pimgSel: NULL;

                    _prev();
                }
            }
            break;
        //キャンバスプレビュー(漫画用)
        case PREVTYPE_CANVAS_COMIC:
            /*
                m_pimgPrev にプレビュー描画。
                draw::updateRect() でプレビューをキャンバスに描画。
                m_rcfPrev : 前回の描画範囲
            */

            m_pimgPrev = draw::allocTileImage(pimgCur->getColType());

            if(!m_pimgPrev->create(g_draw->nImgW, g_draw->nImgH))
            {
                delete m_pimgPrev;
                m_pimgPrev = NULL;
            }
            else
            {
                draw::setBeforeDraw_filter(TRUE);
                draw::setBeforeDraw_filterDrawFlash();

                g_draw->drawinfo.funcDrawPixel = &CTileImage::setPixel_subdraw;
                g_draw->work.funcUpdatePrev    = _comicprev;

                m_rcfPrev.clear();

                m_pinfo->pimgSrc = pimgCur;
                m_pinfo->pimgDst = m_pimgPrev;

                _prev();
            }
            break;
    }
}

//! レイアウト・ウィジェット作成

void CFilterDlg::_create_widget(AXBuf *pbuf,int prevw,int prevh)
{
    AXLayout *plTop,*pl2,*plm,*plh;
    AXWindow *pwin;

    //トップレイアウト

    setLayout(plTop = new AXLayoutVert);
    plTop->setSpacing(10);

    //----------- プレビュー+ウィジェット

    plTop->addItem(pl2 = new AXLayoutHorz(LF_EXPAND_W, 10));

    //プレビュー

    if(m_nPrevType == PREVTYPE_INDLG)
        pl2->addItem(m_pwidPrev = new CFilterPrev(this, 0, WID_PREV, 0, prevw, prevh, draw::getCurImg()));

    //ウィジェット

    pl2->addItem(plm = new AXLayoutMatrix(2, LF_EXPAND_W));

    _create_subwidget(pbuf, plm);

    //---------- ボタン

    plTop->addItem(plh = new AXLayoutHorz(LF_EXPAND_W, 3));
    plh->setPaddingTop(15);

    //プレビューチェック

    if(m_nPrevType == PREVTYPE_INDLG || m_nPrevType == PREVTYPE_CANVAS_COLOR)
        plh->addItem(new AXCheckButton(this, 0, 0, WID_CK_PREV, 0, STRID_PREVIEW, m_bPreview));

    //OK

    plh->addItem(pwin = new AXButton(this, 0, LF_EXPAND_X|LF_RIGHT, WID_BTT_OK, 0, TRSYS_GROUPID, TRSYS_OK));

    setDefaultButton(pwin);

    //キャンセル

    plh->addItem(new AXButton(this, 0, 0, WID_BTT_CANCEL, 0, TRSYS_GROUPID, TRSYS_CANCEL));

    //リセット

    if(m_btTopFlags & TOPF_RESET_BUTTON)
        plh->addItem(new AXButton(this, 0, 0, WID_BTT_RESET, 0, STRID_RESET));
}

//! 各定義ウィジェット作成

void CFilterDlg::_create_subwidget(AXBuf *pbuf,AXLayout *plm)
{
    int wtype,size,ret,id;

    _trgroup(strid::GROUP_DLG_FILTER);

    while(1)
    {
        //ウィジェットタイプ

        wtype = pbuf->getBYTE();
        if(wtype == 255) break;

        //

        if(wtype >= 0x80)
        {
            //------ 独自定義ウィジェット

            switch(wtype)
            {
                //キャンバス範囲外クリッピング
                case 0x80:
                    _create_def_clipping(plm);
                    break;
                //レベル補正
                case 0x81:
                    _create_def_level(plm);
                    break;
            }
        }
        else
        {
            //------ 各タイプのウィジェット

            //BYTE:ラベル名文字列ID (0でなし)

            id = pbuf->getBYTE();

            //ラベル

            if(id)
                plm->addItem(new AXLabel(this, 0, LF_CENTER_Y, MAKE_DW4(0,0,6,6), id));
            else
                plm->addItem(new AXSpacerItem);

            //データサイズ

            size = pbuf->getBYTE();

            //

            switch(wtype)
            {
                //バー
                case 0:
                    ret = _create_bar(pbuf, plm, size);
                    break;
                //チェックボタン
                case 1:
                    ret = _create_check(pbuf, plm, size);
                    break;
                //コンボボックス
                case 2:
                    ret = _create_combo(pbuf, plm, size);
                    break;
                //ラベル
                case 3:
                    plm->addItem(new AXLabel(this, AXLabel::LS_BORDER, LF_CENTER_Y, MAKE_DW4(0,0,0,6), pbuf->getWORD()));
                    ret = TRUE;
                    break;
                default:
                    ret = FALSE;
                    break;
            }

            if(!ret) break;
        }
    }
}


//! キャンバス範囲外クリッピング

void CFilterDlg::_create_def_clipping(AXLayout *plm)
{
    plm->addItem(new AXSpacerItem);
    plm->addItem(new AXCheckButton(this, 0, 0, WID_CK_CLIP, MAKE_DW4(0,0,0,6), STRID_CLIPPING, TRUE));

    m_pinfo->bClipping = TRUE;
}

//! レベル補正

void CFilterDlg::_create_def_level(AXLayout *plm)
{
    AXMem mem;

    plm->addItem(m_pwidLevel = new CFilterWidgetLevel(this, WID_LEVEL, 0));

    //

    if(mem.allocClear(257 * sizeof(DWORD)))
    {
        draw::getCurImg()->getHistogram(mem);

        m_pwidLevel->drawAll(mem);
    }

    m_pwidLevel->getVal(m_pinfo->valbar);
}

//! バー作成（バー + エディット）

BOOL CFilterDlg::_create_bar(AXBuf *pbuf,AXLayout *plm,int size)
{
    AXLayout *pl;
    short barw,min,max,def,center;
    BYTE dig,defType;
    char sig;
    int n,hsv[3];

    if(m_nBarCnt >= MAXNUM_BAR) return FALSE;

    //値

    pbuf->getWORD(&barw);
    pbuf->getWORD(&min);
    pbuf->getWORD(&max);
    if(size > 6) pbuf->getWORD(&def); else def = min;
    if(size > 8) pbuf->getWORD(&center); else center = min;
    if(size > 10) pbuf->getBYTE(&dig); else dig = 0;
    if(size > 11) pbuf->getBYTE(&sig); else sig = 0;
    if(size > 12) pbuf->getBYTE(&defType); else defType = 0;

    //初期値

    if(defType == 1)
        def = g_draw->nImgDPI;
    else if(defType >= 2 && defType <= 4)
    {
        //描画色のH,S,V

        AXRGBtoHSV(g_conf->dwDrawCol, hsv);
        def = hsv[defType - 2];
    }

    //

    m_cSigBar[m_nBarCnt] = sig;
    m_sDefBar[m_nBarCnt] = def;

    if(m_pConfItem && sig)
    {
        if(m_pConfItem->getVal(sig, &n))
            def = n;
    }

    //

    plm->addItem(pl = new AXLayoutHorz(LF_EXPAND_W, 6));
    pl->setPaddingBottom(6);

    //バー

    pl->addItem(m_pbar[m_nBarCnt] = new CFilterBar(this, LF_EXPAND_W|LF_CENTER_Y, WID_BAR_TOP + m_nBarCnt, 0,
                                            min, max, def, center));

    m_pbar[m_nBarCnt]->setMinWidth(barw);

    //エディット

    pl->addItem(m_peditBar[m_nBarCnt] = new AXLineEdit(this, AXLineEdit::ES_SPIN, LF_CENTER_Y,
                                                WID_BAREDIT_TOP + m_nBarCnt, 0));


    m_peditBar[m_nBarCnt]->setInit(6, min, max, dig, def);

    //

    m_pinfo->valbar[m_nBarCnt] = def;

    m_nBarCnt++;

    return TRUE;
}

//! チェックボタン作成

BOOL CFilterDlg::_create_check(AXBuf *pbuf,AXLayout *plm,int size)
{
    WORD wStrID;
    BYTE bCheck;
    char sig;
    int n;

    if(m_nCheckCnt >= MAXNUM_CHECK) return FALSE;

    //値

    pbuf->getWORD(&wStrID);
    if(size > 2) pbuf->getBYTE(&bCheck); else bCheck = FALSE;
    if(size > 3) pbuf->getBYTE(&sig); else sig = 0;

    m_cSigCheck[m_nCheckCnt] = sig;

    if(m_pConfItem && sig)
    {
        if(m_pConfItem->getVal(sig, &n))
            bCheck = n;
    }

    //作成

    plm->addItem(new AXCheckButton(this, 0, 0, WID_CHECK_TOP + m_nCheckCnt, MAKE_DW4(0,0,0,6), wStrID, bCheck));

    //

    m_pinfo->valcheck[m_nCheckCnt] = (bCheck)? 1: 0;

    m_nCheckCnt++;

    return TRUE;
}

//! コンボボックス作成

BOOL CFilterDlg::_create_combo(AXBuf *pbuf,AXLayout *plm,int size)
{
    BYTE cnt,def;
    WORD strtop;
    char sig;
    int n;

    if(m_nComboCnt >= MAXNUM_COMBO) return FALSE;

    //値

    pbuf->getBYTE(&cnt);
    pbuf->getWORD(&strtop);
    if(size > 3) pbuf->getBYTE(&def); else def = 0;
    if(size > 4) pbuf->getBYTE(&sig); else sig = 0;

    m_cSigCombo[m_nComboCnt] = sig;

    if(m_pConfItem && sig)
    {
        if(m_pConfItem->getVal(sig, &n))
            def = n;
    }

    //コンボボックス

    plm->addItem(m_pcombo[m_nComboCnt] = new AXComboBox(this, 0, LF_CENTER_Y,
                                                        WID_COMBO_TOP + m_nComboCnt, MAKE_DW4(0,0,0,6)));

    //項目

    m_pcombo[m_nComboCnt]->addItemMulTr(strtop, cnt);
    m_pcombo[m_nComboCnt]->setAutoWidth();
    m_pcombo[m_nComboCnt]->setCurSel(def);

    //

    m_pinfo->valcombo[m_nComboCnt] = def;

    m_nComboCnt++;

    return TRUE;
}

