/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CCustomGradSel - カスタムグラデーション選択
*/


#include "CCustomGradSel.h"

#include "CGradList.h"

#include "AXString.h"
#include "AXPopupWindow.h"
#include "AXScrollBar.h"
#include "AXClipboard.h"
#include "AXApp.h"
#include "AXAppRes.h"
#include "AXKey.h"


//----------------------------

class CCustomGradPopup:public AXPopupWindow
{
protected:
    enum
    {
        ITEM_H    = 19,
        ITEM_HCNT = 6
    };

    AXImage         m_img;
    AXScrollBar     *m_pScr;
    CGradListItem   *m_pSel;

    void _draw(BOOL bGrad=TRUE);
    CGradListItem *_getItem(int y);
    BOOL _isGradArea(AXHD_MOUSE *phd);

public:
    CCustomGradPopup(int w);

    CGradListItem *getSelItem() { return m_pSel; }

    virtual void endPopup(BOOL bCancel);

    virtual BOOL onPaint(AXHD_PAINT *phd);
    virtual BOOL onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam);
    virtual BOOL onButtonDown(AXHD_MOUSE *phd);
    virtual BOOL onMouseMove(AXHD_MOUSE *phd);
    virtual BOOL onMouseWheel(AXHD_MOUSE *phd,BOOL bUp);
    virtual BOOL onKeyDown(AXHD_KEY *phd);
};

//----------------------------


//***************************************
// CCustomGradSel
//***************************************


CCustomGradSel::CCustomGradSel(AXWindow *pParent,UINT uItemID,int selno)
    : AXWindow(pParent, 0, LF_EXPAND_W|LF_CENTER_Y|LF_FIX_H, uItemID, 0)
{
    m_nMinW = 70;
    m_nH    = 19;

    m_pSel = (CGradListItem *)GRADLIST->getItem(selno);

    draw();
}

//! 新規追加

void CCustomGradSel::addNewGrad(const AXString *pstr)
{
    CGradListItem *p;

    p = GRADLIST->addNewGrad(pstr);
    if(p)
    {
        m_pSel = p;
        draw();

        getNotify()->onNotify(this, 0, GRADLIST->getPos(p));
    }
}

//! 複製

void CCustomGradSel::copyGrad()
{
    CGradListItem *p;

    if(m_pSel)
    {
        p = GRADLIST->copyGrad(m_pSel);
        if(p)
        {
            m_pSel = p;
            draw();

            getNotify()->onNotify(this, 0, GRADLIST->getPos(p));
        }
    }
}

//! 現在の選択削除

void CCustomGradSel::delGrad()
{
    CGradListItem *p;

    if(m_pSel)
    {
        //削除後の選択位置

        p = m_pSel->next();
        if(!p) p = m_pSel->prev();

        //削除

        GRADLIST->deleteItem(m_pSel);

        GRADLIST->updateON();

        //

        m_pSel = p;

        draw();
        getNotify()->onNotify(this, 0, (p)? GRADLIST->getPos(p): 0);
    }
}

//! クリップボードにコピー

void CCustomGradSel::copyClipb()
{
    AXString str;

    if(!m_pSel) return;

    if(m_pSel->getTextFormat(&str))
        axclipb->setText(m_id, str);
}

//! クリップボードから新規貼り付け

void CCustomGradSel::pasteClipb()
{
    AXString str;

    if(axclipb->getText(m_id, &str))
        addNewGrad(&str);
}


//=======================


//! 描画

void CCustomGradSel::draw()
{
    int i,x,y,btsize = m_nH - 2;

    //枠

    m_img.box(0, 0, m_nW, m_nH, 0);

    //ボタン

    x = m_nW - 1 - btsize;

    m_img.fillBox(x, 1, btsize, btsize, 0xffffff);
    m_img.box(x + 1, 2, btsize - 2, btsize - 2, 0);

    x = x + (btsize - 5) / 2 + 2;
    y = 3 + (btsize - 3) / 2;

    for(i = 0; i < 3; i++)
        m_img.lineH(x - i, y - i, 1 + i * 2, 0);

    //グラデーション

    x = m_nW - 2 - btsize;
    y = m_nH - 2;

    if(!m_pSel)
        m_img.fillBox(1, 1, x, y, 0xffffff);
    else
    {
        m_img.box(1, 1, x, y, 0xffffff);
        m_pSel->drawPrev(&m_img, 2, 2, x - 2, y - 2);
    }

    //

    redraw();
}

//! 描画

BOOL CCustomGradSel::onPaint(AXHD_PAINT *phd)
{
    m_img.put(m_id);

    return TRUE;
}

//! サイズ変更時

BOOL CCustomGradSel::onSize()
{
    m_img.recreate(m_nW, m_nH, 32, 0);
    draw();

    return TRUE;
}

//! ボタン押し時

BOOL CCustomGradSel::onButtonDown(AXHD_MOUSE *phd)
{
    CCustomGradPopup *pwin;
    AXRectSize rcs;

    if(phd->button == BUTTON_LEFT && GRADLIST->getCnt())
    {
        //ポップアップ選択

        getWindowRect(&rcs);

        pwin = new CCustomGradPopup(rcs.w);

        pwin->runPopup(rcs.x, rcs.y + rcs.h);

        if(pwin->getSelItem())
        {
            m_pSel = pwin->getSelItem();
            draw();

            getNotify()->onNotify(this, 0, GRADLIST->getPos(m_pSel));
        }

        delete pwin;
    }

    return TRUE;
}




//***************************************
// CCustomGradPopup - リストポップアップ
//***************************************


CCustomGradPopup::CCustomGradPopup(int w)
    : AXPopupWindow(NULL, 0)
{
    int cnt,scrw;

    m_pSel = NULL;

    cnt = GRADLIST->getCnt();

    resize(w, ITEM_H * (cnt < ITEM_HCNT ? cnt: ITEM_HCNT) + 2);

    //スクロール

    m_pScr = new AXScrollBar(this, AXScrollBar::SBS_VERT, 0);
    m_pScr->calcDefSize();

    scrw = m_pScr->getDefW();

    m_pScr->setStatus(0, cnt, ITEM_HCNT);
    m_pScr->moveresize(m_nW - 1 - scrw, 1, scrw, m_nH - 2);

    //

    m_img.create(m_nW - 2 - scrw, m_nH - 2);

    _draw();
}

//! グラデーション部分の範囲か

BOOL CCustomGradPopup::_isGradArea(AXHD_MOUSE *phd)
{
    return (phd->x >= 1 && phd->x < m_nW - 1 &&
            phd->y >= 1 && phd->y < m_nH - 1);
}

//! Y位置からアイテム取得

CGradListItem *CCustomGradPopup::_getItem(int y)
{
    if(y < 1 || y >= m_nH - 1)
        return NULL;
    else
    {
        y = m_pScr->getPos() + (y - 1) / ITEM_H;

        return (CGradListItem *)GRADLIST->getItem(y);
    }
}

//! ポップアップ終了

void CCustomGradPopup::endPopup(BOOL bCancel)
{
    if(bCancel)
        m_pSel = NULL;

    AXPopupWindow::endPopup(bCancel);
}

//! 描画

BOOL CCustomGradPopup::onPaint(AXHD_PAINT *phd)
{
    drawBox(0, 0, m_nW, m_nH, AXAppRes::BLACK);

    m_img.put(m_id, 1, 1);

    return TRUE;
}

//! 通知

BOOL CCustomGradPopup::onNotify(AXWindow *pwin,UINT uNotify,ULONG uParam)
{
    if(pwin == m_pScr && uNotify == AXScrollBar::SBN_SCROLL)
        _draw();

    return TRUE;
}

//! ボタン押し

BOOL CCustomGradPopup::onButtonDown(AXHD_MOUSE *phd)
{
    CGradListItem *p;

    if(_isGradArea(phd))
    {
        p = _getItem(phd->y);

        if(p)
        {
            m_pSel = p;
            endPopup(FALSE);
        }
    }

    return AXPopupWindow::onButtonDown(phd);
}

//! マウス移動

BOOL CCustomGradPopup::onMouseMove(AXHD_MOUSE *phd)
{
    CGradListItem *p;

    if(_isGradArea(phd))
    {
        p = _getItem(phd->y);

        if(p != m_pSel)
        {
            m_pSel = p;
            _draw(FALSE);
        }
    }

    return AXPopupWindow::onMouseMove(phd);
}

//! マウスホイール（垂直スクロール）

BOOL CCustomGradPopup::onMouseWheel(AXHD_MOUSE *phd,BOOL bUp)
{
    if(m_pScr->movePos(bUp? -1: 1))
        _draw();

    return TRUE;
}

//! キー押し

BOOL CCustomGradPopup::onKeyDown(AXHD_KEY *phd)
{
    //スペース、ENTER 時は決定して終了

    if(ISKEY_SPACE(phd->keysym) || ISKEY_ENTER(phd->keysym))
    {
        endPopup(FALSE);
        return TRUE;
    }

    return AXPopupWindow::onKeyDown(phd);
}

//! 描画

void CCustomGradPopup::_draw(BOOL bGrad)
{
    int i,y,w;
    DWORD col;
    CGradListItem *pitem;

    pitem = (CGradListItem *)GRADLIST->getItem(m_pScr->getPos());

    y = 0;
    w = m_img.getWidth();

    for(i = 0; i < ITEM_HCNT && pitem; i++)
    {
        col = (pitem == m_pSel)? 0xff0000: 0xffffff;

        m_img.box(0, y, w, ITEM_H, col);
        m_img.box(1, y + 1, w - 2, ITEM_H - 2, col);

        if(bGrad)
            pitem->drawPrev(&m_img, 2, y + 2, w - 4, ITEM_H - 4);

        pitem = pitem->next();
        y += ITEM_H;
    }

    redraw();
}
